<?php
/**
 * Handle all svg icons in one place.
 *
 * @package WPConsent
 */

/**
 * Get an SVG icon by name with width, height and viewbox options.
 *
 * @param string $name The name of the icon.
 * @param int    $width The width.
 * @param int    $height The height.
 * @param string $viewbox The viewbox, will be auto-built from width and height if not set.
 * @param string $color The color.
 *
 * @return string
 */
function wpconsent_get_icon( $name, $width = 20, $height = 20, $viewbox = '', $color = '' ) {
	$icons = wpconsent_icons();

	if ( ! isset( $icons[ $name ] ) ) {
		return '';
	}

	if ( empty( $viewbox ) ) {
		$viewbox = sprintf( '0 0 %1$s %2$s', $width, $height );
	}

	$icon = $icons[ $name ];
	if ( ! empty( $color ) ) {
		// Do a smart regex to replace the color in the fill attribute.
		$icon = preg_replace( '/fill="[^"]*"/', 'fill="' . $color . '"', $icon );
	}

	return sprintf(
		'<svg class="wpconsent-icon wpconsent-icon-%5$s" width="%1$s" height="%2$s" viewBox="%3$s" fill="none" xmlns="http://www.w3.org/2000/svg">%4$s</svg>',
		$width,
		$height,
		$viewbox,
		$icon,
		$name
	);
}

/**
 * Echo an icon in a safe mode.
 *
 * @param string $name The name of the icon.
 * @param int    $width The width.
 * @param int    $height The height.
 * @param string $viewbox The viewbox, will be auto-built from width and height if not set.
 * @param string $color The fill color (optional).
 *
 * @return void
 */
function wpconsent_icon( $name, $width = 20, $height = 20, $viewbox = '', $color = '' ) {
	$icon = wpconsent_get_icon( $name, $width, $height, $viewbox, $color );

	if ( ! empty( $icon ) ) {
		echo wp_kses(
			$icon,
			wpconsent_get_icon_allowed_tags()
		);
	}
}

/**
 * Get the allowed tags for svg icons, to be used in wp_kses.
 *
 * @return array
 * @see wpconsent_icon()
 * @see wp_kses()
 */
function wpconsent_get_icon_allowed_tags() {
	return array(
		'svg'      => array(
			'class'           => true,
			'aria-hidden'     => true,
			'aria-labelledby' => true,
			'role'            => true,
			'xmlns'           => true,
			'width'           => true,
			'height'          => true,
			'viewbox'         => true,
		),
		'g'        => array(
			'fill'      => true,
			'clip-path' => true,
			'transform' => true,
			'stroke'    => true,
		),
		'title'    => array( 'title' => true ),
		'path'     => array(
			'd'         => true,
			'fill'      => true,
			'fill-rule' => true,
			'clip-rule' => true,
		),
		'circle'   => array(
			'cx'           => true,
			'cy'           => true,
			'r'            => true,
			'stroke'       => true,
			'stroke-width' => true,
			'fill'         => true,
		),
		'rect'     => array(
			'x'      => true,
			'y'      => true,
			'width'  => true,
			'height' => true,
			'fill'   => true,
		),
		'clipPath' => array(
			'id' => true,
		),
		'defs'     => array(),
	);
}

/**
 * Get the whole array of WPConsent SVG icons.
 *
 * @return array
 */
function wpconsent_icons() {
	return (array) apply_filters(
		'wpconsent_icons',
		array(
			'logo'          => '<g clip-path="url(#clip0_19_2)"><path d="M7.72238 43.8545L32.7493 43.8276C34.3127 43.8276 35.7547 42.9785 36.5229 41.6173L56.2399 6.6308C56.2399 6.6308 56.2264 6.6308 56.213 6.6308C56.6173 5.97042 56.8599 5.20223 56.8599 4.36665C56.8599 1.99468 54.9596 0.0674582 52.6011 0.0135498H27.5472C25.9838 0.0135498 24.5418 0.862606 23.7736 2.22379L3.9488 37.3855C2.31807 40.2831 4.40702 43.8545 7.73586 43.8545H7.72238Z" fill="white"/><path d="M144.42 50C144.42 49.1913 144.191 48.4366 143.801 47.7897L120.337 6.17249L102.83 37.1428L108.935 47.9784C109.272 48.5984 109.461 49.3126 109.461 50.0808C109.461 50.7817 109.272 51.4285 108.976 52.0215L85.8625 93.0593C85.3504 93.7736 85.0404 94.6496 85.0404 95.6064C85.0404 98.0188 86.9811 99.9595 89.38 100H114.367C115.93 100 117.372 99.1509 118.14 97.7897L144.003 51.9137C144.003 51.9137 143.989 51.9002 143.976 51.8868C144.245 51.3207 144.42 50.6873 144.42 50Z" fill="white"/><path d="M99.7977 30.0135L116.873 0H84.6764C84.6764 0 84.6764 0 84.6764 0.0134771C84.6225 0.0134771 84.5686 0 84.5281 0C82.8031 0 81.3206 1.01078 80.6198 2.46631L46.6305 62.5606C45.8893 63.9084 44.4203 64.8248 42.7761 64.8248C41.1319 64.8248 39.5955 63.8679 38.8677 62.4528L33.0995 52.2237C32.3314 50.8491 30.8758 50.0135 29.3125 50.0135L4.33944 50.0943C1.02408 50.0943 -1.0514 53.6792 0.579332 56.5633L23.8273 97.8032C24.5955 99.1644 26.0376 100.013 27.6009 100.013H57.3314C57.3718 100.013 57.3987 100.013 57.4392 100.013C59.0969 100.013 60.5254 99.0836 61.2667 97.7089C61.2667 97.7089 61.2667 97.7089 61.2801 97.7224L82.6009 60.283L95.5254 37.5741L99.326 30.8895L99.8111 30.027L99.7977 30.0135Z" fill="white"/></g>',
			'logo-text'     => '<g><path d="M36.9757 2.83892C36.9757 1.2968 35.7402 0.0438371 34.2069 0.00878906H17.9183C16.9019 0.00878906 15.9644 0.560795 15.4649 1.44576L2.56726 24.3058C1.50705 26.1897 2.86516 28.5116 5.02938 28.5116H21.3004C22.3168 28.4941 23.2544 27.9508 23.7538 27.0659L36.5726 4.3197C36.5726 4.3197 36.5638 4.3197 36.5551 4.3197C36.8179 3.89036 36.9757 3.39092 36.9757 2.84768V2.83892Z" fill="#0065FF"/><path d="M93.8936 32.507C93.8936 31.9813 93.7359 31.4906 93.4906 31.07L78.2359 4.01294L66.8541 24.148L70.8233 31.1927C71.0423 31.5957 71.165 32.0601 71.165 32.5596C71.165 33.0152 71.0423 33.4358 70.8496 33.8213L55.8227 60.4928C55.4898 60.9572 55.2882 61.5267 55.2882 62.1488C55.2882 63.7172 56.55 64.979 58.1096 65.0053H74.3544C75.3708 65.0053 76.3083 64.4533 76.8077 63.5683L93.622 33.7424C93.622 33.7424 93.6133 33.7337 93.6045 33.7249C93.7797 33.3569 93.8936 32.9451 93.8936 32.4982V32.507Z" fill="#0065FF"/><path d="M78.2447 4.00424L75.9841 0L64.8826 19.513L64.5672 20.0738L62.0963 24.4197C62.587 23.5698 63.507 22.9915 64.5585 22.9915C65.5047 22.9915 66.3371 23.4559 66.8541 24.1569V24.1393L78.2359 4.00424H78.2447Z" fill="#004CBA"/><path d="M55.0517 0C55.0517 0 55.0517 0 55.0517 0.00876201C55.0166 0.00876201 54.9816 0 54.9553 0C53.8338 0 52.8699 0.657151 52.4143 1.60345L30.3165 40.6732C29.8346 41.5494 28.8796 42.1453 27.8106 42.1453C26.7416 42.1453 25.7428 41.5232 25.2696 40.6031L21.5195 33.9528C21.02 33.0591 20.0738 32.5158 19.0574 32.5158L2.82136 32.5684C0.665908 32.5684 -0.683441 34.8991 0.376762 36.7741L15.4912 63.5859C15.9907 64.4708 16.9282 65.0229 17.9446 65.0229H37.2736C37.2999 65.0229 37.3174 65.0229 37.3437 65.0229C38.4214 65.0229 39.3502 64.4183 39.8321 63.5245C39.8321 63.5245 39.8321 63.5246 39.8408 63.5333L53.7023 39.1925L62.1051 24.4285L64.576 20.0825L64.8914 19.5218L75.9841 0H55.0517Z" fill="#0065FF"/></g><g><path d="M160.844 13.9054C162.798 13.9054 164.27 15.3774 164.27 17.121C164.27 17.857 164.007 18.5405 163.85 19.1275L154.79 48.5854C154.264 50.3728 152.836 51.2227 151.048 51.2227C149.261 51.2227 147.938 50.3816 147.307 48.5854L139.193 26.4525L131.027 48.5854C130.396 50.3728 129.126 51.2227 127.338 51.2227C125.551 51.2227 124.123 50.3816 123.544 48.5854L114.519 19.1188C114.309 18.5405 114.046 17.857 114.046 17.1123C114.046 15.3774 115.571 13.9054 117.525 13.9054C119.049 13.9054 120.53 14.7991 120.898 16.1747L127.592 39.946L136.021 16.2273C136.652 14.4924 137.809 13.9054 139.132 13.9054C140.455 13.9054 141.717 14.4836 142.348 16.2273L150.777 39.946L157.418 16.1747C157.786 14.8078 159.267 13.9054 160.844 13.9054ZM182.846 14.0631C191.064 14.0631 196.339 19.3816 196.339 26.0232C196.339 32.6648 191.073 37.8782 182.846 37.8782H176.414V47.7355C176.414 49.6894 174.837 51.3191 172.883 51.3191C170.929 51.3191 169.3 49.6894 169.3 47.7355V17.9622C169.3 15.6928 170.929 14.0631 173.199 14.0631H182.846ZM182.53 31.8237C186.482 31.8237 189.435 30.4042 189.435 26.0232C189.435 21.6422 186.482 20.1176 182.53 20.1176H176.414V31.8149H182.53V31.8237ZM228.715 43.7312C229.819 43.7312 230.774 44.5198 230.774 45.7377C230.774 46.7366 230.143 47.5252 228.925 48.3225C226.393 50.2239 223.177 51.3279 219.278 51.3279C210.052 51.3279 200.834 43.6348 200.834 32.5159C200.834 21.3969 209.999 13.6951 219.383 13.6951C223.177 13.6951 226.34 14.7991 228.925 16.6479C230.134 17.489 230.774 18.225 230.774 19.2852C230.774 20.4944 229.827 21.2917 228.715 21.2917C227.821 21.2917 227.19 20.8186 226.498 20.2929C225.026 19.0837 222.336 17.9184 219.331 17.9184C212.268 17.9184 205.469 24.0342 205.469 32.5159C205.469 40.9975 212.268 47.1134 219.278 47.1134C222.388 47.1134 225.026 45.9568 226.498 44.7389C227.181 44.2131 227.865 43.74 228.715 43.74V43.7312ZM247.614 23.2281C253.73 23.2281 261.371 27.9158 261.371 37.2999C261.371 46.684 253.73 51.3191 247.614 51.3191C241.498 51.3191 233.911 46.7366 233.911 37.2999C233.911 27.8632 241.551 23.2281 247.614 23.2281ZM247.614 27.0747C243.242 27.0747 238.23 30.5006 238.23 37.2999C238.23 44.0992 243.233 47.4726 247.614 47.4726C251.995 47.4726 256.998 44.0467 256.998 37.2999C256.998 30.5532 252.048 27.0747 247.614 27.0747ZM280.849 23.2807C285.799 23.2807 291.126 26.2861 291.126 35.0306V48.9972C291.126 50.1538 290.128 51.214 288.91 51.214C287.692 51.214 286.745 50.1625 286.745 48.9972V35.0306C286.745 29.2827 283.267 27.1272 279.578 27.1272C274.207 27.1272 271.464 31.6572 270.991 34.6625V48.9972C270.991 50.1538 269.992 51.214 268.827 51.214C267.662 51.214 266.61 50.1625 266.61 48.9972V25.6026C266.61 24.3935 267.609 23.3858 268.827 23.3858C270.045 23.3858 270.991 24.3847 270.991 25.6026V29.8697C271.307 28.0823 274.575 23.2807 280.849 23.2807ZM306.355 23.2281C310.306 23.2281 313.47 24.9104 315.152 27.4952C315.467 28.0209 315.678 28.5467 315.678 29.1775C315.678 30.229 314.889 30.8598 313.89 30.8598C313.154 30.8598 312.576 30.5444 311.831 29.5455C310.78 28.0209 309.036 26.8556 306.346 26.8556C303.183 26.8556 301.238 28.4328 301.238 30.7547C301.238 32.4896 302.394 33.9704 305.505 34.7589L308.878 35.6001C313.566 36.8618 316.782 38.8683 316.782 43.0828C316.782 48.4014 311.726 51.1965 306.241 51.1965C301.764 51.1965 298.18 49.19 296.909 46.6665C296.699 46.2985 296.121 45.5099 296.121 44.5023C296.121 43.556 296.752 42.6535 297.908 42.6535C298.855 42.6535 299.38 43.1792 299.81 43.9678C301.019 46.4474 303.498 47.4463 306.241 47.4463C309.667 47.4463 312.462 46.0269 312.462 43.3369C312.462 41.2253 310.885 40.0687 308.09 39.3327L304.138 38.2813C299.187 36.967 296.971 34.2244 296.971 30.7985C296.971 26.216 301.133 23.2106 306.355 23.2106V23.2281ZM333.71 23.2281C339.721 23.2281 346.94 27.9684 346.94 36.5113C346.94 38.1936 345.942 38.991 344.145 38.991L324.755 38.8858C325.228 44.7914 329.758 47.5777 334.297 47.5777C336.724 47.5777 338.669 46.8943 340.413 45.5712C341.201 45.0455 341.674 44.7827 342.262 44.7827C343.26 44.7827 343.891 45.4661 343.891 46.4124C343.891 46.9907 343.628 47.569 342.735 48.3663C340.623 50.2151 337.521 51.214 334.2 51.214C327.454 51.214 320.234 46.7892 320.234 37.2999C320.234 27.8107 327.717 23.2281 333.727 23.2281H333.71ZM333.71 27.0747C329.811 27.0747 325.228 29.712 324.755 35.6088H342.673C342.2 29.7033 337.67 27.0747 333.719 27.0747H333.71ZM366.427 23.2807C371.378 23.2807 376.705 26.2861 376.705 35.0306V48.9972C376.705 50.1538 375.706 51.214 374.488 51.214C373.27 51.214 372.324 50.1625 372.324 48.9972V35.0306C372.324 29.2827 368.845 27.1272 365.157 27.1272C359.786 27.1272 357.043 31.6572 356.57 34.6625V48.9972C356.57 50.1538 355.571 51.214 354.406 51.214C353.24 51.214 352.189 50.1625 352.189 48.9972V25.6026C352.189 24.3935 353.188 23.3858 354.406 23.3858C355.624 23.3858 356.57 24.3847 356.57 25.6026V29.8697C356.885 28.0823 360.154 23.2807 366.427 23.2807ZM392.993 47.1046C394.15 46.8417 395 47.5777 395 48.6292C395 50.5305 392.415 51.214 390.049 51.214C387.254 51.214 384.564 50.2151 384.564 45.2558V27.1886H382.243C381.139 27.1886 380.394 26.4525 380.394 25.3398C380.394 24.227 381.13 23.5523 382.243 23.5523H384.564V19.0224C384.564 17.8658 385.563 16.8582 386.781 16.8582C387.999 16.8582 388.945 17.857 388.945 19.0224V23.5523H393.16C394.211 23.5523 395.009 24.2358 395.009 25.3398C395.009 26.4438 394.22 27.1886 393.16 27.1886H388.945V45.1069C388.945 46.7366 389.629 47.4288 390.952 47.4288C391.741 47.4288 392.424 47.2185 393.011 47.1134L392.993 47.1046Z" fill="#11293E"/></g>',
			'logo-mono'     => '<g clip-path="url(#a)" fill="#fff"><path d="M4.84 1.006a.34.34 0 0 0-.333-.34H2.564a.336.336 0 0 0-.292.172L.735 3.566a.336.336 0 0 0 .293.501h1.94a.342.342 0 0 0 .292-.173l1.531-2.713H4.79a.333.333 0 0 0 .05-.175ZM11.628 4.545a.33.33 0 0 0-.048-.171L9.76 1.144 8.403 3.547l.474.84a.33.33 0 0 1 .003.314L7.09 7.88a.341.341 0 0 0 .272.539h1.937a.336.336 0 0 0 .292-.17l2.005-3.558-.003-.002a.34.34 0 0 0 .034-.145h.002Z"/><path d="M9.76 1.144 9.493.667 8.17 2.994l-.038.066-.294.518a.34.34 0 0 1 .569-.03L9.76 1.144ZM6.995.667h-.012a.336.336 0 0 0-.303.19l-2.636 4.66a.345.345 0 0 1-.299.175.348.348 0 0 1-.303-.183l-.447-.792a.337.337 0 0 0-.294-.171l-1.936.006a.336.336 0 0 0-.293.5l1.802 3.199c.06.105.171.17.293.17h2.314c.129 0 .239-.071.297-.178L6.832 5.34l1.002-1.762.294-.518.038-.066L9.49.667H6.995ZM13.896 1.921c0-.148.053-.274.157-.378a.518.518 0 0 1 .38-.155c.136 0 .247.038.334.114.087.077.144.161.17.253l1.065 3.843 1.041-3.825a.48.48 0 0 1 .188-.28.516.516 0 0 1 .627.002c.096.072.16.167.188.285l1.041 3.825 1.076-3.856a.523.523 0 0 1 .501-.36.51.51 0 0 1 .37.151.508.508 0 0 1 .154.374.505.505 0 0 1-.027.163L19.596 7.32a.518.518 0 0 1-.488.375.47.47 0 0 1-.31-.11.546.546 0 0 1-.182-.285l-1.057-3.747-1.053 3.749a.525.525 0 0 1-.19.283.51.51 0 0 1-.325.11.514.514 0 0 1-.315-.102.5.5 0 0 1-.19-.264l-1.564-5.242a.514.514 0 0 1-.027-.165h.002ZM21.94 7.167V1.992c0-.146.05-.27.154-.374a.514.514 0 0 1 .377-.154h1.745c.566 0 1.026.159 1.383.478.357.318.536.785.536 1.399s-.177 1.077-.533 1.398c-.355.321-.817.484-1.385.484H23v1.943a.512.512 0 0 1-.152.375c-.1.1-.226.151-.375.151a.514.514 0 0 1-.378-.154.506.506 0 0 1-.155-.374v.003Zm1.059-2.896h1.134c.287 0 .509-.08.668-.243.16-.162.239-.392.239-.686 0-.294-.081-.526-.242-.685-.16-.16-.382-.239-.666-.239h-1.135v1.855L23 4.27ZM26.842 4.549c0-1.064.249-1.866.749-2.404.5-.539 1.202-.81 2.107-.81.432 0 .828.088 1.191.26.362.173.651.41.869.712.042.06.063.123.063.192a.287.287 0 0 1-.29.295.303.303 0 0 1-.241-.12 1.81 1.81 0 0 0-.671-.553 2.018 2.018 0 0 0-.92-.215c-.714 0-1.264.22-1.645.662-.38.441-.571 1.103-.571 1.984 0 .88.187 1.593.56 2.01.375.418.927.626 1.656.626.318 0 .62-.063.905-.19.285-.128.513-.314.686-.562a.276.276 0 0 1 .241-.12c.084 0 .153.029.207.086a.287.287 0 0 1 .083.21c0 .063-.018.12-.054.174-.2.293-.49.527-.869.704-.38.177-.78.265-1.2.265-.919 0-1.623-.26-2.116-.781-.494-.521-.74-1.328-.74-2.422v-.003ZM33.042 7.19c-.337-.37-.507-.92-.507-1.654 0-.734.168-1.281.504-1.646.336-.365.809-.548 1.417-.548.607 0 1.083.183 1.42.548.334.365.502.914.502 1.646 0 .732-.168 1.285-.503 1.654-.334.369-.809.553-1.42.553-.61 0-1.075-.184-1.413-.553Zm.087-1.654c0 1.118.443 1.676 1.327 1.676.884 0 1.326-.558 1.326-1.676 0-1.118-.442-1.667-1.326-1.667-.884 0-1.327.555-1.327 1.667ZM37.29 7.404V3.686c0-.084.029-.153.086-.207a.293.293 0 0 1 .21-.083c.084 0 .153.027.207.083a.286.286 0 0 1 .083.207v.294c.093-.185.247-.338.465-.458.217-.12.47-.181.755-.181.271 0 .508.031.712.096.204.064.366.147.488.247.121.101.218.226.293.371.075.146.124.291.15.434.025.142.037.3.037.47v2.444a.276.276 0 0 1-.082.207.286.286 0 0 1-.207.082.29.29 0 0 1-.21-.082.277.277 0 0 1-.086-.207V5.078c0-.193-.015-.36-.047-.498a1.104 1.104 0 0 0-.165-.38.718.718 0 0 0-.337-.256 1.503 1.503 0 0 0-.545-.085c-.813 0-1.22.427-1.22 1.282v2.262a.276.276 0 0 1-.083.207.286.286 0 0 1-.206.082.29.29 0 0 1-.21-.082.277.277 0 0 1-.086-.207l-.002.001ZM41.584 6.899c0-.075.025-.137.076-.188a.254.254 0 0 1 .188-.076c.075 0 .14.03.2.09.3.3.69.451 1.162.451.287 0 .52-.052.7-.159.18-.106.27-.27.27-.493a.463.463 0 0 0-.104-.3.747.747 0 0 0-.273-.21 3.832 3.832 0 0 0-.387-.155c-.144-.05-.291-.1-.44-.153a3.603 3.603 0 0 1-.44-.186 2.328 2.328 0 0 1-.387-.242.974.974 0 0 1-.273-.344 1.08 1.08 0 0 1-.102-.48c0-.363.13-.638.392-.824.26-.186.597-.279 1.008-.279.635 0 1.099.195 1.39.586.045.06.067.118.067.174a.25.25 0 0 1-.078.187.263.263 0 0 1-.19.077.26.26 0 0 1-.193-.081 1.426 1.426 0 0 0-.43-.288 1.46 1.46 0 0 0-.566-.096c-.236 0-.43.043-.584.129-.153.087-.23.228-.23.424 0 .116.035.218.103.31a.749.749 0 0 0 .27.221c.113.057.242.113.387.164l.44.155c.147.052.294.114.44.186.145.072.274.15.388.232a.915.915 0 0 1 .275.33c.069.137.103.291.103.464 0 .269-.07.495-.21.681-.14.186-.324.321-.554.405-.23.083-.493.125-.79.125-.346 0-.66-.068-.938-.201-.28-.134-.49-.288-.626-.46a.303.303 0 0 1-.063-.175l-.002-.001ZM45.48 5.536c0-.7.166-1.24.496-1.622.33-.381.794-.572 1.39-.572.235 0 .446.029.635.087.19.059.346.138.47.239.123.1.23.211.317.33.088.119.154.25.2.389.044.14.076.268.095.387a2.447 2.447 0 0 1 .023.558 1.02 1.02 0 0 1-.038.177.26.26 0 0 1-.11.152.396.396 0 0 1-.21.05h-2.673c0 .455.123.817.369 1.087s.584.405 1.013.405c.238 0 .44-.034.607-.105.167-.07.33-.166.493-.288a.307.307 0 0 1 .165-.058.25.25 0 0 1 .187.078.263.263 0 0 1 .077.19.294.294 0 0 1-.1.21 1.61 1.61 0 0 1-.25.2 2.02 2.02 0 0 1-.48.207c-.22.072-.452.106-.7.106-.628 0-1.114-.181-1.458-.543-.343-.361-.516-.917-.516-1.664h-.002Zm.595-.339h2.462c0-.179-.018-.34-.054-.488a1.413 1.413 0 0 0-.177-.417.83.83 0 0 0-.362-.31 1.352 1.352 0 0 0-.579-.112c-.39 0-.696.118-.918.353-.222.236-.347.56-.374.974h.002ZM50.017 7.404V3.686c0-.084.028-.153.085-.207a.293.293 0 0 1 .21-.083c.085 0 .154.027.208.083a.287.287 0 0 1 .082.207v.294c.093-.185.248-.338.465-.458.218-.12.47-.181.755-.181.272 0 .509.031.713.096.204.064.366.147.488.247.121.101.217.226.292.371.075.146.125.291.15.434.026.142.038.3.038.47v2.444a.276.276 0 0 1-.083.207.287.287 0 0 1-.207.082.29.29 0 0 1-.21-.082.277.277 0 0 1-.085-.207V5.078c0-.193-.015-.36-.047-.498a1.106 1.106 0 0 0-.165-.38.718.718 0 0 0-.338-.256 1.504 1.504 0 0 0-.544-.085c-.814 0-1.22.427-1.22 1.282v2.262a.276.276 0 0 1-.083.207.287.287 0 0 1-.207.082.29.29 0 0 1-.21-.082.277.277 0 0 1-.085-.207l-.002.001ZM54.277 3.938a.263.263 0 0 1-.078-.195.27.27 0 0 1 .078-.196.26.26 0 0 1 .194-.081h.357v-.96a.27.27 0 0 1 .09-.203.284.284 0 0 1 .205-.087c.078 0 .149.028.206.085a.279.279 0 0 1 .085.206v.96h.6c.077 0 .142.027.193.081a.276.276 0 0 1 .078.197.264.264 0 0 1-.078.195.262.262 0 0 1-.194.078h-.599v2.525c0 .194.048.344.146.45.096.106.225.158.387.158.081 0 .149.026.204.078a.256.256 0 0 1 .083.195.251.251 0 0 1-.083.194.284.284 0 0 1-.204.078c-.354 0-.629-.105-.824-.317-.195-.211-.292-.49-.292-.836V4.018h-.357a.258.258 0 0 1-.194-.078l-.003-.002Z"/></g><defs><clipPath id="a"><path fill="#fff" transform="translate(.429 .667)" d="M0 0h55.857v7.753H0z"/></clipPath></defs>',
			'close'         => '<path d="M1.68421 12L0 10.3158L4.33083 5.98496L0 1.68421L1.68421 0L6.01504 4.33083L10.3158 0L12 1.68421L7.66917 5.98496L12 10.3158L10.3158 12L6.01504 7.66917L1.68421 12Z" fill="white"/>',
			'layout'        => '<path d="M6 14C5.45 14 4.97917 13.8042 4.5875 13.4125C4.19583 13.0208 4 12.55 4 12V2C4 1.45 4.19583 0.979167 4.5875 0.5875C4.97917 0.195833 5.45 0 6 0H16C16.55 0 17.0208 0.195833 17.4125 0.5875C17.8042 0.979167 18 1.45 18 2V12C18 12.55 17.8042 13.0208 17.4125 13.4125C17.0208 13.8042 16.55 14 16 14H6ZM6 12H16V2H6V12ZM2 16V18C1.45 18 0.979167 17.8042 0.5875 17.4125C0.195833 17.0208 0 16.55 0 16H2ZM0 14V12H2V14H0ZM0 10V8H2V10H0ZM0 6V4H2V6H0ZM4 18V16H6V18H4ZM8 18V16H10V18H8ZM12 18V16H14V18H12Z" fill="#005AE0"/>',
			'style'         => '<path d="M7.05002 19L0.900024 12.85C0.733358 12.6833 0.608358 12.5 0.525024 12.3C0.441691 12.1 0.400024 11.8917 0.400024 11.675C0.400024 11.4583 0.441691 11.25 0.525024 11.05C0.608358 10.85 0.733358 10.6667 0.900024 10.5L6.65002 4.775L4.00002 2.125L5.55002 0.5L15.55 10.5C15.7167 10.6667 15.8375 10.85 15.9125 11.05C15.9875 11.25 16.025 11.4583 16.025 11.675C16.025 11.8917 15.9875 12.1 15.9125 12.3C15.8375 12.5 15.7167 12.6833 15.55 12.85L9.40002 19C9.23336 19.1667 9.05002 19.2917 8.85002 19.375C8.65002 19.4583 8.44169 19.5 8.22502 19.5C8.00836 19.5 7.80002 19.4583 7.60002 19.375C7.40002 19.2917 7.21669 19.1667 7.05002 19ZM8.22502 6.35L2.87502 11.7H13.575L8.22502 6.35ZM18.2 19.5C17.6 19.5 17.0917 19.2875 16.675 18.8625C16.2584 18.4375 16.05 17.9167 16.05 17.3C16.05 16.85 16.1625 16.425 16.3875 16.025C16.6125 15.625 16.8667 15.2333 17.15 14.85L18.2 13.5L19.3 14.85C19.5667 15.2333 19.8167 15.625 20.05 16.025C20.2834 16.425 20.4 16.85 20.4 17.3C20.4 17.9167 20.1834 18.4375 19.75 18.8625C19.3167 19.2875 18.8 19.5 18.2 19.5Z" fill="#051F31" fill-opacity="0.6"/>',
			'content'       => '<path d="M0.800049 22V18H20.8V22H0.800049ZM4.80005 14H6.20005L14 6.225L12.575 4.8L4.80005 12.6V14ZM2.80005 16V11.75L14 0.575C14.1834 0.391667 14.3959 0.25 14.6375 0.15C14.8792 0.05 15.1334 0 15.4 0C15.6667 0 15.925 0.05 16.175 0.15C16.425 0.25 16.6501 0.4 16.85 0.6L18.225 2C18.4251 2.18333 18.5709 2.4 18.6625 2.65C18.7542 2.9 18.8 3.15833 18.8 3.425C18.8 3.675 18.7542 3.92083 18.6625 4.1625C18.5709 4.40417 18.4251 4.625 18.225 4.825L7.05005 16H2.80005Z" fill="#051F31" fill-opacity="0.6"/>',
			'chevron-right' => '<path d="M4.6 6L0 1.4L1.4 0L7.4 6L1.4 12L0 10.6L4.6 6Z" fill="#051F31" fill-opacity="0.6"/>',
			'chevron-left'  => '<path d="M3.84167 7.5L0.341675 4L3.84167 0.5L4.65834 1.31667L1.97501 4L4.65834 6.68333L3.84167 7.5Z" fill="#051F31" fill-opacity="0.5"/>',
			'drag'          => '<path d="M0 6V4H16V6H0ZM0 2V0H16V2H0Z" fill="#051F31" fill-opacity="0.7"/>',
			'visibility'    => '<path d="M6.99998 7.83337C7.72915 7.83337 8.34894 7.57817 8.85935 7.06775C9.36977 6.55733 9.62498 5.93754 9.62498 5.20837C9.62498 4.47921 9.36977 3.85942 8.85935 3.349C8.34894 2.83858 7.72915 2.58337 6.99998 2.58337C6.27081 2.58337 5.65102 2.83858 5.1406 3.349C4.63019 3.85942 4.37498 4.47921 4.37498 5.20837C4.37498 5.93754 4.63019 6.55733 5.1406 7.06775C5.65102 7.57817 6.27081 7.83337 6.99998 7.83337ZM6.99998 6.78337C6.56248 6.78337 6.1906 6.63025 5.88435 6.324C5.5781 6.01775 5.42498 5.64587 5.42498 5.20837C5.42498 4.77087 5.5781 4.399 5.88435 4.09275C6.1906 3.7865 6.56248 3.63337 6.99998 3.63337C7.43748 3.63337 7.80935 3.7865 8.1156 4.09275C8.42185 4.399 8.57498 4.77087 8.57498 5.20837C8.57498 5.64587 8.42185 6.01775 8.1156 6.324C7.80935 6.63025 7.43748 6.78337 6.99998 6.78337ZM6.99998 9.58337C5.58053 9.58337 4.28748 9.18719 3.12081 8.39483C1.95415 7.60247 1.10831 6.54032 0.583313 5.20837C1.10831 3.87643 1.95415 2.81428 3.12081 2.02192C4.28748 1.22955 5.58053 0.833374 6.99998 0.833374C8.41942 0.833374 9.71248 1.22955 10.8791 2.02192C12.0458 2.81428 12.8916 3.87643 13.4166 5.20837C12.8916 6.54032 12.0458 7.60247 10.8791 8.39483C9.71248 9.18719 8.41942 9.58337 6.99998 9.58337ZM6.99998 8.41671C8.09859 8.41671 9.10727 8.12747 10.026 7.549C10.9448 6.97053 11.6472 6.19032 12.1333 5.20837C11.6472 4.22643 10.9448 3.44622 10.026 2.86775C9.10727 2.28928 8.09859 2.00004 6.99998 2.00004C5.90137 2.00004 4.89269 2.28928 3.97394 2.86775C3.05519 3.44622 2.35276 4.22643 1.86665 5.20837C2.35276 6.19032 3.05519 6.97053 3.97394 7.549C4.89269 8.12747 5.90137 8.41671 6.99998 8.41671Z" fill="#B4BCC1"/>',
			'edit'          => '<path d="M1.66667 13.8333H2.85417L11 5.6875L9.8125 4.5L1.66667 12.6458V13.8333ZM0 15.5V11.9583L11 0.979167C11.1667 0.826389 11.3507 0.708333 11.5521 0.625C11.7535 0.541667 11.9653 0.5 12.1875 0.5C12.4097 0.5 12.625 0.541667 12.8333 0.625C13.0417 0.708333 13.2222 0.833333 13.375 1L14.5208 2.16667C14.6875 2.31944 14.809 2.5 14.8854 2.70833C14.9618 2.91667 15 3.125 15 3.33333C15 3.55556 14.9618 3.76736 14.8854 3.96875C14.809 4.17014 14.6875 4.35417 14.5208 4.52083L3.54167 15.5H0ZM10.3958 5.10417L9.8125 4.5L11 5.6875L10.3958 5.10417Z" fill="#051F31" fill-opacity="0.7"/>',
			'delete'        => '<path d="M2.61841 15.5C2.16007 15.5 1.76771 15.3368 1.44132 15.0104C1.11494 14.684 0.951741 14.2917 0.951741 13.8333V3H0.118408V1.33333H4.28507V0.5H9.28507V1.33333H13.4517V3H12.6184V13.8333C12.6184 14.2917 12.4552 14.684 12.1288 15.0104C11.8024 15.3368 11.4101 15.5 10.9517 15.5H2.61841ZM10.9517 3H2.61841V13.8333H10.9517V3ZM4.28507 12.1667H5.95174V4.66667H4.28507V12.1667ZM7.61841 12.1667H9.28507V4.66667H7.61841V12.1667Z" fill="#051F31" fill-opacity="0.7"/>',
			'lock'          => '<path d="M19 9.33333H17.6666V6.66667C17.6666 2.98667 14.68 0 11 0C7.31998 0 4.33331 2.98667 4.33331 6.66667V9.33333H2.99998C1.53331 9.33333 0.333313 10.5333 0.333313 12V25.3333C0.333313 26.8 1.53331 28 2.99998 28H19C20.4666 28 21.6666 26.8 21.6666 25.3333V12C21.6666 10.5333 20.4666 9.33333 19 9.33333ZM6.99998 6.66667C6.99998 4.45333 8.78665 2.66667 11 2.66667C13.2133 2.66667 15 4.45333 15 6.66667V9.33333H6.99998V6.66667ZM19 25.3333H2.99998V12H19V25.3333ZM11 21.3333C12.4666 21.3333 13.6666 20.1333 13.6666 18.6667C13.6666 17.2 12.4666 16 11 16C9.53331 16 8.33331 17.2 8.33331 18.6667C8.33331 20.1333 9.53331 21.3333 11 21.3333Z" fill="#8A8A8A"/>',
			'checkmark'     => '<circle class="path circle" fill="none" stroke="#73AF55" stroke-width="6" stroke-miterlimit="10" cx="65.1" cy="65.1" r="62.1"/><polyline class="path check" fill="none" stroke="#73AF55" stroke-width="6" stroke-linecap="round" stroke-miterlimit="10" points="100.2,40.2 51.5,88.8 29.8,67.5 "/>',
			'preferences'   => '<path d="m370-80-16-128q-13-5-24.5-12T307-235l-119 50L78-375l103-78q-1-7-1-13.5v-27q0-6.5 1-13.5L78-585l110-190 119 50q11-8 23-15t24-12l16-128h220l16 128q13 5 24.5 12t22.5 15l119-50 110 190-103 78q1 7 1 13.5v27q0 6.5-2 13.5l103 78-110 190-118-50q-11 8-23 15t-24 12L590-80H370Zm70-80h79l14-106q31-8 57.5-23.5T639-327l99 41 39-68-86-65q5-14 7-29.5t2-31.5q0-16-2-31.5t-7-29.5l86-65-39-68-99 42q-22-23-48.5-38.5T533-694l-13-106h-79l-14 106q-31 8-57.5 23.5T321-633l-99-41-39 68 86 64q-5 15-7 30t-2 32q0 16 2 31t7 30l-86 65 39 68 99-42q22 23 48.5 38.5T427-266l13 106Zm42-180q58 0 99-41t41-99q0-58-41-99t-99-41q-59 0-99.5 41T342-480q0 58 40.5 99t99.5 41Zm-2-140Z" fill="#fff" />',
			'inbox'         => '<path fill-rule="evenodd" clip-rule="evenodd" d="M13.3333 0.5H1.66667C0.75 0.5 0 1.25 0 2.16667V13.8333C0 14.75 0.741667 15.5 1.66667 15.5H13.3333C14.25 15.5 15 14.75 15 13.8333V2.16667C15 1.25 14.25 0.5 13.3333 0.5ZM13.3333 13.8333H1.66667V11.3333H4.63333C5.20833 12.325 6.275 13 7.50833 13C8.74167 13 9.8 12.325 10.3833 11.3333H13.3333V13.8333ZM9.175 9.66667H13.3333V2.16667H1.66667V9.66667H5.84167C5.84167 10.5833 6.59167 11.3333 7.50833 11.3333C8.425 11.3333 9.175 10.5833 9.175 9.66667Z" fill="#777777"/>',
			'info'          => '<path d="M7.66667 4.33333H9.33334V6H7.66667V4.33333ZM7.66667 7.66666H9.33334V12.6667H7.66667V7.66666ZM8.50001 0.166664C3.90001 0.166664 0.166672 3.9 0.166672 8.5C0.166672 13.1 3.90001 16.8333 8.50001 16.8333C13.1 16.8333 16.8333 13.1 16.8333 8.5C16.8333 3.9 13.1 0.166664 8.50001 0.166664ZM8.50001 15.1667C4.82501 15.1667 1.83334 12.175 1.83334 8.5C1.83334 4.825 4.82501 1.83333 8.50001 1.83333C12.175 1.83333 15.1667 4.825 15.1667 8.5C15.1667 12.175 12.175 15.1667 8.50001 15.1667Z" fill="#EBAD35"/>',
			'success'       => '<path d="M8.50001 0.666656C3.90001 0.666656 0.166672 4.39999 0.166672 8.99999C0.166672 13.6 3.90001 17.3333 8.50001 17.3333C13.1 17.3333 16.8333 13.6 16.8333 8.99999C16.8333 4.39999 13.1 0.666656 8.50001 0.666656ZM8.50001 15.6667C4.82501 15.6667 1.83334 12.675 1.83334 8.99999C1.83334 5.32499 4.82501 2.33332 8.50001 2.33332C12.175 2.33332 15.1667 5.32499 15.1667 8.99999C15.1667 12.675 12.175 15.6667 8.50001 15.6667ZM12.325 5.31666L6.83334 10.8083L4.675 8.65832L3.50001 9.83332L6.83334 13.1667L13.5 6.49999L12.325 5.31666Z" fill="#09A347"/>',
			'warning'       => '<path d="M12.73 0H5.27L0 5.27V12.73L5.27 18H12.73L18 12.73V5.27L12.73 0ZM16 11.9L11.9 16H6.1L2 11.9V6.1L6.1 2H11.9L16 6.1V11.9ZM11.83 4.76L9 7.59L6.17 4.76L4.76 6.17L7.59 9L4.76 11.83L6.17 13.24L9 10.41L11.83 13.24L13.24 11.83L10.41 9L13.24 6.17L11.83 4.76Z" fill="#DF2A35"/>',
			'cookie'        => '<path d="M7 12.8333C6.19305 12.8333 5.43472 12.6802 4.725 12.3739C4.01528 12.0677 3.39791 11.6521 2.87291 11.1271C2.34791 10.6021 1.93229 9.98471 1.62604 9.27499C1.31979 8.56526 1.16666 7.80693 1.16666 6.99999C1.16666 6.27082 1.30764 5.55624 1.58958 4.85624C1.87153 4.15624 2.26528 3.53158 2.77083 2.98228C3.27639 2.43297 3.88403 1.99061 4.59375 1.65519C5.30347 1.31978 6.08125 1.15207 6.92708 1.15207C7.13125 1.15207 7.34028 1.16179 7.55416 1.18124C7.76805 1.20068 7.9868 1.23471 8.21041 1.28332C8.12291 1.72082 8.15208 2.13401 8.29791 2.5229C8.44375 2.91179 8.6625 3.23506 8.95416 3.49269C9.24583 3.75033 9.5934 3.92776 9.99687 4.02499C10.4003 4.12221 10.816 4.0979 11.2437 3.95207C10.991 4.52568 11.0274 5.07499 11.3531 5.59999C11.6788 6.12499 12.1625 6.39721 12.8042 6.41665C12.8139 6.5236 12.8212 6.62325 12.826 6.71561C12.8309 6.80797 12.8333 6.90762 12.8333 7.01457C12.8333 7.81179 12.6802 8.56283 12.374 9.26769C12.0677 9.97256 11.6521 10.5899 11.1271 11.1198C10.6021 11.6496 9.98472 12.0677 9.275 12.3739C8.56528 12.6802 7.80694 12.8333 7 12.8333ZM6.125 5.83332C6.36805 5.83332 6.57465 5.74825 6.74479 5.57811C6.91493 5.40797 7 5.20137 7 4.95832C7 4.71526 6.91493 4.50867 6.74479 4.33853C6.57465 4.16839 6.36805 4.08332 6.125 4.08332C5.88194 4.08332 5.67534 4.16839 5.50521 4.33853C5.33507 4.50867 5.25 4.71526 5.25 4.95832C5.25 5.20137 5.33507 5.40797 5.50521 5.57811C5.67534 5.74825 5.88194 5.83332 6.125 5.83332ZM4.95833 8.74999C5.20139 8.74999 5.40798 8.66492 5.57812 8.49478C5.74826 8.32464 5.83333 8.11804 5.83333 7.87499C5.83333 7.63193 5.74826 7.42533 5.57812 7.25519C5.40798 7.08506 5.20139 6.99999 4.95833 6.99999C4.71528 6.99999 4.50868 7.08506 4.33854 7.25519C4.1684 7.42533 4.08333 7.63193 4.08333 7.87499C4.08333 8.11804 4.1684 8.32464 4.33854 8.49478C4.50868 8.66492 4.71528 8.74999 4.95833 8.74999ZM8.75 9.33332C8.91528 9.33332 9.05382 9.27742 9.16562 9.16561C9.27743 9.0538 9.33333 8.91526 9.33333 8.74999C9.33333 8.58471 9.27743 8.44617 9.16562 8.33436C9.05382 8.22256 8.91528 8.16665 8.75 8.16665C8.58472 8.16665 8.44618 8.22256 8.33437 8.33436C8.22257 8.44617 8.16666 8.58471 8.16666 8.74999C8.16666 8.91526 8.22257 9.0538 8.33437 9.16561C8.44618 9.27742 8.58472 9.33332 8.75 9.33332ZM7 11.6667C8.18611 11.6667 9.23854 11.2583 10.1573 10.4417C11.076 9.62499 11.5792 8.58471 11.6667 7.32082C11.1806 7.10693 10.799 6.81526 10.5219 6.44582C10.2448 6.07637 10.0576 5.66318 9.96041 5.20624C9.2118 5.09929 8.57014 4.77846 8.03541 4.24374C7.50069 3.70901 7.17014 3.06735 7.04375 2.31874C6.26597 2.29929 5.58298 2.44026 4.99479 2.74165C4.40659 3.04304 3.91562 3.4295 3.52187 3.90103C3.12812 4.37256 2.83159 4.8854 2.63229 5.43957C2.43298 5.99374 2.33333 6.51388 2.33333 6.99999C2.33333 8.29304 2.78784 9.39408 3.69687 10.3031C4.6059 11.2121 5.70694 11.6667 7 11.6667Z" fill="white" fill-opacity="0.9"/>',
			'plus'          => '<path d="M440-440H200v-80h240v-240h80v240h240v80H520v240h-80v-240Z" fill="#777" />',
			'generate'      => '<path d="M9.5375 15C7.8625 15 6.4375 14.4187 5.2625 13.2563C4.0875 12.0938 3.5 10.675 3.5 9V8.86875L2.3 10.0688L1.25 9.01875L4.25 6.01875L7.25 9.01875L6.2 10.0688L5 8.86875V9C5 10.25 5.44063 11.3125 6.32188 12.1875C7.20313 13.0625 8.275 13.5 9.5375 13.5C9.8625 13.5 10.1812 13.4625 10.4937 13.3875C10.8062 13.3125 11.1125 13.2 11.4125 13.05L12.5375 14.175C12.0625 14.45 11.575 14.6562 11.075 14.7938C10.575 14.9313 10.0625 15 9.5375 15ZM14.75 11.9813L11.75 8.98125L12.8 7.93125L14 9.13125V9C14 7.75 13.5594 6.6875 12.6781 5.8125C11.7969 4.9375 10.725 4.5 9.4625 4.5C9.1375 4.5 8.81875 4.5375 8.50625 4.6125C8.19375 4.6875 7.8875 4.8 7.5875 4.95L6.4625 3.825C6.9375 3.55 7.425 3.34375 7.925 3.20625C8.425 3.06875 8.9375 3 9.4625 3C11.1375 3 12.5625 3.58125 13.7375 4.74375C14.9125 5.90625 15.5 7.325 15.5 9V9.13125L16.7 7.93125L17.75 8.98125L14.75 11.9813Z" fill="#051F31" fill-opacity="0.5"/>',
			'help'          => '<path fill-rule="evenodd" clip-rule="evenodd" d="M2.16667 9.99984C2.16667 5.39984 5.90001 1.6665 10.5 1.6665C15.1 1.6665 18.8333 5.39984 18.8333 9.99984C18.8333 14.5998 15.1 18.3332 10.5 18.3332C5.90001 18.3332 2.16667 14.5998 2.16667 9.99984ZM11.3333 13.3332V14.9998H9.66667V13.3332H11.3333ZM10.5 16.6665C6.82501 16.6665 3.83334 13.6748 3.83334 9.99984C3.83334 6.32484 6.82501 3.33317 10.5 3.33317C14.175 3.33317 17.1667 6.32484 17.1667 9.99984C17.1667 13.6748 14.175 16.6665 10.5 16.6665ZM7.16667 8.33317C7.16667 6.4915 8.65834 4.99984 10.5 4.99984C12.3417 4.99984 13.8333 6.4915 13.8333 8.33317C13.8333 9.40227 13.175 9.9776 12.534 10.5378C11.9259 11.0692 11.3333 11.587 11.3333 12.4998H9.66667C9.66667 10.9821 10.4518 10.3803 11.142 9.85123C11.6835 9.43618 12.1667 9.06585 12.1667 8.33317C12.1667 7.4165 11.4167 6.6665 10.5 6.6665C9.58334 6.6665 8.83334 7.4165 8.83334 8.33317H7.16667Z" fill="#777777"/>',
			'globe'         => '<path d="M480-80q-82 0-155-31.5t-127.5-86Q143-252 111.5-325T80-480q0-83 31.5-155.5t86-127Q252-817 325-848.5T480-880q83 0 155.5 31.5t127 86q54.5 54.5 86 127T880-480q0 82-31.5 155t-86 127.5q-54.5 54.5-127 86T480-80Zm0-82q26-36 45-75t31-83H404q12 44 31 83t45 75Zm-104-16q-18-33-31.5-68.5T322-320H204q29 50 72.5 87t99.5 55Zm208 0q56-18 99.5-55t72.5-87H638q-9 38-22.5 73.5T584-178ZM170-400h136q-3-20-4.5-39.5T300-480q0-21 1.5-40.5T306-560H170q-5 20-7.5 39.5T160-480q0 21 2.5 40.5T170-400Zm216 0h188q3-20 4.5-39.5T580-480q0-21-1.5-40.5T574-560H386q-3 20-4.5 39.5T380-480q0 21 1.5 40.5T386-400Zm268 0h136q5-20 7.5-39.5T800-480q0-21-2.5-40.5T790-560H654q3 20 4.5 39.5T660-480q0 21-1.5 40.5T654-400Zm-16-240h118q-29-50-72.5-87T584-782q18 33 31.5 68.5T638-640Zm-234 0h152q-12-44-31-83t-45-75q-26 36-45 75t-31 83Zm-200 0h118q9-38 22.5-73.5T376-782q-56 18-99.5 55T204-640Z" />',
			'settings'      => '<path d="M440-280h80l12-60q12-5 22.5-10.5T576-364l58 18 40-68-46-40q2-14 2-26t-2-26l46-40-40-68-58 18q-11-8-21.5-13.5T532-620l-12-60h-80l-12 60q-12 5-22.5 10.5T384-596l-58-18-40 68 46 40q-2 14-2 26t2 26l-46 40 40 68 58-18q11 8 21.5 13.5T428-340l12 60Zm40-120q-33 0-56.5-23.5T400-480q0-33 23.5-56.5T480-560q33 0 56.5 23.5T560-480q0 33-23.5 56.5T480-400ZM200-120q-33 0-56.5-23.5T120-200v-560q0-33 23.5-56.5T200-840h560q33 0 56.5 23.5T840-760v560q0 33-23.5 56.5T760-120H200Zm0-80h560v-560H200v560Zm0-560v560-560Z" fill="#000000"/>',
			'cookie-icon'   => '<path d="M480-80q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-75 29-147t81-128.5q52-56.5 125-91T475-881q21 0 43 2t45 7q-9 45 6 85t45 66.5q30 26.5 71.5 36.5t85.5-5q-26 59 7.5 113t99.5 56q1 11 1.5 20.5t.5 20.5q0 82-31.5 154.5t-85.5 127q-54 54.5-127 86T480-80Zm-60-480q25 0 42.5-17.5T480-620q0-25-17.5-42.5T420-680q-25 0-42.5 17.5T360-620q0 25 17.5 42.5T420-560Zm-80 200q25 0 42.5-17.5T400-420q0-25-17.5-42.5T340-480q-25 0-42.5 17.5T280-420q0 25 17.5 42.5T340-360Zm260 40q17 0 28.5-11.5T640-360q0-17-11.5-28.5T600-400q-17 0-28.5 11.5T560-360q0 17 11.5 28.5T600-320ZM480-160q122 0 216.5-84T800-458q-50-22-78.5-60T683-603q-77-11-132-66t-68-132q-80-2-140.5 29t-101 79.5Q201-644 180.5-587T160-480q0 133 93.5 226.5T480-160Zm0-324Z" fill="#000000"/>',
			'lock-icon'     => '<path d="M240-80q-33 0-56.5-23.5T160-160v-400q0-33 23.5-56.5T240-640h40v-80q0-83 58.5-141.5T480-920q83 0 141.5 58.5T680-720v80h40q33 0 56.5 23.5T800-560v400q0 33-23.5 56.5T720-80H240Zm0-80h480v-400H240v400Zm240-120q33 0 56.5-23.5T560-360q0-33-23.5-56.5T480-440q-33 0-56.5 23.5T400-360q0 33 23.5 56.5T480-280ZM360-640h240v-80q0-50-35-85t-85-35q-50 0-85 35t-35 85v80ZM240-160v-400 400Z" fill="#000000"/>',
			'cookie-off'    => '<path d="m815-260-58-58q18-31 29-66.5t14-73.5q-50-22-78.5-60T683-603q-77-11-132-66t-68-132q-49-2-90 10t-76 33l-57-57q61-42 137.5-58.5T563-872q-9 45 6 84.5t45 66.5q30 27 71 37t86-5q-31 69 11 118t96 51q8 72-9.5 138T815-260ZM340-360q-25 0-42.5-17.5T280-420q0-25 17.5-42.5T340-480q25 0 42.5 17.5T400-420q0 25-17.5 42.5T340-360ZM819-28 701-146q-48 32-103.5 49T480-80q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-62 17-117.5T146-701L27-820l57-57L876-85l-57 57ZM480-160q45 0 85.5-12t76.5-33L205-642q-21 36-33 76.5T160-480q0 133 93.5 226.5T480-160Zm-56-264Zm135-137Z" fill="#000000"/>',
			'shield'        => '<path d="M480-80q-139-35-229.5-159.5T160-516v-244l320-120 320 120v244q0 152-90.5 276.5T480-80Zm0-84q104-33 172-132t68-220v-189l-240-90-240 90v189q0 121 68 220t172 132Zm0-316Z" fill="#000000"/>',
			'policy'        => '<path d="M480-80q-139-35-229.5-159.5T160-516v-244l320-120 320 120v244q0 85-29 163.5T688-214L560-342q-18 11-38.5 16.5T480-320q-66 0-113-47t-47-113q0-66 47-113t113-47q66 0 113 47t47 113q0 22-5.5 42.5T618-398l60 60q20-41 31-86t11-92v-189l-240-90-240 90v189q0 121 68 220t172 132q26-8 49.5-20.5T576-214l56 56q-33 27-71.5 47T480-80Zm0-320q33 0 56.5-23.5T560-480q0-33-23.5-56.5T480-560q-33 0-56.5 23.5T400-480q0 33 23.5 56.5T480-400Zm8-77Z" fill="#000000"/>',
			'list'          => '<path d="M320-280q17 0 28.5-11.5T360-320q0-17-11.5-28.5T320-360q-17 0-28.5 11.5T280-320q0 17 11.5 28.5T320-280Zm0-160q17 0 28.5-11.5T360-480q0-17-11.5-28.5T320-520q-17 0-28.5 11.5T280-480q0 17 11.5 28.5T320-440Zm0-160q17 0 28.5-11.5T360-640q0-17-11.5-28.5T320-680q-17 0-28.5 11.5T280-640q0 17 11.5 28.5T320-600Zm120 320h240v-80H440v80Zm0-160h240v-80H440v80Zm0-160h240v-80H440v80ZM200-120q-33 0-56.5-23.5T120-200v-560q0-33 23.5-56.5T200-840h560q33 0 56.5 23.5T840-760v560q0 33-23.5 56.5T760-120H200Zm0-80h560v-560H200v560Zm0-560v560-560Z" fill="#000000"/>',
			'wrench'        => '<path d="M686-132 444-376q-20 8-40.5 12t-43.5 4q-100 0-170-70t-70-170q0-36 10-68.5t28-61.5l146 146 72-72-146-146q29-18 61.5-28t68.5-10q100 0 170 70t70 170q0 23-4 43.5T584-516l244 242q12 12 12 29t-12 29l-84 84q-12 12-29 12t-29-12Zm29-85 27-27-256-256q18-20 26-46.5t8-53.5q0-60-38.5-104.5T386-758l74 74q12 12 12 28t-12 28L332-500q-12 12-28 12t-28-12l-74-74q9 57 53.5 95.5T360-440q26 0 52-8t47-25l256 256ZM472-488Z" fill="#000000"/>',
			'tune'          => '<path d="M440-120v-240h80v80h320v80H520v80h-80Zm-320-80v-80h240v80H120Zm160-160v-80H120v-80h160v-80h80v240h-80Zm160-80v-80h400v80H440Zm160-160v-240h80v80h160v80H680v80h-80Zm-480-80v-80h400v80H120Z" fill="#000000"/>',
			'pencil'        => '<path d="M200-200h57l391-391-57-57-391 391v57Zm-80 80v-170l528-527q12-11 26.5-17t30.5-6q16 0 31 6t26 18l55 56q12 11 17.5 26t5.5 30q0 16-5.5 30.5T817-647L290-120H120Zm640-584-56-56 56 56Zm-141 85-28-29 57 57-29-28Z" fill="#000000"/>',
			'facebook'      => '<path fill="#A7AAAD" d="M16 8.05A8.02 8.02 0 0 0 8 0C3.58 0 0 3.6 0 8.05A8 8 0 0 0 6.74 16v-5.61H4.71V8.05h2.03V6.3c0-2.02 1.2-3.15 3-3.15.9 0 1.8.16 1.8.16v1.98h-1c-1 0-1.31.62-1.31 1.27v1.49h2.22l-.35 2.34H9.23V16A8.02 8.02 0 0 0 16 8.05Z"/>',
			'x'             => '<path fill="#A7AAAD" d="M389.2 48h70.6L305.6 224.2 487 464H345L233.7 318.6 106.5 464H35.8L200.7 275.5 26.8 48H172.4L272.9 180.9 389.2 48zM364.4 421.8h39.1L151.1 88h-42L364.4 421.8z"/>',
		)
	);
}


/**
 * Returns the site domain.
 *
 * @return string
 */
function wpconsent_get_site_domain() {
	return wp_parse_url( home_url(), PHP_URL_HOST );
}
