<?php
/**
 * Plugin Name: WPayme
 * Description: The WPayme plugin adds payment methods like iDEAL, Bancontact, credit card and more to your WordPress site for a variety of payment providers.
 * Version: 1.0.0
 * Requires at least: 5.9
 * Requires PHP: 8.1
 *
 * Author: Wpayme
 * Author URI: https://www.wpayme.com
 *
 * Text Domain: wpayme
 * Domain Path: /languages/
 *
 * Provides: wp-pay/core
 *
 * License: GPLv2 or later
 *
 * GitHub URI: https://github.com/wpayme/wp-wpayme-pay
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2005-2024 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! defined( 'WPAYME_PAY_DEBUG' ) ) {
	define( 'WPAYME_PAY_DEBUG', false );
}

/**
 * Autoload.
 */
if ( file_exists( __DIR__ . '/vendor/autoload_packages.php' ) ) {
	require_once __DIR__ . '/vendor/autoload_packages.php';
}

/**
 * Load core forms functionality bootstrap to ensure classes are available
 */
if ( file_exists( __DIR__ . '/packages/wpayme/wp-wpayme-pay-forms/wp-wpayme-pay-forms.php' ) ) {
	require_once __DIR__ . '/packages/wpayme/wp-wpayme-pay-forms/wp-wpayme-pay-forms.php';
}

/**
 * Bootstrap.
 */
if ( class_exists( '\Wpayme\WordPress\Pay\Plugin' ) ) {
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$action_scheduler = __DIR__ . '/packages/woocommerce/action-scheduler/action-scheduler.php';
	\Wpayme\WordPress\Pay\Plugin::instance(
		[
			'file'                 => __FILE__,
			'options'              => [
				'about_page_file' => __DIR__ . '/admin/page-about.php',
			],
			'action_scheduler'     => file_exists( $action_scheduler ) ? $action_scheduler : null,
			'wpayme_service_url' => 'https://api.wp-pay.org/wp-json/wpayme-pay/v1/payments',
		]
	);
}

if ( class_exists( '\Wpayme\WordPress\Updater\Plugin' ) ) {
	\Wpayme\WordPress\Updater\Plugin::instance()->setup();
}

if ( class_exists( '\Wpayme\WpaymePayAdminReports\Plugin' ) ) {
	\Wpayme\WpaymePayAdminReports\Plugin::instance()->setup();
}

/**
 * Plugin dependencies.
 *
 * @link https://make.wordpress.org/core/2024/03/05/introducing-plugin-dependencies-in-wordpress-6-5/
 * @link https://github.com/wpayme/wp-wpayme-pay-adyen/issues/25
 */
/*
add_filter(
	'wp_plugin_dependencies_slug',
	function ( $slug ) {
		if ( 'wpayme' === $slug ) {
			return dirname( __DIR__ );
		}

		return $slug;
	}
);
*/

// Disable Action Scheduler for this simplified review if the files are missing
add_filter( 'action_scheduler_pre_init', function( $load ) {
	if ( ! file_exists( __DIR__ . '/packages/woocommerce/action-scheduler/action-scheduler.php' ) ) {
		return false;
	}
	return $load;
} );

add_filter(
	'wpayme_pay_modules',
	function ( $modules ) {
		$modules[] = 'forms';
		$modules[] = 'subscriptions';

		return $modules;
	}
);

add_filter(
	'wpayme_pay_plugin_integrations',
	function ( $integrations ) {
		$classes = [
			\Wpayme\WordPress\Pay\Forms\Integration::class,
			// \Wpayme\WordPress\Pay\Extensions\Charitable\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\EasyDigitalDownloads\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\EventEspresso\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\Give\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\WooCommerce\Extension::class,
			\Wpayme\WordPress\Pay\Extensions\GravityForms\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\FormidableForms\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\MemberPress\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\NinjaForms\Extension::class,
			// \Wpayme\WordPress\Pay\Extensions\RestrictContent\Extension::class,
		];

		foreach ( $classes as $class ) {
			if ( class_exists( $class ) && ! array_key_exists( $class, $integrations ) ) {
				$integrations[ $class ] = new $class();
			}
		}

		return $integrations;
	}
);

add_filter(
	'get_post_metadata',
	function ( $value, $post_id, $meta_key, $single ) {
		static $filter = true;

		if ( false === $filter ) {
			return $value;
		}

		if ( '_wpayme_gateway_id' !== $meta_key ) {
			return $value;
		}

		$filter = false;

		$value = get_post_meta( $post_id, $meta_key, $single );

		$filter = true;

		$mode = get_post_meta( $post_id, '_wpayme_gateway_mode', true );

		switch ( $value ) {
			case 'adyen':
				return ( 'test' === $mode ) ? 'adyen-test' : 'adyen';
			case 'ems-ecommerce':
				return ( 'test' === $mode ) ? 'ems-ecommerce-test' : 'ems-ecommerce';
			case 'ing-ideal-advanced-2022':
				return ( 'test' === $mode ) ? 'ing-ideal-advanced-2022-sandbox' : 'ing-ideal-advanced-2022-production';
			case 'multisafepay-connect':
				return ( 'test' === $mode ) ? 'multisafepay-connect-test' : 'multisafepay-connect';
			case 'ogone-directlink':
				return ( 'test' === $mode ) ? 'ingenico-directlink-test' : 'ingenico-directlink';
			case 'ogone-orderstandard':
				return ( 'test' === $mode ) ? 'ingenico-orderstandard-test' : 'ingenico-orderstandard';
			case 'paypal':
				return ( 'test' === $mode ) ? 'paypal-sandbox' : 'paypal';
			case 'rabobank-omnikassa-2':
				return ( 'test' === $mode ) ? 'rabobank-omnikassa-2-sandbox' : 'rabobank-omnikassa-2';
		}

		return $value;
	},
	10,
	4
);

add_filter(
	'wpayme_pay_gateways',
	function ( $gateways ) {
		// Helper to safely add gateway if class exists
		$add_gateway = function( $class, $args = [] ) use ( &$gateways ) {
			// Only add if the class actually exists in the current codebase
			if ( class_exists( $class ) ) {
				$gateways[] = empty($args) ? new $class() : new $class($args);
			}
		};

		// EMS - eCommerce.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/ems-e-commerce/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\EMS\ECommerce\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\EMS\ECommerce\Integration::class, [
				'id'            => 'ems-ecommerce',
				'name'          => 'EMS e-Commerce',
				'mode'          => 'live',
				'action_url'    => 'https://www.ipg-online.com/connect/gateway/processing',
				'dashboard_url' => 'https://www.ipg-online.com/vt/login',
			]);

			$add_gateway( \Wpayme\WordPress\Pay\Gateways\EMS\ECommerce\Integration::class, [
				'id'            => 'ems-ecommerce-test',
				'name'          => 'EMS e-Commerce - Test',
				'mode'          => 'test',
				'action_url'    => 'https://test.ipg-online.com/connect/gateway/processing',
				'dashboard_url' => 'https://test.ipg-online.com/vt/login',
			]);
		}

		// ICEPAY.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/icepay/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\Icepay\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\Icepay\Integration::class );
		}

		// iDEAL Simulator - iDEAL Professional / Advanced / Zelfbouw.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/ideal-advanced-v3/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\IDealAdvancedV3\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\IDealAdvancedV3\Integration::class, [
				'id'           => 'ideal-simulator-ideal-advanced-v3',
				'name'         => 'iDEAL Simulator - iDEAL Professional / Advanced',
				'mode'         => 'test',
				'provider'     => 'ideal-simulator',
				'product_url'  => 'https://www.ideal-checkout.nl/support/ideal-simulator',
				'acquirer_url' => 'https://www.ideal-checkout.nl/simulator/',
				'certificates' => [
					__DIR__ . '/certificates/ideal-checkout-2019-02-27-2024-02-26.cer',
				],
			]);
		}

		// ING - iDEAL Advanced - New platform - Production.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/ideal-advanced-v3/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\IDealAdvancedV3\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\IDealAdvancedV3\Integration::class, [
				'id'            => 'ing-ideal-advanced-2022-production',
				'name'          => 'ING - iDEAL Advanced - New platform - Production',
				'mode'          => 'live',
				'provider'      => 'ing',
				'product_url'   => 'https://www.ing.nl/zakelijk/betalen/geld-ontvangen/ideal/',
				'manual_url'    => 'https://www.wpaymepay.com/en/manuals/how-to-connect-ing-ideal-advanced-v3-to-wordpress-with-wpayme-pay/',
				'dashboard_url' => 'https://ideal-portal.ing.nl/',
				'acquirer_url'  => 'https://ideal-acquiring.ing.nl/ideal/iDEALv3',
				'certificates'  => [
					__DIR__ . '/certificates/ing-new-2020-03-04-2025-01-17.cer',
				],
			]);

			// ING - iDEAL Advanced - New platform - Sandbox.
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\IDealAdvancedV3\Integration::class, [
				'id'            => 'ing-ideal-advanced-2022-sandbox',
				'name'          => 'ING - iDEAL Advanced - New platform - Sandbox',
				'mode'          => 'test',
				'provider'      => 'ing',
				'product_url'   => 'https://www.ing.nl/zakelijk/betalen/geld-ontvangen/ideal/',
				'manual_url'    => 'https://www.wpaymepay.com/en/manuals/how-to-connect-ing-ideal-advanced-v3-to-wordpress-with-wpayme-pay/',
				'dashboard_url' => 'https://sandbox.ideal-portal.ing.nl/',
				'acquirer_url'  => 'https://sandbox.ideal-acquiring.ing.nl/ideal/iDEALv3',
				'certificates'  => [
					__DIR__ . '/certificates/ing-new-sandbox-2020-03-04-2025-01-17.cer',
				],
			]);
		}

		// ING Checkout.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/pay-nl/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\PayNL\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\PayNL\Integration::class, [
				'id'       => 'ing-checkout',
				'name'     => 'ING - ING Checkout',
				'mode'     => 'live',
				'provider' => 'ing',
			]);

			$add_gateway( \Wpayme\WordPress\Pay\Gateways\PayNL\Integration::class, [
				'id'       => 'ing-checkout-test',
				'name'     => 'ING - ING Checkout - Test',
				'mode'     => 'test',
				'provider' => 'ing',
			]);
		}

		// MultiSafePay - Connect.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/multisafepay/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\MultiSafepay\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\MultiSafepay\Integration::class, [
				'id'      => 'multisafepay-connect',
				'name'    => 'MultiSafepay - Connect',
				'mode'    => 'live',
				'api_url' => 'https://api.multisafepay.com/ewx/',
			]);

			$add_gateway( \Wpayme\WordPress\Pay\Gateways\MultiSafepay\Integration::class, [
				'id'      => 'multisafepay-connect-test',
				'name'    => 'MultiSafepay - Connect - Test',
				'mode'    => 'test',
				'api_url' => 'https://testapi.multisafepay.com/ewx/',
			]);
		}

		// Ingenico.
		$is_utf8 = strcasecmp( get_bloginfo( 'charset' ), 'UTF-8' ) === 0;

		// Ingenico - OrderStandard.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/ogone/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\Ingenico\OrderStandard\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\Ingenico\OrderStandard\Integration::class, [
				'id'               => 'ingenico-orderstandard',
				'name'             => 'Ingenico/Ogone - e-Commerce',
				'mode'             => 'live',
				'action_url'       => $is_utf8 ? 'https://secure.ogone.com/ncol/prod/orderstandard_utf8.asp' : 'https://secure.ogone.com/ncol/prod/orderstandard.asp',
				'direct_query_url' => 'https://secure.ogone.com/ncol/prod/querydirect.asp',
			]);

			$add_gateway( \Wpayme\WordPress\Pay\Gateways\Ingenico\OrderStandard\Integration::class, [
				'id'               => 'ingenico-orderstandard-test',
				'name'             => 'Ingenico/Ogone - e-Commerce - Test',
				'mode'             => 'test',
				'action_url'       => $is_utf8 ? 'https://secure.ogone.com/ncol/test/orderstandard_utf8.asp' : 'https://secure.ogone.com/ncol/test/orderstandard.asp',
				'direct_query_url' => 'https://secure.ogone.com/ncol/test/querydirect.asp',
			]);
		}

		// Rabobank - Rabo Smart Pay.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/omnikassa-2/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\OmniKassa2\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\OmniKassa2\Integration::class, [
				'id'      => 'rabobank-omnikassa-2',
				'name'    => 'Rabobank - Rabo Smart Pay',
				'mode'    => 'live',
				'api_url' => 'https://betalen.rabobank.nl/omnikassa-api/',
			]);

			$add_gateway( \Wpayme\WordPress\Pay\Gateways\OmniKassa2\Integration::class, [
				'id'      => 'rabobank-omnikassa-2-sandbox',
				'name'    => 'Rabobank - Rabo Smart Pay - Sandbox',
				'mode'    => 'test',
				'api_url' => 'https://betalen.rabobank.nl/omnikassa-api-sandbox/',
			]);
		}

		// PayPal.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/paypal/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\PayPal\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\PayPal\Integration::class, [
				'id'         => 'paypal',
				'name'       => 'PayPal',
				'mode'       => 'live',
				'webscr_url' => 'https://www.paypal.com/cgi-bin/webscr',
				'ipn_pb_url' => 'https://ipnpb.paypal.com/cgi-bin/webscr',
			]);

			$add_gateway( \Wpayme\WordPress\Pay\Gateways\PayPal\Integration::class, [
				'id'         => 'paypal-sandbox',
				'name'       => 'PayPal - Sandbox',
				'mode'       => 'test',
				'webscr_url' => 'https://www.sandbox.paypal.com/cgi-bin/webscr',
				'ipn_pb_url' => 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr',
			]);
		}

		// Pay.nl.
		if ( file_exists( __DIR__ . '/packages/wp-pay-gateways/pay-nl/src/Integration.php' ) && class_exists( '\Wpayme\WordPress\Pay\Gateways\PayNL\Integration' ) ) {
			$add_gateway( \Wpayme\WordPress\Pay\Gateways\PayNL\Integration::class );
		}

		return $gateways;
	}
);
