<?php
/**
 * Form Post Type
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay\Forms
 */

namespace Wpayme\WordPress\Pay\Forms;

use Wpayme\WordPress\Money\Money;
use Wpayme\WordPress\Money\Parser as MoneyParser;
use Wpayme\WordPress\Pay\Plugin;
use WP_Post;

/**
 * Form Post Type
 *
 * @author  Remco Tolsma
 * @version 2.7.1
 * @since   1.0.0
 */
class FormPostType {
	/**
	 * Post type.
	 *
	 * @var string
	 */
	const POST_TYPE = 'wpayme_pay_form';

	/**
	 * Amount method input fixed.
	 *
	 * @var string
	 */
	const AMOUNT_METHOD_INPUT_FIXED = 'fixed';

	/**
	 * Amount method input only.
	 *
	 * @var string
	 */
	const AMOUNT_METHOD_INPUT_ONLY = 'input_only';

	/**
	 * Amount method choices only.
	 *
	 * @var string
	 */
	const AMOUNT_METHOD_CHOICES_ONLY = 'choices_only';

	/**
	 * Amount method choices and input.
	 *
	 * @var string
	 */
	const AMOUNT_METHOD_CHOICES_AND_INPUT = 'choices_and_input';

	/**
	 * Construct form post type object.
	 */
	public function __construct() {
		/**
		 * Priority of the initial post types function should be set to < 10.
		 *
		 * @link https://core.trac.wordpress.org/ticket/28488.
		 * @link https://core.trac.wordpress.org/changeset/29318.
		 *
		 * @link https://github.com/WordPress/WordPress/blob/4.0/wp-includes/post.php#L167.
		 */
		add_action( 'init', [ $this, 'register_post_type' ], 0 ); // Highest priority.

		add_filter( 'manage_edit-' . self::POST_TYPE . '_columns', [ $this, 'edit_columns' ] );

		add_action( 'manage_' . self::POST_TYPE . '_posts_custom_column', [ $this, 'custom_columns' ], 10, 2 );

		/*
		 * Add meta box, we use priority 9 to make sure it loads before Yoast SEO meta box.
		 * @link https://github.com/Yoast/wordpress-seo/blob/2.3.4/admin/class-metabox.php#L20.
		 */
		add_action( 'add_meta_boxes', [ $this, 'add_meta_boxes' ], 9 );

		add_action( 'save_post_' . self::POST_TYPE, [ $this, 'save_post' ] );

		add_action( 'post_submitbox_misc_actions', [ $this, 'post_submitbox_misc_actions' ] );

		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );

		// Add action to enqueue frontend styles
		add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_styles'));
		
		// Add AJAX handler for getting payment methods
		add_action('wp_ajax_wpayme_get_gateway_payment_methods', array($this, 'ajax_get_gateway_payment_methods'));
	}

	/**
	 * Register post type.
	 *
	 * @return void
	 */
	public function register_post_type() {
		register_post_type(
			self::POST_TYPE,
			[
				'label'              => __( 'Payment Forms', 'wpayme' ),
				'labels'             => [
					'name'                     => __( 'Payment Forms', 'wpayme' ),
					'singular_name'            => __( 'Payment Form', 'wpayme' ),
					'add_new'                  => __( 'Add New', 'wpayme' ),
					'add_new_item'             => __( 'Add New Payment Form', 'wpayme' ),
					'edit_item'                => __( 'Edit Payment Form', 'wpayme' ),
					'new_item'                 => __( 'New Payment Form', 'wpayme' ),
					'all_items'                => __( 'All Payment Forms', 'wpayme' ),
					'view_item'                => __( 'View Payment Form', 'wpayme' ),
					'search_items'             => __( 'Search Payment Forms', 'wpayme' ),
					'not_found'                => __( 'No payment forms found.', 'wpayme' ),
					'not_found_in_trash'       => __( 'No payment forms found in Trash.', 'wpayme' ),
					'menu_name'                => __( 'Payment Forms', 'wpayme' ),
					'filter_items_list'        => __( 'Filter payment forms list', 'wpayme' ),
					'items_list_navigation'    => __( 'Payment forms list navigation', 'wpayme' ),
					'items_list'               => __( 'Payment forms list', 'wpayme' ),

					/*
					 * New Post Type Labels in 5.0.
					 * @link https://make.wordpress.org/core/2018/12/05/new-post-type-labels-in-5-0/
					 */
					'item_published'           => __( 'Payment form published.', 'wpayme' ),
					'item_published_privately' => __( 'Payment form published privately.', 'wpayme' ),
					'item_reverted_to_draft'   => __( 'Payment form reverted to draft.', 'wpayme' ),
					'item_scheduled'           => __( 'Payment form scheduled.', 'wpayme' ),
					'item_updated'             => __( 'Payment form updated.', 'wpayme' ),
				],
				'public'             => true,
				'publicly_queryable' => true,
				'show_ui'            => true,
				'show_in_nav_menus'  => true,
				'show_in_menu'       => false,
				'show_in_admin_bar'  => false,
				'supports'           => [
					'title',
					'revisions',
				],
				'rewrite'            => [
					'slug' => _x( 'payment-forms', 'slug', 'wpayme' ),
				],
				'query_var'          => false,
				'capabilities'       => self::get_capabilities(),
				'map_meta_cap'       => true,
			]
		);
	}

	/**
	 * Edit columns.
	 *
	 * @param array $columns Edit columns.
	 * @return array
	 */
	public function edit_columns( $columns ) {
		$columns = [
			'cb'                              => '<input type="checkbox" />',
			'wpayme_form_thumbnail'           => __( 'Image', 'wpayme' ),
			'title'                           => __( 'Title', 'wpayme' ),
			'wpayme_payment_form_id'          => __( 'ID', 'wpayme' ),
			'wpayme_payment_form_gateway'   => __( 'Gateway', 'wpayme' ),
			'wpayme_payment_form_payments'  => __( 'Payments', 'wpayme' ),
			'wpayme_payment_form_earnings'  => __( 'Earnings', 'wpayme' ),
			'wpayme_payment_form_shortcode' => __( 'Shortcode', 'wpayme' ),
			'date'                            => __( 'Date', 'wpayme' ),
		];

		return $columns;
	}

	/**
	 * Custom columns.
	 *
	 * @param string $column  Column.
	 * @param int    $post_id Post ID.
	 * @return void
	 */
	public function custom_columns( $column, $post_id ) {
		global $post;
		global $wpdb;

		switch ( $column ) {
			case 'wpayme_form_thumbnail':
				$thumbnail_id = get_post_meta( $post_id, '_wpayme_product_thumbnail', true );
				if ( $thumbnail_id ) {
					echo wp_get_attachment_image( $thumbnail_id, [40, 40], false, ['class' => 'wpayme-admin-thumbnail'] );
				} else {
					// Display a default placeholder image
					echo '<img src="data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHZpZXdCb3g9IjAgMCA0MCA0MCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjQwIiBoZWlnaHQ9IjQwIiByeD0iNCIgZmlsbD0iI0YyRjJGMiIvPgo8cGF0aCBkPSJNMjAgMTBDMTYuMTMgMTAgMTMgMTMuMTMgMTMgMTdDMTMgMjAuMzMgMTUuNDMgMjMuMSAxOC41IDIzLjlWMjZIMTVWMjlIMTguNVYzMUgyMS41VjI5SDI1VjI2SDIxLjVWMjMuOUMyNC41NyAyMy4xIDI3IDIwLjMzIDI3IDE3QzI3IDEzLjEzIDIzLjg3IDEwIDIwIDEwWk0yMCAyMUMxNy4yNCwyMSAxNSwxOC43NiAxNSwxN0MxNSwxNS4yNCAxNy4yNCwxMyAyMCwxM0MyMi43NiwxMyAyNSwxNS4yNCAyNSwxN0MyNSwxOC43NiAyMi43NiwyMSAyMCwyMVoiIGZpbGw9IiM2NjY2NjYiLz4KPC9zdmc+" class="wpayme-admin-thumbnail" alt="" />';
				}
				break;
			case 'wpayme_payment_form_id':
				echo '<strong>' . esc_html( $post_id ) . '</strong>';
				break;
			case 'wpayme_payment_form_gateway':
				$config_id = get_post_meta( $post_id, '_wpayme_payment_form_config_id', true );

				if ( ! empty( $config_id ) ) {
					echo esc_html( get_the_title( $config_id ) );
				} else {
					echo '—';
				}

				break;
			case 'wpayme_payment_form_payments':
				/* phpcs:ignore WordPress.DB.DirectDatabaseQuery */
				$value = $wpdb->get_var(
					$wpdb->prepare(
						"
						SELECT
							COUNT( post.ID ) AS value
						FROM
							$wpdb->posts AS post
								LEFT JOIN
							$wpdb->postmeta AS meta_amount
									ON post.ID = meta_amount.post_id AND meta_amount.meta_key = '_wpayme_payment_amount'
								LEFT JOIN
							$wpdb->postmeta AS meta_source
									ON post.ID = meta_source.post_id AND meta_source.meta_key = '_wpayme_payment_source'
								LEFT JOIN
							$wpdb->postmeta AS meta_source_id
									ON post.ID = meta_source_id.post_id AND meta_source_id.meta_key = '_wpayme_payment_source_id'
						WHERE
							post.post_type = 'wpayme_payment'
								AND
							post.post_status = 'payment_completed'
								AND
							meta_source.meta_value = 'payment_form'
								AND
							meta_source_id.meta_value = %s
						GROUP BY
							post.ID
						;
						",
						$post_id
					)
				);

				echo esc_html( number_format_i18n( (int) $value ) );

				break;
			case 'wpayme_payment_form_earnings':
				/* phpcs:ignore WordPress.DB.DirectDatabaseQuery */
				$value = $wpdb->get_var(
					$wpdb->prepare(
						"
						SELECT
							SUM( meta_amount.meta_value ) AS value
						FROM
							$wpdb->posts AS post
								LEFT JOIN
							$wpdb->postmeta AS meta_amount
									ON post.ID = meta_amount.post_id AND meta_amount.meta_key = '_wpayme_payment_amount'
								LEFT JOIN
							$wpdb->postmeta AS meta_source
									ON post.ID = meta_source.post_id AND meta_source.meta_key = '_wpayme_payment_source'
								LEFT JOIN
							$wpdb->postmeta AS meta_source_id
									ON post.ID = meta_source_id.post_id AND meta_source_id.meta_key = '_wpayme_payment_source_id'
						WHERE
							post.post_type = 'wpayme_payment'
								AND
							post.post_status = 'payment_completed'
								AND
							meta_source.meta_value = 'payment_form'
								AND
							meta_source_id.meta_value = %s
						GROUP BY
							post.ID
						;
						",
						$post_id
					)
				);

				if ( empty( $value ) ) {
					$value = 0;
				}

				$money = new Money( $value, 'EUR' );

				echo esc_html( $money->format_i18n() );

				break;
			case 'wpayme_payment_form_shortcode':
				printf(
					'<input onclick="this.setSelectionRange( 0, this.value.length )" type="text" class="wpayme-pay-shortcode-input" readonly="" value="%s" />',
					esc_attr( $this->get_shortcode( $post_id ) )
				);

				break;
		}
	}

	/**
	 * Add meta boxes.
	 *
	 * @param string $post_type Post type.
	 * @return void
	 */
	public function add_meta_boxes($post_type) {
		if ($post_type !== self::POST_TYPE) {
			return;
		}
		
		add_meta_box(
			'wpayme_payment_form_options',
			__('Payment Form Options', 'wpayme'),
			[$this, 'meta_box_form_options'],
			self::POST_TYPE,
			'normal',
			'high'
		);

		add_meta_box(
			'wpayme_payment_form_shortcode',
			__('Shortcode', 'wpayme'),
			array($this, 'meta_box_form_shortcode'),
			self::POST_TYPE,
			'side',
			'default'
		);
	}

	/**
	 * Form options meta box.
	 *
	 * @param WP_Post $post The object for the current post/page.
	 */
	public function meta_box_form_options( $post ) {
		wp_nonce_field( 'wpayme_pay_save_form_options', 'wpayme_pay_nonce' );

		include __DIR__ . '/../views/meta-box-form-options.php';
	}
	
	/**
	 * Form shortcode meta box.
	 *
	 * @param WP_Post $post The object for the current post/page.
	 */
	public function meta_box_form_shortcode($post) {
		printf(
			'<p>%s</p><pre>[wpayme_payment_form id="%s"]</pre>', 
			esc_html__('Copy this shortcode and paste it into your post, page, or text widget content:', 'wpayme'),
			esc_attr($post->ID)
		);
	}

	/**
	 * Save post.
	 *
	 * @param int $post_id Post ID.
	 * @return void
	 */
	public function save_post( $post_id ) {
		// Check if our nonce is set.
		if ( ! \array_key_exists( 'wpayme_pay_nonce', $_POST ) ) {
		// Debugging note
		// error_log('Nonce field wpayme_pay_nonce is missing');
		return;
	}

	$nonce = \sanitize_text_field( \wp_unslash( $_POST['wpayme_pay_nonce'] ) );

	// Verify that the nonce is valid.
	if ( ! wp_verify_nonce( $nonce, 'wpayme_pay_save_form_options' ) ) {
		// Debugging note
		// error_log('Nonce verification failed');
		return;
	}

	// If this is an autosave, our form has not been submitted, so we don't want to do anything.
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		// Debugging note
		// error_log('Skipping save during autosave');
		return;
	}

	// Check the user's permissions.
	if ( ! current_user_can( 'edit_post', $post_id ) ) {
		// Debugging note
		// error_log('User lacks permission to edit post');
		return;
	}

	// OK, its safe for us to save the data now.
	$definition = [
		// General.
		'_wpayme_payment_form_config_id'      => FILTER_SANITIZE_NUMBER_INT,
	];

	$data = \filter_input_array( INPUT_POST, $definition );

	if ( ! \is_array( $data ) ) {
		// Debugging note
		// error_log('filter_input_array did not return an array. Raw POST data: ' . json_encode($_POST));
		// Initialize data array to avoid errors
		$data = [];
	}

	// Debugging - log the amount choices array
	/*
	if (isset($_POST['_wpayme_payment_form_amount_choices'])) {
		error_log('Amount choices in POST: ' . json_encode($_POST['_wpayme_payment_form_amount_choices']));
	} else {
		error_log('Amount choices not found in POST');
	}
	*/

	if ( \array_key_exists( '_wpayme_payment_form_button_text', $_POST ) ) {
		$data['_wpayme_payment_form_button_text'] = \sanitize_text_field( \wp_unslash( $_POST['_wpayme_payment_form_button_text'] ) );
	}

	if ( \array_key_exists( '_wpayme_payment_form_description', $_POST ) ) {
		$data['_wpayme_payment_form_description'] = \sanitize_text_field( \wp_unslash( $_POST['_wpayme_payment_form_description'] ) );
	}

	if ( \array_key_exists( '_wpayme_payment_form_amount_method', $_POST ) ) {
		$data['_wpayme_payment_form_amount_method'] = \sanitize_text_field( \wp_unslash( $_POST['_wpayme_payment_form_amount_method'] ) );
	} else {
		// Set default to fixed_price if not provided
		$data['_wpayme_payment_form_amount_method'] = 'fixed_price';
	}

	// Save fixed price
	if (isset($_POST['wpayme_fixed_price'])) {
		$fixed_price = filter_var( wp_unslash( $_POST['wpayme_fixed_price'] ), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION );
		$fixed_price = (float) $fixed_price;
		update_post_meta($post_id, '_wpayme_fixed_price', $fixed_price);
	}

	// Save amount choices
	$amount_choices = [];
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitization happens in the loop below.
	$posted_choices = isset($_POST['_wpayme_payment_form_amount_choices']) ? \wp_unslash($_POST['_wpayme_payment_form_amount_choices']) : [];

	// Log the raw choices data for debugging
	// error_log('Raw amount choices from POST: ' . json_encode($posted_choices));

	if (is_array($posted_choices)) {
		foreach ($posted_choices as $choice) {
			// Skip empty values and sanitize before casting to float
			if (!empty($choice)) {
				$sanitized_choice = filter_var( $choice, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION );
				$amount_choices[] = (float) $sanitized_choice;
			}
		}
	}

	// Log the processed choices
	// error_log('Processed amount choices: ' . json_encode($amount_choices));

	// Check if we should clear existing choices (when switching from choices to another method)
	// Or if we should keep an empty field (when all fields are cleared but we still use choices)
	$clear_choices_value = isset($_POST['_wpayme_payment_form_clear_choices']) ? sanitize_text_field( wp_unslash( $_POST['_wpayme_payment_form_clear_choices'] ) ) : '';
	$clear_choices = $clear_choices_value === '1';
	$keep_empty_field = $clear_choices_value === 'keep_empty';
	
	// Only update if we have choices to save or if we're explicitly clearing them
	if (!empty($amount_choices)) {
		update_post_meta($post_id, '_wpayme_payment_form_amount_choices', $amount_choices);
		// error_log('Updated amount choices meta for post ' . $post_id);
	} else if ($clear_choices) {
		// Clear out all choices
		update_post_meta($post_id, '_wpayme_payment_form_amount_choices', []);
		// error_log('Cleared all amount choices for post ' . $post_id);
	} else if ($keep_empty_field) {
		// Keep a single empty field - always use an empty string, not any previous value
		update_post_meta($post_id, '_wpayme_payment_form_amount_choices', array(''));
		// error_log('Keeping one empty amount choice field for post ' . $post_id);
	} else {
		// Check the amount method - if it requires choices but none were saved, something went wrong
		$amount_method = isset($_POST['_wpayme_payment_form_amount_method']) ? 
			sanitize_text_field(wp_unslash($_POST['_wpayme_payment_form_amount_method'])) : '';
		
		if ($amount_method === 'choices_only' || $amount_method === 'choices_and_input') {
			// Get existing choices
			$existing_choices = get_post_meta($post_id, '_wpayme_payment_form_amount_choices', true);
			// error_log('No new choices saved. Using existing choices: ' . json_encode($existing_choices));
		}
	}

	// Save field visibility settings
	$form_fields = ['first_name', 'last_name', 'email', 'quantity', 'comment'];
	$field_visibility = [];
	
	foreach ($form_fields as $field) {
		$field_name = 'wpayme_field_' . $field . '_visibility';
		if (isset($_POST[$field_name])) {
			$field_visibility[$field] = sanitize_text_field(\wp_unslash($_POST[$field_name]));
		}
	}
	
	update_post_meta($post_id, '_wpayme_field_visibility', $field_visibility);
	
	// Save field requirements
	$required_fields = [];
	
	foreach ($form_fields as $field) {
		$field_name = 'wpayme_field_' . $field . '_required';
		if (isset($_POST[$field_name])) {
			$field_value = sanitize_text_field( wp_unslash( $_POST[$field_name] ) );
			if ($field_value === 'required') {
				$required_fields[] = sanitize_key( $field );
			}
		}
	}
	
	update_post_meta($post_id, '_wpayme_required_fields', $required_fields);

	// Save product description if available
	if (isset($_POST['wpayme_product_description'])) {
		update_post_meta($post_id, '_wpayme_product_description', wp_kses_post(\wp_unslash($_POST['wpayme_product_description'])));
	}

	// Save shipping fee
	if (isset($_POST['wpayme_shipping_fee'])) {
		$shipping_fee = filter_var( wp_unslash( $_POST['wpayme_shipping_fee'] ), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION );
		update_post_meta($post_id, '_wpayme_shipping_fee', (float) $shipping_fee);
	}
	if (isset($_POST['wpayme_tax_rate'])) {
		$tax_rate = filter_var( wp_unslash( $_POST['wpayme_tax_rate'] ), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION );
		update_post_meta($post_id, '_wpayme_tax_rate', (float) $tax_rate);
	}

	// Save thank you page
	if (isset($_POST['wpayme_thank_you_page'])) {
		update_post_meta($post_id, '_wpayme_thank_you_page', absint($_POST['wpayme_thank_you_page']));
	}

	// Save button appearance
	if (isset($_POST['wpayme_button_text'])) {
		update_post_meta($post_id, '_wpayme_button_text', sanitize_text_field(\wp_unslash($_POST['wpayme_button_text'])));
	}
	
	// Save button CSS class
	if (isset($_POST['wpayme_button_class'])) {
		update_post_meta($post_id, '_wpayme_button_class', sanitize_text_field(\wp_unslash($_POST['wpayme_button_class'])));
	}
	
	// Save button text color
	if (isset($_POST['wpayme_button_text_color'])) {
		update_post_meta($post_id, '_wpayme_button_text_color', sanitize_hex_color(\wp_unslash($_POST['wpayme_button_text_color'])));
	}

	if (isset($_POST['wpayme_button_color'])) {
		update_post_meta($post_id, '_wpayme_button_color', sanitize_hex_color(\wp_unslash($_POST['wpayme_button_color'])));
	}

	// Save surcharge settings
	if (isset($_POST['wpayme_surcharge_type'])) {
		update_post_meta($post_id, '_wpayme_surcharge_type', 
			sanitize_text_field(\wp_unslash($_POST['wpayme_surcharge_type']))
		);
	}

	if (isset($_POST['wpayme_surcharge_amount'])) {
		$surcharge_amount = filter_var( wp_unslash( $_POST['wpayme_surcharge_amount'] ), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION );
		update_post_meta($post_id, '_wpayme_surcharge_amount', (float) $surcharge_amount);
	}

	if (isset($_POST['wpayme_surcharge_label'])) {
		update_post_meta($post_id, '_wpayme_surcharge_label', 
			sanitize_text_field(\wp_unslash($_POST['wpayme_surcharge_label']))
		);
	}

	// Save payment type
	if (isset($_POST['wpayme_payment_type'])) {
		update_post_meta($post_id, '_wpayme_payment_type', 
			sanitize_text_field(\wp_unslash($_POST['wpayme_payment_type']))
		);
	}

	// Save currency
	if (isset($_POST['wpayme_currency'])) {
		update_post_meta($post_id, '_wpayme_currency', 
			sanitize_text_field(\wp_unslash($_POST['wpayme_currency']))
		);
	}
	
	// Save payment variants
	$variants = [];
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitization happens in the loop below.
	$posted_variants = isset($_POST['wpayme_payment_variants']) ? \wp_unslash($_POST['wpayme_payment_variants']) : [];
	
	if (is_array($posted_variants)) {
		foreach ($posted_variants as $variant) {
			if (isset($variant['name']) && !empty($variant['name'])) {
				$price = 0;
				if (isset($variant['price'])) {
					$price = filter_var( $variant['price'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION );
					$price = (float) $price;
				}
				$variants[] = [
					'name' => sanitize_text_field($variant['name']),
					'price' => $price,
					'description' => isset($variant['description']) ? sanitize_text_field($variant['description']) : '',
				];
			}
		}
	}
	
	update_post_meta($post_id, '_wpayme_payment_variants', $variants);
		
		// Set updated meta data.
		foreach ( $data as $key => $value ) {
			\update_post_meta( $post_id, $key, $value );
		}
	}

	/**
	 * Get shortcode of the specified form post ID.
	 *
	 * @param int|null $post_id Post ID.
	 * @return string
	 */
	private function get_shortcode( $post_id = null ) {
		$post_id = ( null === $post_id ) ? get_the_ID() : $post_id;

		$shortcode = sprintf( '[wpayme_payment_form id="%s"]', esc_attr( strval( $post_id ) ) );

		return $shortcode;
	}

	/**
	 * Post submit box miscellaneous actions.
	 *
	 * @link https://github.com/WordPress/WordPress/blob/5.3/wp-admin/includes/meta-boxes.php#L283-L293
	 * @return void
	 */
	public function post_submitbox_misc_actions() {
		if ( self::POST_TYPE !== get_post_type() ) {
			return;
		}

		?>

		<?php
	}

	/**
	 * Get capabilities for this post type.
	 *
	 * @return array
	 */
	public static function get_capabilities() {
		return [
			'edit_post'              => 'edit_form',
			'read_post'              => 'read_form',
			'delete_post'            => 'delete_form',
			'edit_posts'             => 'edit_forms',
			'edit_others_posts'      => 'edit_others_forms',
			'publish_posts'          => 'publish_forms',
			'read_private_posts'     => 'read_private_forms',
			'read'                   => 'read',
			'delete_posts'           => 'delete_forms',
			'delete_private_posts'   => 'delete_private_forms',
			'delete_published_posts' => 'delete_published_forms',
			'delete_others_posts'    => 'delete_others_forms',
			'edit_private_posts'     => 'edit_private_forms',
			'edit_published_posts'   => 'edit_published_forms',
			'create_posts'           => 'create_forms',
		];
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @param string $hook The current admin page.
	 * @return void
	 */
	public function enqueue_admin_scripts($hook) {
		global $post_type;
		
		if ('wpayme_pay_form' !== $post_type) {
			return;
		}

		// Get current post ID
		$post_id = get_the_ID();
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if (!$post_id && isset($_GET['post'])) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$post_id = absint($_GET['post']);
		}
		
		// Ensure WordPress rich editor scripts and styles are loaded
		wp_enqueue_editor();

		// Enqueue media uploader
		wp_enqueue_media();
		
		// Enqueue color picker
		wp_enqueue_style('wp-color-picker');
		wp_enqueue_script('wp-color-picker');
		
		// Ensure dashicons are enqueued
		wp_enqueue_style('dashicons');
		
		// Enqueue jQuery UI for dialog
		wp_enqueue_style('wp-jquery-ui-dialog');
		wp_enqueue_script('jquery-ui-dialog');
		
		// Get plugin version
		$version = defined('WPAYME_PAY_VERSION') ? WPAYME_PAY_VERSION : '1.0.0';
		
		// Enqueue admin styles
		wp_enqueue_style(
			'wpayme-form-admin-style',
			plugins_url('assets/css/admin-style.css', dirname(__FILE__)),
			array(),
			filemtime(plugin_dir_path(dirname(__FILE__)) . 'assets/css/admin-style.css')
		);
		
		// Enqueue payment methods styles
		wp_enqueue_style(
			'wpayme-payment-methods-style',
			plugins_url('css/admin-payment-methods.css', dirname(__FILE__)),
			array('dashicons', 'wp-jquery-ui-dialog'),
			$version
		);
		
		// Enqueue admin scripts
		wp_enqueue_script(
			'wpayme-form-admin',
			plugins_url('assets/js/form-admin.js', dirname(__FILE__)),
			array('jquery', 'wp-color-picker'),
			$version,
			true
		);
		
		// Add localization data
		wp_localize_script('wpayme-form-admin', 'wpaymeL10n', array(
			'fixedPriceLabel' => __('Fixed Price Amount', 'wpayme'),
			'enterFixedPrice' => __('Enter fixed price', 'wpayme'),
			'amountChoices' => __('Amount Choices', 'wpayme'),
			'addChoice' => __('Add Amount Choice', 'wpayme'),
			'enterAmount' => __('Enter amount', 'wpayme'),
		));
		
		// Add data
		wp_localize_script('wpayme-form-admin', 'wpaymeData', array(
			'fixedPrice' => $post_id ? get_post_meta($post_id, '_wpayme_fixed_price', true) : '',
			'amountChoices' => $post_id ? $this->get_amount_choices($post_id) : array(),
		));
		
		// Enqueue payment methods script
		wp_enqueue_script(
			'wpayme-payment-methods-script',
			plugins_url('js/admin-payment-methods.js', dirname(__FILE__)),
			array('jquery', 'jquery-ui-dialog'),
			$version,
			true
		);
		
		// Localize script with AJAX URL and nonce
		wp_localize_script(
			'wpayme-payment-methods-script',
			'wpaymePayFormsAdmin',
			array(
				'ajaxUrl' => admin_url('admin-ajax.php'),
				'nonce' => wp_create_nonce('wpayme_pay_forms_admin'),
			)
		);
	}
	
	/**
	 * Get formatted amount choices for a form
	 *
	 * @param int $post_id Post ID
	 * @return array Formatted amount choices
	 */
	private function get_amount_choices($post_id) {
		$choices = get_post_meta($post_id, '_wpayme_payment_form_amount_choices', true);
		
		// Ensure choices is an array
		if (!is_array($choices)) {
			return array();
		}
		
		// Format the values for display
		$formatted_choices = array();
		
		foreach ($choices as $value) {
			// Include all values including empty strings (to retain the empty field)
			$formatted_choices[] = $value;
		}
		
		return $formatted_choices;
	}

	public function enqueue_frontend_styles() {
		// Enqueue the form styles
		wp_enqueue_style(
			'wpayme-pay-form-styles',
			plugins_url('assets/css/form-style.css', dirname(__FILE__)),
			array(),
			filemtime(plugin_dir_path(dirname(__FILE__)) . 'assets/css/form-style.css')
		);
		
		// Enqueue payment methods styles
		wp_enqueue_style(
			'wpayme-pay-frontend-payment-methods',
			plugins_url('css/frontend-payment-methods.css', dirname(__FILE__)),
			array(),
			defined('WPAYME_PAY_VERSION') ? WPAYME_PAY_VERSION : '1.0.0'
		);
		
		// Enqueue payment methods script
		wp_enqueue_script(
			'wpayme-pay-frontend-payment-methods',
			plugins_url('js/frontend-payment-methods.js', dirname(__FILE__)),
			array('jquery'),
			defined('WPAYME_PAY_VERSION') ? WPAYME_PAY_VERSION : '1.0.0',
			true
		);
		
		// Localize script with AJAX URL and nonce
		wp_localize_script(
			'wpayme-pay-frontend-payment-methods',
			'wpaymePayFrontend',
			array(
				'ajaxUrl' => admin_url('admin-ajax.php'),
				'nonce' => wp_create_nonce('wpayme_pay_forms_frontend'),
				'loadingText' => __('Loading payment methods...', 'wpayme'),
				'noMethodsText' => __('No payment methods available.', 'wpayme'),
				'errorText' => __('Error loading payment methods.', 'wpayme'),
				'recurringText' => __('Recurring', 'wpayme')
			)
		);
	}

	/**
	 * AJAX handler for getting payment methods for a gateway configuration
	 */
	public function ajax_get_gateway_payment_methods() {
		// Check nonce and permissions.
		$is_admin = false;
		$is_frontend = false;

		if ( isset( $_POST['nonce'] ) ) {
			$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ) );
			$is_admin = wp_verify_nonce( $nonce, 'wpayme_pay_forms_admin' );
			$is_frontend = wp_verify_nonce( $nonce, 'wpayme_pay_forms_frontend' );
		}

		// For admin requests, also check user capabilities.
		if ( $is_admin && ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => 'Insufficient permissions.' ) );
			return;
		}

		if ( ! $is_admin && ! $is_frontend ) {
			wp_send_json_error( array( 'message' => 'Security check failed.' ) );
			return;
		}

		// Check if config_id is provided and sanitize it.
		if ( ! isset( $_POST['config_id'] ) || empty( $_POST['config_id'] ) ) {
			wp_send_json_error( array( 'message' => 'No gateway configuration ID provided.' ) );
			return;
		}

		$config_id = intval( $_POST['config_id'] );
		
		// Get gateway
		$gateway = \Wpayme\WordPress\Pay\Plugin::get_gateway($config_id);
		
		if (null === $gateway) {
			wp_send_json_error(array('message' => 'Gateway configuration not found.'));
			return;
		}
		
		// Get payment methods
		$payment_methods = $gateway->get_payment_methods()->getIterator();
		
		// Sort payment methods by name
		$payment_methods->uasort(
			function ($a, $b) {
				return strnatcasecmp($a->get_name(), $b->get_name());
			}
		);
		
		// Format payment methods for JSON response
		$methods = array();
		
		foreach ($payment_methods as $payment_method) {
			$methods[] = array(
				'id' => $payment_method->get_id(),
				'name' => $payment_method->get_name(),
				'status' => $payment_method->get_status(),
				'supports_recurring' => $payment_method->supports('recurring'),
			);
		}
		
		wp_send_json_success(array('methods' => $methods));
	}
}
