<?php
// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound

/**
 * Meta Box Subscription Update
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay
 */

use Wpayme\WordPress\Html\Element;
use Wpayme\WordPress\Pay\Plugin;
use Wpayme\WordPress\Pay\Payments\PaymentStatus;
use Wpayme\WordPress\Pay\Subscriptions\SubscriptionPostType;
use Wpayme\WordPress\Pay\Subscriptions\SubscriptionStatus;
use Wpayme\WordPress\Pay\Util;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! isset( $post ) ) {
	return;
}

$wpayme_subscription = \get_wpayme_subscription( (int) get_the_ID() );

if ( null === $wpayme_subscription ) {
	return;
}

$wpayme_states = [
	SubscriptionStatus::OPEN      => _x( 'Pending', 'Subscription status', 'wpayme' ),
	SubscriptionStatus::CANCELLED => _x( 'Cancelled', 'Subscription status', 'wpayme' ),
	SubscriptionStatus::EXPIRED   => _x( 'Expired', 'Subscription status', 'wpayme' ),
	SubscriptionStatus::FAILURE   => _x( 'Failed', 'Subscription status', 'wpayme' ),
	SubscriptionStatus::ON_HOLD   => _x( 'On Hold', 'Subscription status', 'wpayme' ),
	SubscriptionStatus::ACTIVE    => _x( 'Active', 'Subscription status', 'wpayme' ),
	SubscriptionStatus::COMPLETED => _x( 'Completed', 'Subscription status', 'wpayme' ),
	// Map payment status `Success` for backwards compatibility.
	PaymentStatus::SUCCESS        => _x( 'Active', 'Subscription status', 'wpayme' ),
];

ksort( $wpayme_states );

$wpayme_states_options = [
	SubscriptionStatus::ACTIVE,
	SubscriptionStatus::CANCELLED,
	SubscriptionStatus::ON_HOLD,
];

// WordPress by default doesn't allow `post_author` values of `0`, that's why we use a dash (`-`).
// @link https://github.com/WordPress/WordPress/blob/4.9.5/wp-admin/includes/post.php#L56-L64.
$wpayme_post_author = get_post_field( 'post_author' );
$wpayme_post_author = empty( $wpayme_post_author ) ? '-' : $wpayme_post_author;


?>
<input type="hidden" name="post_author_override" value="<?php echo esc_attr( $wpayme_post_author ); ?>" />

<div class="wpayme-pay-inner">
	<div class="wpayme-pay-minor-actions">
		<div class="misc-pub-section misc-pub-post-status">
			<?php echo esc_html( __( 'Status:', 'wpayme' ) ); ?>

			<?php

			$wpayme_status_object = get_post_status_object( $post->post_status );

			$wpayme_status_label = isset( $wpayme_status_object, $wpayme_status_object->label ) ? $wpayme_status_object->label : '—';

			?>

			<span id="wpayme-pay-post-status-display"><?php echo esc_html( $wpayme_status_label ); ?></span>

			<?php if ( 'subscr_completed' !== $post->post_status ) : ?>

				<a href="#wpayme-pay-post-status" class="edit-wpayme-pay-post-status hide-if-no-js" role="button">
					<span aria-hidden="true"><?php esc_html_e( 'Edit', 'wpayme' ); ?></span>
					<span class="screen-reader-text"><?php esc_html_e( 'Edit status', 'wpayme' ); ?></span>
				</a>

				<div id="wpayme-pay-post-status-input" class="hide-if-js">
					<label for="wpayme-pay-post-status" class="screen-reader-text"><?php esc_html_e( 'Set status', 'wpayme' ); ?></label>
					<select id="wpayme-pay-post-status" name="wpayme_subscription_status">
						<?php

						foreach ( $wpayme_states as $subscription_status => $label ) {
							if ( ! in_array( $subscription_status, $wpayme_states_options, true ) && $subscription_status !== $wpayme_subscription->get_status() ) {
								continue;
							}

							printf(
								'<option value="%s" %s>%s</option>',
								esc_attr( $subscription_status ),
								selected( $subscription_status, $wpayme_subscription->get_status(), false ),
								esc_html( $label )
							);
						}

						?>
					</select>

					<a href="#wpayme-pay-post-status" class="save-wpayme-pay-post-status hide-if-no-js button"><?php esc_html_e( 'OK', 'wpayme' ); ?></a>
					<a href="#wpayme-pay-post-status" class="cancel-wpayme-pay-post-status hide-if-no-js button-cancel"><?php esc_html_e( 'Cancel', 'wpayme' ); ?></a>
				</div>

				<?php if ( null !== $wpayme_subscription && in_array( $wpayme_subscription->get_status(), [ SubscriptionStatus::FAILURE, SubscriptionStatus::ON_HOLD ], true ) ) : ?>

					<div id="wpayme-pay-post-status-notice" class="notice inline">
						<p>
							<?php

							echo \wp_kses_post(
								\sprintf(
									'%s <a href="#wpayme_subscription_notes">%s</a>',
									\__( 'Recurring payments will not be created until manual reactivation of this subscription.', 'wpayme' ),
									\__( 'See subscription and payment notes for details about status changes.', 'wpayme' )
								)
							);

							?>
						</p>
					</div>

				<?php endif; ?>

			<?php endif; ?>
		</div>

		<div class="misc-pub-section curtime">
			<?php

			$wpayme_next_payment_date = $wpayme_subscription->get_next_payment_date();

			?>

			<span id="timestamp">
				<?php echo esc_html( __( 'Next payment:', 'wpayme' ) ); ?>
			</span>

			<span id="wpayme-pay-next-payment-date-display"><?php echo esc_html( null === $wpayme_next_payment_date ? '—' : $wpayme_next_payment_date->format_i18n( \__( 'D j M Y', 'wpayme' ) ) ); ?></span>

			<?php if ( 'woocommerce' !== $wpayme_subscription->get_source() ) : ?>

				<a href="#wpayme-pay-next-payment-date" class="edit-wpayme-pay-next-payment-date hide-if-no-js" role="button">
					<span aria-hidden="true"><?php esc_html_e( 'Edit', 'wpayme' ); ?></span>
					<span class="screen-reader-text"><?php esc_html_e( 'Edit next payment date', 'wpayme' ); ?></span>
				</a>

			<?php endif; ?>

			<div id="wpayme-pay-next-payment-date-input" class="hide-if-js">
				<input type="hidden" name="hidden_wpayme_pay_next_payment_date" id="hidden_wpayme_pay_next_payment_date" value="<?php echo \esc_attr( null === $wpayme_next_payment_date ? '' : $wpayme_next_payment_date->format( 'Y-m-d' ) ); ?>" />
				<label for="wpayme-pay-next-payment-date" class="screen-reader-text"><?php esc_html_e( 'Set date', 'wpayme' ); ?></label>

				<?php

				$wpayme_element = new Element(
					'input',
					[
						'id'       => 'wpayme-pay-next-payment-date',
						'name'     => 'wpayme_subscription_next_payment_date',
						'type'     => 'date',
						'value'    => null === $wpayme_next_payment_date ? '' : $wpayme_next_payment_date->format( 'Y-m-d' ),
						'data-min' => ( new DateTimeImmutable( 'tomorrow' ) )->format( 'Y-m-d' ),
					]
				);

				$wpayme_element->output();

				?>

				<a href="#wpayme-pay-next-payment-date" class="save-wpayme-pay-next-payment-date hide-if-no-js button"><?php esc_html_e( 'OK', 'wpayme' ); ?></a>
				<a href="#wpayme-pay-next-payment-date" class="cancel-wpayme-pay-next-payment-date hide-if-no-js button-cancel"><?php esc_html_e( 'Cancel', 'wpayme' ); ?></a>
			</div>

			<?php

			$wpayme_today = new DateTimeImmutable( 'today midnight', new DateTimeZone( Plugin::TIMEZONE ) );

			if ( SubscriptionStatus::ACTIVE === $wpayme_subscription->get_status() && null !== $wpayme_next_payment_date && $wpayme_next_payment_date < $wpayme_today ) :
				?>

				<div id="wpayme-pay-next-payment-date-error" class="error inline">
					<p><?php echo esc_html( __( 'Set the next payment date to a future date to continue payments for this subscription.', 'wpayme' ) ); ?></p>
				</div>

			<?php endif; ?>

			<div id="wpayme-pay-next-payment-date-min-error" class="hidden error inline">
				<p><?php echo esc_html( __( 'Please select a future date.', 'wpayme' ) ); ?></p>
			</div>

			<div id="wpayme-pay-next-payment-date-notice" class="hidden notice inline">
				<p>
					<?php

					\printf(
						/* translators: %s subscription source description */
						\esc_html( \__( 'Editing the next payment date does not affect the current status or validity of %s.', 'wpayme' ) ),
						\wp_kses(
							$wpayme_subscription->get_source_text(),
							[
								'a' => [
									'href' => true,
								],
							]
						)
					);

					?>
				</p>
			</div>
		</div>
	</div>
</div>

<div class="wpayme-pay-major-actions">
	<div class="wpayme-pay-action">
		<?php

		wp_nonce_field( 'wpayme_subscription_update', 'wpayme_subscription_nonce' );

		printf(
			'<input type="hidden" name="wpayme_subscription_id" value="%s" />',
			esc_attr( (string) $wpayme_subscription->get_id() )
		);

		submit_button(
			__( 'Update', 'wpayme' ),
			'primary',
			'wpayme_subscription_update',
			false
		);

		?>
	</div>

	<div class="clear"></div>
</div>
