<?php
/**
 * Meta Box Subscription Info
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay
 * @var \Wpayme\WordPress\Pay\Plugin $plugin Plugin.
 * @var \Wpayme\WordPress\Pay\Subscriptions\Subscription $subscription Subscription.
 */

use Wpayme\WordPress\DateTime\DateTime;
use Wpayme\WordPress\Pay\Core\PaymentMethods;
use Wpayme\WordPress\Pay\Subscriptions\SubscriptionStatus;
use Wpayme\WordPress\Pay\Util;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$wpayme_subscription_id = $subscription->get_id();

$wpayme_customer = $subscription->get_customer();
$wpayme_user_id  = is_null( $wpayme_customer ) ? null : $wpayme_customer->get_user_id();

$wpayme_phase = $subscription->get_display_phase();

?>
<table class="form-table">
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Date', 'wpayme' ); ?>
		</th>
		<td>
			<?php the_time( __( 'l jS \o\f F Y, h:ia', 'wpayme' ) ); ?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'ID', 'wpayme' ); ?>
		</th>
		<td>
			<?php echo esc_html( (string) $subscription->get_id() ); ?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Description', 'wpayme' ); ?>
		</th>
		<td>
			<?php echo esc_html( (string) $subscription->get_description() ); ?>
		</td>
	</tr>

	<?php if ( null !== $subscription->config_id ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Gateway', 'wpayme' ); ?>
			</th>
			<td>
				<?php edit_post_link( get_the_title( $subscription->config_id ), '', '', $subscription->config_id ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Payment Method', 'wpayme' ); ?>
		</th>
		<td>
			<?php
			$wpayme_payment_method = $subscription->get_payment_method();

			// Icon.
			$wpayme_icon_url = PaymentMethods::get_icon_url( $wpayme_payment_method );

			if ( null !== $wpayme_icon_url ) {
				\printf(
					'<span class="wpayme-pay-tip" title="%2$s"><img src="%1$s" alt="%2$s" title="%2$s" width="32" valign="bottom" /></span> ',
					\esc_url( $wpayme_icon_url ),
					\esc_attr( (string) PaymentMethods::get_name( $wpayme_payment_method ) )
				);
			}

			// Name.
			echo esc_html( (string) PaymentMethods::get_name( $wpayme_payment_method ) );

			?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Amount', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			if ( null !== $wpayme_phase ) :

				echo esc_html( $wpayme_phase->get_amount()->format_i18n() );

			endif;

			?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php echo esc_html__( 'Recurrence', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$wpayme_total_periods = ( null === $wpayme_phase ) ? null : $wpayme_phase->get_total_periods();

			if ( null === $wpayme_phase || 1 === $wpayme_total_periods ) {
				// No recurrence.
				echo '—';
			} elseif ( null === $wpayme_total_periods ) {
				// Infinite.
				echo esc_html( strval( Util::format_recurrences( $wpayme_phase->get_interval() ) ) );
			} else {
				// Fixed number of recurrences.
				printf(
					'%s (%s)',
					esc_html( strval( Util::format_recurrences( $wpayme_phase->get_interval() ) ) ),
					esc_html( strval( Util::format_frequency( $wpayme_total_periods ) ) )
				);
			}

			?>
		</td>
	</tr>

	<?php

	// Show next payment (delivery) date if subscription is active.
	if ( SubscriptionStatus::ACTIVE === $subscription->get_status() ) :

		?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Next Payment Date', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				$wpayme_next_payment_date = $subscription->get_next_payment_date();

				echo empty( $wpayme_next_payment_date ) ? '—' : esc_html( $wpayme_next_payment_date->format_i18n( __( 'D j M Y', 'wpayme' ) ) );

				?>
			</td>
		</tr>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Next Payment Delivery Date', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				$wpayme_next_payment_delivery_date = $subscription->get_next_payment_delivery_date();

				echo empty( $wpayme_next_payment_delivery_date ) ? '—' : esc_html( $wpayme_next_payment_delivery_date->format_i18n( __( 'D j M Y', 'wpayme' ) ) );

				?>
			</td>
		</tr>

	<?php endif; ?>

	<?php

	$wpayme_customer = $subscription->get_customer();

	if ( null !== $wpayme_customer ) :

		$wpayme_text = \strval( $wpayme_customer->get_name() );

		if ( empty( $wpayme_text ) ) :

			$wpayme_text = $wpayme_customer->get_email();

		endif;

		if ( ! empty( $wpayme_text ) ) :

			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Customer', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo \esc_html( $wpayme_text ); ?>
				</td>
			</tr>

		<?php endif; ?>

	<?php endif; ?>

	<?php if ( null !== $wpayme_user_id ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'User', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				$wpayme_user_text = sprintf( '#%s', $wpayme_user_id );

				// User display name.
				$wpayme_user = new WP_User( $wpayme_user_id );

				$wpayme_display_name = $wpayme_user->display_name;

				if ( ! empty( $wpayme_display_name ) ) {
					$wpayme_user_text .= sprintf( ' - %s', $wpayme_display_name );
				}

				printf(
					'<a href="%s">%s</a>',
					esc_url( get_edit_user_link( $wpayme_user_id ) ),
					esc_html( $wpayme_user_text )
				);

				?>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Source', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			echo wp_kses(
				$subscription->get_source_text(),
				[
					'a'  => [
						'href' => true,
					],
					'br' => [],
				]
			);

			?>
		</td>
	</tr>

	<tr>
		<th scope="row" rowspan="3">
			<?php esc_html_e( 'Customer action links', 'wpayme' ); ?>

			<span class="dashicons dashicons-editor-help wpayme-pay-tip" title="<?php echo esc_attr__( 'These actions links can be shared with the customer.', 'wpayme' ); ?>" tabindex="0"></span>

			<?php
			/**
			 * Clipboard feature.
			 * 
			 * @link https://github.com/WordPress/WordPress/blob/68e3310c024d7fceb84a5028e955ad163de6bd45/wp-includes/js/plupload/handlers.js#L364-L393
			 * @link https://translate.wordpress.org/projects/wp/dev/nl/default/?filters%5Bstatus%5D=either&filters%5Boriginal_id%5D=10763746&filters%5Btranslation_id%5D=91929960
			 * @link https://translate.wordpress.org/projects/wp/dev/nl/default/?filters%5Bstatus%5D=either&filters%5Boriginal_id%5D=6831324&filters%5Btranslation_id%5D=58732256
			 */
			?>
		</th>
		<td>
			<?php // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedClassFound -- HTML class attribute. ?>
			<div class="wpayme-pay-action-link">
				<div>
					<?php

					$wpayme_url = $subscription->get_cancel_url();

					\printf(
						'<a class="wpayme-pay-action-link-anchor" href="%s">%s</a>',
						\esc_attr( $wpayme_url ),
						\esc_html__( 'Customer subscription cancel page →', 'wpayme' )
					);

					?>
				</div>

				<div>
					<span class="dashicons dashicons-editor-help wpayme-pay-tip" title="<?php echo \esc_attr__( 'This page can be shared with the customer and gives the customer the option to cancel this subscription.', 'wpayme' ); ?>" tabindex="0"></span>
				</div>

				<div class="wpayme-pay-action-link-clipboard">
					<button type="button" class="button button-small wpayme-pay-copy-url" data-clipboard-text="<?php echo \esc_url( $wpayme_url ); ?>"><?php \esc_html_e( 'Copy URL to clipboard', 'wpayme' ); ?></button>
					<span class="success hidden" aria-hidden="true"><?php \esc_html_e( 'Copied!', 'wpayme' ); ?></span>
				</div>
			</div>
		</td>
	</tr>
	<tr>
		<td>
			<div class="wpayme-pay-action-link">
				<div>
					<?php

					$wpayme_url = $subscription->get_renewal_url();

					\printf(
						'<a class="wpayme-pay-action-link-anchor" href="%s">%s</a>',
						\esc_attr( $wpayme_url ),
						\esc_html__( 'Customer subscription renew page →', 'wpayme' )
					);

					?>
				</div>

				<div>
					<span class="dashicons dashicons-editor-help wpayme-pay-tip" title="<?php echo \esc_attr__( 'This page can be shared with the customer and gives the customer the option to (early) renew the subscription.', 'wpayme' ); ?>" tabindex="0"></span>
				</div>

				<div class="wpayme-pay-action-link-clipboard">
					<button type="button" class="button button-small wpayme-pay-copy-url" data-clipboard-text="<?php echo \esc_url( $wpayme_url ); ?>"><?php \esc_html_e( 'Copy URL to clipboard', 'wpayme' ); ?></button>
					<span class="success hidden" aria-hidden="true"><?php \esc_html_e( 'Copied!', 'wpayme' ); ?></span>
				</div>
			</div>
		</td>
	</tr>
	<tr>
		<td>
			<div class="wpayme-pay-action-link">
				<div>
					<?php

					$wpayme_url = $subscription->get_mandate_selection_url();

					\printf(
						'<a class="wpayme-pay-action-link-anchor" href="%s">%s</a>',
						\esc_attr( $wpayme_url ),
						\esc_html__( 'Customer change payment method page →', 'wpayme' )
					);

					?>
				</div>

				<div>
					<span class="dashicons dashicons-editor-help wpayme-pay-tip" title="<?php echo \esc_attr__( 'This link can be shared with the customer and gives the customer the opportunity to change the payment method. This is useful if a credit card expires or if a customer wants to have the charge debited from another account.', 'wpayme' ); ?>" tabindex="0"></span>
				</div>

				<div class="wpayme-pay-action-link-clipboard">
					<button type="button" class="button button-small wpayme-pay-copy-url" data-clipboard-text="<?php echo \esc_url( $wpayme_url ); ?>"><?php \esc_html_e( 'Copy URL to clipboard', 'wpayme' ); ?></button>
					<span class="success hidden" aria-hidden="true"><?php \esc_html_e( 'Copied!', 'wpayme' ); ?></span>
				</div>
			</div>
		</td>
	</tr>

	<?php if ( $plugin->is_debug_mode() ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'REST API URL', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				/**
				 * REST API URL.
				 *
				 * @link https://developer.wordpress.org/rest-api/using-the-rest-api/authentication/#cookie-authentication
				 */
				$wpayme_rest_api_url = rest_url( 'wpayme-pay/v1/subscriptions/' . $wpayme_subscription_id );

				$wpayme_rest_api_nonce_url = wp_nonce_url( $wpayme_rest_api_url, 'wp_rest' );

				printf(
					'<a href="%s">%s</a>',
					esc_url( $wpayme_rest_api_nonce_url ),
					esc_html( $wpayme_rest_api_url )
				);

				?>
			</td>
		</tr>

	<?php endif; ?>

</table>