<?php
// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound

/**
 * Meta Box Payment Info
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay
 * @var \Wpayme\WordPress\Pay\Plugin $plugin Plugin.
 * @var \Wpayme\WordPress\Pay\Payments\Payment $payment Payment.
 */

use Wpayme\WordPress\Pay\Core\PaymentMethods;
use Wpayme\WordPress\Pay\Gender;
use Wpayme\WordPress\Pay\Payments\PaymentStatus;
use Wpayme\WordPress\Pay\Plugin;
use Wpayme\WordPress\Pay\VatNumbers\VatNumberValidationService;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

?>
<table class="form-table">
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Date', 'wpayme' ); ?>
		</th>
		<td>
			<?php echo esc_html( $payment->date->format_i18n() ); ?>
		</td>
	</tr>

	<?php

	$wpayme_failure_reason = $payment->get_failure_reason();

	if ( PaymentStatus::FAILURE === $payment->get_status() && null !== $wpayme_failure_reason ) :

		?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Status', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				$wpayme_status_label = $payment->get_status_label();

				echo \esc_html( ( null === $wpayme_status_label ) ? '—' : $wpayme_status_label );

				printf(
					' — %s',
					esc_html( $wpayme_failure_reason )
				);

				?>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'ID', 'wpayme' ); ?>
		</th>
		<td>
			<?php echo esc_html( (string) $payment->get_id() ); ?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Order ID', 'wpayme' ); ?>
		</th>
		<td>
			<?php echo esc_html( (string) $payment->get_order_id() ); ?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Description', 'wpayme' ); ?>
		</th>
		<td>
			<?php echo esc_html( (string) $payment->get_description() ); ?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Amount', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			echo esc_html( $payment->get_total_amount()->format_i18n() );

			?>
		</td>
	</tr>

	<?php

	$wpayme_amount_refunded = $payment->get_refunded_amount();

	if ( $wpayme_amount_refunded->get_value() > 0 ) :

		?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Refunded Amount', 'wpayme' ); ?>
			</th>
			<td>
				<?php echo esc_html( $wpayme_amount_refunded->format_i18n() ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<?php

	$wpayme_charged_back_amount = $payment->get_charged_back_amount();

	if ( null !== $wpayme_charged_back_amount && $wpayme_charged_back_amount->get_value() > 0 ) :

		?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Charged Back Amount', 'wpayme' ); ?>
			</th>
			<td>
				<?php echo esc_html( $wpayme_charged_back_amount->format_i18n() ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Transaction ID', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$wpayme_payments_post_type = \Wpayme\WordPress\Pay\Admin\AdminPaymentPostType::POST_TYPE;

			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			do_action( 'manage_' . $wpayme_payments_post_type . '_posts_custom_column', 'wpayme_payment_transaction', $payment->get_id() );

			?>
		</td>
	</tr>

	<?php

	$wpayme_purchase_id = $payment->get_meta( 'purchase_id' );

	if ( $wpayme_purchase_id ) :
		?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Purchase ID', 'wpayme' ); ?>
			</th>
			<td>
				<?php echo esc_html( $wpayme_purchase_id ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<?php if ( null !== $payment->config_id ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Gateway', 'wpayme' ); ?>
			</th>
			<td>
				<?php edit_post_link( get_the_title( $payment->config_id ), '', '', $payment->config_id ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Payment Method', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$payment_method = $payment->get_payment_method();

			// Name.
			$name = PaymentMethods::get_name( $payment_method );
			$name = ( null === $name ) ? $payment_method : $name;

			$gateway = Plugin::get_gateway( (int) $payment->get_config_id() );

			if ( null !== $gateway && null !== $payment_method ) {
				$method = $gateway->get_payment_method( $payment_method );

				if ( null !== $method ) {
					$name = $method->get_name();
				}
			}

			// Icon.
			$icon_url = PaymentMethods::get_icon_url( $payment_method );

			if ( null !== $icon_url ) {
				\printf(
					'<span class="wpayme-pay-tip" title="%2$s"><img src="%1$s" alt="%2$s" title="%2$s" width="32" valign="bottom" /></span> ',
					\esc_url( $icon_url ),
					\esc_attr( (string) $name )
				);
			}

			// Name.
			echo esc_html( (string) $name );

			// Issuer.
			$issuer = $payment->get_meta( 'issuer' );

			if ( $issuer ) {
				echo esc_html( sprintf( ' (`%s`)', $issuer ) );
			}

			?>
		</td>
	</tr>

	<?php

	$wpayme_consumer_bank_details = $payment->get_consumer_bank_details();

	if ( null !== $wpayme_consumer_bank_details ) :

		$wpayme_consumer_name = $wpayme_consumer_bank_details->get_name();

		if ( null !== $wpayme_consumer_name ) :

			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Account Holder', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_consumer_name ); ?>
				</td>
			</tr>

			<?php

		endif;

		$wpayme_account_holder_city = $wpayme_consumer_bank_details->get_city();

		if ( null !== $wpayme_account_holder_city ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Account Holder City', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_account_holder_city ); ?>
				</td>
			</tr>

			<?php

		endif;

		$wpayme_account_holder_country = $wpayme_consumer_bank_details->get_country();

		if ( null !== $wpayme_account_holder_country ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Account Holder Country', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_account_holder_country ); ?>
				</td>
			</tr>

			<?php

		endif;

		$wpayme_account_number = $wpayme_consumer_bank_details->get_account_number();

		if ( null !== $wpayme_account_number ) :

			if ( PaymentMethods::CREDIT_CARD === $payment->get_payment_method() && 4 === strlen( $wpayme_account_number ) ) {
				$wpayme_account_number = sprintf( '●●●● ●●●● ●●●● %d', $wpayme_account_number );
			}

			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Account Number', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_account_number ); ?>
				</td>
			</tr>

			<?php

		endif;

		$wpayme_iban = $wpayme_consumer_bank_details->get_iban();

		if ( null !== $wpayme_iban ) :
			?>

			<tr>
				<th scope="row">
					<?php

					printf(
						'<abbr title="%s">%s</abbr>',
						esc_attr( _x( 'International Bank Account Number', 'IBAN abbreviation title', 'wpayme' ) ),
						esc_html__( 'IBAN', 'wpayme' )
					);

					?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_iban ); ?>
				</td>
			</tr>

			<?php

		endif;

		$wpayme_bic = $wpayme_consumer_bank_details->get_bic();

		if ( null !== $wpayme_bic ) :
			?>

			<tr>
				<th scope="row">
					<?php

					printf(
						'<abbr title="%s">%s</abbr>',
						esc_attr( _x( 'Bank Identifier Code', 'BIC abbreviation title', 'wpayme' ) ),
						esc_html__( 'BIC', 'wpayme' )
					);

					?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_bic ); ?>
				</td>
			</tr>

			<?php

		endif;

	endif;

	?>

	<?php

	$wpayme_bank_transfer_recipient = $payment->get_bank_transfer_recipient_details();

	?>

	<?php if ( null !== $wpayme_bank_transfer_recipient ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Bank Transfer Recipient', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				echo wp_kses(
					wpautop( $wpayme_bank_transfer_recipient ),
					[
						'p'  => [],
						'br' => [],
					]
				);

				?>
			</td>
		</tr>

	<?php endif; ?>

	<?php

	$wpayme_customer = $payment->get_customer();

	if ( null !== $wpayme_customer ) :

		$wpayme_text = \strval( $wpayme_customer->get_name() );

		if ( empty( $wpayme_text ) ) :

			$wpayme_text = $wpayme_customer->get_email();

		endif;

		if ( ! empty( $wpayme_text ) ) :

			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Customer', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo \esc_html( $wpayme_text ); ?>
				</td>
			</tr>

			<?php
		endif;

		$wpayme_company_name = $wpayme_customer->get_company_name();

		if ( null !== $wpayme_company_name ) :
			?>

			<tr>
				<th scope="row">
					<?php \esc_html_e( 'Company', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo \esc_html( $wpayme_company_name ); ?>
				</td>
			</tr>

			<?php
		endif;

		$wpayme_vat_number = $wpayme_customer->get_vat_number();

		if ( null !== $wpayme_vat_number ) :
			$wpayme_vat_number_validity = $wpayme_vat_number->get_validity();

			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'VAT Number', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo \esc_html( $wpayme_vat_number ); ?>
				</td>
			</tr>

			<?php if ( null !== $wpayme_vat_number_validity ) : ?>

				<tr>
					<th scope="row">
						<?php esc_html_e( 'VAT Number Validity', 'wpayme' ); ?>
					</th>
					<td style="padding-top: 20px">
						<table class="wpayme-pay-form-sub-table">
							<tr>
								<th scope="row">
									<?php esc_html_e( 'VAT Number', 'wpayme' ); ?>
								</th>
								<td>
									<?php echo esc_html( $wpayme_vat_number_validity->get_vat_number() ); ?>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<?php esc_html_e( 'Request Date', 'wpayme' ); ?>
								</th>
								<td>
									<?php echo esc_html( $wpayme_vat_number_validity->get_request_date()->format( 'd-m-Y' ) ); ?>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<?php esc_html_e( 'Valid', 'wpayme' ); ?>
								</th>
								<td>
									<?php echo esc_html( $wpayme_vat_number_validity->is_valid() ? __( 'Yes', 'wpayme' ) : __( 'No', 'wpayme' ) ); ?>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<?php esc_html_e( 'Name', 'wpayme' ); ?>
								</th>
								<td>
									<?php echo esc_html( (string) $wpayme_vat_number_validity->get_name() ); ?>
								</td>
							</tr>
							<tr>
								<th scope="row">
									<?php esc_html_e( 'Address', 'wpayme' ); ?>
								</th>
								<td>
									<?php

									echo \wp_kses(
										\nl2br( (string) $wpayme_vat_number_validity->get_address() ),
										[
											'br' => [],
										]
									);

									?>
								</td>
							</tr>

							<?php

							$wpayme_service = $wpayme_vat_number_validity->get_service();

							if ( null !== $wpayme_service ) :
								?>

								<tr>
									<th scope="row">
										<?php esc_html_e( 'Validation Service', 'wpayme' ); ?>
									</th>
									<td>
										<?php

										switch ( $wpayme_service ) {
											case VatNumberValidationService::VIES:
												echo esc_html( __( 'VIES', 'wpayme' ) );

												break;
											default:
												echo \esc_html( $wpayme_service );

												break;
										}

										?>
									</td>
								</tr>

							<?php endif; ?>

						</table>
					</td>
				</tr>

				<?php
			endif;

		endif;

		$wpayme_birth_date = $wpayme_customer->get_birth_date();

		if ( null !== $wpayme_birth_date ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Date of birth', 'wpayme' ); ?>
				</th>
				<td>
					<?php

					echo esc_html( $wpayme_birth_date->format_i18n( __( 'D j M Y', 'wpayme' ) ) )

					?>
				</td>
			</tr>

		<?php endif; ?>

		<?php

		$wpayme_gender = $wpayme_customer->get_gender();

		if ( null !== $wpayme_gender ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Gender', 'wpayme' ); ?>
				</th>
				<td>
					<?php

					switch ( $wpayme_gender ) {
						case Gender::FEMALE:
							echo esc_html( __( 'Female', 'wpayme' ) );

							break;
						case Gender::MALE:
							echo esc_html( __( 'Male', 'wpayme' ) );

							break;
						case Gender::OTHER:
							echo esc_html( __( 'Other', 'wpayme' ) );

							break;
					}

					?>
				</td>
			</tr>

		<?php endif; ?>

		<?php

		$wpayme_user_id = $wpayme_customer->get_user_id();

		if ( null !== $wpayme_user_id ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'User', 'wpayme' ); ?>
				</th>
				<td>
					<?php

					$wpayme_user_text = sprintf( '#%s', $wpayme_user_id );

					// User display name.
					$wpayme_user = new WP_User( $wpayme_user_id );

					$wpayme_display_name = $wpayme_user->display_name;

					if ( ! empty( $wpayme_display_name ) ) {
						$wpayme_user_text .= sprintf( ' - %s', $wpayme_display_name );
					}

					printf(
						'<a href="%s">%s</a>',
						esc_url( get_edit_user_link( $wpayme_user_id ) ),
						esc_html( $wpayme_user_text )
					);

					?>
				</td>
			</tr>

		<?php endif; ?>

		<?php

		$wpayme_ip_address = $wpayme_customer->get_ip_address();

		if ( null !== $wpayme_ip_address ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'IP Address', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $wpayme_ip_address ); ?>
				</td>
			</tr>

		<?php endif; ?>

	<?php endif; ?>

	<?php if ( null !== $payment->get_billing_address() ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Billing Address', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				$address = $payment->get_billing_address();

				echo wp_kses(
					nl2br( (string) $address ),
					[
						'br' => [],
					]
				);

				?>
			</td>
		</tr>

	<?php endif; ?>

	<?php if ( null !== $payment->get_shipping_address() ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Shipping Address', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				$address = $payment->get_shipping_address();

				echo wp_kses(
					nl2br( (string) $address ),
					[
						'br' => [],
					]
				);

				?>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Source', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			echo wp_kses(
				$payment->get_source_text(),
				[
					'a'  => [
						'href' => true,
					],
					'br' => [],
				]
			);

			?>
		</td>
	</tr>

	<?php if ( 'membership' === $payment->get_source() ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'User ID', 'wpayme' ); ?>
			</th>
			<td>
				<?php echo esc_html( (string) $payment->get_meta( 'membership_user_id' ) ); ?>
			</td>
		</tr>
		<tr>
			<th scope="row">
				<?php esc_html_e( 'Subscription ID', 'wpayme' ); ?>
			</th>
			<td>
				<?php echo esc_html( (string) $payment->get_meta( 'membership_subscription_id' ) ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<?php

	$ogone_alias = $payment->get_meta( 'ogone_alias' );

	?>

	<?php if ( ! empty( $ogone_alias ) && \is_string( $ogone_alias ) ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'Ingenico Alias', 'wpayme' ); ?>
			</th>
			<td>
				<?php echo esc_html( $ogone_alias ); ?>
			</td>
		</tr>

	<?php endif; ?>

	<?php if ( $plugin->is_debug_mode() ) : ?>

		<?php if ( null !== $payment->get_version() ) : ?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Version', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $payment->get_version() ); ?>
				</td>
			</tr>

		<?php endif ?>

		<?php if ( null !== $payment->get_mode() ) : ?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'Mode', 'wpayme' ); ?>
				</th>
				<td>
					<?php

					switch ( $payment->get_mode() ) {
						case 'live':
							esc_html_e( 'Live', 'wpayme' );

							break;
						case 'test':
							esc_html_e( 'Test', 'wpayme' );

							break;
						default:
							echo esc_html( $payment->get_mode() );

							break;
					}

					?>
				</td>
			</tr>

		<?php endif ?>

	<?php endif; ?>

	<?php

	$customer = $payment->get_customer();

	if ( null !== $customer ) :

		$user_agent = $customer->get_user_agent();

		if ( null !== $user_agent ) :
			?>

			<tr>
				<th scope="row">
					<?php esc_html_e( 'User Agent', 'wpayme' ); ?>
				</th>
				<td>
					<?php echo esc_html( $user_agent ); ?>
				</td>
			</tr>

		<?php endif; ?>

	<?php endif; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Action URL', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$url = $payment->get_action_url();

			if ( null !== $url ) {
				printf(
					'<a href="%s" target="_blank">%s</a>',
					esc_attr( $url ),
					esc_html( $url )
				);
			}

			?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Pay Redirect URL', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$url = $payment->get_pay_redirect_url();

			printf(
				'<a href="%s">%s</a>',
				esc_attr( $url ),
				esc_html( $url )
			);

			?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Return URL', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$url = $payment->get_return_url();

			printf(
				'<a href="%s">%s</a>',
				esc_attr( $url ),
				esc_html( $url )
			);

			?>
		</td>
	</tr>
	<tr>
		<th scope="row">
			<?php esc_html_e( 'Return Redirect URL', 'wpayme' ); ?>
		</th>
		<td>
			<?php

			$url = $payment->get_return_redirect_url();

			printf(
				'<a href="%s">%s</a>',
				esc_attr( $url ),
				esc_html( $url )
			);

			?>
		</td>
	</tr>

	<?php if ( $plugin->is_debug_mode() ) : ?>

		<tr>
			<th scope="row">
				<?php esc_html_e( 'REST API URL', 'wpayme' ); ?>
			</th>
			<td>
				<?php

				/**
				 * REST API URL.
				 *
				 * @link https://developer.wordpress.org/rest-api/using-the-rest-api/authentication/#cookie-authentication
				 */
				$rest_api_url = rest_url( 'wpayme-pay/v1/payments/' . $payment->get_id() );

				$rest_api_nonce_url = wp_nonce_url( $rest_api_url, 'wp_rest' );

				printf(
					'<a href="%s">%s</a>',
					esc_url( $rest_api_nonce_url ),
					esc_html( $rest_api_url )
				);

				?>
			</td>
		</tr>

	<?php endif; ?>
</table>
