<?php
/**
 * Meta Box Gateway Test
 *
 * @author Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license GPL-3.0-or-later
 * @package Wpayme\WordPress\Pay
 * @var \WP_Post $post WordPress post.
 */

use Wpayme\WordPress\Money\Currencies;
use Wpayme\WordPress\Money\Currency;
use Wpayme\WordPress\Pay\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$wpayme_gateway = Plugin::get_gateway( $post->ID );

if ( null === $wpayme_gateway ) {
	printf(
		'<em>%s</em>',
		esc_html( __( 'Please save the entered account details of your payment provider, to make a test payment.', 'wpayme' ) )
	);

	return;
}

wp_nonce_field( 'test_pay_gateway', 'wpayme_pay_test_nonce' );

$wpayme_currency_default = Currency::get_instance( 'EUR' );

$wpayme_payment_methods = $wpayme_gateway->get_payment_methods(
	[
		'status' => [
			'',
			'active',
		],
	]
);

?>
<table class="form-table">
	<tr>
		<th scope="row">
			<label for="wpayme-pay-test-payment-methods">
				<?php esc_html_e( 'Payment Method', 'wpayme' ); ?>
			</label>
		</th>
		<td>
			<select id="wpayme-pay-test-payment-methods" name="wpayme_pay_test_payment_method">
				<?php if ( count( $wpayme_payment_methods ) > 1 ) : ?>

					<option value=""><?php esc_html_e( '— Choose payment method —', 'wpayme' ); ?></option>

				<?php endif; ?>

				<?php

				foreach ( $wpayme_payment_methods as $wpayme_payment_method ) {
					printf(
						'<option value="%s" data-is-recurring="%d">%s</option>',
						esc_attr( $wpayme_payment_method->get_id() ),
						esc_attr( $wpayme_payment_method->supports( 'recurring' ) ? '1' : ' 0' ),
						esc_html( $wpayme_payment_method->get_name() )
					);
				}

				?>
			</select>
		</td>
	</tr>

	<?php foreach ( $wpayme_payment_methods as $wpayme_payment_method ) : ?>

		<?php foreach ( $wpayme_payment_method->get_fields() as $wpayme_field ) : ?>

			<tr class="wpayme-pay-cloak wpayme-pay-test-payment-method <?php echo esc_attr( $wpayme_payment_method->get_id() ); ?>">
				<th scope="row">
					<?php echo esc_html( $wpayme_field->get_label() ); ?>
				</th>
				<td>
					<?php

					try {
						$wpayme_field->output();
					} catch ( \Exception $wpayme_exception ) {
						echo '<em>';

						printf(
							/* translators: %s: Exception message. */
							esc_html__( 'This field could not be displayed due to the following error message: "%s".', 'wpayme' ),
							esc_html( $wpayme_exception->getMessage() )
						);

						echo '</em>';

						?>
						<div class="error">
							<p>
								<?php

								echo wp_kses(
									sprintf(
										/* translators: 1: Field label, 2: Payment method name */
										__( '<strong>Wpayme Pay</strong> — An error occurred within the "%1$s" field of the "%2$s" payment method.', 'wpayme' ),
										esc_html( $wpayme_field->get_label() ),
										esc_html( $wpayme_payment_method->get_name() )
									),
									[
										'strong' => [],
									]
								);

								?>
							</p>

							<dl>
								<dt><?php esc_html_e( 'Message', 'wpayme' ); ?></dt>
								<dd><?php echo esc_html( $wpayme_exception->getMessage() ); ?></dd>

								<?php if ( 0 !== $wpayme_exception->getCode() ) : ?>

									<dt><?php esc_html_e( 'Code', 'wpayme' ); ?></dt>
									<dd><?php echo esc_html( $wpayme_exception->getCode() ); ?></dd>

								<?php endif; ?>
							</dl>
						</div>
						<?php
					}

					?>
				</td>
			</tr>

		<?php endforeach; ?>

	<?php endforeach; ?>

	<tr>
		<th scope="row">
			<?php esc_html_e( 'Amount', 'wpayme' ); ?>
		</th>
		<td>
			<select name="test_currency_code">
				<?php

				foreach ( Currencies::get_currencies() as $wpayme_currency ) {
					$wpayme_label = $wpayme_currency->get_alphabetic_code();

					$wpayme_symbol = $wpayme_currency->get_symbol();

					if ( null !== $wpayme_symbol ) {
						$wpayme_label = sprintf( '%s (%s)', $wpayme_label, $wpayme_symbol );
					}

					printf(
						'<option value="%s" %s>%s</option>',
						esc_attr( $wpayme_currency->get_alphabetic_code() ),
						selected( $wpayme_currency->get_alphabetic_code(), $wpayme_currency_default->get_alphabetic_code(), false ),
						esc_html( $wpayme_label )
					);
				}

				?>
			</select>

			<input name="test_amount" id="test_amount" class="regular-text code wpayme-pay-form-control" value="" type="number" step="any" size="6" autocomplete="off" />
		</td>
	</tr>

	<?php if ( $wpayme_gateway->supports( 'recurring' ) ) : ?>

		<?php

		$wpayme_options = [
			''  => __( '— Select Repeat —', 'wpayme' ),
			'D' => __( 'Daily', 'wpayme' ),
			'W' => __( 'Weekly', 'wpayme' ),
			'M' => __( 'Monthly', 'wpayme' ),
			'Y' => __( 'Annually', 'wpayme' ),
		];

		$wpayme_options_interval_suffix = [
			'D' => __( 'days', 'wpayme' ),
			'W' => __( 'weeks', 'wpayme' ),
			'M' => __( 'months', 'wpayme' ),
			'Y' => __( 'year', 'wpayme' ),
		];

		?>
		<tr>
			<th scope="row">
				<label for="wpayme-pay-test-subscription">
					<?php esc_html_e( 'Subscription', 'wpayme' ); ?>
				</label>
			</th>
			<td>
				<fieldset>
					<legend class="screen-reader-text"><span><?php esc_html_e( 'Test Subscription', 'wpayme' ); ?></span></legend>

					<label for="wpayme-pay-test-subscription">
						<input name="wpayme_pay_test_subscription" id="wpayme-pay-test-subscription" value="1" type="checkbox" />
						<?php esc_html_e( 'Start a subscription for this payment.', 'wpayme' ); ?>
					</label>
				</fieldset>

				<?php

				\wp_add_inline_script(
					'wpayme-pay-admin',
					"
					jQuery( document ).ready( function( $ ) {
						$( '#wpayme-pay-test-subscription' ).change( function() {
							$( '.wpayme-pay-test-subscription' ).toggle( $( this ).prop( 'checked' ) );
						} );
					} );
					"
				);

				?>
			</td>
		</tr>
		<tr class="wpayme-pay-cloak wpayme-pay-test-subscription">
			<th scope="row">
				<label for="wpayme_pay_test_repeat_frequency"><?php esc_html_e( 'Frequency', 'wpayme' ); ?></label>
			</th>
			<td>
				<select id="wpayme_pay_test_repeat_frequency" name="wpayme_pay_test_repeat_frequency">
					<?php

					foreach ( $wpayme_options as $wpayme_key => $wpayme_label ) {
						$wpayme_interval_suffix = '';

						if ( isset( $wpayme_options_interval_suffix[ $wpayme_key ] ) ) {
							$wpayme_interval_suffix = $wpayme_options_interval_suffix[ $wpayme_key ];
						}

						printf(
							'<option value="%s" data-interval-suffix="%s">%s</option>',
							esc_attr( $wpayme_key ),
							esc_attr( $wpayme_interval_suffix ),
							esc_html( $wpayme_label )
						);
					}

					?>
				</select>
			</td>
		</tr>
		<tr class="wpayme-pay-cloak wpayme-pay-test-subscription">
			<th scope="row">
				<label for="wpayme_pay_test_repeat_interval"><?php esc_html_e( 'Repeat every', 'wpayme' ); ?></label>
			</th>
			<td>
				<select id="wpayme_pay_test_repeat_interval" name="wpayme_pay_test_repeat_interval">
					<?php

					foreach ( range( 1, 30 ) as $wpayme_value ) {
						printf(
							'<option value="%s">%s</option>',
							esc_attr( (string) $wpayme_value ),
							esc_html( (string) $wpayme_value )
						);
					}

					?>
				</select>

				<span id="wpayme_pay_test_repeat_interval_suffix"><?php esc_html_e( 'days/weeks/months/year', 'wpayme' ); ?></span>
			</td>
		</tr>
		<tr class="wpayme-pay-cloak wpayme-pay-test-subscription">
			<th scope="row">
				<?php esc_html_e( 'Ends On', 'wpayme' ); ?>
			</th>
			<td>
				<div>
					<input type="radio" id="wpayme_pay_ends_never" name="wpayme_pay_ends_on" value="never" checked="checked" />

					<label for="wpayme_pay_ends_never">
						<?php esc_html_e( 'Never', 'wpayme' ); ?>
					</label>
				</div>
				<div>
					<input type="radio" id="wpayme_pay_ends_count" name="wpayme_pay_ends_on" value="count" />

					<label for="wpayme_pay_ends_count">
						<?php

						$wpayme_allowed_html = [
							'input' => [
								'id'    => true,
								'name'  => true,
								'type'  => true,
								'value' => true,
								'size'  => true,
								'class' => true,
							],
						];

						echo wp_kses(
							sprintf(
								/* translators: %s: Input field for number times */
								__( 'After %s times', 'wpayme' ),
								sprintf( '<input type="number" name="wpayme_pay_ends_on_count" value="%s" min="1" />', esc_attr( '' ) )
							),
							$wpayme_allowed_html
						);

						?>
					</label>
				</div>

				<div>
					<input type="radio" id="wpayme_pay_ends_date" name="wpayme_pay_ends_on" value="date" />

					<label for="wpayme_pay_ends_date">
						<?php

						echo wp_kses(
							sprintf(
								/* translators: %s: input HTML */
								__( 'On %s', 'wpayme' ),
								sprintf( '<input type="date" id="wpayme_pay_ends_on_date" name="wpayme_pay_ends_on_date" value="%s" />', esc_attr( '' ) )
							),
							$wpayme_allowed_html
						);

						?>
					</label>
				</div>
			</td>
		</tr>

	<?php endif; ?>

	<tr>
		<td>

		</td>
		<td>
			<?php submit_button( __( 'Test', 'wpayme' ), 'secondary', 'test_pay_gateway', false ); ?>
		</td>
	</tr>

</table>

<?php wp_print_scripts( 'wpayme-pay-gateway-test' ); ?>
