<?php
/**
 * Admin Settings
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay\Admin
 */

namespace Wpayme\WordPress\Pay\Admin;

use Wpayme\WordPress\Html\Element;
use Wpayme\WordPress\Pay\Plugin;
use Wpayme\WordPress\Pay\Util;

/**
 * WordPress iDEAL admin
 *
 * @author  Remco Tolsma
 * @version 2.2.6
 * @since   1.0.0
 */
class AdminSettings {
	/**
	 * Plugin.
	 *
	 * @var Plugin
	 */
	private $plugin;

	/**
	 * Constructs and initialize an admin object.
	 *
	 * @param Plugin $plugin Plugin.
	 */
	public function __construct( Plugin $plugin ) {
		$this->plugin = $plugin;

		// Actions.
		add_action( 'admin_init', [ $this, 'admin_init' ] );
	}

	/**
	 * Admin initialize.
	 *
	 * @return void
	 */
	public function admin_init() {
		// Settings - General.
		add_settings_section(
			'wpayme_pay_general',
			__( 'General', 'wpayme' ),
			function () {
			},
			'wpayme_pay'
		);

		// Default Config.
		add_settings_field(
			'wpayme_pay_config_id',
			__( 'Default Gateway', 'wpayme' ),
			[ $this, 'input_page' ],
			'wpayme_pay',
			'wpayme_pay_general',
			[
				'post_type'        => 'wpayme_gateway',
				'show_option_none' => __( '— Select a gateway —', 'wpayme' ),
				'label_for'        => 'wpayme_pay_config_id',
			]
		);

		// Remove data on uninstall.
		add_settings_field(
			'wpayme_pay_uninstall_clear_data',
			__( 'Remove Data', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay',
			'wpayme_pay_general',
			[
				'legend'      => __( 'Remove Data', 'wpayme' ),
				'description' => __( 'Remove all plugin data on uninstall', 'wpayme' ),
				'label_for'   => 'wpayme_pay_uninstall_clear_data',
				'classes'     => 'regular-text',
				'type'        => 'checkbox',
			]
		);

		// Debug mode.
		$debug_mode_args = [
			'legend'      => \__( 'Debug Mode', 'wpayme' ),
			'description' => \__( 'Enable debug mode', 'wpayme' ),
			'label_for'   => 'wpayme_pay_debug_mode',
			'type'        => 'checkbox',
		];

		if ( defined( 'WPAYME_PAY_DEBUG' ) && WPAYME_PAY_DEBUG ) {
			$debug_mode_args['value']    = true;
			$debug_mode_args['disabled'] = \disabled( WPAYME_PAY_DEBUG, true, false );
		}

		\add_settings_field(
			'wpayme_pay_debug_mode',
			\__( 'Debug Mode', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay',
			'wpayme_pay_general',
			$debug_mode_args
		);

		\add_settings_field(
			'wpayme_pay_subscriptions_processing_disabled',
			\__( 'Disable Recurring Payments', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay',
			'wpayme_pay_general',
			[
				'legend'      => \__( 'Disable starting recurring payments at gateway', 'wpayme' ),
				'description' => \__( 'Disable starting recurring payments at gateway', 'wpayme' ),
				'label_for'   => 'wpayme_pay_subscriptions_processing_disabled',
				'type'        => 'checkbox',
			]
		);

		if ( version_compare( $this->plugin->get_version(), '10', '>=' ) ) {
			// Settings - Payment Methods.
			\add_settings_section(
				'wpayme_pay_payment_methods',
				\__( 'Payment Methods', 'wpayme' ),
				function () {
				},
				'wpayme_pay'
			);

			foreach ( $this->plugin->get_payment_methods() as $payment_method ) {
				$id = 'wpayme_pay_payment_method_' . $payment_method->get_id() . '_status';

				add_settings_field(
					$id,
					$payment_method->get_name(),
					[ $this, 'select_payment_method_status' ],
					'wpayme_pay',
					'wpayme_pay_payment_methods',
					[
						'label_for' => $id,
					]
				);
			}
		}
		
		// Register email settings
		$this->register_email_settings();
	}
	
	/**
	 * Register email settings fields.
	 *
	 * @return void
	 */
	public function register_email_settings() {
		// Register the email settings page
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_stripe_receipts', [ 'sanitize_callback' => 'rest_sanitize_boolean' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_buyer_enabled', [ 'sanitize_callback' => 'rest_sanitize_boolean' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_buyer_content_type', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_from_address', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_buyer_subject', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_buyer_body', [ 'sanitize_callback' => 'wp_kses_post' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_seller_enabled', [ 'sanitize_callback' => 'rest_sanitize_boolean' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_seller_addresses', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_seller_content_type', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_seller_subject', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_seller_body', [ 'sanitize_callback' => 'wp_kses_post' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_error_enabled', [ 'sanitize_callback' => 'rest_sanitize_boolean' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_error_addresses', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_rate_limit_enabled', [ 'sanitize_callback' => 'rest_sanitize_boolean' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_rate_limit_addresses', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		register_setting( 'wpayme_pay_email', 'wpayme_pay_email_parallel_sending', [ 'sanitize_callback' => 'rest_sanitize_boolean' ] );
		
		// Email Settings - Main section
		add_settings_section(
			'wpayme_pay_email_settings',
			__( 'Email Notifications', 'wpayme' ),
			function() {
				echo '<p>' . esc_html__( 'Configure how payment notification emails are sent to buyers and sellers.', 'wpayme' ) . '</p>';
			},
			'wpayme_pay_email'
		);
		
		// Stripe Receipts
		add_settings_field(
			'wpayme_pay_email_stripe_receipts',
			__( 'Send Receipt Email From Stripe', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay_email',
			'wpayme_pay_email_settings',
			[
				'legend'      => __( 'Stripe Receipts', 'wpayme' ),
				'description' => __( 'If checked, Stripe will send email receipts to your customers whenever they make successful payment. Note: Receipts are not sent in test mode.', 'wpayme' ),
				'label_for'   => 'wpayme_pay_email_stripe_receipts',
				'type'        => 'checkbox',
			]
		);
		
		// Buyer Email Settings Section
		add_settings_section(
			'wpayme_pay_email_buyer_settings',
			__( 'Buyer Email Settings', 'wpayme' ),
			function() {},
			'wpayme_pay_email'
		);
		
		// Enable Buyer Emails
		add_settings_field(
			'wpayme_pay_email_buyer_enabled',
			__( 'Send Emails to Buyer After Purchase', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay_email',
			'wpayme_pay_email_buyer_settings',
			[
				'legend'      => __( 'Enable Buyer Emails', 'wpayme' ),
				'description' => __( 'If checked the plugin will send an email to the buyer with the sale details. If digital goods are purchased then the email will contain the download links for the purchased products.', 'wpayme' ),
				'label_for'   => 'wpayme_pay_email_buyer_enabled',
				'type'        => 'checkbox',
			]
		);
		
		// Buyer Email Content Type
		add_settings_field(
			'wpayme_pay_email_buyer_content_type',
			__( 'Buyer Email Content Type', 'wpayme' ),
			[ $this, 'select_content_type' ],
			'wpayme_pay_email',
			'wpayme_pay_email_buyer_settings',
			[
				'label_for' => 'wpayme_pay_email_buyer_content_type',
				'description' => __( 'Choose which format of email to send.', 'wpayme' ),
			]
		);
		
		// From Email Address
		add_settings_field(
			'wpayme_pay_email_from_address',
			__( 'From Email Address', 'wpayme' ),
			[ $this, 'input_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_buyer_settings',
			[
				'label_for'   => 'wpayme_pay_email_from_address',
				'description' => __( 'Example: Your Name <sales@your-domain.com> This is the email address that will be used to send the email to the buyer. This name and email address will appear in the from field of the email.', 'wpayme' ),
				'classes'     => 'regular-text',
				'type'        => 'text',
			]
		);
		
		// Buyer Email Subject
		add_settings_field(
			'wpayme_pay_email_buyer_subject',
			__( 'Buyer Email Subject', 'wpayme' ),
			[ $this, 'input_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_buyer_settings',
			[
				'label_for'   => 'wpayme_pay_email_buyer_subject',
				'description' => __( 'This is the subject of the email that will be sent to the buyer.', 'wpayme' ),
				'classes'     => 'regular-text',
				'type'        => 'text',
			]
		);
		
		// Buyer Email Body
		add_settings_field(
			'wpayme_pay_email_buyer_body',
			__( 'Buyer Email Body', 'wpayme' ),
			[ $this, 'wysiwyg_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_buyer_settings',
			[
				'label_for'   => 'wpayme_pay_email_buyer_body',
				'description' => __( 'This is the body of the email that will be sent to the buyer. Do not change the text within the braces {}.', 'wpayme' ),
				'rows'        => 10,
				'default'     => "Hello\n\nThank you for your purchase! You ordered the following item(s):\n\n{product_details}\n\nRegards,\nThe Team",
			]
		);
		
		// Add shortcode hint for buyer email
		add_settings_field(
			'wpayme_pay_email_buyer_shortcodes',
			__( 'Available Tags', 'wpayme' ),
			[ $this, 'email_shortcodes_hint' ],
			'wpayme_pay_email',
			'wpayme_pay_email_buyer_settings',
			[
				'tags' => [
					'{product_details}' => __( 'Details of the purchased products', 'wpayme' ),
					'{transaction_id}' => __( 'The unique transaction ID', 'wpayme' ),
					'{transaction_date}' => __( 'Date of the transaction', 'wpayme' ),
					'{payer_email}' => __( 'Email address of the buyer', 'wpayme' ),
					'{first_name}' => __( 'First name of the buyer', 'wpayme' ),
					'{last_name}' => __( 'Last name of the buyer', 'wpayme' ),
					'{payment_amount}' => __( 'Total payment amount', 'wpayme' ),
					'{payment_method}' => __( 'Payment method used', 'wpayme' ),
					'{site_title}' => __( 'Your website title', 'wpayme' ),
				]
			]
		);
		
		// Seller Email Settings Section
		add_settings_section(
			'wpayme_pay_email_seller_settings',
			__( 'Seller Email Settings', 'wpayme' ),
			function() {},
			'wpayme_pay_email'
		);
		
		// Enable Seller Emails
		add_settings_field(
			'wpayme_pay_email_seller_enabled',
			__( 'Send Emails to Seller After Purchase', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay_email',
			'wpayme_pay_email_seller_settings',
			[
				'legend'      => __( 'Enable Seller Emails', 'wpayme' ),
				'description' => __( 'If checked the plugin will send an email to the seller with the sale details.', 'wpayme' ),
				'label_for'   => 'wpayme_pay_email_seller_enabled',
				'type'        => 'checkbox',
			]
		);
		
		// Seller Email Addresses
		add_settings_field(
			'wpayme_pay_email_seller_addresses',
			__( 'Notification Email Address', 'wpayme' ),
			[ $this, 'input_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_seller_settings',
			[
				'label_for'   => 'wpayme_pay_email_seller_addresses',
				'description' => __( 'This is the email address where the seller will be notified of product sales. You can put multiple email addresses separated by comma (,) in the above field to send the notification to multiple email addresses.', 'wpayme' ),
				'classes'     => 'regular-text',
				'type'        => 'text',
			]
		);
		
		// Seller Email Content Type
		add_settings_field(
			'wpayme_pay_email_seller_content_type',
			__( 'Seller Email Content Type', 'wpayme' ),
			[ $this, 'select_content_type' ],
			'wpayme_pay_email',
			'wpayme_pay_email_seller_settings',
			[
				'label_for' => 'wpayme_pay_email_seller_content_type',
				'description' => __( 'Choose which format of email to send.', 'wpayme' ),
			]
		);
		
		// Seller Email Subject
		add_settings_field(
			'wpayme_pay_email_seller_subject',
			__( 'Seller Email Subject', 'wpayme' ),
			[ $this, 'input_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_seller_settings',
			[
				'label_for'   => 'wpayme_pay_email_seller_subject',
				'description' => __( 'This is the subject of the email that will be sent to the seller for record.', 'wpayme' ),
				'classes'     => 'regular-text',
				'type'        => 'text',
			]
		);
		
		// Seller Email Body
		add_settings_field(
			'wpayme_pay_email_seller_body',
			__( 'Seller Email Body', 'wpayme' ),
			[ $this, 'wysiwyg_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_seller_settings',
			[
				'label_for'   => 'wpayme_pay_email_seller_body',
				'description' => __( 'This is the body of the email that will be sent to the seller. Do not change the text within the braces {}.', 'wpayme' ),
				'rows'        => 10,
				'default'     => "Dear Seller\n\nThis mail is to notify you of a product sale.\n\n{product_details}\n\nThe sale was made to {payer_email}\n\nThanks",
			]
		);
		
		// Add shortcode hint for seller email
		add_settings_field(
			'wpayme_pay_email_seller_shortcodes',
			__( 'Available Tags', 'wpayme' ),
			[ $this, 'email_shortcodes_hint' ],
			'wpayme_pay_email',
			'wpayme_pay_email_seller_settings',
			[
				'tags' => [
					'{product_details}' => __( 'Details of the purchased products', 'wpayme' ),
					'{transaction_id}' => __( 'The unique transaction ID', 'wpayme' ),
					'{transaction_date}' => __( 'Date of the transaction', 'wpayme' ),
					'{payer_email}' => __( 'Email address of the buyer', 'wpayme' ),
					'{first_name}' => __( 'First name of the buyer', 'wpayme' ),
					'{last_name}' => __( 'Last name of the buyer', 'wpayme' ),
					'{payment_amount}' => __( 'Total payment amount', 'wpayme' ),
					'{payment_method}' => __( 'Payment method used', 'wpayme' ),
					'{site_title}' => __( 'Your website title', 'wpayme' ),
				]
			]
		);
		
		// Transaction Error Email Settings
		add_settings_section(
			'wpayme_pay_email_error_settings',
			__( 'Transaction Error Email Settings', 'wpayme' ),
			function() {},
			'wpayme_pay_email'
		);
		
		// Enable Error Emails
		add_settings_field(
			'wpayme_pay_email_error_enabled',
			__( 'Send Email On Payment Failure', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay_email',
			'wpayme_pay_email_error_settings',
			[
				'legend'      => __( 'Enable Error Emails', 'wpayme' ),
				'description' => __( 'If checked, plugin will send a notification email when error occurred during payment processing. The email will be sent to the email address specified below.', 'wpayme' ),
				'label_for'   => 'wpayme_pay_email_error_enabled',
				'type'        => 'checkbox',
			]
		);
		
		// Error Email Addresses
		add_settings_field(
			'wpayme_pay_email_error_addresses',
			__( 'Send Error Email To', 'wpayme' ),
			[ $this, 'input_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_error_settings',
			[
				'label_for'   => 'wpayme_pay_email_error_addresses',
				'description' => __( 'Enter recipient address of error email.', 'wpayme' ),
				'classes'     => 'regular-text',
				'type'        => 'text',
			]
		);
		
		// Rate Limit Email Settings
		add_settings_section(
			'wpayme_pay_email_rate_limit_settings',
			__( 'Daily Transaction Rate Limit Email Settings', 'wpayme' ),
			function() {},
			'wpayme_pay_email'
		);
		
		// Enable Rate Limit Emails
		add_settings_field(
			'wpayme_pay_email_rate_limit_enabled',
			__( 'Send Email On Daily Transaction Rate Limit', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay_email',
			'wpayme_pay_email_rate_limit_settings',
			[
				'legend'      => __( 'Enable Rate Limit Emails', 'wpayme' ),
				'description' => __( 'If checked, the plugin will send a notification email if the daily transaction rate limit is reached. This is only applicable if you are not using any captcha option. The email will be sent to the email address specified below.', 'wpayme' ),
				'label_for'   => 'wpayme_pay_email_rate_limit_enabled',
				'type'        => 'checkbox',
			]
		);
		
		// Rate Limit Email Addresses
		add_settings_field(
			'wpayme_pay_email_rate_limit_addresses',
			__( 'Send Transaction Rate Limit Email To', 'wpayme' ),
			[ $this, 'input_element' ],
			'wpayme_pay_email',
			'wpayme_pay_email_rate_limit_settings',
			[
				'label_for'   => 'wpayme_pay_email_rate_limit_addresses',
				'description' => __( 'Enter the recipient address for the rate limit notification email.', 'wpayme' ),
				'classes'     => 'regular-text',
				'type'        => 'text',
			]
		);
		
		// Additional Email Settings
		add_settings_section(
			'wpayme_pay_email_additional_settings',
			__( 'Additional Email Settings', 'wpayme' ),
			function() {},
			'wpayme_pay_email'
		);
		
		// Parallel Email Sending
		add_settings_field(
			'wpayme_pay_email_parallel_sending',
			__( 'Send Emails in Parallel', 'wpayme' ),
			[ $this, 'input_checkbox' ],
			'wpayme_pay_email',
			'wpayme_pay_email_additional_settings',
			[
				'legend'      => __( 'Parallel Email Sending', 'wpayme' ),
				'description' => __( 'This option is not recommended for most sites. Enabling this option can speed up the checkout process for customers. Test this before enabling on production as it may not work properly on some servers.', 'wpayme' ),
				'label_for'   => 'wpayme_pay_email_parallel_sending',
				'type'        => 'checkbox',
			]
		);
	}

	/**
	 * Input text.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function input_element( $args ) {
		$defaults = [
			'type'        => 'text',
			'classes'     => 'regular-text',
			'description' => '',
		];

		$args = wp_parse_args( $args, $defaults );

		$name  = $args['label_for'];
		$value = get_option( $name );

		$element = new Element(
			'input',
			[
				'name'  => $name,
				'id'    => $name,
				'type'  => $args['type'],
				'class' => $args['classes'],
				'value' => $value,
			]
		);

		$element->output();

		if ( ! empty( $args['description'] ) ) {
			printf(
				'<p class="description">%s</p>',
				esc_html( $args['description'] )
			);
		}
	}

	/**
	 * Toggle switch element.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function toggle_switch( $args ) {
		$id    = $args['label_for'];
		$name  = $args['label_for'];
		$value = \array_key_exists( 'value', $args ) ? $args['value'] : get_option( $name );

		echo '<div class="wpayme-toggle-container">';
		echo '<label class="wpayme-toggle-switch">';
		
		$attributes = [
			'name'  => $name,
			'id'    => $id,
			'type'  => 'checkbox',
			'value' => '1',
		];

		if ( $value ) {
			$attributes['checked'] = 'checked';
		}

		if ( \array_key_exists( 'disabled', $args ) && $args['disabled'] ) {
			$attributes['disabled'] = 'disabled';
		}

		$element = new Element( 'input', $attributes );
		$element->output();

		echo '<span class="wpayme-toggle-slider"></span>';
		echo '</label>';
		
		echo '<span class="wpayme-toggle-label">' . esc_html( $args['description'] ) . '</span>';
		echo '</div>';
	}

	/**
	 * Input checkbox.
	 *
	 * @link https://github.com/WordPress/WordPress/blob/4.9.1/wp-admin/options-writing.php#L60-L68
	 * @link https://github.com/WordPress/WordPress/blob/4.9.1/wp-admin/options-reading.php#L110-L141
	 * @param array $args Arguments.
	 * @return void
	 */
	public function input_checkbox( $args ) {
		// Use the toggle switch instead of the standard checkbox
		$this->toggle_switch( $args );
	}

	/**
	 * Input page.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function input_page( $args ) {
		$name = $args['label_for'];

		$selected = get_option( $name, '' );

		if ( false === $selected ) {
			$selected = '';
		}

		wp_dropdown_pages(
			[
				'name'             => esc_attr( $name ),
				'post_type'        => esc_attr( isset( $args['post_type'] ) ? $args['post_type'] : 'page' ),
				'selected'         => esc_attr( $selected ),
				'show_option_none' => esc_attr( isset( $args['show_option_none'] ) ? $args['show_option_none'] : __( '— Select a page —', 'wpayme' ) ),
				'class'            => 'regular-text',
			]
		);
	}

	/**
	 * Select payment method status.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function select_payment_method_status( $args ) {
		$name = $args['label_for'];

		$selected = get_option( $name, '' );

		$statuses = [
			''         => '',
			'active'   => \__( 'Active', 'wpayme' ),
			'inactive' => \__( 'Inactive', 'wpayme' ),
		];

		\printf(
			'<select id="%s" name="%s">',
			\esc_attr( $name ),
			\esc_attr( $name )
		);

		foreach ( $statuses as $status => $label ) {
			\printf(
				'<option value="%s" %s>%s</option>',
				\esc_attr( $status ),
				\selected( $status, $selected, false ),
				\esc_html( $label )
			);
		}

		echo '</select>';
	}

	/**
	 * Render a textarea field.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function textarea_element( $args ) {
		$defaults = [
			'classes'     => 'large-text',
			'description' => '',
			'rows'        => 5,
			'default'     => '',
		];

		$args = wp_parse_args( $args, $defaults );

		$name  = $args['label_for'];
		$value = get_option( $name );
		
		// If no value is set yet, use the default value
		if ( false === $value && ! empty( $args['default'] ) ) {
			$value = $args['default'];
		}

		printf(
			'<textarea name="%1$s" id="%1$s" class="%2$s" rows="%3$s">%4$s</textarea>',
			esc_attr( $name ),
			esc_attr( $args['classes'] ),
			esc_attr( $args['rows'] ),
			esc_textarea( $value )
		);

		if ( ! empty( $args['description'] ) ) {
			printf(
				'<p class="description">%s</p>',
				esc_html( $args['description'] )
			);
		}
	}
	
	/**
	 * Render a WYSIWYG editor field.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function wysiwyg_element( $args ) {
		$defaults = [
			'description' => '',
			'rows'        => 10,
			'default'     => '',
			'media_buttons' => true,
			'teeny'       => false,
			'quicktags'   => true,
		];

		$args = wp_parse_args( $args, $defaults );

		$name  = $args['label_for'];
		$value = get_option( $name );
		
		// If no value is set yet, use the default value
		if ( false === $value && ! empty( $args['default'] ) ) {
			$value = $args['default'];
		}
		
		// Ensure the WordPress editor scripts are loaded
		wp_enqueue_editor();
		
		// Output the editor
		wp_editor(
			$value,
			$name,
			array(
				'textarea_name' => $name,
				'textarea_rows' => $args['rows'],
				'media_buttons' => $args['media_buttons'],
				'teeny'         => $args['teeny'],
				'quicktags'     => $args['quicktags'],
				'editor_class'  => 'wpayme-wysiwyg-editor',
			)
		);

		if ( ! empty( $args['description'] ) ) {
			printf(
				'<p class="description">%s</p>',
				esc_html( $args['description'] )
			);
		}
	}

	/**
	 * Render a dropdown for email content type selection.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function select_content_type( $args ) {
		$name  = $args['label_for'];
		$value = get_option( $name, 'text/html' );
		
		$options = [
			'text/html'  => __( 'HTML', 'wpayme' ),
			'text/plain' => __( 'Plain Text', 'wpayme' ),
		];
		
		echo '<select name="' . esc_attr( $name ) . '" id="' . esc_attr( $name ) . '">';
		
		foreach ( $options as $option_value => $option_label ) {
			echo '<option value="' . esc_attr( $option_value ) . '" ' . selected( $value, $option_value, false ) . '>' . esc_html( $option_label ) . '</option>';
		}
		
		echo '</select>';
		
		if ( ! empty( $args['description'] ) ) {
			printf(
				'<p class="description">%s</p>',
				esc_html( $args['description'] )
			);
		}
	}
	
	/**
	 * Show available email shortcode tags with description.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function email_shortcodes_hint( $args ) {
		$tags = isset( $args['tags'] ) ? $args['tags'] : [];
		
		if ( empty( $tags ) ) {
			return;
		}
		
		echo '<div class="wpayme-email-tags-wrapper">';
		echo '<details>';
		echo '<summary>' . esc_html__( 'Click here to see available tags', 'wpayme' ) . '</summary>';
		echo '<div class="wpayme-email-tags">';
		echo '<p>' . esc_html__( 'You can use the following email tags in the email body field:', 'wpayme' ) . '</p>';
		echo '<table class="wpayme-email-tags-table">';
		
		foreach ( $tags as $tag => $description ) {
			echo '<tr>';
			echo '<td><code>' . esc_html( $tag ) . '</code></td>';
			echo '<td>' . esc_html( $description ) . '</td>';
			echo '</tr>';
		}
		
		echo '</table>';
		echo '</div>';
		echo '</details>';
		echo '</div>';
	}
}
