<?php
/**
 * Fields
 *
 * @author    Wpayme <hi@wpayme.com>
 * @copyright 2024-2025 Wpayme
 * @license   GPL-3.0-or-later
 * @package   Wpayme\WordPress\Pay\Extensions\GravityForms
 */

namespace Wpayme\WordPress\Pay\Extensions\GravityForms;

use GF_Fields;
use Wpayme\WordPress\Pay\Core\Util as Core_Util;

/**
 * Title: WordPress pay extension Gravity Forms fields
 * Description:
 * Copyright: 2024-2025 Wpayme
 * Company: Wpayme
 *
 * @author  Remco Tolsma
 * @version 2.0.0
 * @since   1.0.0
 */
class Fields {
	/**
	 * Issuers field type.
	 */
	const ISSUERS_FIELD_TYPE = 'ideal_issuer_drop_down';

	/**
	 * Payment methods field type.
	 */
	const PAYMENT_METHODS_FIELD_TYPE = 'wpayme_pay_payment_method_selector';

	/**
	 * Construct and initialize custom Gravity Forms fields.
	 */
	public function __construct() {
		// Enable credit card field.
		add_filter( 'gform_enable_credit_card_field', '__return_true' );

		// Register custom fields.
		if ( Core_Util::class_method_exists( 'GF_Fields', 'register' ) ) {
			GF_Fields::register( new IssuersField() );

			// We do some voodoo in the payment methods field class which requires the `gform_gf_field_create` filter added in Gravity Forms 1.9.19.
			if ( GravityForms::version_compare( '1.9.19', '>=' ) ) {
				GF_Fields::register( new PaymentMethodsField() );
			}
		}

		// Add extra fields settings.
		add_action( 'gform_field_standard_settings', [ $this, 'field_standard_settings' ], 10, 2 );
	}

	/**
	 * Add pay field group to the Gravity Forms field groups.
	 *
	 * @see   https://github.com/wp-premium/gravityforms/blob/1.9.19/form_detail.php#L2353-L2368
	 * @since 1.4.7
	 *
	 * @param array $field_groups Field groups.
	 *
	 * @return array
	 */
	public static function add_pay_field_group( $field_groups ) {
		if ( ! isset( $field_groups['wpayme_pay_fields'] ) ) {
			$field_groups['wpayme_pay_fields'] = [
				'name'   => 'wpayme_pay_fields',
				'label'  => __( 'Payment Fields', 'wpayme' ),
				'fields' => [],
			];
		}

		return $field_groups;
	}

	/**
	 * Field standard settings.
	 *
	 * @link https://github.com/wp-premium/gravityforms/blob/1.9.19/form_detail.php#L525
	 *
	 * @param int $position Position of the field settings.
	 * @param int $form_id  Form ID.
	 */
	public function field_standard_settings( $position, $form_id ) {
		if ( 10 !== $position ) {
			return;
		}

		$feeds = FeedsDB::get_feeds_by_form_id( $form_id );

		if ( empty( $feeds ) ) {
			return;
		}

		?>
		<li class="wpayme_pay_config_field_setting field_setting">
			<label for="wpayme_pay_config_field" class="section_label">
				<?php \esc_html_e( 'Payment Gateway Configuration', 'wpayme' ); ?>

				<?php \gform_tooltip( 'form_field_wpayme_pay_config' ); ?>
			</label>

			<select id="wpayme_pay_config_field"
					onchange="SetFieldProperty( 'wpaymePayConfigId', jQuery( this ).val() );" class="fieldwidth-3">
				<option value=""><?php \esc_html_e( '— Use Payment Feed Setting —', 'wpayme' ); ?></option>
				<?php

				$config_ids = \wp_list_pluck( $feeds, 'config_id' );

				$config_ids = \array_unique( $config_ids );

				foreach ( $config_ids as $config_id ) {
					\printf(
						'<option value="%s">%s</option>',
						\esc_attr( $config_id ),
						\esc_html( \get_the_title( $config_id ) )
					);
				}

				?>
			</select>
		</li>

		<li class="wpayme_pay_display_field_setting field_setting">
			<label for="wpayme_pay_display_field" class="section_label">
				<?php \esc_html_e( 'Display Mode', 'wpayme' ); ?>

				<?php \gform_tooltip( 'form_field_wpayme_pay_display' ); ?>
			</label>

			<select id="wpayme_pay_display_field" onchange="SetFieldProperty( 'wpaymePayDisplayMode', jQuery( this ).val() );" class="fieldwidth-3">
				<option value=""><?php \esc_html_e( '— Use field default —', 'wpayme' ); ?></option>
				<option value="select"><?php echo \esc_html_x( 'Select', 'Field display mode', 'wpayme' ); ?></option>
				<option value="icons-24"><?php echo \esc_html_x( 'List with icons', 'Field display mode', 'wpayme' ); ?></option>
				<option value="icons-64"><?php echo \esc_html_x( 'Small icons', 'Field display mode', 'wpayme' ); ?></option>
				<option value="icons-125"><?php echo \esc_html_x( 'Large icons', 'Field display mode', 'wpayme' ); ?></option>
			</select>
		</li>
		<?php
	}
}
