<?php

namespace WeDevs\Wpuf\Fields;

/**
 * Turnstile Field Class
 */
class Form_Field_Cloudflare_Turnstile extends Field_Contract {
    public function __construct() {
        $this->name       = __( 'Cloudflare Turnstile', 'wp-user-frontend' );
        $this->input_type = 'cloudflare_turnstile';
        $this->icon       = 'cloud';

        wp_enqueue_script( 'wpuf-turnstile' );
    }

    /**
     * Render the field in frontend
     *
     * @since 4.0.13
     *
     * @param array $field_settings
     * @param int   $form_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        $enable_turnstile = wpuf_get_option( 'enable_turnstile', 'wpuf_general', 'off' );
        $site_key         = wpuf_get_option( 'turnstile_site_key', 'wpuf_general', '' );
        $theme            = ! empty( $field_settings['turnstile_theme'] ) ? $field_settings['turnstile_theme'] : 'light';
        $size             = ! empty( $field_settings['turnstile_size'] ) ? $field_settings['turnstile_size'] : 'normal';
        $action           = ! empty( $field_settings['turnstile_type'] ) ? $field_settings['turnstile_type'] : 'non-interactive';

        if ( wpuf_is_checkbox_or_toggle_on( $enable_turnstile ) || empty( $site_key ) ) {
            return;
        }

        $action = 'non_interactive' === $action ? 'non-interactive' : $action;
        ?>

        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <div
                <?php if ( 'invisible' === $action ) { ?>
                    style="display: none;"
                <?php } ?>
                id="wpuf-turnstile"
                class="wpuf-fields wpuf-turnstile"></div>
        </li>

        <script>
            window.onloadTurnstileCallback = function () {
                turnstile.render("#wpuf-turnstile", {
                    sitekey: "<?php echo esc_js( $site_key ); ?>",
                    theme:"<?php echo esc_js( $theme ); ?>",
                    size:"<?php echo esc_js( $size ); ?>"
                });
            };
        </script>

        <?php
    }

    /**
     * Custom validator
     *
     * @since 4.0.13
     *
     * @return array
     */
    public function get_validator() {
        return [
            'callback'      => 'has_turnstile_api_keys',
            'button_class'  => 'button-faded',
            'icon'          => WPUF_ASSET_URI . '/images/key-rounded.svg',
            'msg_title'     => sprintf(
                '<span class="wpuf-text-primary">%s</span>',
                __( 'Site key and Secret key', 'wp-user-frontend' )
            ),
            'msg' => sprintf(
                '<span class="wpuf-text-xl wpuf-text-gray-500">%1$s <a class="wpuf-text-primary wpuf-italic wpuf-font-bold" style="text-decoration: underline;" href="%2$s" target="_blank">%3$s</a> %4$s <a class="wpuf-text-primary" href="%5$s" target="_blank">%6$s</a></span>',
                __( 'You need to set Site key and Secret key in', 'wp-user-frontend' ),
                admin_url( 'admin.php?page=wpuf-settings' ),
                __( 'Settings', 'wp-user-frontend' ),
                __( 'in order to use "Cloudflare Turnstile" field.', 'wp-user-frontend' ),
                'https://developers.cloudflare.com/turnstile/',
                __( 'Click here to get the these key', 'wp-user-frontend' )
            ),
            'cta' => sprintf(
                '<a class="%1$s" href="%2$s" target="_blank">%3$s</a>',
                'wpuf-px-[20px] wpuf-py-[10px] !wpuf-border !wpuf-border-solid !wpuf-border-gray-300 !wpuf-rounded-md hover:!wpuf-bg-gray-50 !wpuf-text-black',
                admin_url( 'admin.php?page=wpuf-settings' ),
                __( 'Go to Setting', 'wp-user-frontend' )
            ),
        ];
    }

    /**
     * Get field options setting
     *
     * @since 4.0.13
     *
     * @return array
     */
    public function get_options_settings() {
        $settings = [
            [
                'name'          => 'label',
                'title'         => __( 'Title', 'wp-user-frontend' ),
                'type'          => 'text',
                'section'       => 'basic',
                'priority'      => 10,
                'help_text'     => __( 'Title of the section', 'wp-user-frontend' ),
            ],
            [
                'name'          => 'turnstile_theme',
                'title'         => 'Turnstile Theme',
                'type'          => 'radio',
                'options'       => [
                    'light' => __( 'Light', 'wp-user-frontend' ),
                    'dark'  => __( 'Dark', 'wp-user-frontend' ),
                ],
                'default'       => 'light',
                'section'       => 'basic',
                'priority'      => 12,
                'help_text'     => __( 'Select turnstile theme', 'wp-user-frontend' ),
            ],
            [
                'name'          => 'turnstile_size',
                'title'         => 'Turnstile Size',
                'type'          => 'radio',
                'options'       => [
                    'normal'   => __( 'Normal [Width: 300px, Height: 65px]', 'wp-user-frontend' ),
                    'flexible' => __( 'Flexible [Width: 100% (min: 300px), Height: 65px]', 'wp-user-frontend' ),
                    'compact'  => __( 'Compact [Width: 150px, Height: 140px]', 'wp-user-frontend' ),
                ],
                'default'       => 'normal',
                'section'       => 'basic',
                'priority'      => 13,
                'help_text'     => __( 'Select turnstile size', 'wp-user-frontend' ),
            ],
            [
                'name'      => 'turnstile_type',
                'title'     => 'Turnstile type',
                'type'      => 'radio',
                'options'   => [
                    'managed'         => __( 'Managed (recommended)', 'wp-user-frontend' ),
                    'non_interactive' => __( 'Non-Interactive', 'wp-user-frontend' ),
                    'invisible'       => __( 'Invisible', 'wp-user-frontend' ),
                ],
                'default'   => 'managed',
                'section'   => 'advanced',
                'priority'  => 11,
                'help_text' => __( 'Select turnstile type', 'wp-user-frontend' ),
            ],
        ];

        return apply_filters( 'wpuf_turnstile_field_option_settings', $settings );
    }

    /**
     * Get the field props
     *
     * @since 4.0.13
     *
     * @return array
     */
    public function get_field_props() {

        $props = [
            'input_type'      => 'cloudflare_turnstile',
            'template'        => $this->get_type(),
            'label'           => '',
            'name'            => '',
            'turnstile_type'  => 'managed',
            'turnstile_theme' => 'light',
            'turnstile_size'  => 'normal',
            'is_new'          => true,
            'is_meta'         => 'yes',
            'id'              => 0,
            'wpuf_cond'       => null,
        ];

        return $props;
    }
}
