<?php

namespace SLCA\GravityFormSignature;

use wpCloud\StatelessMedia\Compatibility;

/**
 * Class GravityFormSignature
 */
class GravityFormSignature extends Compatibility {
  const GF_SIGNATURE_PATH = 'gravity_forms/signatures/';

  protected $id = 'gravity-form-signature';
  protected $title = 'Gravity Forms Signature Add-On';
  protected $constant = 'WP_STATELESS_COMPATIBILITY_GF_SIG';
  protected $description = 'Enables support for signature images generated by the signature add-on for Gravity Forms.';
  protected $plugin_file = 'gravityformssignature/signature.php';
  protected $plugin_version;

  /**
   * @param $sm
   */
  public function module_init($sm) {
    $this->plugin_version = defined('GF_SIGNATURE_VERSION') ? GF_SIGNATURE_VERSION : '';

    add_filter('gform_save_field_value', array($this, 'gform_save_field_value'), 10, 5);
    add_filter('gform_signature_delete_file_pre_delete_entry', array($this, 'delete_signature'), 10, 4);
    add_filter('gform_signature_url', array($this, 'get_signature_url'), 10, 4);

    add_filter('sm:sync::syncArgs', array($this, 'sync_args'), 10, 4);

  }

  /**
   * On gform save field value sync file to GCS and alter the file url to GCS link.
   * @param $value
   * @param $lead
   * @param $field
   * @param $form
   * @param $input_id
   * @return array|false|mixed|string
   */
  public function gform_save_field_value($value, $lead, $field, $form, $input_id) {
    if (empty($value)) return $value;

    $type = \GFFormsModel::get_input_type($field);
    
    if ($type == 'signature') {
      $is_stateless = ud_get_stateless_media()->get('sm.mode') === 'stateless';

      try {
        $folder = \GFSignature::get_signatures_folder();
        $file_path = $folder . $value;

        // For stateless mode there is no way to override signature upload path in GFSignature::get_signatures_folder()
        // so we have to move the file to the proper location
        if ( $is_stateless ) {
          $old_path = $file_path;
          $file_path = ud_get_stateless_media()->get_gs_path() . '/' . self::GF_SIGNATURE_PATH . $value;

          if ( !class_exists('\WP_Filesystem_Direct') ) {
            require_once(ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php');
            require_once(ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php');
          }

          $filesystem = new \WP_Filesystem_Direct( false );
          $filesystem->move($old_path, $file_path, true);
        }

        $name = self::GF_SIGNATURE_PATH . $value;
        $name = apply_filters('wp_stateless_file_name', $name, 0);

        do_action( 'sm:sync::syncFile', $name, $file_path, true );
      } catch (\Throwable $th) {
      }
    }
    return $value;
  }

  /**
   * [newer version of Gravity Forms Signature]
   * Get GSC URL for signature image.
   *
   * Also doing sync on the fly for previous entries.
   */
  public function get_signature_url($url, $filename, $form_id, $field_id) {
    $mode = ud_get_stateless_media()->get('sm.mode');

    if ( $mode === 'disabled' ) {
      return $url;
    }

    // Sync signature file
    try {
      $folder = \GFSignature::get_signatures_folder();
      $absolute_path = $folder . $filename . '.png';
      $name = apply_filters('wp_stateless_file_name', $absolute_path, false);
      do_action('sm:sync::syncFile', $name, $absolute_path);

      if ( $mode !== 'backup' ) {
        $url = ud_get_stateless_media()->get_gs_host() . '/' . $name;
      }
    } catch (\Throwable $th) {
    }
    
    return $url;
  }

  /**
   * Deleting signature file from GCS.
   */
  public function delete_signature($return, $form, $lead_id, $field_id) {
    try {
      $lead = \RGFormsModel::get_lead($lead_id);
      $name = rgar($lead, $field_id);

      do_action('sm:sync::deleteFile', self::GF_SIGNATURE_PATH . $name);
    } catch (\Throwable $th) {
    }
    return $return;
  }

  /**
   * Update args when uploading/syncing GF file to GCS.
   * 
   * @param array $args
   * @param string $name
   * @param string $file
   * @param bool $force
   * 
   * @return array
   */
  public function sync_args($args, $name, $file, $force) {
    if ( strpos($name, self::GF_SIGNATURE_PATH) === false ) {
      return $args;
    }

    if ( ud_get_stateless_media()->is_mode('stateless') ) {
      $args['name_with_root'] = false;
    }

    $args['source'] = 'Gravity Forms Signature';
    $args['source_version'] = $this->plugin_version;

    return $args;
  }
}
