<?php

/**
 * @package WP Product Feed Manager/User Interface/Functions
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Starts the Feed Editor page for a standard feed.
 */
/** @noinspection PhpUnused */
function wppfm_open_product_feed_page() {
	$add_new_feed_page = new WPPFM_Add_Feed_Editor_Page;
	$add_new_feed_page->show();
}

/**
 * Starts the feed list page.
 */
function wppfm_feed_list_page() {
	wppfm_check_prerequisites();

	$add_feed_list_page = new WPPFM_Add_Feed_List_Page();
	$add_feed_list_page->show();
}

/**
 * Starts the correct Feed Editor page.
 */
function wppfm_feed_editor_page() {
	wppfm_check_prerequisites();

	$feed_type = wppfm_get_url_parameter('feed-type' ) ?? 'product-feed';

	switch ( $feed_type ) {
		case 'google-product-review-feed':
			wppfm_rf_open_review_feed_page();
			break;
		case 'google-merchant-promotions-feed':
			wppfm_pf_open_promotions_feed_page();
			break;
		default:
			wppfm_open_product_feed_page();
			break;
	}
}

/**
 * Starts the settings page.
 */
function wppfm_settings_page() {
	wppfm_check_prerequisites();

	$add_settings_page = new WPPFM_Add_Settings_Page();
	$add_settings_page->show();
}

/**
 * Starts the support page.
 */
function wppfm_support_page() {
	$add_support_page = new WPPFM_Add_Support_Page();
	$add_support_page->show();
}


/**
 * Returns an array of possible feed types that can be altered using the wppfm_feed_types filter.
 *
 * @return array with possible feed types.
 */
function wppfm_list_feed_type_text() {

	return apply_filters(
		'wppfm_feed_types',
		array(
			'1' => 'Product Feed',
			'4' => 'Google Local Product Inventory Feed',
			'5' => 'Google Dynamic Remarketing Feed',
			'6' => 'Google Vehicle Ads Feed',
			'7' => 'Google Dynamic Search Ads Feed',
			'8' => 'Google Local Product Feed',
		)
	);
}

/**
 * Returns a string with the plugin's WordPress menu icon.
 *
 * @return string with the icon in svg format.
 */
function wppfm_get_menu_icon_svg(  ) {
	/** @noinspection SpellCheckingInspection */
	return 'data:image/svg+xml;base64,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';
	//return 'data:image/svg+xml;base64,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';
}

/**
 * Checks if the plugin's prerequisites are met.
 *
 * @return void
 */
function wppfm_check_prerequisites() {
	if ( ! wppfm_wc_installed_and_active() ) {
		wppfm_you_have_no_woocommerce_installed_message();
		exit;
	}

	if ( ! wppfm_wc_min_version_required() ) {
		wppfm_update_your_woocommerce_version_message();
		exit;
	}
}

