<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class WC_Stripe_Express_Checkout_Button_States
 *
 * Provides a map between WC states and Payment Request API states.
 * The list is based on libaddressinput: https://github.com/google/libaddressinput,
 * which is used by Chromium based browsers in the native Payment Request address dialog.
 *
 * Known issues/inconsistencies:
 *    1. WC provides a dropdown list of states, but there's no state field in Chrome for the following countries:
 *        AO (Angola), BD (Bangladesh), BG (Bulgaria), BJ (Benin), BO (Bolivia), DO (Dominican Republic),
 *        DZ (Algeria), GH (Ghana), GT (Guatemala), HU (Hungary), KE (Kenya), LA (Laos),
 *        LR (Liberia), LT (Lithuania), MD (Moldova), NA (Namibia), NP (Nepal), PK (Pakistan),
 *        PY (Paraguay), RO (Romania), TZ (Tanzania), UG (Uganda), UM (United States Minor Outlying Islands),
 *        ZA (South Africa), ZM (Zambia).
 *    2. Chrome does not provide a dropdown list of states for 161 countries in total, out of the 249 countries WC supports,
 *        so the countries in which the state field is required by WC, and not provided by the browser are not supported.
 *    3. Chrome does not provide a zip/postal code field for 60 out of the 249 countries WC supports. Only for 5 countries
 *        the zip code field is missing while the state field is present: BS (Bahamas), PA (Panama), SC (Seychelles),
 *        SR (Suriname), TV (Tuvalu). Several other countries provide an optional zip code field.
 *    4. WC expects it, but there's no city field in the Payment Request API for the following:
 *        JP (Japan), MO (Macao), TN (Tunisia), KY (Cayman Islands), GI (Gibraltar).
 *    5. The following countries are not listed in WC:
 *        AC (Ascension Island), IC (Canary Islands), EA (Ceuta & Melilla), CP (Clipperton Island), DG (Diego Garcia),
 *        TA (Tristan da Cunha), XK (Kosovo).
 *
 * @since 10.3.0
 */
class WC_Stripe_Express_Checkout_Button_States {
	/**
	 * A list of states which is compatible with Chromium based browsers for the Payment Request API.
	 * If the input comes from Chrome, we will always match with `code`, but if the request comes from
	 * Safari or other browsers which don't provide a dropdown list, we can match one of the following:
	 *    - [0] = code (string)
	 *    - [1] = name (string)
	 *    - [2] = localName (string|null)
	 *
	 * @var array
	 */
	const STATES = [
		// Afghanistan.
		'AF' => [],
		// Angola.
		'AO' => [],
		// Argentina.
		'AR' => [
			'C' => [ 'Ciudad Autónoma de Buenos Aires', 'Ciudad Autónoma de Buenos Aires', null ],
			'B' => [ 'Buenos Aires', 'Buenos Aires', null ],
			'K' => [ 'Catamarca', 'Catamarca', null ],
			'H' => [ 'Chaco', 'Chaco', null ],
			'U' => [ 'Chubut', 'Chubut', null ],
			'X' => [ 'Córdoba', 'Córdoba', null ],
			'W' => [ 'Corrientes', 'Corrientes', null ],
			'E' => [ 'Entre Ríos', 'Entre Ríos', null ],
			'P' => [ 'Formosa', 'Formosa', null ],
			'Y' => [ 'Jujuy', 'Jujuy', null ],
			'L' => [ 'La Pampa', 'La Pampa', null ],
			'F' => [ 'La Rioja', 'La Rioja', null ],
			'M' => [ 'Mendoza', 'Mendoza', null ],
			'N' => [ 'Misiones', 'Misiones', null ],
			'Q' => [ 'Neuquén', 'Neuquén', null ],
			'R' => [ 'Río Negro', 'Río Negro', null ],
			'A' => [ 'Salta', 'Salta', null ],
			'J' => [ 'San Juan', 'San Juan', null ],
			'D' => [ 'San Luis', 'San Luis', null ],
			'Z' => [ 'Santa Cruz', 'Santa Cruz', null ],
			'S' => [ 'Santa Fe', 'Santa Fe', null ],
			'G' => [ 'Santiago del Estero', 'Santiago del Estero', null ],
			'V' => [ 'Tierra del Fuego', 'Tierra del Fuego', null ],
			'T' => [ 'Tucumán', 'Tucumán', null ],
		],
		// Austria.
		'AT' => [],
		// Australia.
		'AU' => [
			'ACT' => [ 'ACT', 'Australian Capital Territory', null ],
			'NSW' => [ 'NSW', 'New South Wales', null ],
			'NT'  => [ 'NT', 'Northern Territory', null ],
			'QLD' => [ 'QLD', 'Queensland', null ],
			'SA'  => [ 'SA', 'South Australia', null ],
			'TAS' => [ 'TAS', 'Tasmania', null ],
			'VIC' => [ 'VIC', 'Victoria', null ],
			'WA'  => [ 'WA', 'Western Australia', null ],
			// [ 'JBT', 'Jervis Bay Territory', null ],
		],
		// Aland Islands.
		'AX' => [],
		// Bangladesh.
		'BD' => [],
		// Belgium.
		'BE' => [],
		// Bulgaria.
		'BG' => [],
		// Bahrain.
		'BH' => [],
		// Burundi.
		'BI' => [],
		// Benin.
		'BJ' => [],
		// Bolivia.
		'BO' => [],
		// Brazil.
		'BR' => [
			'AC' => [ 'AC', 'Acre', null ],
			'AL' => [ 'AL', 'Alagoas', null ],
			'AP' => [ 'AP', 'Amapá', null ],
			'AM' => [ 'AM', 'Amazonas', null ],
			'BA' => [ 'BA', 'Bahia', null ],
			'CE' => [ 'CE', 'Ceará', null ],
			'DF' => [ 'DF', 'Distrito Federal', null ],
			'ES' => [ 'ES', 'Espírito Santo', null ],
			'GO' => [ 'GO', 'Goiás', null ],
			'MA' => [ 'MA', 'Maranhão', null ],
			'MT' => [ 'MT', 'Mato Grosso', null ],
			'MS' => [ 'MS', 'Mato Grosso do Sul', null ],
			'MG' => [ 'MG', 'Minas Gerais', null ],
			'PA' => [ 'PA', 'Pará', null ],
			'PB' => [ 'PB', 'Paraíba', null ],
			'PR' => [ 'PR', 'Paraná', null ],
			'PE' => [ 'PE', 'Pernambuco', null ],
			'PI' => [ 'PI', 'Piauí', null ],
			'RJ' => [ 'RJ', 'Rio de Janeiro', null ],
			'RN' => [ 'RN', 'Rio Grande do Norte', null ],
			'RS' => [ 'RS', 'Rio Grande do Sul', null ],
			'RO' => [ 'RO', 'Rondônia', null ],
			'RR' => [ 'RR', 'Roraima', null ],
			'SC' => [ 'SC', 'Santa Catarina', null ],
			'SP' => [ 'SP', 'São Paulo', null ],
			'SE' => [ 'SE', 'Sergipe', null ],
			'TO' => [ 'TO', 'Tocantins', null ],
		],
		// Canada.
		'CA' => [
			'AB' => [ 'AB', 'Alberta', 'Alberta' ],
			'BC' => [ 'BC', 'British Columbia', 'Colombie-Britannique' ],
			'MB' => [ 'MB', 'Manitoba', 'Manitoba' ],
			'NB' => [ 'NB', 'New Brunswick', 'Nouveau-Brunswick' ],
			'NL' => [ 'NL', 'Newfoundland and Labrador', 'Terre-Neuve-et-Labrador' ],
			'NT' => [ 'NT', 'Northwest Territories', 'Territoires du Nord-Ouest' ],
			'NS' => [ 'NS', 'Nova Scotia', 'Nouvelle-Écosse' ],
			'NU' => [ 'NU', 'Nunavut', 'Nunavut' ],
			'ON' => [ 'ON', 'Ontario', 'Ontario' ],
			'PE' => [ 'PE', 'Prince Edward Island', 'Île-du-Prince-Édouard' ],
			'QC' => [ 'QC', 'Quebec', 'Québec' ],
			'SK' => [ 'SK', 'Saskatchewan', 'Saskatchewan' ],
			'YT' => [ 'YT', 'Yukon', 'Yukon' ],
		],
		// Switzerland.
		'CH' => [],
		// China.
		'CN' => [
			'CN1'  => [ 'Yunnan Sheng', 'Yunnan Sheng', '云南省' ],
			'CN2'  => [ 'Beijing Shi', 'Beijing Shi', '北京市' ],
			'CN3'  => [ 'Tianjin Shi', 'Tianjin Shi', '天津市' ],
			'CN4'  => [ 'Hebei Sheng', 'Hebei Sheng', '河北省' ],
			'CN5'  => [ 'Shanxi Sheng', 'Shanxi Sheng', '山西省' ],
			'CN6'  => [ 'Neimenggu Zizhiqu', 'Neimenggu Zizhiqu', '内蒙古' ],
			'CN7'  => [ 'Liaoning Sheng', 'Liaoning Sheng', '辽宁省' ],
			'CN8'  => [ 'Jilin Sheng', 'Jilin Sheng', '吉林省' ],
			'CN9'  => [ 'Heilongjiang Sheng', 'Heilongjiang Sheng', '黑龙江省' ],
			'CN10' => [ 'Shanghai Shi', 'Shanghai Shi', '上海市' ],
			'CN11' => [ 'Jiangsu Sheng', 'Jiangsu Sheng', '江苏省' ],
			'CN12' => [ 'Zhejiang Sheng', 'Zhejiang Sheng', '浙江省' ],
			'CN13' => [ 'Anhui Sheng', 'Anhui Sheng', '安徽省' ],
			'CN14' => [ 'Fujian Sheng', 'Fujian Sheng', '福建省' ],
			'CN15' => [ 'Jiangxi Sheng', 'Jiangxi Sheng', '江西省' ],
			'CN16' => [ 'Shandong Sheng', 'Shandong Sheng', '山东省' ],
			'CN17' => [ 'Henan Sheng', 'Henan Sheng', '河南省' ],
			'CN18' => [ 'Hubei Sheng', 'Hubei Sheng', '湖北省' ],
			'CN19' => [ 'Hunan Sheng', 'Hunan Sheng', '湖南省' ],
			'CN20' => [ 'Guangdong Sheng', 'Guangdong Sheng', '广东省' ],
			'CN21' => [ 'Guangxi Zhuangzuzizhiqu', 'Guangxi Zhuangzuzizhiqu', '广西' ],
			'CN22' => [ 'Hainan Sheng', 'Hainan Sheng', '海南省' ],
			'CN23' => [ 'Chongqing Shi', 'Chongqing Shi', '重庆市' ],
			'CN24' => [ 'Sichuan Sheng', 'Sichuan Sheng', '四川省' ],
			'CN25' => [ 'Guizhou Sheng', 'Guizhou Sheng', '贵州省' ],
			'CN26' => [ 'Shaanxi Sheng', 'Shaanxi Sheng', '陕西省' ],
			'CN27' => [ 'Gansu Sheng', 'Gansu Sheng', '甘肃省' ],
			'CN28' => [ 'Qinghai Sheng', 'Qinghai Sheng', '青海省' ],
			'CN29' => [ 'Ningxia Huizuzizhiqu', 'Ningxia Huizuzizhiqu', '宁夏' ],
			'CN30' => [ 'Macau', 'Macau', '澳门' ],
			'CN31' => [ 'Xizang Zizhiqu', 'Xizang Zizhiqu', '西藏' ],
			'CN32' => [ 'Xinjiang Weiwuerzizhiqu', 'Xinjiang Weiwuerzizhiqu', '新疆' ],
			// [ 'Taiwan', 'Taiwan', '台湾' ],
			// [ 'Hong Kong', 'Hong Kong', '香港' ],
		],
		// Czech Republic.
		'CZ' => [],
		// Germany.
		'DE' => [],
		// Denmark.
		'DK' => [],
		// Dominican Republic.
		'DO' => [],
		// Algeria.
		'DZ' => [],
		// Estonia.
		'EE' => [],
		// Egypt.
		'EG' => [
			'EGALX' => [ 'Alexandria Governorate', 'Alexandria Governorate', 'الإسكندرية' ],
			'EGASN' => [ 'Aswan Governorate', 'Aswan Governorate', 'أسوان' ],
			'EGAST' => [ 'Asyut Governorate', 'Asyut Governorate', 'أسيوط' ],
			'EGBA'  => [ 'Red Sea Governorate', 'Red Sea Governorate', 'البحر الأحمر' ],
			'EGBH'  => [ 'El Beheira Governorate', 'El Beheira Governorate', 'البحيرة' ],
			'EGBNS' => [ 'Beni Suef Governorate', 'Beni Suef Governorate', 'بني سويف' ],
			'EGC'   => [ 'Cairo Governorate', 'Cairo Governorate', 'القاهرة' ],
			'EGDK'  => [ 'Dakahlia Governorate', 'Dakahlia Governorate', 'الدقهلية' ],
			'EGDT'  => [ 'Damietta Governorate', 'Damietta Governorate', 'دمياط' ],
			'EGFYM' => [ 'Faiyum Governorate', 'Faiyum Governorate', 'الفيوم' ],
			'EGGH'  => [ 'Gharbia Governorate', 'Gharbia Governorate', 'الغربية' ],
			'EGGZ'  => [ 'Giza Governorate', 'Giza Governorate', 'الجيزة' ],
			'EGIS'  => [ 'Ismailia Governorate', 'Ismailia Governorate', 'الإسماعيلية' ],
			'EGJS'  => [ 'South Sinai Governorate', 'South Sinai Governorate', 'جنوب سيناء' ],
			'EGKB'  => [ 'Qalyubia Governorate', 'Qalyubia Governorate', 'القليوبية' ],
			'EGKFS' => [ 'Kafr El Sheikh Governorate', 'Kafr El Sheikh Governorate', 'كفر الشيخ' ],
			'EGKN'  => [ 'Qena Governorate', 'Qena Governorate', 'قنا' ],
			'EGLX'  => [ 'Luxor Governorate', 'Luxor Governorate', 'الأقصر' ],
			'EGMN'  => [ 'Menia Governorate', 'Menia Governorate', 'المنيا' ],
			'EGMNF' => [ 'Menofia Governorate', 'Menofia Governorate', 'المنوفية' ],
			'EGMT'  => [ 'Matrouh Governorate', 'Matrouh Governorate', 'مطروح' ],
			'EGPTS' => [ 'Port Said Governorate', 'Port Said Governorate', 'بورسعيد' ],
			'EGSHG' => [ 'Sohag Governorate', 'Sohag Governorate', 'سوهاج' ],
			'EGSHR' => [ 'Ash Sharqia Governorate', 'Ash Sharqia Governorate', 'الشرقية' ],
			'EGSIN' => [ 'North Sinai Governorate', 'North Sinai Governorate', 'شمال سيناء' ],
			'EGSUZ' => [ 'Suez Governorate', 'Suez Governorate', 'السويس' ],
			'EGWAD' => [ 'New Valley Governorate', 'New Valley Governorate', 'الوادي الجديد' ],
		],
		// Spain.
		'ES' => [
			'C'  => [ 'A Coruña', 'A Coruña', null ],
			'VI' => [ 'Álava', 'Álava', null ],
			'AB' => [ 'Albacete', 'Albacete', null ],
			'A'  => [ 'Alicante', 'Alicante', null ],
			'AL' => [ 'Almería', 'Almería', null ],
			'O'  => [ 'Asturias', 'Asturias', null ],
			'AV' => [ 'Ávila', 'Ávila', null ],
			'BA' => [ 'Badajoz', 'Badajoz', null ],
			'PM' => [ 'Balears', 'Balears', null ],
			'B'  => [ 'Barcelona', 'Barcelona', null ],
			'BU' => [ 'Burgos', 'Burgos', null ],
			'CC' => [ 'Cáceres', 'Cáceres', null ],
			'CA' => [ 'Cádiz', 'Cádiz', null ],
			'S'  => [ 'Cantabria', 'Cantabria', null ],
			'CS' => [ 'Castellón', 'Castellón', null ],
			'CE' => [ 'Ceuta', 'Ceuta', null ],
			'CR' => [ 'Ciudad Real', 'Ciudad Real', null ],
			'CO' => [ 'Córdoba', 'Córdoba', null ],
			'CU' => [ 'Cuenca', 'Cuenca', null ],
			'GI' => [ 'Girona', 'Girona', null ],
			'GR' => [ 'Granada', 'Granada', null ],
			'GU' => [ 'Guadalajara', 'Guadalajara', null ],
			'SS' => [ 'Guipúzcoa', 'Guipúzcoa', null ],
			'H'  => [ 'Huelva', 'Huelva', null ],
			'HU' => [ 'Huesca', 'Huesca', null ],
			'J'  => [ 'Jaén', 'Jaén', null ],
			'LO' => [ 'La Rioja', 'La Rioja', null ],
			'GC' => [ 'Las Palmas', 'Las Palmas', null ],
			'LE' => [ 'León', 'León', null ],
			'L'  => [ 'Lleida', 'Lleida', null ],
			'LU' => [ 'Lugo', 'Lugo', null ],
			'M'  => [ 'Madrid', 'Madrid', null ],
			'MA' => [ 'Málaga', 'Málaga', null ],
			'ML' => [ 'Melilla', 'Melilla', null ],
			'MU' => [ 'Murcia', 'Murcia', null ],
			'NA' => [ 'Navarra', 'Navarra', null ],
			'OR' => [ 'Ourense', 'Ourense', null ],
			'P'  => [ 'Palencia', 'Palencia', null ],
			'PO' => [ 'Pontevedra', 'Pontevedra', null ],
			'SA' => [ 'Salamanca', 'Salamanca', null ],
			'TF' => [ 'Santa Cruz de Tenerife', 'Santa Cruz de Tenerife', null ],
			'SG' => [ 'Segovia', 'Segovia', null ],
			'SE' => [ 'Sevilla', 'Sevilla', null ],
			'SO' => [ 'Soria', 'Soria', null ],
			'T'  => [ 'Tarragona', 'Tarragona', null ],
			'TE' => [ 'Teruel', 'Teruel', null ],
			'TO' => [ 'Toledo', 'Toledo', null ],
			'V'  => [ 'Valencia', 'Valencia', null ],
			'VA' => [ 'Valladolid', 'Valladolid', null ],
			'BI' => [ 'Vizcaya', 'Vizcaya', null ],
			'ZA' => [ 'Zamora', 'Zamora', null ],
			'Z'  => [ 'Zaragoza', 'Zaragoza', null ],
		],
		// Finland.
		'FI' => [],
		// France.
		'FR' => [],
		// French Guiana.
		'GF' => [],
		// Ghana.
		'GH' => [],
		// Guadeloupe.
		'GP' => [],
		// Greece.
		'GR' => [],
		// Guatemala.
		'GT' => [],
		// Hong Kong.
		'HK' => [
			'HONG KONG'       => [ 'Hong Kong Island', 'Hong Kong Island', '香港島' ],
			'KOWLOON'         => [ 'Kowloon', 'Kowloon', '九龍' ],
			'NEW TERRITORIES' => [ 'New Territories', 'New Territories', '新界' ],
		],
		// Hungary.
		'HU' => [],
		// Indonesia.
		'ID' => [
			'AC' => [ 'Aceh', 'Aceh', null ],
			'SU' => [ 'Sumatera Utara', 'Sumatera Utara', null ],
			'SB' => [ 'Sumatera Barat', 'Sumatera Barat', null ],
			'RI' => [ 'Riau', 'Riau', null ],
			'KR' => [ 'Kepulauan Riau', 'Kepulauan Riau', null ],
			'JA' => [ 'Jambi', 'Jambi', null ],
			'SS' => [ 'Sumatera Selatan', 'Sumatera Selatan', null ],
			'BB' => [ 'Kepulauan Bangka Belitung', 'Kepulauan Bangka Belitung', null ],
			'BE' => [ 'Bengkulu', 'Bengkulu', null ],
			'LA' => [ 'Lampung', 'Lampung', null ],
			'JK' => [ 'DKI Jakarta', 'DKI Jakarta', null ],
			'JB' => [ 'Jawa Barat', 'Jawa Barat', null ],
			'BT' => [ 'Banten', 'Banten', null ],
			'JT' => [ 'Jawa Tengah', 'Jawa Tengah', null ],
			'JI' => [ 'Jawa Timur', 'Jawa Timur', null ],
			'YO' => [ 'Daerah Istimewa Yogyakarta', 'Daerah Istimewa Yogyakarta', null ],
			'BA' => [ 'Bali', 'Bali', null ],
			'NB' => [ 'Nusa Tenggara Barat', 'Nusa Tenggara Barat', null ],
			'NT' => [ 'Nusa Tenggara Timur', 'Nusa Tenggara Timur', null ],
			'KB' => [ 'Kalimantan Barat', 'Kalimantan Barat', null ],
			'KT' => [ 'Kalimantan Tengah', 'Kalimantan Tengah', null ],
			'KI' => [ 'Kalimantan Timur', 'Kalimantan Timur', null ],
			'KS' => [ 'Kalimantan Selatan', 'Kalimantan Selatan', null ],
			'KU' => [ 'Kalimantan Utara', 'Kalimantan Utara', null ],
			'SA' => [ 'Sulawesi Utara', 'Sulawesi Utara', null ],
			'ST' => [ 'Sulawesi Tengah', 'Sulawesi Tengah', null ],
			'SG' => [ 'Sulawesi Tenggara', 'Sulawesi Tenggara', null ],
			'SR' => [ 'Sulawesi Barat', 'Sulawesi Barat', null ],
			'SN' => [ 'Sulawesi Selatan', 'Sulawesi Selatan', null ],
			'GO' => [ 'Gorontalo', 'Gorontalo', null ],
			'MA' => [ 'Maluku', 'Maluku', null ],
			'MU' => [ 'Maluku Utara', 'Maluku Utara', null ],
			'PA' => [ 'Papua', 'Papua', null ],
			'PB' => [ 'Papua Barat', 'Papua Barat', null ],
			// [ 'Kalimantan Tengah', 'Kalimantan Tengah', null ],
			// [ 'Kalimantan Timur', 'Kalimantan Timur', null ],
		],
		// Ireland.
		'IE' => [
			'CW' => [ 'Co. Carlow', 'Co. Carlow', null ],
			'CN' => [ 'Co. Cavan', 'Co. Cavan', null ],
			'CE' => [ 'Co. Clare', 'Co. Clare', null ],
			'CO' => [ 'Co. Cork', 'Co. Cork', null ],
			'DL' => [ 'Co. Donegal', 'Co. Donegal', null ],
			'D'  => [ 'Co. Dublin', 'Co. Dublin', null ],
			'G'  => [ 'Co. Galway', 'Co. Galway', null ],
			'KY' => [ 'Co. Kerry', 'Co. Kerry', null ],
			'KE' => [ 'Co. Kildare', 'Co. Kildare', null ],
			'KK' => [ 'Co. Kilkenny', 'Co. Kilkenny', null ],
			'LS' => [ 'Co. Laois', 'Co. Laois', null ],
			'LM' => [ 'Co. Leitrim', 'Co. Leitrim', null ],
			'LK' => [ 'Co. Limerick', 'Co. Limerick', null ],
			'LD' => [ 'Co. Longford', 'Co. Longford', null ],
			'LH' => [ 'Co. Louth', 'Co. Louth', null ],
			'MO' => [ 'Co. Mayo', 'Co. Mayo', null ],
			'MH' => [ 'Co. Meath', 'Co. Meath', null ],
			'MN' => [ 'Co. Monaghan', 'Co. Monaghan', null ],
			'OY' => [ 'Co. Offaly', 'Co. Offaly', null ],
			'RN' => [ 'Co. Roscommon', 'Co. Roscommon', null ],
			'SO' => [ 'Co. Sligo', 'Co. Sligo', null ],
			'TA' => [ 'Co. Tipperary', 'Co. Tipperary', null ],
			'WD' => [ 'Co. Waterford', 'Co. Waterford', null ],
			'WH' => [ 'Co. Westmeath', 'Co. Westmeath', null ],
			'WX' => [ 'Co. Wexford', 'Co. Wexford', null ],
			'WW' => [ 'Co. Wicklow', 'Co. Wicklow', null ],
		],
		// Israel.
		'IL' => [],
		// Isle of Man.
		'IM' => [],
		// India.
		'IN' => [
			'AP' => [ 'Andhra Pradesh', 'Andhra Pradesh', null ],
			'AR' => [ 'Arunachal Pradesh', 'Arunachal Pradesh', null ],
			'AS' => [ 'Assam', 'Assam', null ],
			'BR' => [ 'Bihar', 'Bihar', null ],
			'CT' => [ 'Chhattisgarh', 'Chhattisgarh', null ],
			'GA' => [ 'Goa', 'Goa', null ],
			'GJ' => [ 'Gujarat', 'Gujarat', null ],
			'HR' => [ 'Haryana', 'Haryana', null ],
			'HP' => [ 'Himachal Pradesh', 'Himachal Pradesh', null ],
			'JK' => [ 'Jammu and Kashmir', 'Jammu & Kashmir', null ],
			'JH' => [ 'Jharkhand', 'Jharkhand', null ],
			'KA' => [ 'Karnataka', 'Karnataka', null ],
			'KL' => [ 'Kerala', 'Kerala', null ],
			// 'LA' => __( 'Ladakh', 'woocommerce' ),
			'MP' => [ 'Madhya Pradesh', 'Madhya Pradesh', null ],
			'MH' => [ 'Maharashtra', 'Maharashtra', null ],
			'MN' => [ 'Manipur', 'Manipur', null ],
			'ML' => [ 'Meghalaya', 'Meghalaya', null ],
			'MZ' => [ 'Mizoram', 'Mizoram', null ],
			'NL' => [ 'Nagaland', 'Nagaland', null ],
			'OR' => [ 'Odisha', 'Odisha', null ],
			'PB' => [ 'Punjab', 'Punjab', null ],
			'RJ' => [ 'Rajasthan', 'Rajasthan', null ],
			'SK' => [ 'Sikkim', 'Sikkim', null ],
			'TN' => [ 'Tamil Nadu', 'Tamil Nadu', null ],
			'TS' => [ 'Telangana', 'Telangana', null ],
			'TR' => [ 'Tripura', 'Tripura', null ],
			'UK' => [ 'Uttarakhand', 'Uttarakhand', null ],
			'UP' => [ 'Uttar Pradesh', 'Uttar Pradesh', null ],
			'WB' => [ 'West Bengal', 'West Bengal', null ],
			'AN' => [ 'Andaman and Nicobar Islands', 'Andaman & Nicobar', null ],
			'CH' => [ 'Chandigarh', 'Chandigarh', null ],
			'DN' => [ 'Dadra and Nagar Haveli', 'Dadra & Nagar Haveli', null ],
			'DD' => [ 'Daman and Diu', 'Daman & Diu', null ],
			'DL' => [ 'Delhi', 'Delhi', null ],
			'LD' => [ 'Lakshadweep', 'Lakshadweep', null ],
			'PY' => [ 'Puducherry', 'Puducherry', null ],
		],
		// Iran.
		'IR' => [
			'KHZ' => [ 'Khuzestan Province', 'Khuzestan Province', 'استان خوزستان' ],
			'THR' => [ 'Tehran Province', 'Tehran Province', 'استان تهران' ],
			'ILM' => [ 'Ilam Province', 'Ilam Province', 'استان ایلام' ],
			'BHR' => [ 'Bushehr Province', 'Bushehr Province', 'استان بوشهر' ],
			'ADL' => [ 'Ardabil Province', 'Ardabil Province', 'استان اردبیل' ],
			'ESF' => [ 'Isfahan Province', 'Isfahan Province', 'استان اصفهان' ],
			'YZD' => [ 'Yazd Province', 'Yazd Province', 'استان یزد' ],
			'KRH' => [ 'Kermanshah Province', 'Kermanshah Province', 'استان کرمانشاه' ],
			'KRN' => [ 'Kerman Province', 'Kerman Province', 'استان کرمان' ],
			'HDN' => [ 'Hamadan Province', 'Hamadan Province', 'استان همدان' ],
			'GZN' => [ 'Qazvin Province', 'Qazvin Province', 'استان قزوین' ],
			'ZJN' => [ 'Zanjan Province', 'Zanjan Province', 'استان زنجان' ],
			'LRS' => [ 'Lorestan Province', 'Lorestan Province', 'استان لرستان' ],
			'ABZ' => [ 'Alborz Province', 'Alborz Province', 'استان البرز' ],
			'EAZ' => [ 'East Azerbaijan Province', 'East Azerbaijan Province', 'استان آذربایجان شرقی' ],
			'WAZ' => [ 'West Azerbaijan Province', 'West Azerbaijan Province', 'استان آذربایجان غربی' ],
			'CHB' => [ 'Chaharmahal and Bakhtiari Province', 'Chaharmahal and Bakhtiari Province', 'استان چهارمحال و بختیاری' ],
			'SKH' => [ 'South Khorasan Province', 'South Khorasan Province', 'استان خراسان جنوبی' ],
			'RKH' => [ 'Razavi Khorasan Province', 'Razavi Khorasan Province', 'استان خراسان رضوی' ],
			'NKH' => [ 'North Khorasan Province', 'North Khorasan Province', 'استان خراسان شمالی' ],
			'SMN' => [ 'Semnan Province', 'Semnan Province', 'استان سمنان' ],
			'FRS' => [ 'Fars Province', 'Fars Province', 'استان فارس' ],
			'QHM' => [ 'Qom Province', 'Qom Province', 'استان قم' ],
			'KRD' => [ 'Kurdistan Province', 'Kurdistan Province', 'استان کردستان' ],
			'KBD' => [ 'Kohgiluyeh and Boyer-Ahmad Province', 'Kohgiluyeh and Boyer-Ahmad Province', 'استان کهگیلویه و بویراحمد' ],
			'GLS' => [ 'Golestan Province', 'Golestan Province', 'استان گلستان' ],
			'GIL' => [ 'Gilan Province', 'Gilan Province', 'استان گیلان' ],
			'MZN' => [ 'Mazandaran Province', 'Mazandaran Province', 'استان مازندران' ],
			'MKZ' => [ 'Markazi Province', 'Markazi Province', 'استان مرکزی' ],
			'HRZ' => [ 'Hormozgan Province', 'Hormozgan Province', 'استان هرمزگان' ],
			'SBN' => [ 'Sistan and Baluchestan Province', 'Sistan and Baluchestan Province', 'استان سیستان و بلوچستان' ],
		],
		// Iceland.
		'IS' => [],
		// Italy.
		'IT' => [
			'AG' => [ 'AG', 'Agrigento', null ],
			'AL' => [ 'AL', 'Alessandria', null ],
			'AN' => [ 'AN', 'Ancona', null ],
			'AO' => [ 'AO', 'Aosta', null ],
			'AR' => [ 'AR', 'Arezzo', null ],
			'AP' => [ 'AP', 'Ascoli Piceno', null ],
			'AT' => [ 'AT', 'Asti', null ],
			'AV' => [ 'AV', 'Avellino', null ],
			'BA' => [ 'BA', 'Bari', null ],
			'BT' => [ 'BT', 'Barletta-Andria-Trani', null ],
			'BL' => [ 'BL', 'Belluno', null ],
			'BN' => [ 'BN', 'Benevento', null ],
			'BG' => [ 'BG', 'Bergamo', null ],
			'BI' => [ 'BI', 'Biella', null ],
			'BO' => [ 'BO', 'Bologna', null ],
			'BZ' => [ 'BZ', 'Bolzano', null ],
			'BS' => [ 'BS', 'Brescia', null ],
			'BR' => [ 'BR', 'Brindisi', null ],
			'CA' => [ 'CA', 'Cagliari', null ],
			'CL' => [ 'CL', 'Caltanissetta', null ],
			'CB' => [ 'CB', 'Campobasso', null ],
			'CE' => [ 'CE', 'Caserta', null ],
			'CT' => [ 'CT', 'Catania', null ],
			'CZ' => [ 'CZ', 'Catanzaro', null ],
			'CH' => [ 'CH', 'Chieti', null ],
			'CO' => [ 'CO', 'Como', null ],
			'CS' => [ 'CS', 'Cosenza', null ],
			'CR' => [ 'CR', 'Cremona', null ],
			'KR' => [ 'KR', 'Crotone', null ],
			'CN' => [ 'CN', 'Cuneo', null ],
			'EN' => [ 'EN', 'Enna', null ],
			'FM' => [ 'FM', 'Fermo', null ],
			'FE' => [ 'FE', 'Ferrara', null ],
			'FI' => [ 'FI', 'Firenze', null ],
			'FG' => [ 'FG', 'Foggia', null ],
			'FC' => [ 'FC', 'Forlì-Cesena', null ],
			'FR' => [ 'FR', 'Frosinone', null ],
			'GE' => [ 'GE', 'Genova', null ],
			'GO' => [ 'GO', 'Gorizia', null ],
			'GR' => [ 'GR', 'Grosseto', null ],
			'IM' => [ 'IM', 'Imperia', null ],
			'IS' => [ 'IS', 'Isernia', null ],
			'SP' => [ 'SP', 'La Spezia', null ],
			'AQ' => [ 'AQ', "L'Aquila", null ],
			'LT' => [ 'LT', 'Latina', null ],
			'LE' => [ 'LE', 'Lecce', null ],
			'LC' => [ 'LC', 'Lecco', null ],
			'LI' => [ 'LI', 'Livorno', null ],
			'LO' => [ 'LO', 'Lodi', null ],
			'LU' => [ 'LU', 'Lucca', null ],
			'MC' => [ 'MC', 'Macerata', null ],
			'MN' => [ 'MN', 'Mantova', null ],
			'MS' => [ 'MS', 'Massa-Carrara', null ],
			'MT' => [ 'MT', 'Matera', null ],
			'ME' => [ 'ME', 'Messina', null ],
			'MI' => [ 'MI', 'Milano', null ],
			'MO' => [ 'MO', 'Modena', null ],
			'MB' => [ 'MB', 'Monza e Brianza', null ],
			'NA' => [ 'NA', 'Napoli', null ],
			'NO' => [ 'NO', 'Novara', null ],
			'NU' => [ 'NU', 'Nuoro', null ],
			'OR' => [ 'OR', 'Oristano', null ],
			'PD' => [ 'PD', 'Padova', null ],
			'PA' => [ 'PA', 'Palermo', null ],
			'PR' => [ 'PR', 'Parma', null ],
			'PV' => [ 'PV', 'Pavia', null ],
			'PG' => [ 'PG', 'Perugia', null ],
			'PU' => [ 'PU', 'Pesaro e Urbino', null ],
			'PE' => [ 'PE', 'Pescara', null ],
			'PC' => [ 'PC', 'Piacenza', null ],
			'PI' => [ 'PI', 'Pisa', null ],
			'PT' => [ 'PT', 'Pistoia', null ],
			'PN' => [ 'PN', 'Pordenone', null ],
			'PZ' => [ 'PZ', 'Potenza', null ],
			'PO' => [ 'PO', 'Prato', null ],
			'RG' => [ 'RG', 'Ragusa', null ],
			'RA' => [ 'RA', 'Ravenna', null ],
			'RC' => [ 'RC', 'Reggio Calabria', null ],
			'RE' => [ 'RE', 'Reggio Emilia', null ],
			'RI' => [ 'RI', 'Rieti', null ],
			'RN' => [ 'RN', 'Rimini', null ],
			'RM' => [ 'RM', 'Roma', null ],
			'RO' => [ 'RO', 'Rovigo', null ],
			'SA' => [ 'SA', 'Salerno', null ],
			'SS' => [ 'SS', 'Sassari', null ],
			'SV' => [ 'SV', 'Savona', null ],
			'SI' => [ 'SI', 'Siena', null ],
			'SR' => [ 'SR', 'Siracusa', null ],
			'SO' => [ 'SO', 'Sondrio', null ],
			'SU' => [ 'SU', 'Sud Sardegna', null ],
			'TA' => [ 'TA', 'Taranto', null ],
			'TE' => [ 'TE', 'Teramo', null ],
			'TR' => [ 'TR', 'Terni', null ],
			'TO' => [ 'TO', 'Torino', null ],
			'TP' => [ 'TP', 'Trapani', null ],
			'TN' => [ 'TN', 'Trento', null ],
			'TV' => [ 'TV', 'Treviso', null ],
			'TS' => [ 'TS', 'Trieste', null ],
			'UD' => [ 'UD', 'Udine', null ],
			'VA' => [ 'VA', 'Varese', null ],
			'VE' => [ 'VE', 'Venezia', null ],
			'VB' => [ 'VB', 'Verbano-Cusio-Ossola', null ],
			'VC' => [ 'VC', 'Vercelli', null ],
			'VR' => [ 'VR', 'Verona', null ],
			'VV' => [ 'VV', 'Vibo Valentia', null ],
			'VI' => [ 'VI', 'Vicenza', null ],
			'VT' => [ 'VT', 'Viterbo', null ],
		],
		// Jamaica.
		'JM' => [
			'JM-01' => [ 'Kingston', 'Kingston', null ],
			'JM-02' => [ 'St. Andrew', 'St. Andrew', null ],
			'JM-03' => [ 'St. Thomas', 'St. Thomas', null ],
			'JM-04' => [ 'Portland', 'Portland', null ],
			'JM-05' => [ 'St. Mary', 'St. Mary', null ],
			'JM-06' => [ 'St. Ann', 'St. Ann', null ],
			'JM-07' => [ 'Trelawny', 'Trelawny', null ],
			'JM-08' => [ 'St. James', 'St. James', null ],
			'JM-09' => [ 'Hanover', 'Hanover', null ],
			'JM-10' => [ 'Westmoreland', 'Westmoreland', null ],
			'JM-11' => [ 'St. Elizabeth', 'St. Elizabeth', null ],
			'JM-12' => [ 'Manchester', 'Manchester', null ],
			'JM-13' => [ 'Clarendon', 'Clarendon', null ],
			'JM-14' => [ 'St. Catherine', 'St. Catherine', null ],
		],
		// Japan.
		'JP' => [
			'JP01' => [ 'Hokkaido', 'Hokkaido', '北海道' ],
			'JP02' => [ 'Aomori', 'Aomori', '青森県' ],
			'JP03' => [ 'Iwate', 'Iwate', '岩手県' ],
			'JP04' => [ 'Miyagi', 'Miyagi', '宮城県' ],
			'JP05' => [ 'Akita', 'Akita', '秋田県' ],
			'JP06' => [ 'Yamagata', 'Yamagata', '山形県' ],
			'JP07' => [ 'Fukushima', 'Fukushima', '福島県' ],
			'JP08' => [ 'Ibaraki', 'Ibaraki', '茨城県' ],
			'JP09' => [ 'Tochigi', 'Tochigi', '栃木県' ],
			'JP10' => [ 'Gunma', 'Gunma', '群馬県' ],
			'JP11' => [ 'Saitama', 'Saitama', '埼玉県' ],
			'JP12' => [ 'Chiba', 'Chiba', '千葉県' ],
			'JP13' => [ 'Tokyo', 'Tokyo', '東京都' ],
			'JP14' => [ 'Kanagawa', 'Kanagawa', '神奈川県' ],
			'JP15' => [ 'Niigata', 'Niigata', '新潟県' ],
			'JP16' => [ 'Toyama', 'Toyama', '富山県' ],
			'JP17' => [ 'Ishikawa', 'Ishikawa', '石川県' ],
			'JP18' => [ 'Fukui', 'Fukui', '福井県' ],
			'JP19' => [ 'Yamanashi', 'Yamanashi', '山梨県' ],
			'JP20' => [ 'Nagano', 'Nagano', '長野県' ],
			'JP21' => [ 'Gifu', 'Gifu', '岐阜県' ],
			'JP22' => [ 'Shizuoka', 'Shizuoka', '静岡県' ],
			'JP23' => [ 'Aichi', 'Aichi', '愛知県' ],
			'JP24' => [ 'Mie', 'Mie', '三重県' ],
			'JP25' => [ 'Shiga', 'Shiga', '滋賀県' ],
			'JP26' => [ 'Kyoto', 'Kyoto', '京都府' ],
			'JP27' => [ 'Osaka', 'Osaka', '大阪府' ],
			'JP28' => [ 'Hyogo', 'Hyogo', '兵庫県' ],
			'JP29' => [ 'Nara', 'Nara', '奈良県' ],
			'JP30' => [ 'Wakayama', 'Wakayama', '和歌山県' ],
			'JP31' => [ 'Tottori', 'Tottori', '鳥取県' ],
			'JP32' => [ 'Shimane', 'Shimane', '島根県' ],
			'JP33' => [ 'Okayama', 'Okayama', '岡山県' ],
			'JP34' => [ 'Hiroshima', 'Hiroshima', '広島県' ],
			'JP35' => [ 'Yamaguchi', 'Yamaguchi', '山口県' ],
			'JP36' => [ 'Tokushima', 'Tokushima', '徳島県' ],
			'JP37' => [ 'Kagawa', 'Kagawa', '香川県' ],
			'JP38' => [ 'Ehime', 'Ehime', '愛媛県' ],
			'JP39' => [ 'Kochi', 'Kochi', '高知県' ],
			'JP40' => [ 'Fukuoka', 'Fukuoka', '福岡県' ],
			'JP41' => [ 'Saga', 'Saga', '佐賀県' ],
			'JP42' => [ 'Nagasaki', 'Nagasaki', '長崎県' ],
			'JP43' => [ 'Kumamoto', 'Kumamoto', '熊本県' ],
			'JP44' => [ 'Oita', 'Oita', '大分県' ],
			'JP45' => [ 'Miyazaki', 'Miyazaki', '宮崎県' ],
			'JP46' => [ 'Kagoshima', 'Kagoshima', '鹿児島県' ],
			'JP47' => [ 'Okinawa', 'Okinawa', '沖縄県' ],
		],
		// Kenya.
		'KE' => [],
		// South Korea.
		'KR' => [],
		// Kuwait.
		'KW' => [],
		// Laos.
		'LA' => [],
		// Lebanon.
		'LB' => [],
		// Sri Lanka.
		'LK' => [],
		// Liberia.
		'LR' => [],
		// Lithuania.
		'LT' => [],
		// Luxembourg.
		'LU' => [],
		// Moldova.
		'MD' => [],
		// Martinique.
		'MQ' => [],
		// Malta.
		'MT' => [],
		// Mexico.
		'MX' => [
			'DF' => [ 'CDMX', 'Ciudad de México', null ],
			'JA' => [ 'Jal.', 'Jalisco', null ],
			'NL' => [ 'N.L.', 'Nuevo León', null ],
			'AG' => [ 'Ags.', 'Aguascalientes', null ],
			'BC' => [ 'B.C.', 'Baja California', null ],
			'BS' => [ 'B.C.S.', 'Baja California Sur', null ],
			'CM' => [ 'Camp.', 'Campeche', null ],
			'CS' => [ 'Chis.', 'Chiapas', null ],
			'CH' => [ 'Chih.', 'Chihuahua', null ],
			'CO' => [ 'Coah.', 'Coahuila de Zaragoza', null ],
			'CL' => [ 'Col.', 'Colima', null ],
			'DG' => [ 'Dgo.', 'Durango', null ],
			'GT' => [ 'Gto.', 'Guanajuato', null ],
			'GR' => [ 'Gro.', 'Guerrero', null ],
			'HG' => [ 'Hgo.', 'Hidalgo', null ],
			'MX' => [ 'Méx.', 'Estado de México', null ],
			'MI' => [ 'Mich.', 'Michoacán', null ],
			'MO' => [ 'Mor.', 'Morelos', null ],
			'NA' => [ 'Nay.', 'Nayarit', null ],
			'OA' => [ 'Oax.', 'Oaxaca', null ],
			'PU' => [ 'Pue.', 'Puebla', null ],
			'QT' => [ 'Qro.', 'Querétaro', null ],
			'QR' => [ 'Q.R.', 'Quintana Roo', null ],
			'SL' => [ 'S.L.P.', 'San Luis Potosí', null ],
			'SI' => [ 'Sin.', 'Sinaloa', null ],
			'SO' => [ 'Son.', 'Sonora', null ],
			'TB' => [ 'Tab.', 'Tabasco', null ],
			'TM' => [ 'Tamps.', 'Tamaulipas', null ],
			'TL' => [ 'Tlax.', 'Tlaxcala', null ],
			'VE' => [ 'Ver.', 'Veracruz', null ],
			'YU' => [ 'Yuc.', 'Yucatán', null ],
			'ZA' => [ 'Zac.', 'Zacatecas', null ],
		],
		// Malaysia.
		'MY' => [
			'JHR' => [ 'Johor', 'Johor', null ],
			'KDH' => [ 'Kedah', 'Kedah', null ],
			'KTN' => [ 'Kelantan', 'Kelantan', null ],
			'LBN' => [ 'Labuan', 'Labuan', null ],
			'MLK' => [ 'Melaka', 'Melaka', null ],
			'NSN' => [ 'Negeri Sembilan', 'Negeri Sembilan', null ],
			'PHG' => [ 'Pahang', 'Pahang', null ],
			'PNG' => [ 'Pulau Pinang', 'Pulau Pinang', null ],
			'PRK' => [ 'Perak', 'Perak', null ],
			'PLS' => [ 'Perlis', 'Perlis', null ],
			'SBH' => [ 'Sabah', 'Sabah', null ],
			'SWK' => [ 'Sarawak', 'Sarawak', null ],
			'SGR' => [ 'Selangor', 'Selangor', null ],
			'TRG' => [ 'Terengganu', 'Terengganu', null ],
			'PJY' => [ 'Putrajaya', 'Putrajaya', null ],
			'KUL' => [ 'Kuala Lumpur', 'Kuala Lumpur', null ],
		],
		// Mozambique.
		'MZ' => [
			'MZP'   => [ 'Cabo Delgado', 'Cabo Delgado', null ],
			'MZG'   => [ 'Gaza', 'Gaza', null ],
			'MZI'   => [ 'Inhambane', 'Inhambane', null ],
			'MZB'   => [ 'Manica', 'Manica', null ],
			'MZL'   => [ 'Maputo', 'Maputo', null ],
			'MZMPM' => [ 'Cidade de Maputo', 'Cidade de Maputo', null ],
			'MZN'   => [ 'Nampula', 'Nampula', null ],
			'MZA'   => [ 'Niassa', 'Niassa', null ],
			'MZS'   => [ 'Sofala', 'Sofala', null ],
			'MZT'   => [ 'Tete', 'Tete', null ],
			'MZQ'   => [ 'Zambezia', 'Zambezia', null ],
		],
		// Namibia.
		'NA' => [],
		// Nigeria.
		'NG' => [
			'AB' => [ 'Abia', 'Abia', null ],
			'FC' => [ 'Federal Capital Territory', 'Federal Capital Territory', null ],
			'AD' => [ 'Adamawa', 'Adamawa', null ],
			'AK' => [ 'Akwa Ibom', 'Akwa Ibom', null ],
			'AN' => [ 'Anambra', 'Anambra', null ],
			'BA' => [ 'Bauchi', 'Bauchi', null ],
			'BY' => [ 'Bayelsa', 'Bayelsa', null ],
			'BE' => [ 'Benue', 'Benue', null ],
			'BO' => [ 'Borno', 'Borno', null ],
			'CR' => [ 'Cross River', 'Cross River', null ],
			'DE' => [ 'Delta', 'Delta', null ],
			'EB' => [ 'Ebonyi', 'Ebonyi', null ],
			'ED' => [ 'Edo', 'Edo', null ],
			'EK' => [ 'Ekiti', 'Ekiti', null ],
			'EN' => [ 'Enugu', 'Enugu', null ],
			'GO' => [ 'Gombe', 'Gombe', null ],
			'IM' => [ 'Imo', 'Imo', null ],
			'JI' => [ 'Jigawa', 'Jigawa', null ],
			'KD' => [ 'Kaduna', 'Kaduna', null ],
			'KN' => [ 'Kano', 'Kano', null ],
			'KT' => [ 'Katsina', 'Katsina', null ],
			'KE' => [ 'Kebbi', 'Kebbi', null ],
			'KO' => [ 'Kogi', 'Kogi', null ],
			'KW' => [ 'Kwara', 'Kwara', null ],
			'LA' => [ 'Lagos', 'Lagos', null ],
			'NA' => [ 'Nasarawa', 'Nasarawa', null ],
			'NI' => [ 'Niger', 'Niger', null ],
			'OG' => [ 'Ogun State', 'Ogun State', null ],
			'ON' => [ 'Ondo', 'Ondo', null ],
			'OS' => [ 'Osun', 'Osun', null ],
			'OY' => [ 'Oyo', 'Oyo', null ],
			'PL' => [ 'Plateau', 'Plateau', null ],
			'RI' => [ 'Rivers', 'Rivers', null ],
			'SO' => [ 'Sokoto', 'Sokoto', null ],
			'TA' => [ 'Taraba', 'Taraba', null ],
			'YO' => [ 'Yobe', 'Yobe', null ],
			'ZA' => [ 'Zamfara', 'Zamfara', null ],
		],
		// Netherlands.
		'NL' => [],
		// Norway.
		'NO' => [],
		// Nepal.
		'NP' => [],
		// New Zealand.
		'NZ' => [],
		// Peru.
		'PE' => [
			'CAL' => [ 'Callao', 'Callao', null ],
			'LMA' => [ 'Municipalidad Metropolitana de Lima', 'Municipalidad Metropolitana de Lima', null ],
			'AMA' => [ 'Amazonas', 'Amazonas', null ],
			'ANC' => [ 'Áncash', 'Áncash', null ],
			'APU' => [ 'Apurímac', 'Apurímac', null ],
			'ARE' => [ 'Arequipa', 'Arequipa', null ],
			'AYA' => [ 'Ayacucho', 'Ayacucho', null ],
			'CAJ' => [ 'Cajamarca', 'Cajamarca', null ],
			'CUS' => [ 'Cuzco', 'Cuzco', null ],
			'HUV' => [ 'Huancavelica', 'Huancavelica', null ],
			'HUC' => [ 'Huánuco', 'Huánuco', null ],
			'ICA' => [ 'Ica', 'Ica', null ],
			'JUN' => [ 'Junín', 'Junín', null ],
			'LAL' => [ 'La Libertad', 'La Libertad', null ],
			'LAM' => [ 'Lambayeque', 'Lambayeque', null ],
			'LIM' => [ 'Gobierno Regional de Lima', 'Gobierno Regional de Lima', null ],
			'LOR' => [ 'Loreto', 'Loreto', null ],
			'MDD' => [ 'Madre de Dios', 'Madre de Dios', null ],
			'MOQ' => [ 'Moquegua', 'Moquegua', null ],
			'PAS' => [ 'Pasco', 'Pasco', null ],
			'PIU' => [ 'Piura', 'Piura', null ],
			'PUN' => [ 'Puno', 'Puno', null ],
			'SAM' => [ 'San Martín', 'San Martín', null ],
			'TAC' => [ 'Tacna', 'Tacna', null ],
			'TUM' => [ 'Tumbes', 'Tumbes', null ],
			'UCA' => [ 'Ucayali', 'Ucayali', null ],
		],
		// Philippines.
		'PH' => [
			'ABR' => [ 'Abra', 'Abra', null ],
			'AGN' => [ 'Agusan del Norte', 'Agusan del Norte', null ],
			'AGS' => [ 'Agusan del Sur', 'Agusan del Sur', null ],
			'AKL' => [ 'Aklan', 'Aklan', null ],
			'ALB' => [ 'Albay', 'Albay', null ],
			'ANT' => [ 'Antique', 'Antique', null ],
			'APA' => [ 'Apayao', 'Apayao', null ],
			'AUR' => [ 'Aurora', 'Aurora', null ],
			'BAS' => [ 'Basilan', 'Basilan', null ],
			'BAN' => [ 'Bataan', 'Bataan', null ],
			'BTN' => [ 'Batanes', 'Batanes', null ],
			'BTG' => [ 'Batangas', 'Batangas', null ],
			'BEN' => [ 'Benguet', 'Benguet', null ],
			'BIL' => [ 'Biliran', 'Biliran', null ],
			'BOH' => [ 'Bohol', 'Bohol', null ],
			'BUK' => [ 'Bukidnon', 'Bukidnon', null ],
			'BUL' => [ 'Bulacan', 'Bulacan', null ],
			'CAG' => [ 'Cagayan', 'Cagayan', null ],
			'CAN' => [ 'Camarines Norte', 'Camarines Norte', null ],
			'CAS' => [ 'Camarines Sur', 'Camarines Sur', null ],
			'CAM' => [ 'Camiguin', 'Camiguin', null ],
			'CAP' => [ 'Capiz', 'Capiz', null ],
			'CAT' => [ 'Catanduanes', 'Catanduanes', null ],
			'CAV' => [ 'Cavite', 'Cavite', null ],
			'CEB' => [ 'Cebu', 'Cebu', null ],
			'COM' => [ 'Compostela Valley', 'Compostela Valley', null ],
			'NCO' => [ 'Cotabato', 'Cotabato', null ],
			'DAV' => [ 'Davao del Norte', 'Davao del Norte', null ],
			'DAS' => [ 'Davao del Sur', 'Davao del Sur', null ],
			'DAC' => [ 'Davao Occidental', 'Davao Occidental', null ],
			'DAO' => [ 'Davao Oriental', 'Davao Oriental', null ],
			'DIN' => [ 'Dinagat Islands', 'Dinagat Islands', null ],
			'EAS' => [ 'Eastern Samar', 'Eastern Samar', null ],
			'GUI' => [ 'Guimaras', 'Guimaras', null ],
			'IFU' => [ 'Ifugao', 'Ifugao', null ],
			'ILN' => [ 'Ilocos Norte', 'Ilocos Norte', null ],
			'ILS' => [ 'Ilocos Sur', 'Ilocos Sur', null ],
			'ILI' => [ 'Iloilo', 'Iloilo', null ],
			'ISA' => [ 'Isabela', 'Isabela', null ],
			'KAL' => [ 'Kalinga', 'Kalinga', null ],
			'LUN' => [ 'La Union', 'La Union', null ],
			'LAG' => [ 'Laguna', 'Laguna', null ],
			'LAN' => [ 'Lanao del Norte', 'Lanao del Norte', null ],
			'LAS' => [ 'Lanao del Sur', 'Lanao del Sur', null ],
			'LEY' => [ 'Leyte', 'Leyte', null ],
			'MAG' => [ 'Maguindanao', 'Maguindanao', null ],
			'MAD' => [ 'Marinduque', 'Marinduque', null ],
			'MAS' => [ 'Masbate', 'Masbate', null ],
			'MSC' => [ 'Misamis Occidental', 'Misamis Occidental', null ],
			'MSR' => [ 'Misamis Oriental', 'Misamis Oriental', null ],
			'MOU' => [ 'Mountain Province', 'Mountain Province', null ],
			'NEC' => [ 'Negros Occidental', 'Negros Occidental', null ],
			'NER' => [ 'Negros Oriental', 'Negros Oriental', null ],
			'NSA' => [ 'Northern Samar', 'Northern Samar', null ],
			'NUE' => [ 'Nueva Ecija', 'Nueva Ecija', null ],
			'NUV' => [ 'Nueva Vizcaya', 'Nueva Vizcaya', null ],
			'MDC' => [ 'Mindoro Occidental', 'Mindoro Occidental', null ],
			'MDR' => [ 'Mindoro Oriental', 'Mindoro Oriental', null ],
			'PLW' => [ 'Palawan', 'Palawan', null ],
			'PAM' => [ 'Pampanga', 'Pampanga', null ],
			'PAN' => [ 'Pangasinan', 'Pangasinan', null ],
			'QUE' => [ 'Quezon Province', 'Quezon Province', null ],
			'QUI' => [ 'Quirino', 'Quirino', null ],
			'RIZ' => [ 'Rizal', 'Rizal', null ],
			'ROM' => [ 'Romblon', 'Romblon', null ],
			'WSA' => [ 'Samar', 'Samar', null ],
			'SAR' => [ 'Sarangani', 'Sarangani', null ],
			'SIQ' => [ 'Siquijor', 'Siquijor', null ],
			'SOR' => [ 'Sorsogon', 'Sorsogon', null ],
			'SCO' => [ 'South Cotabato', 'South Cotabato', null ],
			'SLE' => [ 'Southern Leyte', 'Southern Leyte', null ],
			'SUK' => [ 'Sultan Kudarat', 'Sultan Kudarat', null ],
			'SLU' => [ 'Sulu', 'Sulu', null ],
			'SUN' => [ 'Surigao del Norte', 'Surigao del Norte', null ],
			'SUR' => [ 'Surigao del Sur', 'Surigao del Sur', null ],
			'TAR' => [ 'Tarlac', 'Tarlac', null ],
			'TAW' => [ 'Tawi-Tawi', 'Tawi-Tawi', null ],
			'ZMB' => [ 'Zambales', 'Zambales', null ],
			'ZAN' => [ 'Zamboanga del Norte', 'Zamboanga del Norte', null ],
			'ZAS' => [ 'Zamboanga del Sur', 'Zamboanga del Sur', null ],
			'ZSI' => [ 'Zamboanga Sibuguey', 'Zamboanga Sibuguey', null ],
			'00'  => [ 'Metro Manila', 'Metro Manila', null ],
		],
		// Pakistan.
		'PK' => [],
		// Poland.
		'PL' => [],
		// Puerto Rico.
		'PR' => [],
		// Portugal.
		'PT' => [],
		// Paraguay.
		'PY' => [],
		// Reunion.
		'RE' => [],
		// Romania.
		'RO' => [],
		// Serbia.
		'RS' => [],
		// Sweden.
		'SE' => [],
		// Singapore.
		'SG' => [],
		// Slovenia.
		'SI' => [],
		// Slovakia.
		'SK' => [],
		// Thailand.
		'TH' => [
			'TH-37' => [ 'Amnat Charoen', 'Amnat Charoen', 'อำนาจเจริญ' ],
			'TH-15' => [ 'Ang Thong', 'Ang Thong', 'อ่างทอง' ],
			'TH-14' => [ 'Phra Nakhon Si Ayutthaya', 'Phra Nakhon Si Ayutthaya', 'พระนครศรีอยุธยา' ],
			'TH-10' => [ 'Bangkok', 'Bangkok', 'กรุงเทพมหานคร' ],
			'TH-38' => [ 'Bueng Kan', 'Bueng Kan', 'จังหวัด บึงกาฬ' ],
			'TH-31' => [ 'Buri Ram', 'Buri Ram', 'บุรีรัมย์' ],
			'TH-24' => [ 'Chachoengsao', 'Chachoengsao', 'ฉะเชิงเทรา' ],
			'TH-18' => [ 'Chai Nat', 'Chai Nat', 'ชัยนาท' ],
			'TH-36' => [ 'Chaiyaphum', 'Chaiyaphum', 'ชัยภูมิ' ],
			'TH-22' => [ 'Chanthaburi', 'Chanthaburi', 'จันทบุรี' ],
			'TH-50' => [ 'Chiang Rai', 'Chiang Rai', 'เชียงราย' ],
			'TH-57' => [ 'Chiang Mai', 'Chiang Mai', 'เชียงใหม่' ],
			'TH-20' => [ 'Chon Buri', 'Chon Buri', 'ชลบุรี' ],
			'TH-86' => [ 'Chumpon', 'Chumpon', 'ชุมพร' ],
			'TH-46' => [ 'Kalasin', 'Kalasin', 'กาฬสินธุ์' ],
			'TH-62' => [ 'Kamphaeng Phet', 'Kamphaeng Phet', 'กำแพงเพชร' ],
			'TH-71' => [ 'Kanchanaburi', 'Kanchanaburi', 'กาญจนบุรี' ],
			'TH-40' => [ 'Khon Kaen', 'Khon Kaen', 'ขอนแก่น' ],
			'TH-81' => [ 'Krabi', 'Krabi', 'กระบี่' ],
			'TH-52' => [ 'Lampang', 'Lampang', 'ลำปาง' ],
			'TH-51' => [ 'Lamphun', 'Lamphun', 'ลำพูน' ],
			'TH-42' => [ 'Loei', 'Loei', 'เลย' ],
			'TH-16' => [ 'Lop Buri', 'Lop Buri', 'ลพบุรี' ],
			'TH-58' => [ 'Mae Hong Son', 'Mae Hong Son', 'แม่ฮ่องสอน' ],
			'TH-44' => [ 'Maha Sarakham', 'Maha Sarakham', 'มหาสารคาม' ],
			'TH-49' => [ 'Mukdahan', 'Mukdahan', 'มุกดาหาร' ],
			'TH-26' => [ 'Nakhon Nayok', 'Nakhon Nayok', 'นครนายก' ],
			'TH-73' => [ 'Nakhon Pathom', 'Nakhon Pathom', 'นครปฐม' ],
			'TH-48' => [ 'Nakhon Phanom', 'Nakhon Phanom', 'นครพนม' ],
			'TH-30' => [ 'Nakhon Ratchasima', 'Nakhon Ratchasima', 'นครราชสีมา' ],
			'TH-60' => [ 'Nakhon Sawan', 'Nakhon Sawan', 'นครสวรรค์' ],
			'TH-80' => [ 'Nakhon Si Thammarat', 'Nakhon Si Thammarat', 'นครศรีธรรมราช' ],
			'TH-55' => [ 'Nan', 'Nan', 'น่าน' ],
			'TH-96' => [ 'Narathiwat', 'Narathiwat', 'นราธิวาส' ],
			'TH-39' => [ 'Nong Bua Lam Phu', 'Nong Bua Lam Phu', 'หนองบัวลำภู' ],
			'TH-43' => [ 'Nong Khai', 'Nong Khai', 'หนองคาย' ],
			'TH-12' => [ 'Nonthaburi', 'Nonthaburi', 'นนทบุรี' ],
			'TH-13' => [ 'Pathum Thani', 'Pathum Thani', 'ปทุมธานี' ],
			'TH-94' => [ 'Pattani', 'Pattani', 'ปัตตานี' ],
			'TH-82' => [ 'Phang Nga', 'Phang Nga', 'พังงา' ],
			'TH-93' => [ 'Phattalung', 'Phattalung', 'พัทลุง' ],
			'TH-56' => [ 'Phayao', 'Phayao', 'พะเยา' ],
			'TH-67' => [ 'Phetchabun', 'Phetchabun', 'เพชรบูรณ์' ],
			'TH-76' => [ 'Phetchaburi', 'Phetchaburi', 'เพชรบุรี' ],
			'TH-66' => [ 'Phichit', 'Phichit', 'พิจิตร' ],
			'TH-65' => [ 'Phitsanulok', 'Phitsanulok', 'พิษณุโลก' ],
			'TH-54' => [ 'Phrae', 'Phrae', 'แพร่' ],
			'TH-83' => [ 'Phuket', 'Phuket', 'ภูเก็ต' ],
			'TH-25' => [ 'Prachin Buri', 'Prachin Buri', 'ปราจีนบุรี' ],
			'TH-77' => [ 'Prachuap Khiri Khan', 'Prachuap Khiri Khan', 'ประจวบคีรีขันธ์' ],
			'TH-85' => [ 'Ranong', 'Ranong', 'ระนอง' ],
			'TH-70' => [ 'Ratchaburi', 'Ratchaburi', 'ราชบุรี' ],
			'TH-21' => [ 'Rayong', 'Rayong', 'ระยอง' ],
			'TH-45' => [ 'Roi Et', 'Roi Et', 'ร้อยเอ็ด' ],
			'TH-27' => [ 'Sa Kaeo', 'Sa Kaeo', 'สระแก้ว' ],
			'TH-47' => [ 'Sakon Nakhon', 'Sakon Nakhon', 'สกลนคร' ],
			'TH-11' => [ 'Samut Prakan', 'Samut Prakan', 'สมุทรปราการ' ],
			'TH-74' => [ 'Samut Sakhon', 'Samut Sakhon', 'สมุทรสาคร' ],
			'TH-75' => [ 'Samut Songkhram', 'Samut Songkhram', 'สมุทรสงคราม' ],
			'TH-19' => [ 'Saraburi', 'Saraburi', 'สระบุรี' ],
			'TH-91' => [ 'Satun', 'Satun', 'สตูล' ],
			'TH-17' => [ 'Sing Buri', 'Sing Buri', 'สิงห์บุรี' ],
			'TH-33' => [ 'Si Sa Ket', 'Si Sa Ket', 'ศรีสะเกษ' ],
			'TH-90' => [ 'Songkhla', 'Songkhla', 'สงขลา' ],
			'TH-64' => [ 'Sukhothai', 'Sukhothai', 'สุโขทัย' ],
			'TH-72' => [ 'Suphanburi', 'Suphanburi', 'สุพรรณบุรี' ],
			'TH-84' => [ 'Surat Thani', 'Surat Thani', 'สุราษฎร์ธานี' ],
			'TH-32' => [ 'Surin', 'Surin', 'สุรินทร์' ],
			'TH-63' => [ 'Tak', 'Tak', 'ตาก' ],
			'TH-92' => [ 'Trang', 'Trang', 'ตรัง' ],
			'TH-23' => [ 'Trat', 'Trat', 'ตราด' ],
			'TH-34' => [ 'Ubon Ratchathani', 'Ubon Ratchathani', 'อุบลราชธานี' ],
			'TH-41' => [ 'Udon Thani', 'Udon Thani', 'อุดรธานี' ],
			'TH-61' => [ 'Uthai Thani', 'Uthai Thani', 'อุทัยธานี' ],
			'TH-53' => [ 'Uttaradit', 'Uttaradit', 'อุตรดิตถ์' ],
			'TH-95' => [ 'Yala', 'Yala', 'ยะลา' ],
			'TH-35' => [ 'Yasothon', 'Yasothon', 'ยโสธร' ],
		],
		// Turkey.
		'TR' => [
			'TR01' => [ 'Adana', 'Adana', null ],
			'TR02' => [ 'Adıyaman', 'Adıyaman', null ],
			'TR03' => [ 'Afyon', 'Afyon', null ],
			'TR04' => [ 'Ağrı', 'Ağrı', null ],
			'TR05' => [ 'Amasya', 'Amasya', null ],
			'TR06' => [ 'Ankara', 'Ankara', null ],
			'TR07' => [ 'Antalya', 'Antalya', null ],
			'TR08' => [ 'Artvin', 'Artvin', null ],
			'TR09' => [ 'Aydın', 'Aydın', null ],
			'TR10' => [ 'Balıkesir', 'Balıkesir', null ],
			'TR11' => [ 'Bilecik', 'Bilecik', null ],
			'TR12' => [ 'Bingöl', 'Bingöl', null ],
			'TR13' => [ 'Bitlis', 'Bitlis', null ],
			'TR14' => [ 'Bolu', 'Bolu', null ],
			'TR15' => [ 'Burdur', 'Burdur', null ],
			'TR16' => [ 'Bursa', 'Bursa', null ],
			'TR17' => [ 'Çanakkale', 'Çanakkale', null ],
			'TR18' => [ 'Çankırı', 'Çankırı', null ],
			'TR19' => [ 'Çorum', 'Çorum', null ],
			'TR20' => [ 'Denizli', 'Denizli', null ],
			'TR21' => [ 'Diyarbakır', 'Diyarbakır', null ],
			'TR22' => [ 'Edirne', 'Edirne', null ],
			'TR23' => [ 'Elazığ', 'Elazığ', null ],
			'TR24' => [ 'Erzincan', 'Erzincan', null ],
			'TR25' => [ 'Erzurum', 'Erzurum', null ],
			'TR26' => [ 'Eskişehir', 'Eskişehir', null ],
			'TR27' => [ 'Gaziantep', 'Gaziantep', null ],
			'TR28' => [ 'Giresun', 'Giresun', null ],
			'TR29' => [ 'Gümüşhane', 'Gümüşhane', null ],
			'TR30' => [ 'Hakkari', 'Hakkari', null ],
			'TR31' => [ 'Hatay', 'Hatay', null ],
			'TR32' => [ 'Isparta', 'Isparta', null ],
			'TR33' => [ 'Mersin', 'Mersin', null ],
			'TR34' => [ 'İstanbul', 'İstanbul', null ],
			'TR35' => [ 'İzmir', 'İzmir', null ],
			'TR36' => [ 'Kars', 'Kars', null ],
			'TR37' => [ 'Kastamonu', 'Kastamonu', null ],
			'TR38' => [ 'Kayseri', 'Kayseri', null ],
			'TR39' => [ 'Kırklareli', 'Kırklareli', null ],
			'TR40' => [ 'Kırşehir', 'Kırşehir', null ],
			'TR41' => [ 'Kocaeli', 'Kocaeli', null ],
			'TR42' => [ 'Konya', 'Konya', null ],
			'TR43' => [ 'Kütahya', 'Kütahya', null ],
			'TR44' => [ 'Malatya', 'Malatya', null ],
			'TR45' => [ 'Manisa', 'Manisa', null ],
			'TR46' => [ 'Kahramanmaraş', 'Kahramanmaraş', null ],
			'TR47' => [ 'Mardin', 'Mardin', null ],
			'TR48' => [ 'Muğla', 'Muğla', null ],
			'TR49' => [ 'Muş', 'Muş', null ],
			'TR50' => [ 'Nevşehir', 'Nevşehir', null ],
			'TR51' => [ 'Niğde', 'Niğde', null ],
			'TR52' => [ 'Ordu', 'Ordu', null ],
			'TR53' => [ 'Rize', 'Rize', null ],
			'TR54' => [ 'Sakarya', 'Sakarya', null ],
			'TR55' => [ 'Samsun', 'Samsun', null ],
			'TR56' => [ 'Siirt', 'Siirt', null ],
			'TR57' => [ 'Sinop', 'Sinop', null ],
			'TR58' => [ 'Sivas', 'Sivas', null ],
			'TR59' => [ 'Tekirdağ', 'Tekirdağ', null ],
			'TR60' => [ 'Tokat', 'Tokat', null ],
			'TR61' => [ 'Trabzon', 'Trabzon', null ],
			'TR62' => [ 'Tunceli', 'Tunceli', null ],
			'TR63' => [ 'Şanlıurfa', 'Şanlıurfa', null ],
			'TR64' => [ 'Uşak', 'Uşak', null ],
			'TR65' => [ 'Van', 'Van', null ],
			'TR66' => [ 'Yozgat', 'Yozgat', null ],
			'TR67' => [ 'Zonguldak', 'Zonguldak', null ],
			'TR68' => [ 'Aksaray', 'Aksaray', null ],
			'TR69' => [ 'Bayburt', 'Bayburt', null ],
			'TR70' => [ 'Karaman', 'Karaman', null ],
			'TR71' => [ 'Kırıkkale', 'Kırıkkale', null ],
			'TR72' => [ 'Batman', 'Batman', null ],
			'TR73' => [ 'Şırnak', 'Şırnak', null ],
			'TR74' => [ 'Bartın', 'Bartın', null ],
			'TR75' => [ 'Ardahan', 'Ardahan', null ],
			'TR76' => [ 'Iğdır', 'Iğdır', null ],
			'TR77' => [ 'Yalova', 'Yalova', null ],
			'TR78' => [ 'Karabük', 'Karabük', null ],
			'TR79' => [ 'Kilis', 'Kilis', null ],
			'TR80' => [ 'Osmaniye', 'Osmaniye', null ],
			'TR81' => [ 'Düzce', 'Düzce', null ],
		],
		// Tanzania.
		'TZ' => [],
		// Ukraine.
		'UA' => [
			'UA-05' => [ 'Vinnytsia Oblast', 'Vinnytsia Oblast', 'Вінницька область' ],
			'UA-07' => [ 'Volyn Oblast', 'Volyn Oblast', 'Волинська область' ],
			'UA-09' => [ 'Luhansk Oblast', 'Luhansk Oblast', 'Луганська область' ],
			'UA-12' => [ 'Dnipropetrovsk Oblast', 'Dnipropetrovsk Oblast', 'Дніпропетровська область' ],
			'UA-14' => [ 'Donetsk Oblast', 'Donetsk Oblast', 'Донецька область' ],
			'UA-18' => [ 'Zhytomyr Oblast', 'Zhytomyr Oblast', 'Житомирська область' ],
			'UA-21' => [ 'Zakarpattia Oblast', 'Zakarpattia Oblast', 'Закарпатська область' ],
			'UA-23' => [ 'Zaporizhzhia Oblast', 'Zaporizhzhia Oblast', 'Запорізька область' ],
			'UA-26' => [ 'Ivano-Frankivsk Oblast', 'Ivano-Frankivsk Oblast', 'Івано-Франківська область' ],
			'UA-30' => [ 'Kyiv', 'Kyiv', 'Київ' ],
			'UA-32' => [ 'Kyiv Oblast', 'Kyiv Oblast', 'Київська область' ],
			'UA-35' => [ 'Kirovohrad Oblast', 'Kirovohrad Oblast', 'Кіровоградська область' ],
			'UA-40' => [ 'Sevastopol', 'Sevastopol', 'Севастополь' ],
			'UA-43' => [ 'Crimea', 'Crimea', 'Автономна Республіка Крим' ],
			'UA-46' => [ 'Lviv Oblast', 'Lviv Oblast', 'Львівська область' ],
			'UA-48' => [ 'Mykolaiv Oblast', 'Mykolaiv Oblast', 'Миколаївська область' ],
			'UA-51' => [ 'Odesa Oblast', 'Odesa Oblast', 'Одеська область' ],
			'UA-53' => [ 'Poltava Oblast', 'Poltava Oblast', 'Полтавська область' ],
			'UA-56' => [ 'Rivne Oblast', 'Rivne Oblast', 'Рівненська область' ],
			'UA-59' => [ 'Sumy Oblast', 'Sumy Oblast', 'Сумська область' ],
			'UA-61' => [ 'Ternopil Oblast', 'Ternopil Oblast', 'Тернопільська область' ],
			'UA-63' => [ 'Kharkiv Oblast', 'Kharkiv Oblast', 'Харківська область' ],
			'UA-65' => [ 'Kherson Oblast', 'Kherson Oblast', 'Херсонська область' ],
			'UA-68' => [ 'Khmelnytskyi Oblast', 'Khmelnytskyi Oblast', 'Хмельницька область' ],
			'UA-71' => [ 'Cherkasy Oblast', 'Cherkasy Oblast', 'Черкаська область' ],
			'UA-74' => [ 'Chernihiv Oblast', 'Chernihiv Oblast', 'Чернігівська область' ],
			'UA-77' => [ 'Chernivtsi Oblast', 'Chernivtsi Oblast', 'Чернівецька область' ],
		],
		// Uganda.
		'UG' => [],
		// United States Minor Outlying Islands.
		'UM' => [],
		// United States.
		'US' => [
			'AL' => [ 'AL', 'Alabama', null ],
			'AK' => [ 'AK', 'Alaska', null ],
			'AZ' => [ 'AZ', 'Arizona', null ],
			'AR' => [ 'AR', 'Arkansas', null ],
			'CA' => [ 'CA', 'California', null ],
			'CO' => [ 'CO', 'Colorado', null ],
			'CT' => [ 'CT', 'Connecticut', null ],
			'DE' => [ 'DE', 'Delaware', null ],
			'DC' => [ 'DC', 'District of Columbia', null ],
			'FL' => [ 'FL', 'Florida', null ],
			'GA' => [ 'GA', 'Georgia', null ],
			'HI' => [ 'HI', 'Hawaii', null ],
			'ID' => [ 'ID', 'Idaho', null ],
			'IL' => [ 'IL', 'Illinois', null ],
			'IN' => [ 'IN', 'Indiana', null ],
			'IA' => [ 'IA', 'Iowa', null ],
			'KS' => [ 'KS', 'Kansas', null ],
			'KY' => [ 'KY', 'Kentucky', null ],
			'LA' => [ 'LA', 'Louisiana', null ],
			'ME' => [ 'ME', 'Maine', null ],
			'MD' => [ 'MD', 'Maryland', null ],
			'MA' => [ 'MA', 'Massachusetts', null ],
			'MI' => [ 'MI', 'Michigan', null ],
			'MN' => [ 'MN', 'Minnesota', null ],
			'MS' => [ 'MS', 'Mississippi', null ],
			'MO' => [ 'MO', 'Missouri', null ],
			'MT' => [ 'MT', 'Montana', null ],
			'NE' => [ 'NE', 'Nebraska', null ],
			'NV' => [ 'NV', 'Nevada', null ],
			'NH' => [ 'NH', 'New Hampshire', null ],
			'NJ' => [ 'NJ', 'New Jersey', null ],
			'NM' => [ 'NM', 'New Mexico', null ],
			'NY' => [ 'NY', 'New York', null ],
			'NC' => [ 'NC', 'North Carolina', null ],
			'ND' => [ 'ND', 'North Dakota', null ],
			'OH' => [ 'OH', 'Ohio', null ],
			'OK' => [ 'OK', 'Oklahoma', null ],
			'OR' => [ 'OR', 'Oregon', null ],
			'PA' => [ 'PA', 'Pennsylvania', null ],
			'RI' => [ 'RI', 'Rhode Island', null ],
			'SC' => [ 'SC', 'South Carolina', null ],
			'SD' => [ 'SD', 'South Dakota', null ],
			'TN' => [ 'TN', 'Tennessee', null ],
			'TX' => [ 'TX', 'Texas', null ],
			'UT' => [ 'UT', 'Utah', null ],
			'VT' => [ 'VT', 'Vermont', null ],
			'VA' => [ 'VA', 'Virginia', null ],
			'WA' => [ 'WA', 'Washington', null ],
			'WV' => [ 'WV', 'West Virginia', null ],
			'WI' => [ 'WI', 'Wisconsin', null ],
			'WY' => [ 'WY', 'Wyoming', null ],
			'AA' => [ 'AA', 'Armed Forces (AA)', null ],
			'AE' => [ 'AE', 'Armed Forces (AE)', null ],
			'AP' => [ 'AP', 'Armed Forces (AP)', null ],
			//[ 'AS', 'American Samoa', null ],
			//[ 'GU', 'Guam', null ],
			//[ 'MH', 'Marshall Islands', null ],
			//[ 'FM', 'Micronesia', null ],
			//[ 'MP', 'Northern Mariana Islands', null ],
			//[ 'PW', 'Palau', null ],
			//[ 'PR', 'Puerto Rico', null ],
			//[ 'VI', 'Virgin Islands', null ],
		],
		// Vietnam.
		'VN' => [],
		// Mayotte.
		'YT' => [],
		// South Africa.
		'ZA' => [],
		// Zambia.
		'ZM' => [],
	];
}
