<?php
/**
 * WooCommerce Braintree Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@woocommerce.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Braintree Gateway to newer
 * versions in the future. If you wish to customize WooCommerce Braintree Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/braintree/
 *
 * @package   WC-Braintree/Integrations
 * @author    WooCommerce
 * @copyright Copyright: (c) 2016-2020, Automattic, Inc.
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace WC_Braintree\Integrations;

use WC_Braintree\PayPal\Buttons;
use SkyVerge\WooCommerce\PluginFramework\v5_15_10 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * Product Add-ons integration class.
 *
 * @since 2.3.0
 */
class Product_Addons {


	/**
	 * Constructs the product add-ons integration class.
	 *
	 * @since 2.3.0
	 */
	public function __construct() {

		add_filter( 'wc_braintree_paypal_product_button_should_validate_product_data', '__return_true', 10, 2 );
		add_filter( 'wc_braintree_paypal_product_button_validate_product_data', [ $this, 'validate_product_data' ], 10, 2 );
		add_action( 'wc_braintree_paypal_before_product_button_add_to_cart', [ $this, 'prepare_product_data_for_addons' ], 10, 4 );
	}


	/**
	 * Validates the product data for displaying the product button.
	 *
	 * @since 2.3.0
	 *
	 * @param bool            $is_valid       whether the product data is valid.
	 * @param Buttons\Product $product_button product button instance.
	 * @return bool
	 */
	public function validate_product_data( $is_valid, $product_button ) {

		if ( $is_valid && isset( $GLOBALS['Product_Addon_Cart'] ) ) {

			$cart_handler = $GLOBALS['Product_Addon_Cart'];

			if ( $cart_handler instanceof \WC_Product_Addons_Cart ) {

				$product_id = (int) Framework\SV_WC_Helper::get_posted_value( 'product_id' );
				$serialized = Framework\SV_WC_Helper::get_posted_value( 'cart_form' );
				$post_data  = [];

				if ( ! empty( $serialized ) ) {
					parse_str( $serialized, $post_data );
				}

				$quantity = isset( $post_data['quantity'] ) ? (int) $post_data['quantity'] : 1;

				// keep a reference to currently-enqueued notices, to restore to this point after validation.
				$notices = wc_get_notices();

				$is_valid = (bool) $cart_handler->validate_add_cart_item( $is_valid, $product_id, $quantity, $post_data );

				// restore notices from before validation, to prevent notices generated by this validation from displaying on subsequent page loads.
				wc_set_notices( $notices );
			}
		}

		return $is_valid;
	}


	/**
	 * Prepares the data for product add-ons before adding the product to the cart.
	 *
	 * @since 2.3.0
	 *
	 * @param int   $product_id product ID.
	 * @param int   $quantity quantity.
	 * @param int   $variation_id product variation ID.
	 * @param array $cart_data cart data.
	 */
	public function prepare_product_data_for_addons( $product_id, $quantity, $variation_id, $cart_data ) {

		// transpose product add-on data to $_POST for compatibility.
		foreach ( $cart_data as $key => $value ) {

			if ( 0 === stripos( $key, 'addon-' ) ) {
				$_POST[ $key ] = $value;
			}
		}
	}
}
