<?php
/**
 * Build setting page(tab)
 *
 * @package WolfOz taxonomy metaboxes
 */

namespace WolfOz\Admin;

if ( ! class_exists( \WolfOz\Admin\Settings::class ) ) {

	/**
	 * Base setting class
	 *
	 * @author wolfoz
	 */
	class Settings {
		/**
		 * Sluggable name of class
		 *
		 * @var string
		 */
		protected static $_name = 'wolfoz-settings';

		/**
		 * Initialized flag
		 *
		 * @var boolean
		 */
		protected static $_initialized = false;

		/**
		 * Slug
		 *
		 * @var string
		 */
		protected static $_slug = 'wo-adv-ui';
		/**
		 * Option sections
		 *
		 * @var array
		 */
		protected static $_sections = array();

		/**
		 * Option tabs
		 *
		 * @var array
		 */
		protected static $_tabs = array();

		/**
		 * Return name
		 *
		 * @return string
		 * @uses esc_attr()
		 */
		final public static function name(): string {
			return esc_attr( self::$_name );
		}

		/**
		 * Return slug
		 *
		 * @return string
		 * @uses esc_attr()
		 */
		final public static function slug(): string {
			return esc_attr( self::$_slug );
		}

		/**
		 * Initialization, add hooks, load text domain etc
		 */
		public static function initialize() {
			if ( ! self::$_initialized ) {
				if ( is_admin() ) {
					add_action(
						'admin_menu',
						function () {
							add_options_page(
								esc_html( __( 'Extended ui', 'wolfoz-taxonomy-mbtree' ) ),
								esc_html( __( 'Extended ui', 'wolfoz-taxonomy-mbtree' ) ),
								'manage_options',
								esc_attr( self::slug() ),
								array( __CLASS__, 'render' )
							);
						}
					);
				}
				load_textdomain( self::name(), dirname( __DIR__, 2 ) . '/languages/' . self::name() . '-' . get_user_locale() . '.mo' );
				self::$_initialized = true;
			}
		}

		/**
		 * Add setting page
		 *
		 * @param string $name the section name.
		 * @param string $title the section title.
		 */
		public static function section( string $name, string $title ): void {
			self::$_sections[ $name ] = esc_html( $title );
			self::$_tabs[ $name ]     = wp_create_nonce( $name );
		}

		/**
		 * Generates 'settings' link for plugin in plugins list
		 *
		 * @param string $slug plugin slug.
		 * @return string
		 */
		public static function link( string $slug ): string {
			$default = array_key_first( self::$_sections );
			if ( array_key_exists( $slug, self::$_sections ) ) {

				return sprintf(
					'<a href="options-general.php?page=%s">%s</a>',
					esc_attr( self::slug() ) . ( $slug === $default ? '' : '&tab=' . esc_attr( $slug ) ),
					esc_html( __( 'Settings', 'wolfoz-taxonomy-mbtree' ) )
				);
			}
			return '';
		}

		/**
		 * Render settings page
		 */
		public static function render() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}

			echo '<div class="wrap">';
			echo '<h1>' . esc_html( __( 'Extended ui settings', 'wolfoz-taxonomy-mbtree' ) ) . '</h1>';

			$active = $default = array_key_first( self::$_sections );
			// Set active tab.
			if ( isset( $_REQUEST['tab'] ) ) {
				$tab = sanitize_text_field( wp_unslash( $_REQUEST['tab'] ) );
				$key = array_search( $tab, self::$_tabs, true );
				if ( ( false !== $key ) && wp_verify_nonce( $tab, $key ) ) {
					$active = $key;
				}
			}

			echo '<nav class="nav-tab-wrapper">';
			foreach ( self::$_sections as $name => $title ) {
				echo '<a class="nav-tab' . ( $name === $active ? ' nav-tab-active' : '' )
					. '" href="options-general.php?page=' . esc_attr( self::slug() )
					. ( $name === $default ? '' : '&tab=' . esc_attr( self::$_tabs[ $name ] ) ) . '">'
					. esc_html( $title ) . '</a>';
			}
			echo '</nav>';

			echo '<form class="wo-form" method="post" action="options.php" novalidate="novalidate">';
			settings_fields( $active );
			do_settings_sections( $active );
			submit_button();
			echo '</form>';

			echo '<div class="clear"></div></div><!-- end wrap -->';
		}
	}
}
