<?php
/**
 * Metaboxes
 *
 * @package WolfOz taxonomy metaboxes
 */

namespace WolfOz\Admin;

use WolfOz\Admin;

require_once 'Settings.php';


/**
 * Metaboxes class
 *
 * @author wolfoz
 */
class Metaboxes {
	const SUFFIXES = array( '_tvw' );

	/**
	 * The plugin file name
	 *
	 * @var string
	 */
	protected static $_plugin_file = '';

	/**
	 * Taxonomies
	 *
	 * @var array
	 */
	protected static $_taxonomies = array();

	/**
	 * The plugin name
	 *
	 * @var string
	 */
	private static $_name = 'wolfoz-taxonomy-mbtree';

	/**
	 * The plugin version
	 *
	 * @var string
	 */
	private static $_ver = '1.0.14';

	/**
	 * The pplugin url
	 *
	 * @var string
	 */
	private static $_url = '';

	/**
	 * The plugin slug
	 *
	 * @var string
	 */
	private static $_slug = 'wo-mb-tvw-ui';

	/**
	 * Activate plugin
	 */
	public static function activate() {
		$name = 'wo_ui';
		foreach ( self::SUFFIXES as $suff ) {
			if ( ! get_option( $name . $suff, null ) ) {
				update_option( $name . $suff, array() );
			}
		}
	}

	/**
	 * Deactivate plugin
	 */
	public static function deactivate() {
	}

	/**
	 * Get taxonomies
	 *
	 * @return array
	 */
	public static function taxonomies(): array {
		if ( empty( self::$_taxonomies ) ) {
			foreach ( get_taxonomies( array(), 'objects' ) as $tax ) {
				if ( ( $tax->public || ! $tax->_builtin ) && $tax->show_ui ) {
					self::$_taxonomies[ $tax->name ] = $tax;
				}
			}
		}
		return self::$_taxonomies;
	}

	/**
	 * Return the plugin name
	 *
	 * @return string
	 * @uses esc_attr()
	 */
	public static function name(): string {
		return esc_attr( static::$_name );
	}

	/**
	 * Return the plugin slug
	 *
	 * @return string
	 * @uses esc_attr()
	 */
	public static function slug(): string {
		return esc_attr( static::$_slug );
	}

	/**
	 * Return the plugin version
	 *
	 * @return string
	 */
	public static function version(): string {
		return self::$_ver;
	}

	/**
	 * Get the plugin url
	 *
	 * @return string
	 */
	public static function url(): string {
		return static::$_url;
	}


	/**
	 * Common initialization.
	 *
	 * @param string $file plugin file.
	 */
	public static function initialize( string $file ) {
		self::$_plugin_file = $file;
		self::$_name        = basename( self::$_plugin_file, '.php' );
		self::$_url         = plugins_url( '/', self::$_plugin_file );

		register_activation_hook( self::$_plugin_file, array( __CLASS__, 'activate' ) );
		register_deactivation_hook( self::$_plugin_file, array( __CLASS__, 'deactivate' ) );

		Settings::initialize();

		if ( is_admin() ) {
			add_action( 'admin_init', array( __CLASS__, 'initialize_backend' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'load_resources' ) );
		}
	}

	/**
	 * Backend(admin) initialization.
	 */
	public static function initialize_backend() {

		require_once 'Walkers/TermsChecklist.php';

		load_textdomain( 'wolfoz-taxonomy-mbtree', dirname( self::$_plugin_file ) . '/languages/' . self::name() . '-' . get_user_locale() . '.mo' );

		Settings::section( self::slug(), esc_html( __( 'Taxonomy metaboxes', 'wolfoz-taxonomy-mbtree' ) ) );
		self::register_options();
		add_filter(
			'plugin_action_links_' . plugin_basename( self::$_plugin_file ),
			function ( $links ) {
				$links['settings'] = Settings::link( self::slug() );
				return $links;
			},
			10,
			1
		);

		// metabox walker.
		add_filter( 'wp_terms_checklist_args', array( __CLASS__, 'render' ), 10, 2 );
	}

	/**
	 * Load resources - scripts, styles, etc
	 */
	public static function load_resources() {

		wp_enqueue_script( self::name(), self::url() . 'assets/js/admin/' . self::name() . '.js', array( 'jquery' ), self::version(), true );

		wp_enqueue_style( self::name(), self::url() . 'assets/css/admin/' . self::name() . '.css', array( 'list-tables', 'wp-admin', 'jquery-ui-style' ), self::version() );
	}

	/**
	 * Register options
	 */
	public static function register_options() {
		$section = 'metabox_settings';
		add_settings_section(
			$section,
			esc_html( __( 'Taxonomies metaboxes tree ui', 'wolfoz-taxonomy-mbtree' ) ),
			function () {
				// echo 'Enable taxonomy ui trees';
			},
			esc_attr( self::slug() )
		);

		$name = 'wo_ui';
		register_setting(
			self::slug(),
			$name . '_tvw',
			array(
				'sanitize_callback' => function ( $values ) {
					$result = array();
					if ( is_array( $values ) ) {
						foreach ( $values as $tax => $params ) {
							$tax = sanitize_text_field( $tax );
							if ( ! array_key_exists( $tax, $result ) ) {
								$result[ $tax ] = array();
							}
							foreach ( (array) $params as $key => $value ) {
								$key = sanitize_text_field( $key );
								$result[ $tax ][ $key ] = ( 'on' === $value ) ? 'yes' : 'no';
							}
						}
					}
					return $result;
				},
			)
		);

		$tax_states = get_option( $name . '_tvw', array() );
		if ( ! is_array( $tax_states ) ) {
			$tax_states = array();
		}

		foreach ( self::taxonomies() as $tax ) {
			if ( ! array_key_exists( $tax->name, $tax_states ) ) {
				$tax_states[ $tax->name ] = array();
			}
			if ( ! is_array( $tax_states[ $tax->name ] ) ) {
				$tax_states[ $tax->name ] = array();
			}
			foreach ( array( 'enabled', 'parents', 'children', 'expanded' ) as $key ) {
				if ( ! array_key_exists( $key, $tax_states[ $tax->name ] ) ) {
					$tax_states[ $tax->name ][ $key ] = 'no';
				}
			}

			add_settings_field(
				$name . '_' . $tax->name . '_tvw',
				$tax->label,
				function ( $args ) {
					if ( ! array_key_exists( 'label_text', $args ) ) {
						$args['label_text'] = esc_html( __( 'enable metabox treeview', 'wolfoz-taxonomy-mbtree' ) );
					}
					if ( $args['taxonomy']->hierarchical ) {
						printf(
							'<label for="%1$s-%2$s-id"><input type="checkbox" name="%1$s[%2$s][enabled]" id="%1$s-%2$s-id" %3$s> %4$s</label>  ',
							esc_attr( $args['name'] ),
							esc_attr( $args['taxonomy']->name ),
							checked( $args['state']['enabled'], 'yes', false ),
							esc_html( $args['label_text'] )
						);
						printf(
							'<label for="%1$s-%2$s-expanded-id"><input type="checkbox" name="%1$s[%2$s][expanded]" id="%1$s-%2$s-expanded-id" %3$s> %4$s</label> ',
							esc_attr( $args['name'] ),
							esc_attr( $args['taxonomy']->name ),
							checked( $args['state']['expanded'], 'yes', false ),
							esc_html( __( 'shows expanded tree', 'wolfoz-taxonomy-mbtree' ) )
						);
						echo '<br />';
						printf(
							'<label for="%1$s-%2$s-parents-id"><input type="checkbox" name="%1$s[%2$s][parents]" id="%1$s-%2$s-parents-id" %3$s> %4$s</label> ',
							esc_attr( $args['name'] ),
							esc_attr( $args['taxonomy']->name ),
							checked( $args['state']['parents'], 'yes', false ),
							esc_html( __( 'autoselect parents on selecting an element', 'wolfoz-taxonomy-mbtree' ) )
						);
						printf(
							' <label for="%1$s-%2$s-children-id"><input type="checkbox" name="%1$s[%2$s][children]" id="%1$s-%2$s-children-id" %3$s> %4$s</label>',
							esc_attr( $args['name'] ),
							esc_attr( $args['taxonomy']->name ),
							checked( $args['state']['children'], 'yes', false ),
							esc_html( __( 'autodeselect children on deselecting an element', 'wolfoz-taxonomy-mbtree' ) )
						);
					} else {
						printf( '<label> %s</label>', esc_html( __( 'One-level taxonomy type', 'wolfoz-taxonomy-mbtree' ) ) );
					}
				},
				self::slug(),
				$section,
				array(
					'name'     => $name . '_tvw',
					'taxonomy' => $tax,
					'state'    => $tax_states[ $tax->name ],
				)
			);
		}
	}

	/**
	 * Replace metabox renderer(walker)
	 *
	 * @param array $args   options.
	 * @param mixed $post_id    post id.
	 * @return \WolfOz\Admin\Walkers\TermsChecklist.
	 */
	public static function render( array $args, $post_id ) {
		$trees = get_option( 'wo_ui_tvw' );
		if ( ! is_array( $trees ) ) {
			$trees = array();
		}
		if ( is_array( $args ) && array_key_exists( 'taxonomy', $args )
				&& array_key_exists( $args['taxonomy'], $trees )
				&& array_key_exists( 'enabled', $trees[ $args['taxonomy'] ] ) && 'yes' === $trees[ $args['taxonomy'] ]['enabled'] ) {  // checkbox.
					$args['checked_ontop'] = false;

					$options           = $trees[ $args['taxonomy'] ];
					$options['params'] = $args;
			if ( ! empty( $post_id ) ) {
				$options['post'] = $post_id;
			}

			$args['walker'] = new Walkers\TermsChecklist( $options );
		}
		return $args;
	}
}
