<?php
/**
 * Search Results Page
 *
 * Handles displaying the plugin search results in the admin area.
 *
 * @package WithoutReviewsFinder
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Display the search results page
 */
function wrfnder_display_search_results_page() {
    $plugins    = [];
    $start_page = isset( $_POST['wrfnder_start_page'] ) ? intval( $_POST['wrfnder_start_page'] ) : 1;

    if ( isset( $_POST['wrfnder_find_plugins'] ) ) {
        $nonce = isset( $_POST['wrfnder_find_plugins_nonce'] )
            ? sanitize_text_field( wp_unslash( $_POST['wrfnder_find_plugins_nonce'] ) )
            : '';

        if ( wp_verify_nonce( $nonce, 'wrfnder_find_plugins_action' ) ) {
            $plugins = wrfnder_get_plugins_with_no_reviews( 50, 100, $start_page );
        } else {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Security check failed. Please try again.', 'without-reviews-finder' ) . '</p></div>';
        }
    }
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Search Plugins Without Reviews', 'without-reviews-finder' ); ?></h1>

        <p>
            <?php esc_html_e( 'Each search will display up to 50 plugins from WordPress.org that currently have no reviews.', 'without-reviews-finder' ); ?>
        </p>

        <?php if ( empty( $plugins ) ) : ?>
            
            <form method="post" id="wrfnder-form">
                <?php wp_nonce_field( 'wrfnder_find_plugins_action', 'wrfnder_find_plugins_nonce' ); ?>
                <input type="hidden" name="wrfnder_start_page" value="<?php echo esc_attr( $start_page ); ?>" />
                <button type="submit" name="wrfnder_find_plugins" class="button button-primary">
                    <?php esc_html_e( 'Find Plugins', 'without-reviews-finder' ); ?>
                </button>
            </form>
        <?php endif; ?>

        <p style="margin-top:10px; font-style:italic; color:#666;">
            <?php
            if ( empty( $plugins ) ) {
                esc_html_e(
                    'Please wait after clicking the search button. Depending on network speed, it might take some time for the results to appear.',
                    'without-reviews-finder'
                );
            } else {
                esc_html_e(
                    'Use the "Search for More" button at the bottom to view additional plugins. Please wait after clicking the button. Depending on network speed, it might take some time for the results to appear.',
                    'without-reviews-finder'
                );
            }
            ?>
        </p>

        <?php if ( ! empty( $plugins ) ) : ?>
            <h2><?php esc_html_e( 'Search Results', 'without-reviews-finder' ); ?></h2>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Plugin Name', 'without-reviews-finder' ); ?></th>
                        <th><?php esc_html_e( 'Description', 'without-reviews-finder' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( $plugins as $plugin ) : ?>
                        <tr>
                            <td>
                                <a href="https://wordpress.org/plugins/<?php echo esc_attr( $plugin['slug'] ); ?>" target="_blank">
                                    <?php echo esc_html( $plugin['name'] ); ?>
                                </a>
                            </td>
                            <td><?php echo esc_html( $plugin['description'] ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <form method="post" style="margin-top:15px;">
                <?php wp_nonce_field( 'wrfnder_find_plugins_action', 'wrfnder_find_plugins_nonce' ); ?>
                <input type="hidden" name="wrfnder_start_page" value="<?php echo esc_attr( $start_page + 1 ); ?>" />
                <button type="submit" name="wrfnder_find_plugins" class="button button-secondary">
                    <?php esc_html_e( 'Search for More', 'without-reviews-finder' ); ?>
                </button>
            </form>
        <?php elseif ( isset( $_POST['wrfnder_find_plugins'] ) ) : ?>
            <p><?php esc_html_e( 'No plugins found without reviews.', 'without-reviews-finder' ); ?></p>
        <?php endif; ?>
    </div>
    <?php
}