<?php
/**
 * Plugin Name: Wise Stock – Restock Intelligence & Sales Forecast for WooCommerce
 * Description: Improve your stock management with advanced sales forecasting and automated restocking intelligence to prevent stockouts before they happen.
 * Version: 1.1.1
 * Author: Kashberg
 * Author URI: https://kashberg.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wisestock
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * WC requires at least: 7.1
 * WC tested up to: 10.3
 *
 * @package WiseStock
 */
 
// Prevent direct access.
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants.
define('WISESTOCK_VERSION', '1.1.1');
define('WISESTOCK_PLUGIN_FILE', __FILE__);
define('WISESTOCK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WISESTOCK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WISESTOCK_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Declare compatibility with WooCommerce custom order tables.
 */
add_action('before_woocommerce_init', function () {
    if (class_exists('\\Automattic\\WooCommerce\\Utilities\\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', WISESTOCK_PLUGIN_FILE, true);
    }
});

/**
 * Main plugin class.
 *
 * This is the main entry point for the Wise Stock plugin.
 * It handles plugin initialization, activation, deactivation, and
 * coordinates all core functionality including velocity calculations,
 * alert management, and supplier notifications.
 */
class WISESTOCK {

    /**
     * Plugin instance.
     *
     * @var WISESTOCK
     */
    private static $instance = null;

    /**
     * Get plugin instance.
     *
     * @return WISESTOCK
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks.
     */
    private function init_hooks() {
        add_action('init', array($this, 'initialize'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        register_uninstall_hook(__FILE__, array(__CLASS__, 'uninstall'));
    }

    /**
     * Initialize plugin.
     */
    public function initialize() {
        // Load plugin textdomain for translations
        $this->load_textdomain();

        // Check for WooCommerce dependency
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // Include required files and initialize core functionality
        $this->includes();
        $this->init_core();
    }

    /**
     * Include required files.
     */
    private function includes() {
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-core.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-velocity-calculator.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-notification-manager.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-supplier-manager.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-dashboard-widget.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-demo-data-manager.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-plugin-updater.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-api.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-timebound-settings.php';
        require_once WISESTOCK_PLUGIN_DIR . 'includes/class-wisestock-timebound-admin.php';
    }

    /**
     * Initialize core functionality.
     */
    private function init_core() {
        // Initialize plugin updater first
        WISESTOCK_Plugin_Updater::get_instance();
        
        // Initialize core functionality
        WISESTOCK_Core::get_instance();
    }

    /**
     * Load plugin textdomain.
     */
    private function load_textdomain() {
        load_plugin_textdomain(
            'wisestock', 
            false, 
            dirname(plugin_basename(__FILE__)) . '/languages/'
        );
    }

    /**
     * Plugin activation.
     */
    public function activate() {
        if (!class_exists('WooCommerce')) {
            deactivate_plugins(plugin_basename(__FILE__));
            wp_die(
                esc_html__('WiseStock requires WooCommerce to be installed and active.', 'wisestock'),
                esc_html__('Plugin Activation Error', 'wisestock'),
                array('back_link' => true)
            );
        }

        $this->includes();
        $activation_result = WISESTOCK_Core::get_instance()->activate();

        if (!$activation_result) {
            deactivate_plugins(plugin_basename(__FILE__));
            wp_die(
                esc_html__('WiseStock activation failed. Please check the error logs for details.', 'wisestock'),
                esc_html__('Plugin Activation Error', 'wisestock'),
                array('back_link' => true)
            );
        }
    }

    /**
     * Plugin deactivation.
     */
    public function deactivate() {
        $this->includes();
        WISESTOCK_Core::get_instance()->deactivate();
    }

    /**
     * Plugin uninstall.
     *
     * This method is called when the plugin is uninstalled.
     * It checks the user setting and deletes database tables if enabled.
     *
     * @since 1.0.0
     */
    public static function uninstall() {
        if (!current_user_can('activate_plugins')) {
            return;
        }

        $defaults = WISESTOCK_Core::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        $delete_data_on_uninstall = $options['delete_data_on_uninstall'];

        if (!$delete_data_on_uninstall) {
            self::cleanup_options_and_cron();
            return;
        }

        global $wpdb;

        try {
            $tables = array(
                $wpdb->prefix . 'wisestock_velocity',
                $wpdb->prefix . 'wisestock_notifications',
                $wpdb->prefix . 'wisestock_suppliers',
                $wpdb->prefix . 'wisestock_demo_products',
            );

            foreach ($tables as $table) {
                $table_exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
                if ($table_exists) {
                    $wpdb->query('DROP TABLE IF EXISTS `' . esc_sql($table) . '`');
                }
            }

            self::cleanup_options_and_cron();

            $wpdb->delete($wpdb->postmeta, array('meta_key' => '_wisestock_default_supplier'), array('%s'));
            $wpdb->delete($wpdb->postmeta, array('meta_key' => '_wisestock_velocity_data'), array('%s'));
            $wpdb->delete($wpdb->postmeta, array('meta_key' => '_wisestock_last_calculated'), array('%s'));
        } catch (Exception $e) {
            do_action('wisestock_uninstall_error', $e);
        }
    }

    /**
     * Clean up WordPress options and cron jobs.
     *
     * @since 1.0.0
     */
    private static function cleanup_options_and_cron() {
        delete_option('wisestock_options');
        delete_option('wisestock_db_version');
        delete_option('wisestock_activation_status');
        delete_option('wisestock_activation_date');
        delete_option('wisestock_installation_completed');

        wp_clear_scheduled_hook('wisestock_daily_calculations');
        wp_clear_scheduled_hook('wisestock_hourly_notifications');

        wp_cache_flush();

        delete_transient('wisestock_velocity_cache');
        delete_transient('wisestock_notifications_cache');
        delete_transient('wisestock_suppliers_cache');
    }

    /**
     * WooCommerce missing notice.
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="notice notice-error">
            <p>
                <?php
				/* translators: %s: Link to install WooCommerce. */
                printf(esc_html__('WiseStock requires WooCommerce to be installed and active. %s', 'wisestock'), '<a href="' . esc_url(admin_url('plugin-install.php?s=woocommerce&tab=search&type=term')) . '">' . esc_html__('Install WooCommerce', 'wisestock') . '</a>');
                ?>
            </p>
        </div>
        <?php
    }
}

// Initialize the plugin.
WISESTOCK::get_instance();

/**
 * Get Wise Stock API instance.
 *
 * @return WISESTOCK_API
 * @since 1.0.0
 */
function wisestock_api() {
    return WISESTOCK_API::get_instance();
}
