<?php
/**
 * Velocity Calculator Class
 *
 * This class handles sales velocity calculations using weighted rolling averages
 * and provides predictive analytics for stockout predictions and restock recommendations.
 * It implements the core algorithm: 7-day (50%), 14-day (30%), 30-day (20%).
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Velocity Calculator Class
 */
class WISESTOCK_Velocity_Calculator {

    /**
     * Velocity weights for different time periods
     *
     * @var array
     */
    private $velocity_weights;

    /**
     * Confidence calculation settings
     *
     * @var array
     */
    private $confidence_settings;

    /**
     * Constructor
     */
    public function __construct() {
        // Get velocity weights and confidence settings from store type preset
        $store_settings = WISESTOCK_Core::get_store_type_settings();
        
        $this->velocity_weights = $store_settings['velocity_weights'];
        $this->confidence_settings = $store_settings['confidence_settings'];

        // Allow weights to be filtered by hooks
        $this->velocity_weights = apply_filters('wisestock_velocity_weights', $this->velocity_weights);

        // Allow confidence settings to be filtered by hooks
        $this->confidence_settings = apply_filters('wisestock_confidence_settings', $this->confidence_settings);
    }

    /**
     * Record a sale for velocity calculation
     *
     * @param int $product_id Product ID
     * @param int $quantity Quantity sold
     * @since 1.0.0
     */
    public function record_sale($product_id, $quantity) {
        // Hook to override behavior
        $result = apply_filters('wisestock_before_record_sale', null, $product_id, $quantity);
        if ($result !== null) {
            return $result;
        }
        
        // Original stock-dependent logic
        $product = wc_get_product($product_id);
        if (!$product || !$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_velocity';
        $sanitized_table = esc_sql($table_name);
        $today = current_time('Y-m-d');

        // Check if record exists for today
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM `{$sanitized_table}` WHERE product_id = %d AND date = %s", $product_id, $today));

        if ($existing) {
            // Update existing record
            $wpdb->update(
                $table_name,
                array('quantity_sold' => $quantity),
                array('id' => $existing),
                array('%d'),
                array('%d')
            );
        } else {
            // Insert new record
            $wpdb->insert(
                $table_name,
                array(
                    'product_id' => $product_id,
                    'date' => $today,
                    'quantity_sold' => $quantity
                ),
                array('%d', '%s', '%d')
            );
        }

        // Recalculate running average
        $this->update_running_average($product_id);
        
        // Trigger action after sale is recorded
        do_action('wisestock_after_record_sale', $product_id, $quantity);
    }

    /**
     * Calculate velocity for a product over specified days
     *
     * @param int $product_id Product ID
     * @param int $days Number of days to calculate (default: 7)
     * @return array Velocity data with confidence score
     * @since 1.0.0
     */
    public function calculate_velocity($product_id, $days = 7) {
        // Check cache first for performance
        $cache_key = 'wisestock_velocity_' . $product_id . '_' . $days;
        $cached_result = get_transient($cache_key);
        if ($cached_result !== false) {
            return $cached_result;
        }
        // Hook to override velocity calculation
        $result = apply_filters('wisestock_calculate_velocity', null, $product_id, $days);
        if ($result !== null) {
            return $result;
        }

        global $wpdb;
        
        $table_name = $wpdb->prefix . 'wisestock_velocity';
        $sanitized_table = esc_sql($table_name);
        $end_date = current_time('Y-m-d');
        $start_timestamp = strtotime("-$days days", strtotime($end_date));
        $start_date = gmdate('Y-m-d', $start_timestamp);

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $results = $wpdb->get_results($wpdb->prepare("SELECT quantity_sold, date FROM `{$sanitized_table}` WHERE product_id = %d AND date BETWEEN %s AND %s ORDER BY date ASC", $product_id, $start_date, $end_date));

        if (empty($results)) {
            return array(
                'velocity' => 0,
                'confidence' => 0,
                'days_analyzed' => 0,
                'total_sold' => 0,
                'weekend_pattern' => null,
                'weekday_pattern' => null
            );
        }

        $total_sold = array_sum(wp_list_pluck($results, 'quantity_sold'));
        $velocity = $total_sold / $days;
        $confidence = $this->calculate_confidence($results, $days);

        // Calculate weekend/weekday patterns
        $patterns = $this->calculate_weekend_weekday_patterns($results);

        $result = array(
            'velocity' => round($velocity, 2),
            'confidence' => round($confidence, 2),
            'days_analyzed' => count($results),
            'total_sold' => $total_sold,
            'weekend_pattern' => $patterns['weekend'],
            'weekday_pattern' => $patterns['weekday']
        );
        
        // Cache result for 1 hour to improve performance
        set_transient($cache_key, $result, HOUR_IN_SECONDS);
        
        return $result;
    }

    /**
     * Get available data collection methods
     *
     * @return array Available data collection methods
     * @since 1.0.0
     */
    public function get_data_collection_methods() {
        $default_methods = array(
            'stock_management' => array($this, 'record_sale')
        );
        
        return apply_filters('wisestock_data_collection_methods', $default_methods);
    }

    /**
     * Calculate weighted velocity using multiple time periods
     *
     * @param int $product_id Product ID
     * @return array Weighted velocity data
     * @since 1.0.0
     */
    public function calculate_weighted_velocity($product_id) {
        $velocities = array();
        $total_weight = 0;
        $weighted_sum = 0;

        foreach ($this->velocity_weights as $period => $weight) {
            $days = (int) str_replace('_day', '', $period);
            $velocity_data = $this->calculate_velocity($product_id, $days);

            $velocities[$period] = $velocity_data;
            $weighted_sum += $velocity_data['velocity'] * $weight;
            $total_weight += $weight;
        }

        $weighted_velocity = $total_weight > 0 ? $weighted_sum / $total_weight : 0;

        // Calculate overall confidence based on the best available data
        $overall_confidence = 0;
        $best_confidence = 0;

        foreach ($velocities as $period => $data) {
            if ($data['confidence'] > $best_confidence) {
                $best_confidence = $data['confidence'];
            }
        }

        // Use the best confidence, but ensure it's reasonable for historical data
        $overall_confidence = max($this->confidence_settings['min_confidence'], $best_confidence);

        return array(
            'weighted_velocity' => round($weighted_velocity, 2),
            'period_velocities' => $velocities,
            'weights' => $this->velocity_weights,
            'confidence' => $overall_confidence
        );
    }

    /**
     * Predict days until stockout
     *
     * @param int $product_id Product ID
     * @return int Days until stockout (0 if no stock, -1 if increasing stock)
     * @since 1.0.0
     */
    public function predict_stockout($product_id) {
        $product = wc_get_product($product_id);
        if (!$product || !$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
            return -1;
        }

        $current_stock = $product->get_stock_quantity();
        if (0 >= $current_stock) {
			return 0;
		}

        $velocity_data = $this->calculate_weighted_velocity($product_id);
        $daily_velocity = $velocity_data['weighted_velocity'];

        if (0 >= $daily_velocity) {
			return -1; // No sales velocity, stock won't deplete
		}

        return (int) ceil($current_stock / $daily_velocity);
    }

    /**
     * Get restock recommendation
     *
     * @param int $product_id Product ID
     * @return array Restock recommendation data
     * @since 1.0.0
     */
    public function get_restock_recommendation($product_id) {
        $product = wc_get_product($product_id);
        if (!$product || !$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
            return array(
                'recommended_quantity' => 0,
                'urgency' => 'none',
                'reasoning' => 'Product not managing stock or is digital product'
            );
        }

        $velocity_data = $this->calculate_weighted_velocity($product_id);
        $daily_velocity = $velocity_data['weighted_velocity'];
        $current_stock = $product->get_stock_quantity();
        $lead_time = $this->get_lead_time($product_id);

        if (0 >= $daily_velocity) {
            return array(
                'recommended_quantity' => 0,
                'urgency' => 'none',
                'reasoning' => 'No sales velocity detected'
            );
        }

        // Get prediction window and confidence
        $optimal_window = $this->get_optimal_prediction_window( $product_id );
        $timebound_velocity = $this->get_timebound_velocity( $product_id, $optimal_window );
        $confidence = $timebound_velocity['confidence'] ?? $velocity_data['confidence'];
        
        // Calculate volatility for risk adjustment
        $volatility = $this->calculate_velocity_volatility( $velocity_data['period_velocities'] );
        
        // No market change adjustments - removed
        $market_adjustment = 1.0;
        
        // Calculate base quantities
        $safety_stock = $daily_velocity * 7; // 7 days base safety stock
        $lead_time_stock = $daily_velocity * $lead_time;
        
        // Apply confidence-based buffer (lower confidence = higher buffer)
        $confidence_buffer_factor = ( 1 - $confidence ) * 0.3; // Max 30% buffer
        $confidence_buffer = ( $safety_stock + $lead_time_stock ) * $confidence_buffer_factor;
        
        // Apply volatility buffer (higher volatility = higher buffer)
        $volatility_buffer_factor = $volatility * 0.25; // Max 25% buffer for high volatility
        $volatility_buffer = ( $safety_stock + $lead_time_stock ) * $volatility_buffer_factor;
        
        // Calculate comprehensive recommended quantity
        $base_quantity = $safety_stock + $lead_time_stock;
        $total_buffer = $confidence_buffer + $volatility_buffer;
        $adjusted_quantity = $base_quantity + $total_buffer;
        
        // Apply market adjustment
        $adjusted_quantity *= $market_adjustment;
        
        // Subtract current stock
        $recommended_quantity = max(0, $adjusted_quantity - $current_stock);

        // Determine urgency
        $days_to_stockout = $this->predict_stockout($product_id);
        $urgency = $this->determine_urgency($days_to_stockout, $lead_time);
        
        // Calculate risk level
        $risk_level = $this->calculate_risk_level( $confidence );

        $recommendation = array(
            'recommended_quantity' => (int) round($recommended_quantity),
            'urgency' => $urgency,
            'days_to_stockout' => $days_to_stockout,
            'daily_velocity' => $daily_velocity,
            'safety_stock' => (int) round($safety_stock),
            'lead_time' => $lead_time,
            'reasoning' => $this->generate_comprehensive_reasoning(
                $days_to_stockout,
                $lead_time,
                $daily_velocity,
                $confidence,
                $volatility
            ),
            // Enhanced fields
            'prediction_window_days' => $optimal_window,
            'confidence' => $confidence,
            'volatility' => $volatility,
            'risk_level' => $risk_level,
            'confidence_buffer' => (int) round($confidence_buffer),
            'volatility_buffer' => (int) round($volatility_buffer),

            'recommendation_valid_until' => gmdate( 'Y-m-d', strtotime( "+{$optimal_window} days" ) )
        );

        // Apply filter to allow customization of restock recommendations
        return apply_filters('wisestock_restock_recommendation', $recommendation, $product_id);
    }

    /**
     * Generate comprehensive reasoning for restock recommendation
     *
     * @param int $days_to_stockout Days to stockout
     * @param int $lead_time Lead time in days
     * @param float $daily_velocity Daily velocity
     * @param float $confidence Confidence score
     * @param float $volatility Volatility score

     * @return string Comprehensive reasoning text
     * @since 1.0.2
     */
    private function generate_comprehensive_reasoning( $days_to_stockout, $lead_time, $daily_velocity, $confidence, $volatility ) {
        $confidence_text = round( $confidence * 100, 1 ) . '%';
        $volatility_text = round( $volatility * 100, 1 ) . '%';
        
        // Base urgency message
        if ( $days_to_stockout <= 0 ) {
            $urgency_message = __( 'CRITICAL: Out of stock! ', 'wisestock' );
        } elseif ( $days_to_stockout <= 3 ) {
            $urgency_message = sprintf(
                /* translators: %d: days to stockout */
                __( 'CRITICAL: Stock runs out in %d days. ', 'wisestock' ),
                $days_to_stockout
            );
        } elseif ( $days_to_stockout <= $lead_time ) {
            $urgency_message = sprintf(
                /* translators: %d: days to stockout */
                __( 'WARNING: Stock runs out in %d days. ', 'wisestock' ),
                $days_to_stockout
            );
        } else {
            $urgency_message = '';
        }
        
        // Confidence and volatility context
        $confidence_message = sprintf(
            /* translators: %s: confidence percentage */
            __( 'Prediction confidence: %s. ', 'wisestock' ),
            $confidence_text
        );
        
        $volatility_message = '';
        if ( $volatility > 0.3 ) {
            $volatility_message = sprintf(
                /* translators: %s: volatility percentage */
                __( 'HIGH volatility (%s) - added safety buffer. ', 'wisestock' ),
                $volatility_text
            );
        } elseif ( $volatility > 0.15 ) {
            $volatility_message = sprintf(
                /* translators: %s: volatility percentage */
                __( 'Moderate volatility (%s). ', 'wisestock' ),
                $volatility_text
            );
        }
        
        return $urgency_message . $confidence_message . $volatility_message;
    }

    /**
     * Update running average for a product
     *
     * @param int $product_id Product ID
     * @since 1.0.0
     */
    private function update_running_average($product_id) {
        $velocity_data = $this->calculate_weighted_velocity($product_id);
        $running_average = $velocity_data['weighted_velocity'];

        global $wpdb;

        $table_name = $wpdb->prefix . 'wisestock_velocity';
        $today = current_time('Y-m-d');

        $wpdb->update(
            $table_name,
            array('running_average' => $running_average),
            array(
                'product_id' => $product_id,
                'date' => $today
            ),
            array('%f'),
            array('%d', '%s')
        );

        // Clear velocity-related caches
        $this->clear_velocity_caches($product_id);
        
        // Trigger velocity updated action
        do_action('wisestock_velocity_updated', $product_id, $velocity_data);
    }

    /**
     * Update velocities for all products
     *
     * @since 1.0.0
     */
    public function update_all_velocities() {
        $products = wc_get_products(array(
            'limit' => -1,
            'status' => 'publish',
            'manage_stock' => true
        ));

        foreach ($products as $product) {
            // Double-check stock management is enabled and product is not digital
            if (!$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
                continue;
            }
            $this->update_running_average($product->get_id());
        }
    }

    /**
     * Calculate confidence score based on data consistency
     *
     * @param array $results Sales data results
     * @param int $expected_days Expected number of days
     * @return float Confidence score (0-1)
     * @since 1.0.0
     */
    private function calculate_confidence($results, $expected_days) {
        $actual_days = count($results);

        if (0 === $actual_days) {
			return 0;
		}

        // Base confidence on data completeness, but be more generous
        $data_completeness = min(1, $actual_days / max($expected_days, 1));

        // For sparse data scenarios
        // So we'll be more generous with confidence scores
        if (3 <= $actual_days) {
			// Good data - calculate consistency
			$quantities = wp_list_pluck($results, 'quantity_sold');
			$mean = array_sum($quantities) / count($quantities);

			if (0 < $mean) {
                $variance = 0;
                foreach ($quantities as $quantity) {
                    $variance += pow($quantity - $mean, 2);
                }
                $variance /= count($quantities);

                // Lower variance = higher confidence, but cap the penalty
                $consistency = max($this->confidence_settings['good_data_min_confidence'], 1 - ($variance / max($mean * 2, 1)));
            } else {
                $consistency = $this->confidence_settings['sparse_data_confidence']; // Use setting for zero sales
            }
		} elseif (1 <= $actual_days) {
			// Some data - use moderate confidence from settings
			$consistency = $this->confidence_settings['sparse_data_confidence'];
		} else {
			// No data
			$consistency = 0;
		}

        // Combine completeness and consistency, but don't be too harsh
        $confidence = $data_completeness * $consistency;

        // Ensure minimum confidence for any data from settings
        return max($this->confidence_settings['min_confidence'], min(1, $confidence));
    }

    /**
     * Get lead time for a product
     *
     * @param int $product_id Product ID
     * @return int Lead time in days
     * @since 1.0.0
     */
    private function get_lead_time($product_id) {
        // Check if product has supplier with lead time
        $supplier_manager = new WISESTOCK_Supplier_Manager();
        $supplier = $supplier_manager->get_product_supplier($product_id);

        if ($supplier && isset($supplier->lead_time_days)) {
            return (int) $supplier->lead_time_days;
        }

        // Default lead time
        return 7;
    }

    /**
     * Determine urgency level
     *
     * @param int $days_to_stockout Days until stockout
     * @param int $lead_time Lead time in days
     * @return string Urgency level
     * @since 1.0.0
     */
    private function determine_urgency($days_to_stockout, $lead_time) {
        if (0 >= $days_to_stockout) {
			return 'critical';
		} elseif (3 >= $days_to_stockout) {
			return 'critical';
		} elseif ($lead_time >= $days_to_stockout) {
			return 'warning';
		} elseif ($lead_time + 7 >= $days_to_stockout) {
			return 'mild';
		}

		return 'none';
	}

    /**
     * Generate reasoning for recommendation
     *
     * @param int $days_to_stockout Days until stockout
     * @param int $lead_time Lead time in days
     * @param float $daily_velocity Daily sales velocity
     * @return string Reasoning text
     * @since 1.0.0
     */
    private function generate_reasoning($days_to_stockout, $lead_time, $daily_velocity) {
        if (0 >= $days_to_stockout) {
			return esc_html__('Product is out of stock and needs immediate restocking.', 'wisestock');
		} elseif ($lead_time >= $days_to_stockout) {
			/* translators: 1: Days to stockout. 2: Lead time in days. */
			return sprintf(esc_html__('Product will stockout in %1$d days, but lead time is %2$d days. Immediate action required.', 'wisestock'), $days_to_stockout, $lead_time);
		} elseif ($lead_time + 7 >= $days_to_stockout) {
			$available_days = max($days_to_stockout - $lead_time, 0);
			/* translators: 1: Days to stockout. 2: Recommended order window in days. */
			return sprintf(esc_html__('Product will stockout in %1$d days. Recommended to order within %2$d days to maintain stock.', 'wisestock'), $days_to_stockout, $available_days);
		}

		/* translators: %.2f: Daily sales velocity. */
		return sprintf(esc_html__('Product has healthy stock levels. Daily velocity: %.2f units/day.', 'wisestock'), $daily_velocity);
    }

    /**
     * Calculate weekend/weekday patterns
     *
     * @param array $results Sales data results
     * @return array Weekend and weekday patterns
     * @since 1.0.0
     */
    private function calculate_weekend_weekday_patterns($results) {
        $weekend_sales = array();
        $weekday_sales = array();

        foreach ($results as $result) {
            $day_of_week = gmdate('N', strtotime($result->date . ' UTC')); // 1 = Monday, 7 = Sunday
            $quantity = $result->quantity_sold;

            if ($day_of_week >= 6) { // Saturday (6) or Sunday (7)
                $weekend_sales[] = $quantity;
            } else {
                $weekday_sales[] = $quantity;
            }
        }

        $weekend_avg = !empty($weekend_sales) ? array_sum($weekend_sales) / count($weekend_sales) : 0;
        $weekday_avg = !empty($weekday_sales) ? array_sum($weekday_sales) / count($weekday_sales) : 0;

        return array(
            'weekend' => array(
                'average' => round($weekend_avg, 2),
                'total_days' => count($weekend_sales),
                'total_sales' => array_sum($weekend_sales)
            ),
            'weekday' => array(
                'average' => round($weekday_avg, 2),
                'total_days' => count($weekday_sales),
                'total_sales' => array_sum($weekday_sales)
            )
        );
    }

    /**
     * Clear velocity-related caches for a product
     *
     * @param int $product_id Product ID
     * @since 1.0.0
     */
    private function clear_velocity_caches($product_id) {
        // Clear velocity calculation caches for different day periods
        $cache_keys = array(
            'wisestock_velocity_' . $product_id . '_7',
            'wisestock_velocity_' . $product_id . '_14',
            'wisestock_velocity_' . $product_id . '_30'
        );
        
        foreach ($cache_keys as $cache_key) {
            delete_transient($cache_key);
        }
    }

    // =====================================================================
    // TIMEBOUND PREDICTION METHODS
    // =====================================================================

    /**
     * Get timebound velocity with prediction window
     *
     * @param int $product_id Product ID
     * @param int $prediction_window_days Optional prediction window
     * @return array Timebound velocity data
     * @since 1.0.1
     */
    public function get_timebound_velocity( $product_id, $prediction_window_days = null ) {
        // Use default prediction window if not specified
        if ( null === $prediction_window_days ) {
            $prediction_window_days = $this->get_optimal_prediction_window( $product_id );
        }

        // Get base velocity data
        $base_velocity = $this->calculate_weighted_velocity( $product_id );
        
        // Apply timebound enhancements
        $timebound_data = array(
            'product_id' => $product_id,
            'prediction_window_days' => $prediction_window_days,
            'weighted_velocity' => $base_velocity['weighted_velocity'],
            'base_confidence' => $base_velocity['confidence'],
            'timebound_enabled' => true
        );

        // Apply confidence decay if enabled
        $timebound_settings = $this->get_timebound_settings();
        if ( $timebound_settings['confidence_decay_enabled'] ) {
            $timebound_data['confidence'] = $this->apply_confidence_decay(
                $base_velocity['confidence'],
                $product_id,
                $prediction_window_days
            );
        } else {
            $timebound_data['confidence'] = $base_velocity['confidence'];
        }

        // Calculate predicted sales for the window
        $timebound_data['predicted_sales'] = round( $base_velocity['weighted_velocity'] * $prediction_window_days, 0 );

        // Calculate enhanced timebound metrics from historical data
        $volatility = $this->calculate_velocity_volatility( $base_velocity['period_velocities'] );
        $trend_direction = $this->calculate_trend_direction( $base_velocity['period_velocities'] );
        $market_sensitivity = $this->calculate_market_sensitivity( $volatility, $base_velocity['weighted_velocity'], $trend_direction );
        
        // Add enhanced metrics to timebound data
        $timebound_data['volatility'] = $volatility;
        $timebound_data['trend_direction'] = $trend_direction;
        $timebound_data['market_sensitivity'] = $market_sensitivity;
        
        // Calculate settings-based confidence decay rate
        $settings_instance = WISESTOCK_Timebound_Settings::get_instance();
        $current_settings = $settings_instance->get_settings();
        $store_preset = $current_settings['store_type_preset'] ?? 'retail_store';
        $store_presets = $settings_instance->get_store_type_presets();
        
        $confidence_decay_rate = 0.12; // Default
        if ( isset( $store_presets[$store_preset]['settings']['confidence_decay_rate'] ) ) {
            $confidence_decay_rate = $store_presets[$store_preset]['settings']['confidence_decay_rate'];
        }
        $timebound_data['confidence_decay_rate'] = $confidence_decay_rate;

        // Add period velocities for transparency
        $timebound_data['period_velocities'] = $base_velocity['period_velocities'];
        $timebound_data['weights'] = $base_velocity['weights'];

        return apply_filters( 'wisestock_timebound_velocity_data', $timebound_data, $product_id );
    }

    /**
     * Get smart stock recommendation with timebound analysis
     *
     * @param int $product_id Product ID
     * @param int $preferred_days Optional preferred prediction window
     * @return array Enhanced stock recommendation
     * @since 1.0.1
     */
    public function get_smart_stock_recommendation( $product_id, $preferred_days = null ) {
        // Get base recommendation
        $base_recommendation = $this->get_restock_recommendation( $product_id );
        
        if ( 0 === $base_recommendation['recommended_quantity'] ) {
            return $base_recommendation; // No enhancement needed
        }

        // Get timebound velocity data
        $timebound_velocity = $this->get_timebound_velocity( $product_id, $preferred_days );
        
        // Enhance recommendation with timebound data
        $enhanced_recommendation = $base_recommendation;
        $enhanced_recommendation['prediction_window_days'] = $timebound_velocity['prediction_window_days'];
        $enhanced_recommendation['confidence'] = $timebound_velocity['confidence'];
        $enhanced_recommendation['predicted_sales'] = $timebound_velocity['predicted_sales'];
        
        // Calculate confidence-based safety buffer
        $safety_buffer = $this->calculate_confidence_based_safety_buffer(
            $base_recommendation['recommended_quantity'],
            $timebound_velocity['confidence']
        );
        $enhanced_recommendation['safety_buffer'] = $safety_buffer;
        
        // Adjust recommended quantity based on confidence
        $enhanced_recommendation['recommended_quantity'] = max(
            $base_recommendation['recommended_quantity'],
            $base_recommendation['recommended_quantity'] + $safety_buffer
        );

        // Add risk assessment
        $enhanced_recommendation['risk_level'] = $this->calculate_risk_level( $timebound_velocity['confidence'] );
        
        // Add validity period
        $enhanced_recommendation['recommendation_valid_until'] = gmdate( 'Y-m-d', strtotime( "+{$timebound_velocity['prediction_window_days']} days" ) );
        
        // Enhanced reasoning
        $enhanced_recommendation['reasoning'] = $this->generate_timebound_reasoning(
            $base_recommendation['days_to_stockout'],
            $base_recommendation['daily_velocity'],
            $timebound_velocity['prediction_window_days'],
            $timebound_velocity['confidence']
        );

        return apply_filters( 'wisestock_timebound_recommendation', $enhanced_recommendation, $product_id );
    }

    /**
     * Get optimal prediction window for a product
     *
     * @param int $product_id Product ID
     * @return int Optimal prediction window in days
     * @since 1.0.1
     */
    public function get_optimal_prediction_window( $product_id ) {
        $timebound_settings = $this->get_timebound_settings();
        
        if ( ! $timebound_settings['auto_window_adjustment'] ) {
            return $timebound_settings['default_prediction_window'];
        }

        // Get product velocity data for analysis
        $velocity_data = $this->calculate_weighted_velocity( $product_id );
        
        // Calculate volatility based on velocity variations
        $volatility = $this->calculate_velocity_volatility( $velocity_data['period_velocities'] );
        
        // Adjust window based on volatility
        $base_window = $timebound_settings['default_prediction_window'];
        
        if ( $volatility > $timebound_settings['volatility_threshold'] ) {
            // High volatility - use shorter window
            $optimal_window = max( 7, $base_window * 0.7 );
        } elseif ( $volatility < ( $timebound_settings['volatility_threshold'] * 0.5 ) ) {
            // Low volatility - can use longer window
            $optimal_window = min( $timebound_settings['max_prediction_window'], $base_window * 1.5 );
        } else {
            // Normal volatility - use default
            $optimal_window = $base_window;
        }

        return (int) $optimal_window;
    }



    /**
     * Get intelligent prediction windows with confidence scores
     *
     * @param int $product_id Product ID
     * @return array Prediction windows with confidence
     * @since 1.0.1
     */
    public function get_intelligent_prediction_windows( $product_id ) {
        $windows = array( 7, 14, 21, 30, 45, 60, 90 );
        $results = array();
        
        $timebound_settings = $this->get_timebound_settings();

        foreach ( $windows as $window ) {
            if ( $window > $timebound_settings['max_prediction_window'] ) {
                continue;
            }

            $timebound_data = $this->get_timebound_velocity( $product_id, $window );
            
            $results[] = array(
                'window_days' => $window,
                'confidence' => $timebound_data['confidence'],
                'predicted_sales' => $timebound_data['predicted_sales'],
                'is_recommended' => $window === $this->get_optimal_prediction_window( $product_id ),
                'risk_level' => $this->calculate_risk_level( $timebound_data['confidence'] )
            );
        }

        // Sort by confidence (highest first)
        usort( $results, function( $a, $b ) {
            return $b['confidence'] <=> $a['confidence'];
        } );

        return $results;
    }

    /**
     * Apply confidence decay based on prediction window and product characteristics
     *
     * @param float $base_confidence Base confidence score
     * @param int $product_id Product ID
     * @param int $prediction_window_days Prediction window
     * @return float Decayed confidence score
     * @since 1.0.1
     */
    private function apply_confidence_decay( $base_confidence, $product_id, $prediction_window_days ) {
        // Get product for category analysis
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return $base_confidence;
        }

        // Get store type preset decay rate (prioritized over product type)
        $settings_instance = WISESTOCK_Timebound_Settings::get_instance();
        $current_settings = $settings_instance->get_settings();
        $store_preset = $current_settings['store_type_preset'] ?? 'retail_store';
        $store_presets = $settings_instance->get_store_type_presets();
        
        // Use store type preset decay rate if available, fallback to product type
        if ( isset( $store_presets[$store_preset]['settings']['confidence_decay_rate'] ) ) {
            $decay_rate = $store_presets[$store_preset]['settings']['confidence_decay_rate'];
        } else {
            $decay_rate = $this->get_product_decay_rate( $product );
        }
        
        // Apply time-based decay (longer predictions = lower confidence)
        $time_decay_factor = 1 - ( ( $prediction_window_days - 7 ) * $decay_rate );
        $time_decay_factor = max( 0.1, min( 1.0, $time_decay_factor ) ); // Clamp between 0.1 and 1.0

        $decayed_confidence = $base_confidence * $time_decay_factor;
        
        $timebound_settings = $this->get_timebound_settings();
        // Ensure minimum threshold
        return max( $timebound_settings['min_confidence_threshold'], $decayed_confidence );
    }

    /**
     * Get product-specific decay rate
     *
     * @param WC_Product $product Product instance  
     * @return float Decay rate
     * @since 1.0.1
     */
    private function get_product_decay_rate( $product ) {
        $timebound_settings = $this->get_timebound_settings();
        $decay_rates = $timebound_settings['product_type_decay_rates'];
        
        // Try to determine product type from categories
        $categories = wp_get_post_terms( $product->get_id(), 'product_cat', array( 'fields' => 'names' ) );
        
        foreach ( $categories as $category ) {
            $category_lower = strtolower( $category );
            
            // Check for fashion keywords
            if ( strpos( $category_lower, 'fashion' ) !== false || 
                 strpos( $category_lower, 'clothing' ) !== false ||
                 strpos( $category_lower, 'apparel' ) !== false ) {
                return $decay_rates['fashion'];
            }
            
            // Check for electronics keywords
            if ( strpos( $category_lower, 'electronic' ) !== false ||
                 strpos( $category_lower, 'tech' ) !== false ||
                 strpos( $category_lower, 'gadget' ) !== false ) {
                return $decay_rates['electronics'];
            }
            
            // Check for health keywords
            if ( strpos( $category_lower, 'health' ) !== false ||
                 strpos( $category_lower, 'medical' ) !== false ||
                 strpos( $category_lower, 'wellness' ) !== false ) {
                return $decay_rates['health'];
            }
            
            // Check for food keywords
            if ( strpos( $category_lower, 'food' ) !== false ||
                 strpos( $category_lower, 'grocery' ) !== false ||
                 strpos( $category_lower, 'beverage' ) !== false ) {
                return $decay_rates['food'];
            }
        }
        
        // Default decay rate for general products
        return $decay_rates['general'];
    }

    /**
     * Calculate velocity volatility
     *
     * @param array $period_velocities Period velocity data
     * @return float Volatility score (0-1)
     * @since 1.0.1
     */
    private function calculate_velocity_volatility( $period_velocities ) {
        $velocities = array();
        
        foreach ( $period_velocities as $period => $data ) {
            if ( isset( $data['velocity'] ) && $data['velocity'] > 0 ) {
                $velocities[] = $data['velocity'];
            }
        }
        
        if ( count( $velocities ) < 2 ) {
            return 0; // No volatility with insufficient data
        }
        
        // Calculate coefficient of variation (standard deviation / mean)
        $mean = array_sum( $velocities ) / count( $velocities );
        
        if ( 0 === $mean ) {
            return 0;
        }
        
        $variance = 0;
        foreach ( $velocities as $velocity ) {
            $variance += pow( $velocity - $mean, 2 );
        }
        
        $std_dev = sqrt( $variance / count( $velocities ) );
        $coefficient_of_variation = $std_dev / $mean;
        
        // Normalize to 0-1 scale (cap at 1.0 for very high volatility)
        return min( 1.0, $coefficient_of_variation );
    }

    /**
     * Calculate trend direction from historical velocity data
     * 
     * Analyzes period velocities to determine if sales trend is increasing, stable, or decreasing
     * using the trend_detection_sensitivity setting from store type presets.
     *
     * @param array $period_velocities Period velocity data (7d, 14d, 30d)
     * @return string Trend direction: 'increasing', 'stable', or 'decreasing'
     * @since 1.0.1
     */
    private function calculate_trend_direction( $period_velocities ) {
        $velocities = array();
        $periods = array( 7, 14, 30 ); // Order matters - recent to older
        
        // Extract velocities in chronological order (recent first)
        foreach ( $periods as $period ) {
            if ( isset( $period_velocities[$period] ) && isset( $period_velocities[$period]['velocity'] ) ) {
                $velocities[] = $period_velocities[$period]['velocity'];
            }
        }
        
        if ( count( $velocities ) < 2 ) {
            return 'stable'; // Not enough data to determine trend
        }
        
        // Get trend detection sensitivity from settings
        $timebound_settings = $this->get_timebound_settings();
        $settings_instance = WISESTOCK_Timebound_Settings::get_instance();
        $current_settings = $settings_instance->get_settings();
        $store_preset = $current_settings['store_type_preset'] ?? 'retail_store';
        $store_presets = $settings_instance->get_store_type_presets();
        
        $sensitivity = 0.6; // Default sensitivity
        if ( isset( $store_presets[$store_preset]['settings']['trend_detection_sensitivity'] ) ) {
            $sensitivity = $store_presets[$store_preset]['settings']['trend_detection_sensitivity'];
        }
        
        // Calculate trend using weighted comparison (recent data has more weight)
        $recent_velocity = $velocities[0]; // 7-day (most recent)
        $older_velocity = $velocities[count($velocities) - 1]; // 30-day (oldest available)
        
        if ( 0 === $older_velocity ) {
            return 'stable'; // Avoid division by zero
        }
        
        // Calculate percentage change from older to recent
        $change_percent = ( $recent_velocity - $older_velocity ) / $older_velocity;
        
        // Apply sensitivity threshold - higher sensitivity = detect smaller trends
        $threshold = ( 1 - $sensitivity ) * 0.3; // Range: 0.03 (high sens) to 0.15 (low sens)
        
        if ( $change_percent > $threshold ) {
            return 'increasing';
        } elseif ( $change_percent < -$threshold ) {
            return 'decreasing';
        } else {
            return 'stable';
        }
    }

    /**
     * Calculate market sensitivity based on actual volatility and velocity patterns
     * 
     * Uses existing settings to determine how sensitive a product is to market changes
     * based on its historical volatility and sales velocity patterns.
     *
     * @param float $volatility Calculated volatility from period data
     * @param float $velocity Current velocity
     * @param string $trend_direction Calculated trend direction
     * @return float Market sensitivity (0.0 to 1.0)
     * @since 1.0.1
     */
    private function calculate_market_sensitivity( $volatility, $velocity, $trend_direction ) {
        // Base sensitivity correlates more strongly with volatility
        $base_sensitivity = 0.3 + ( $volatility * 1.2 ); // Higher correlation with volatility
        
        // Adjust based on trend direction - changing trends indicate market responsiveness
        if ( 'stable' === $trend_direction ) {
            $trend_adjustment = -0.15; // Stable trends = lower market sensitivity
        } else {
            $trend_adjustment = 0.2; // Changing trends = higher market sensitivity
        }
        
        // Velocity influence - higher velocity can indicate market responsiveness
        $velocity_adjustment = min( 0.15, $velocity * 0.03 ); // Increased influence
        
        // Combine factors
        $market_sensitivity = $base_sensitivity + $trend_adjustment + $velocity_adjustment;
        
        // Ensure realistic bounds
        return (float) round( min( 0.85, max( 0.15, $market_sensitivity ) ), 2 );
    }

    /**
     * Calculate confidence-based safety buffer
     *
     * @param int $base_quantity Base recommended quantity
     * @param float $confidence Confidence score
     * @return int Safety buffer units
     * @since 1.0.1
     */
    private function calculate_confidence_based_safety_buffer( $base_quantity, $confidence ) {
        // Lower confidence = higher safety buffer
        $buffer_factor = ( 1 - $confidence ) * 0.3; // Max 30% buffer for 0% confidence
        $safety_buffer = $base_quantity * $buffer_factor;
        
        return (int) round( $safety_buffer );
    }

    /**
     * Calculate risk level based on confidence
     *
     * @param float $confidence Confidence score
     * @return string Risk level (low, medium, high)
     * @since 1.0.1
     */
    private function calculate_risk_level( $confidence ) {
        if ( $confidence >= 0.8 ) {
            return 'low';
        } elseif ( $confidence >= 0.6 ) {
            return 'medium';
        } else {
            return 'high';
        }
    }

    /**
     * Generate timebound-specific reasoning
     *
     * @param int $days_to_stockout Days to stockout
     * @param float $daily_velocity Daily velocity
     * @param int $prediction_window Prediction window
     * @param float $confidence Confidence score
     * @return string Reasoning text
     * @since 1.0.1
     */
    private function generate_timebound_reasoning( $days_to_stockout, $daily_velocity, $prediction_window, $confidence ) {
        $confidence_text = round( $confidence * 100, 1 ) . '%';
        
        if ( $days_to_stockout <= 3 ) {
            return sprintf(
                /* translators: %1$d: days to stockout, %2$d: prediction window, %3$s: confidence percentage */
                __( 'Critical: Stock runs out in %1$d days. Time-bound analysis (%3$s confidence) predicts %2$d-day replenishment need.', 'wisestock' ),
                $days_to_stockout,
                $prediction_window,
                $confidence_text
            );
        } elseif ( $days_to_stockout <= 7 ) {
            return sprintf(
                /* translators: %1$d: days to stockout, %2$d: prediction window, %3$s: confidence percentage */
                __( 'Warning: Stock runs out in %1$d days. Time-bound forecast (%3$s confidence) suggests proactive restocking.', 'wisestock' ),
                $days_to_stockout,
                $prediction_window,
                $confidence_text
            );
        } else {
            return sprintf(
                /* translators: %1$d: prediction window, %2$s: confidence percentage */
                __( 'Info: %1$d-day time-bound prediction (%2$s confidence) indicates upcoming restock opportunity.', 'wisestock' ),
                $prediction_window,
                $confidence_text
            );
        }
    }

    /**
     * Get timebound settings
     *
     * @return array Timebound settings
     * @since 1.0.1
     */
    public function get_timebound_settings() {
        // Use static call to avoid circular dependency during initialization
        $all_defaults = WISESTOCK_Core::get_default_options();
        $default_settings = $all_defaults['timebound_settings'];

        // Get timebound settings from consolidated options
        $main_options = get_option('wisestock_options', array());
        if (isset($main_options['timebound_settings'])) {
            $saved_settings = $main_options['timebound_settings'];
        } else {
            $saved_settings = array();
        }
        $settings = wp_parse_args( $saved_settings, $default_settings );

        // Allow filtering of timebound settings
        return apply_filters( 'wisestock_timebound_settings', $settings );
    }

    /**
     * Update timebound settings
     *
     * @param array $settings New settings
     * @since 1.0.1
     */
    public function update_timebound_settings( $settings ) {
        $current_settings = $this->get_timebound_settings();
        $updated_settings = wp_parse_args( $settings, $current_settings );
        
        // Update consolidated options
        $main_options = get_option('wisestock_options', array());
        $main_options['timebound_settings'] = $updated_settings;
        update_option('wisestock_options', $main_options);
    }
}
