<?php
/**
 * Timebound Settings Manager Class
 *
 * This class manages timebound-specific settings including prediction windows,
 * confidence decay rates, market change thresholds, and store type presets.
 *
 * @package WiseStock
 * @since 1.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Timebound Settings Manager Class
 */
class WISESTOCK_Timebound_Settings {

	/**
	 * Settings instance
	 *
	 * @var WISESTOCK_Timebound_Settings
	 */
	private static $instance = null;

	/**
	 * Get settings instance
	 *
	 * @return WISESTOCK_Timebound_Settings
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Settings will be initialized on first access
		// to avoid database operations during plugin loading
	}

	/**
	 * Initialize settings on plugin activation
	 *
	 * @since 1.0.1
	 */
	public function init_settings() {
		// Check if settings exist in main options
		$main_options = get_option('wisestock_options', array());
		if (!isset($main_options['timebound_settings'])) {
			// Create new defaults
			$default_settings = $this->get_default_timebound_settings();
			$main_options['timebound_settings'] = $default_settings;
			update_option('wisestock_options', $main_options);
		}
	}	/**
	 * Get default timebound settings
	 *
	 * @return array Default settings
	 * @since 1.0.1
	 */
	public function get_default_timebound_settings() {
		return array(
			'default_prediction_window' => 14,
			'max_prediction_window' => 90,
			'min_confidence_threshold' => 0.5,
			'volatility_threshold' => 0.25, // Default to Balanced (25%)
			'confidence_decay_enabled' => true, // Enabled by default
			'auto_window_adjustment' => true, // Enabled by default
			'product_type_decay_rates' => array(
				'fashion' => 0.15,
				'electronics' => 0.08,
				'health' => 0.05,
				'food' => 0.12,
				'general' => 0.10
			),
			'store_type_preset' => 'retail_store' // Default store type
		);
	}

	/**
	 * Get all timebound settings
	 *
	 * @return array Timebound settings
	 * @since 1.0.1
	 */
	public function get_settings() {
		// Read from consolidated wisestock_options
		$main_options = get_option('wisestock_options', array());
		if (isset($main_options['timebound_settings'])) {
			return wp_parse_args($main_options['timebound_settings'], $this->get_default_timebound_settings());
		}
		
		// Return defaults if nothing exists
		return $this->get_default_timebound_settings();
	}

	/**
	 * Update timebound settings
	 *
	 * @param array $new_settings New settings to save
	 * @return bool Success status
	 * @since 1.0.1
	 */
	public function update_settings( $new_settings ) {
		$current_settings = $this->get_settings();
		$updated_settings = wp_parse_args( $new_settings, $current_settings );
		
		// Validate settings before saving
		$validated_settings = $this->validate_settings( $updated_settings );
		
		// Save to consolidated wisestock_options
		$main_options = get_option('wisestock_options', array());
		$main_options['timebound_settings'] = $validated_settings;
		$result = update_option('wisestock_options', $main_options);
		
		// Clear any related caches
		$this->clear_settings_cache();
		
		return $result;
	}

	/**
	 * Check if timebound features are enabled
	 *
	 * @return bool True if enabled
	 * @since 1.0.1
	 */
	public function is_enabled() {
		$settings = $this->get_settings();
		return ! empty( $settings['enabled'] );
	}

	/**
	 * Enable timebound features
	 *
	 * @since 1.0.1
	 */
	public function enable() {
		$this->update_settings( array( 'enabled' => true ) );
	}

	/**
	 * Disable timebound features
	 *
	 * @since 1.0.1
	 */
	public function disable() {
		$this->update_settings( array( 'enabled' => false ) );
	}

    /**
     * Get store type presets for timebound predictions
     *
     * @return array Store type presets
     * @since 1.0.1
     */
    public function get_store_type_presets() {
        return array(
            'retail_store' => array(
                'name' => __( 'General Retail Store', 'wisestock' ),
                'description' => __( 'Balanced approach suitable for most store types with moderate prediction windows', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.12,
                    'min_confidence_threshold' => 0.65,
                    'seasonal_adjustment' => false,
                    'trend_detection_sensitivity' => 0.6
                )
            ),
            'beauty_cosmetics' => array(
                'name' => __( 'Beauty & Cosmetics Store', 'wisestock' ),
                'description' => __( 'Optimized for trend-driven beauty products with high trend sensitivity', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 14,
                    'confidence_decay_rate' => 0.18,
                    'min_confidence_threshold' => 0.55,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.8
                )
            ),
            'consumer_electronics' => array(
                'name' => __( 'Consumer Electronics Store', 'wisestock' ),
                'description' => __( 'Balanced for electronics with moderate trend sensitivity and seasonal patterns', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.12,
                    'min_confidence_threshold' => 0.65,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.65
                )
            ),
            'fast_fashion' => array(
                'name' => __( 'Fast Fashion Store', 'wisestock' ),
                'description' => __( 'Extremely trend-driven with rapid market changes and short prediction windows', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 7,
                    'confidence_decay_rate' => 0.25,
                    'min_confidence_threshold' => 0.45,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.9
                )
            ),
            'fashion_apparel' => array(
                'name' => __( 'Fashion & Apparel Store', 'wisestock' ),
                'description' => __( 'Fashion-focused with high trend sensitivity and seasonal considerations', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 10,
                    'confidence_decay_rate' => 0.2,
                    'min_confidence_threshold' => 0.5,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.75
                )
            ),
            'gourmet_food' => array(
                'name' => __( 'Gourmet Food Store', 'wisestock' ),
                'description' => __( 'Premium food products with quality focus and seasonal patterns', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 14,
                    'confidence_decay_rate' => 0.1,
                    'min_confidence_threshold' => 0.7,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.55
                )
            ),
            'health_wellness' => array(
                'name' => __( 'Health & Wellness Store', 'wisestock' ),
                'description' => __( 'Health products requiring higher confidence with moderate seasonal patterns', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.08,
                    'min_confidence_threshold' => 0.75,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.5
                )
            ),
            'home_decor' => array(
                'name' => __( 'Home Decor & Furniture Store', 'wisestock' ),
                'description' => __( 'Home products with strong seasonal patterns and longer decision cycles', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 35,
                    'confidence_decay_rate' => 0.08,
                    'min_confidence_threshold' => 0.7,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.45
                )
            ),
            'kitchen_cooking' => array(
                'name' => __( 'Kitchen & Cooking Store', 'wisestock' ),
                'description' => __( 'Kitchen products with moderate seasonality and stable demand patterns', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.1,
                    'min_confidence_threshold' => 0.65,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.55
                )
            ),
            'luxury_fashion' => array(
                'name' => __( 'Luxury Fashion Store', 'wisestock' ),
                'description' => __( 'Premium fashion requiring high confidence with seasonal and trend considerations', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 28,
                    'confidence_decay_rate' => 0.06,
                    'min_confidence_threshold' => 0.8,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.65
                )
            ),
            'outdoor_gear' => array(
                'name' => __( 'Outdoor Gear Store', 'wisestock' ),
                'description' => __( 'Outdoor equipment with strong seasonal patterns and stable demand', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 28,
                    'confidence_decay_rate' => 0.08,
                    'min_confidence_threshold' => 0.7,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.5
                )
            ),
            'pet_supplies' => array(
                'name' => __( 'Pet Supplies Store', 'wisestock' ),
                'description' => __( 'Pet products with steady demand and moderate seasonal variations', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.1,
                    'min_confidence_threshold' => 0.65,
                    'seasonal_adjustment' => false,
                    'trend_detection_sensitivity' => 0.55
                )
            ),
            'prescription_pharmacy' => array(
                'name' => __( 'Prescription Pharmacy Store', 'wisestock' ),
                'description' => __( 'Critical medical supplies requiring highest confidence and stability', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 45,
                    'confidence_decay_rate' => 0.05,
                    'min_confidence_threshold' => 0.85,
                    'seasonal_adjustment' => false,
                    'trend_detection_sensitivity' => 0.3
                )
            ),
            'sports_equipment' => array(
                'name' => __( 'Sports Equipment Store', 'wisestock' ),
                'description' => __( 'Sports products with seasonal patterns and moderate trend sensitivity', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.1,
                    'min_confidence_threshold' => 0.65,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.6
                )
            ),
            'tech_gadgets' => array(
                'name' => __( 'Tech Gadgets & Accessories Store', 'wisestock' ),
                'description' => __( 'Fast-moving tech products with high trend sensitivity and rapid changes', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 14,
                    'confidence_decay_rate' => 0.18,
                    'min_confidence_threshold' => 0.55,
                    'seasonal_adjustment' => false,
                    'trend_detection_sensitivity' => 0.8
                )
            ),
            'traditional_medical' => array(
                'name' => __( 'Traditional Medical Supply Store', 'wisestock' ),
                'description' => __( 'Medical supplies requiring maximum reliability and long-term stability', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 42,
                    'confidence_decay_rate' => 0.05,
                    'min_confidence_threshold' => 0.85,
                    'seasonal_adjustment' => false,
                    'trend_detection_sensitivity' => 0.25
                )
            ),
            'veterinary' => array(
                'name' => __( 'Veterinary Medicine Store', 'wisestock' ),
                'description' => __( 'Pet health products requiring high reliability with some seasonal variation', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 35,
                    'confidence_decay_rate' => 0.06,
                    'min_confidence_threshold' => 0.8,
                    'seasonal_adjustment' => false,
                    'trend_detection_sensitivity' => 0.35
                )
            ),
            'wellness_supplements' => array(
                'name' => __( 'Wellness & Supplements Store', 'wisestock' ),
                'description' => __( 'Supplement products with trend awareness and seasonal health patterns', 'wisestock' ),
                'settings' => array(
                    'default_prediction_window' => 21,
                    'confidence_decay_rate' => 0.12,
                    'min_confidence_threshold' => 0.65,
                    'seasonal_adjustment' => true,
                    'trend_detection_sensitivity' => 0.6
                )
            )
        );
    }	/**
	 * Apply store type preset to timebound settings
	 *
	 * @param string $preset_key Store type preset key
	 * @return bool Success status
	 * @since 1.0.1
	 */
	public function apply_store_type_preset( $preset_key ) {
		$presets = $this->get_store_type_presets();
		
		if ( ! isset( $presets[ $preset_key ] ) ) {
			return false;
		}
		
		$preset = $presets[ $preset_key ];
		$current_settings = $this->get_settings();
		
		// Merge preset settings with current settings
		$new_settings = array_merge( $current_settings, $preset['settings'] );
		$new_settings['store_type_preset'] = $preset_key;
		
		return $this->update_settings( $new_settings );
	}

	/**
	 * Get current store type preset
	 *
	 * @return array|false Current preset data or false if none
	 * @since 1.0.1
	 */
	public function get_current_store_type_preset() {
		$settings = $this->get_settings();
		$preset_key = $settings['store_type_preset'] ?? 'retail_store';
		
		$presets = $this->get_store_type_presets();
		return $presets[ $preset_key ] ?? false;
	}

	/**
	 * Get volatility threshold presets
	 *
	 * @return array Volatility threshold presets with business-friendly names
	 * @since 1.0.1
	 */
	public function get_volatility_threshold_presets() {
		return array(
			'conservative' => array(
				'name' => __( 'Conservative', 'wisestock' ),
				'value' => 0.15,
				'description' => __( 'Low volatility tolerance - only flag products with very stable sales patterns as low-volatility. Recommended for stores with steady, predictable demand.', 'wisestock' )
			),
			'balanced' => array(
				'name' => __( 'Balanced', 'wisestock' ),
				'value' => 0.25,
				'description' => __( 'Moderate volatility tolerance - balanced approach suitable for most stores. Flags products with reasonable sales consistency as reliable.', 'wisestock' )
			),
			'aggressive' => array(
				'name' => __( 'Aggressive', 'wisestock' ),
				'value' => 0.40,
				'description' => __( 'High volatility tolerance - accepts more variation in sales patterns. Good for trend-driven stores where some fluctuation is normal.', 'wisestock' )
			)
		);
	}

	/**
	 * Get volatility threshold preset by key
	 *
	 * @param string $preset_key Preset key (conservative, balanced, aggressive)
	 * @return array|false Preset data or false if not found
	 * @since 1.0.1
	 */
	public function get_volatility_threshold_preset( $preset_key ) {
		$presets = $this->get_volatility_threshold_presets();
		return $presets[ $preset_key ] ?? false;
	}

	/**
	 * Get current volatility threshold preset key
	 *
	 * @return string Preset key, defaults to 'balanced' if none match
	 * @since 1.0.1
	 */
	public function get_current_volatility_threshold_preset() {
		$current_threshold = $this->get_setting( 'volatility_threshold', 0.25 );
		$presets = $this->get_volatility_threshold_presets();
		
		foreach ( $presets as $key => $preset ) {
			if ( abs( $current_threshold - $preset['value'] ) < 0.001 ) {
				return $key;
			}
		}
		
		return 'balanced';
	}

	/**
	 * Validate settings before saving
	 *
	 * @param array $settings Settings to validate
	 * @return array Validated settings
	 * @since 1.0.1
	 */
	private function validate_settings( $settings ) {
		// Ensure boolean values are boolean
		$settings['enabled'] = ! empty( $settings['enabled'] );
		$settings['confidence_decay_enabled'] = ! empty( $settings['confidence_decay_enabled'] );
		$settings['auto_window_adjustment'] = ! empty( $settings['auto_window_adjustment'] );

		// Validate numeric ranges
		$settings['default_prediction_window'] = max( 1, min( 365, (int) $settings['default_prediction_window'] ) );
		$settings['max_prediction_window'] = max( $settings['default_prediction_window'], min( 365, (int) $settings['max_prediction_window'] ) );
		$settings['min_confidence_threshold'] = max( 0.0, min( 1.0, (float) $settings['min_confidence_threshold'] ) );
		$settings['volatility_threshold'] = max( 0.0, min( 1.0, (float) $settings['volatility_threshold'] ) );
		
		// Validate decay rates
		if ( isset( $settings['product_type_decay_rates'] ) && is_array( $settings['product_type_decay_rates'] ) ) {
			foreach ( $settings['product_type_decay_rates'] as $type => $rate ) {
				$settings['product_type_decay_rates'][ $type ] = max( 0.0, min( 1.0, (float) $rate ) );
			}
		}
		
		// Validate store type preset
		$valid_presets = array_keys( $this->get_store_type_presets() );
		if ( ! in_array( $settings['store_type_preset'], $valid_presets, true ) ) {
			$settings['store_type_preset'] = 'retail_store';
		}
		
		return $settings;
	}

	/**
	 * Clear settings-related caches
	 *
	 * @since 1.0.1
	 */
	private function clear_settings_cache() {
		// Clear any velocity calculation caches when settings change
		global $wpdb;
		$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_wisestock_velocity_%'" );
		$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_wisestock_velocity_%'" );
		
		// Trigger action for other components to clear their caches
		do_action( 'wisestock_timebound_settings_updated' );
	}

	/**
	 * Get timebound setting value
	 *
	 * @param string $key Setting key
	 * @param mixed $default Default value if not found
	 * @return mixed Setting value
	 * @since 1.0.1
	 */
	public function get_setting( $key, $default = null ) {
		$settings = $this->get_settings();
		return $settings[ $key ] ?? $default;
	}

	/**
	 * Update a single timebound setting
	 *
	 * @param string $key Setting key
	 * @param mixed $value Setting value
	 * @return bool Success status
	 * @since 1.0.1
	 */
	public function update_setting( $key, $value ) {
		$settings = $this->get_settings();
		$settings[ $key ] = $value;
		return $this->update_settings( $settings );
	}

	/**
	 * Reset timebound settings to defaults
	 *
	 * @return bool Success status
	 * @since 1.0.1
	 */
	public function reset_to_defaults() {
		$default_settings = $this->get_default_timebound_settings();
		
		// Update consolidated options
		$main_options = get_option('wisestock_options', array());
		$main_options['timebound_settings'] = $default_settings;
		return update_option('wisestock_options', $main_options);
	}

	/**
	 * Export timebound settings
	 *
	 * @return array Settings array for export
	 * @since 1.0.1
	 */
	public function export_settings() {
		$settings = $this->get_settings();
		$settings['_export_timestamp'] = current_time( 'timestamp' );
		$settings['_export_version'] = WISESTOCK_VERSION;
		return $settings;
	}

	/**
	 * Import timebound settings
	 *
	 * @param array $imported_settings Settings to import
	 * @return bool Success status
	 * @since 1.0.1
	 */
	public function import_settings( $imported_settings ) {
		if ( ! is_array( $imported_settings ) ) {
			return false;
		}
		
		// Remove import metadata
		unset( $imported_settings['_export_timestamp'] );
		unset( $imported_settings['_export_version'] );
		
		return $this->update_settings( $imported_settings );
	}
}