<?php
/**
 * Timebound Admin Class
 *
 * This class handles the admin interface for timebound-specific settings and features.
 * It manages the timebound settings section in the main settings page.
 *
 * @package WiseStock
 * @since 1.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Timebound Admin Class
 */
class WISESTOCK_Timebound_Admin {

	/**
	 * Timebound settings instance
	 *
	 * @var WISESTOCK_Timebound_Settings
	 */
	private $timebound_settings;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->timebound_settings = WISESTOCK_Timebound_Settings::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize admin hooks
	 *
	 * @since 1.0.1
	 */
	private function init_hooks() {
		// Add main prediction settings section (timebound is now the only algorithm)
		add_action( 'wisestock_settings_sections', array( $this, 'render_prediction_settings_section' ) );
		
		// Handle prediction settings save
		add_action( 'wisestock_save_settings', array( $this, 'save_prediction_settings' ) );
		
		// Add admin scripts and styles
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_prediction_admin_assets' ) );
	}

	/**
	 * Render timebound settings section
	 *
	 * @param array $options Current plugin options
	 * @since 1.0.1
	 */
	public function render_prediction_settings_section( $options ) {
		$timebound_settings = $this->timebound_settings->get_settings();
		$store_presets = $this->timebound_settings->get_store_type_presets();
		$current_preset = $this->timebound_settings->get_current_store_type_preset();
		?>
		<!-- Prediction Settings -->
		<div class="wisestock-card">
			<div class="wisestock-card-header">
				<h2 class="wisestock-card-title">
					<?php esc_html_e( 'Prediction Settings', 'wisestock' ); ?>
					<span class="wisestock-feature-badge"><?php esc_html_e( 'Intelligence', 'wisestock' ); ?></span>
				</h2>
				<p class="wisestock-card-subtitle">
					<?php esc_html_e( 'Configure intelligent inventory predictions', 'wisestock' ); ?>
				</p>
			</div>

			<div id="prediction-advanced-settings">

				<!-- Store Type Preset -->
				<div class="wisestock-form-group">
					<label class="wisestock-form-label" for="store_type_preset">
						<?php esc_html_e( 'Store Type', 'wisestock' ); ?>
					</label>
					<select id="store_type_preset" name="store_type" class="wisestock-form-select">
						<?php foreach ( $store_presets as $key => $preset ) : ?>
							<option value="<?php echo esc_attr( $key ); ?>"
									data-description="<?php echo esc_attr( $preset['description'] ); ?>"
									data-default-window="<?php echo esc_attr( $preset['settings']['default_prediction_window'] ); ?>"
									data-confidence-threshold="<?php echo esc_attr( $preset['settings']['min_confidence_threshold'] ); ?>"
									data-confidence-decay="<?php echo esc_attr( $preset['settings']['confidence_decay_rate'] ); ?>"

									<?php selected( $options['store_type'] ?? 'retail_store', $key ); ?>>
								<?php echo esc_html( $preset['name'] ); ?>
							</option>
						<?php endforeach; ?>
					</select>
					<div class="wisestock-form-help" id="store-type-preset-description">
						<?php 
						$selected_store_type = $options['store_type'] ?? 'retail_store';
						$selected_preset = $store_presets[$selected_store_type] ?? $store_presets['retail_store'];
						echo esc_html( $selected_preset['description'] );
						?>
					</div>
				</div>

				<!-- Advanced Options -->
				<div class="wisestock-form-group">
					<label class="wisestock-form-label">
						<input type="checkbox"
							   name="timebound_settings[confidence_decay_enabled]"
							   id="confidence_decay_enabled"
							   value="1"
							   <?php checked( $timebound_settings['confidence_decay_enabled'], true ); ?>>
						<?php esc_html_e( 'Enable Confidence Decay', 'wisestock' ); ?>
					</label>
					<div class="wisestock-form-help">
						<?php esc_html_e( 'Apply confidence decay for longer prediction windows based on product type characteristic. Longer predictions are less reliable, so confidence decreases over time.', 'wisestock' ); ?>
					</div>
				</div>

				<div class="wisestock-form-group">
					<label class="wisestock-form-label">
						<input type="checkbox"
							   name="timebound_settings[auto_window_adjustment]"
							   id="auto_window_adjustment"
							   value="1"
							   <?php checked( $timebound_settings['auto_window_adjustment'], true ); ?>>
						<?php esc_html_e( 'Auto Window Adjustment', 'wisestock' ); ?>
					</label>
					<div class="wisestock-form-help">
						<?php echo wp_kses_post( __( 'Automatically adjust prediction windows based on product velocity volatility. <br> Instead of fixed "Default Prediction Window", system picks best window between "Default Prediction" and "Maximum Prediction Window" per product.', 'wisestock' ) );?>
					</div>
				</div>

				<!-- Advanced Numerical Settings -->
				<div class="wisestock-form-group">
					<label class="wisestock-form-label" for="default_prediction_window">
						<?php esc_html_e( 'Default Prediction Window (Days)', 'wisestock' ); ?>
					</label>
					<input type="number"
						   name="timebound_settings[default_prediction_window]"
						   id="default_prediction_window"
						   class="wisestock-form-input"
						   value="<?php echo esc_attr( $timebound_settings['default_prediction_window'] ?? 14 ); ?>"
						   min="7"
						   max="90"
						   step="1">
					<div class="wisestock-form-help">
						<?php esc_html_e( 'Default number of days to predict ahead for inventory planning.', 'wisestock' ); ?>
					</div>
				</div>

				<div class="wisestock-form-group">
					<label class="wisestock-form-label" for="max_prediction_window">
						<?php esc_html_e( 'Maximum Prediction Window (Days)', 'wisestock' ); ?>
					</label>
					<input type="number"
						   name="timebound_settings[max_prediction_window]"
						   id="max_prediction_window"
						   class="wisestock-form-input"
						   value="<?php echo esc_attr( $timebound_settings['max_prediction_window'] ?? 90 ); ?>"
						   min="14"
						   max="365"
						   step="1">
					<div class="wisestock-form-help">
						<?php esc_html_e( 'Maximum number of days the system can predict ahead.', 'wisestock' ); ?>
					</div>
				</div>

				<div class="wisestock-form-group">
					<label class="wisestock-form-label" for="min_confidence_threshold">
						<?php esc_html_e( 'Confidence Threshold', 'wisestock' ); ?>
					</label>
					<?php 
					$confidence_presets = array(
						'40' => array(
							'name' => __( 'Very Aggressive (40%)', 'wisestock' ),
							'description' => __( 'Accept very low confidence predictions. Good for trend-driven stores or when stockouts are extremely costly.', 'wisestock' )
						),
						'50' => array(
							'name' => __( 'Aggressive (50%)', 'wisestock' ),
							'description' => __( 'Accept lower confidence predictions. Good for fast-moving items or when stockouts are costly.', 'wisestock' )
						),
						'60' => array(
							'name' => __( 'Standard (60%)', 'wisestock' ),
							'description' => __( 'Balanced approach. Good for most retail stores with moderate risk tolerance.', 'wisestock' )
						),
						'70' => array(
							'name' => __( 'Reliable (70%)', 'wisestock' ),
							'description' => __( 'Higher value require more reliable data before making predictions. Good for stores with mixed inventory turnover.', 'wisestock' )
						),
						'80' => array(
							'name' => __( 'Conservative (80%)', 'wisestock' ),
							'description' => __( 'Higher value require more reliable data before making predictions. Good for expensive items or when overstocking is costly.', 'wisestock' )
						),
						'90' => array(
							'name' => __( 'Very Conservative (90%)', 'wisestock' ),
							'description' => __( 'This requires very highly reliable data before making predictions. Good for luxury items or critical business decisions.', 'wisestock' )
						)
					);
					$current_threshold = $timebound_settings['min_confidence_threshold'];
					// Convert decimal back to percentage for dropdown display (like volatility threshold)
					$current_threshold = intval($current_threshold * 100);
					// Convert to string for proper comparison
					$current_threshold = strval($current_threshold);
					?>
					<select name="timebound_settings[min_confidence_threshold]"
							id="min_confidence_threshold"
							class="wisestock-form-select">
						<?php foreach ( $confidence_presets as $value => $preset ) : ?>
							<option value="<?php echo esc_attr( $value ); ?>"
									data-description="<?php echo esc_attr( $preset['description'] ); ?>"
									<?php selected( $current_threshold, $value ); ?>>
								<?php echo esc_html( $preset['name'] ); ?>
							</option>
						<?php endforeach; ?>
					</select>
					<div class="wisestock-form-help" id="confidence-threshold-description">
						<?php 
						$selected_preset = $confidence_presets[$current_threshold] ?? $confidence_presets['70'];
						echo esc_html( $selected_preset['description'] );
						?>
					</div>
				</div>

				<!-- Volatility Thresholds -->
				<div class="wisestock-form-group">
					<label class="wisestock-form-label" for="volatility_threshold_preset">
						<?php esc_html_e( 'Volatility Threshold', 'wisestock' ); ?>
					</label>
						<?php 
						$volatility_presets = $this->timebound_settings->get_volatility_threshold_presets();
						$current_preset = $this->timebound_settings->get_current_volatility_threshold_preset();
					?>
					<select id="volatility_threshold_preset" 
							name="timebound_settings[volatility_threshold_preset]" 
							class="wisestock-form-input">
					<?php foreach ( $volatility_presets as $key => $preset ) : ?>
						<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $current_preset, $key ); ?>>
							<?php echo esc_html( $preset['name'] ); ?>
						</option>
					<?php endforeach; ?>
					</select>
					
					<!-- Hidden field to store the actual threshold value -->
					<input type="hidden" 
						   id="volatility_threshold" 
						   name="timebound_settings[volatility_threshold]" 
						   value="<?php echo esc_attr( $timebound_settings['volatility_threshold'] ); ?>" />
					
					<div id="volatility_description" class="wisestock-form-help">
						<?php 
						$preset_data = $volatility_presets[ $current_preset ] ?? $volatility_presets['balanced'];
						echo esc_html( $preset_data['description'] );
						?>
					</div>
				</div>

			</div>
		</div>
		<?php
	}

	/**
	 * Save timebound settings
	 *
	 * @since 1.0.1
	 */
	public function save_prediction_settings() {
		// Verify nonce
		if ( ! wp_verify_nonce( $_POST['wisestock_settings_nonce'] ?? '', 'wisestock_save_settings' ) ) {
			return;
		}

		// Check permissions
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		// Handle store type setting (main WiseStock option)
		if ( isset( $_POST['store_type'] ) ) {
			$valid_presets = array_keys( $this->timebound_settings->get_store_type_presets() );
			$store_type = in_array( $_POST['store_type'], $valid_presets, true ) ? 
				$_POST['store_type'] : 'retail_store';
			
			// Update main WiseStock options
			$options = get_option( 'wisestock_options', array() );
			$options['store_type'] = $store_type;
			update_option( 'wisestock_options', $options );
			
			// Apply the store type preset to timebound settings (but allow individual overrides below)
			$this->timebound_settings->apply_store_type_preset( $store_type );
		}

		// Handle timebound-specific advanced settings if present
		if ( isset( $_POST['timebound_settings'] ) ) {
			$timebound_data = $_POST['timebound_settings'];
			
			// Sanitize and validate data
			$sanitized_settings = array();
			
			// Boolean settings (timebound is always enabled, so these are just feature toggles)
			$sanitized_settings['confidence_decay_enabled'] = ! empty( $timebound_data['confidence_decay_enabled'] );
			$sanitized_settings['auto_window_adjustment'] = ! empty( $timebound_data['auto_window_adjustment'] );
			
			// Numerical settings
			if ( isset( $timebound_data['default_prediction_window'] ) ) {
				$default_window = absint( $timebound_data['default_prediction_window'] );
				$sanitized_settings['default_prediction_window'] = max( 7, min( 90, $default_window ) );
			}
			
			if ( isset( $timebound_data['max_prediction_window'] ) ) {
				$max_window = absint( $timebound_data['max_prediction_window'] );
				$sanitized_settings['max_prediction_window'] = max( 14, min( 365, $max_window ) );
			}
			
			if ( isset( $timebound_data['min_confidence_threshold'] ) ) {
				$min_confidence = absint( $timebound_data['min_confidence_threshold'] );
				// Validate against allowed preset values
				$allowed_values = array( 40, 50, 60, 70, 80, 90 );
				if ( in_array( $min_confidence, $allowed_values, true ) ) {
					// Convert percentage to decimal (like volatility threshold)
					$sanitized_settings['min_confidence_threshold'] = $min_confidence / 100;
				} else {
					$sanitized_settings['min_confidence_threshold'] = 0.7; // Default fallback as decimal
				}
			}
			
			// Handle volatility threshold (preset only)
			if ( isset( $timebound_data['volatility_threshold_preset'] ) ) {
				$preset = $this->timebound_settings->get_volatility_threshold_preset( $timebound_data['volatility_threshold_preset'] );
				if ( $preset ) {
					$sanitized_settings['volatility_threshold'] = $preset['value'];
				} else {
					$sanitized_settings['volatility_threshold'] = 0.25; // Fallback to balanced
				}
			}

			// Update timebound settings
			$this->timebound_settings->update_settings( $sanitized_settings );
		}
		
		// Clear caches when settings change
		wp_cache_flush();
		
		// Clear timebound-related transients
		global $wpdb;
		$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_wisestock_timebound_%'" );
		$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_wisestock_timebound_%'" );
	}

	/**
	 * Enqueue timebound admin assets
	 *
	 * @param string $hook Current admin page hook
	 * @since 1.0.1
	 */
	public function enqueue_prediction_admin_assets( $hook ) {

		// Only load on wisestock settings page
		if ( false === strpos( $hook, 'wisestock-settings' ) ) {
			return;
		}

		// Enqueue timebound admin script
		wp_enqueue_script(
			'wisestock-timebound-admin',
			WISESTOCK_PLUGIN_URL . 'admin/js/wisestock-timebound-admin.js',
			array( 'jquery', 'wisestock-settings' ),
			WISESTOCK_VERSION,
			true
		);

		// Localize script with timebound data
		wp_localize_script( 'wisestock-timebound-admin', 'wisestock_timebound_admin', array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'wisestock_timebound_nonce' ),
			'strings' => array(
				'confirm_reset' => __( 'Are you sure you want to reset timebound settings to defaults?', 'wisestock' ),
				'settings_saved' => __( 'Timebound settings saved successfully', 'wisestock' ),
				'settings_error' => __( 'Error saving timebound settings', 'wisestock' ),
			)
		) );

		// Enqueue timebound admin styles
		wp_enqueue_style(
			'wisestock-timebound-admin',
			WISESTOCK_PLUGIN_URL . 'admin/css/wisestock-timebound-admin.css',
			array( 'wisestock-admin' ),
			WISESTOCK_VERSION
		);
	}

	/**
	 * Get timebound settings for AJAX requests
	 *
	 * @since 1.0.1
	 */
	public function ajax_get_timebound_settings() {
		// Verify nonce
		if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'wisestock_timebound_nonce' ) ) {
			wp_send_json_error( 'Invalid nonce' );
		}

		// Check permissions
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Insufficient permissions' );
		}

		$settings = $this->timebound_settings->get_settings();
		wp_send_json_success( $settings );
	}

	/**
	 * Reset timebound settings to defaults
	 *
	 * @since 1.0.1
	 */
	public function ajax_reset_timebound_settings() {
		// Verify nonce
		if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'wisestock_timebound_nonce' ) ) {
			wp_send_json_error( 'Invalid nonce' );
		}

		// Check permissions
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Insufficient permissions' );
		}

		$result = $this->timebound_settings->reset_to_defaults();
		
		if ( $result ) {
			wp_send_json_success( 'Timebound settings reset to defaults' );
		} else {
			wp_send_json_error( 'Failed to reset timebound settings' );
		}
	}
}