<?php
/**
 * Supplier Manager Class
 *
 * This class handles supplier management, contact information, and automated
 * supplier notifications for restock recommendations. It manages supplier-product
 * relationships and provides email templates for supplier communications.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Supplier Manager Class
 */
class WISESTOCK_Supplier_Manager {

    /**
     * Constructor
     */
    public function __construct() {
    }

    /**
     * Add a new supplier
     *
     * @param array $supplier_data Supplier data
     * @return int|false Supplier ID on success, false on failure
     * @since 1.0.0
     */
    public function add_supplier($supplier_data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_suppliers';

        // Sanitize input data
        $sanitized_data = array(
            'name' => sanitize_text_field($supplier_data['name']),
            'email' => sanitize_email($supplier_data['email']),
            'product_ids' => json_encode($supplier_data['product_ids']),
            'lead_time_days' => (int) $supplier_data['lead_time_days']
        );

        $result = $wpdb->insert(
            $table_name,
            $sanitized_data,
            array('%s', '%s', '%s', '%d')
        );

        if ($result) {
            // Clear supplier caches
            $this->clear_supplier_caches();
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Update supplier information
     *
     * @param int $supplier_id Supplier ID
     * @param array $supplier_data Supplier data
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function update_supplier($supplier_id, $supplier_data) {
        global $wpdb;
        $table_name = esc_sql($wpdb->prefix . 'wisestock_suppliers');

        // Sanitize input data
        $sanitized_data = array();
        if (isset($supplier_data['name'])) {
            $sanitized_data['name'] = sanitize_text_field($supplier_data['name']);
        }
        if (isset($supplier_data['email'])) {
            $sanitized_data['email'] = sanitize_email($supplier_data['email']);
        }
        if (isset($supplier_data['product_ids'])) {
            $sanitized_data['product_ids'] = json_encode($supplier_data['product_ids']);
        }
        if (isset($supplier_data['lead_time_days'])) {
            $sanitized_data['lead_time_days'] = (int) $supplier_data['lead_time_days'];
        }

        if (empty($sanitized_data)) {
            return false;
        }

        global $wpdb;
        $result = $wpdb->update(
            $table_name,
            $sanitized_data,
            array('id' => $supplier_id),
            array_fill(0, count($sanitized_data), '%s'),
            array('%d')
        );

        if ($result !== false) {
            // Clear supplier caches
            $this->clear_supplier_caches();
        }
        
        return $result !== false;
    }

    /**
     * Delete a supplier
     *
     * @param int $supplier_id Supplier ID
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function delete_supplier($supplier_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_suppliers';
        $result = $wpdb->delete(
            $table_name,
            array('id' => $supplier_id),
            array('%d')
        );

        if ($result !== false) {
            // Clear supplier caches
            $this->clear_supplier_caches();
        }
        
        return $result !== false;
    }

    /**
     * Get supplier by ID
     *
     * @param int $supplier_id Supplier ID
     * @return object|false Supplier object or false if not found
     * @since 1.0.0
     */
    public function get_supplier($supplier_id) {
        global $wpdb;
        $table_name = esc_sql($wpdb->prefix . 'wisestock_suppliers');

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $supplier = $wpdb->get_row($wpdb->prepare('SELECT id, name, email, product_ids, lead_time_days, created_at FROM `' . $table_name . '` WHERE id = %d', $supplier_id));

        if ($supplier) {
            $supplier->product_ids = json_decode($supplier->product_ids, true);
        }

        return $supplier;
    }

    /**
     * Get all suppliers
     *
     * @return array Array of supplier objects
     * @since 1.0.0
     */
    public function get_all_suppliers() {
        // Check cache first for performance
        $cache_key = 'wisestock_all_suppliers';
        $cached_result = get_transient($cache_key);
        if ($cached_result !== false) {
            return $cached_result;
        }
        global $wpdb;
        $table_name = esc_sql($wpdb->prefix . 'wisestock_suppliers');

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $suppliers = $wpdb->get_results( $wpdb->prepare('SELECT id, name, email, product_ids, lead_time_days, created_at FROM `' . $table_name . '` WHERE %d = %d ORDER BY name ASC', 1, 1));

        foreach ($suppliers as $supplier) {
            $supplier->product_ids = json_decode($supplier->product_ids, true);
        }

        // Cache result for 30 minutes to improve performance
        set_transient($cache_key, $suppliers, 30 * MINUTE_IN_SECONDS);

        return $suppliers;
    }

    /**
     * Get supplier for a specific product
     *
     * @param int $product_id Product ID
     * @return object|false Supplier object or false if not found
     * @since 1.0.0
     */
    public function get_product_supplier($product_id) {
        global $wpdb;
        $table_name = esc_sql($wpdb->prefix . 'wisestock_suppliers');

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $suppliers = $wpdb->get_results($wpdb->prepare('SELECT id, name, email, product_ids, lead_time_days, created_at FROM `' . $table_name . '` WHERE %d = %d', 1, 1));

        foreach ($suppliers as $supplier) {
            $product_ids = json_decode($supplier->product_ids, true);
            if (is_array($product_ids) && in_array($product_id, $product_ids)) {
                $supplier->product_ids = $product_ids;
                return $supplier;
            }
        }

        return false;
    }

    /**
     * Send supplier notification for restock recommendation
     *
     * @param int $product_id Product ID
     * @param array $recommendation Restock recommendation data
     * @param int|null $supplier_id Specific supplier ID (optional)
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function send_supplier_notification($product_id, $recommendation, $supplier_id = null) {
        // If specific supplier ID provided, get that supplier
        if ($supplier_id) {
            $supplier = $this->get_supplier($supplier_id);
            // Verify this supplier is assigned to the product
            if (!$supplier) {
                return false;
            }
            if (!in_array($product_id, $supplier->product_ids)) {
                return false;
            }
        } else {
            // Fallback to old behavior for backward compatibility
            $supplier = $this->get_product_supplier($product_id);
        }
        
        if (!$supplier || !$supplier->email) {
            return false;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            return false;
        }

        $email_data = $this->prepare_supplier_email_data($product, $supplier, $recommendation);

        // Get plugin options with defaults
        $defaults = WISESTOCK_Core::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        
        // Use custom subject if provided, otherwise use default
        if (!empty($options['supplier_email_subject'])) {
            $subject = $options['supplier_email_subject'];
            $subject = str_replace('{site_name}', $email_data['site_name'], $subject);
            $subject = str_replace('{urgency}', strtoupper($email_data['urgency']), $subject);
            $subject = str_replace('{product_name}', $email_data['product_name'], $subject);
        } else {
            $subject = $this->get_supplier_email_subject($product, $recommendation);
        }

        // Always use the template method
        $body = $this->get_supplier_email_template($email_data);

        // Set headers with custom from email if provided
        $headers = array('Content-Type: text/html; charset=UTF-8');
        if (!empty($options['supplier_from_email'])) {
            $headers[] = 'From: ' . $options['supplier_from_email'];
        }

        $result = wp_mail($supplier->email, $subject, $body, $headers);

        if ($result) {
            // Trigger supplier notified action
            do_action('wisestock_supplier_notified', $product_id, $supplier->id, $email_data);
        }

        return $result;
    }

    /**
     * Prepare supplier email data
     *
     * @param WC_Product $product Product object
     * @param object $supplier Supplier object
     * @param array $recommendation Restock recommendation data
     * @return array Email data
     * @since 1.0.0
     */
    private function prepare_supplier_email_data($product, $supplier, $recommendation) {
        return array(
            'supplier_name' => $supplier->name,
            'supplier_email' => $supplier->email,
            'product_name' => $product->get_name(),
            'product_sku' => $product->get_sku(),
            'current_stock' => $product->get_stock_quantity(),
            'recommended_quantity' => $recommendation['recommended_quantity'],
            'urgency' => $recommendation['urgency'],
            'days_to_stockout' => $recommendation['days_to_stockout'],
            'daily_velocity' => $recommendation['daily_velocity'],
            'lead_time' => $supplier->lead_time_days,
            'reasoning' => $recommendation['reasoning'],
            'site_name' => get_bloginfo('name'),
            'admin_email' => get_option('admin_email')
        );
    }

    /**
     * Get supplier email subject
     *
     * @param WC_Product $product Product object
     * @param array $recommendation Restock recommendation data
     * @return string Email subject
     * @since 1.0.0
     */
    private function get_supplier_email_subject($product, $recommendation) {
        $site_name = get_bloginfo('name');
        $urgency = strtoupper($recommendation['urgency']);

        /* translators: %s: Product name. */
        $product_subject = sprintf(__('Restock Request for %s', 'wisestock'), $product->get_name());

        /* translators: 1: Site name. 2: Urgency. 3: Email subject. */
        return sprintf('[%1$s] %2$s - %3$s', $site_name, $urgency, $product_subject);
    }

    /**
     * Get supplier email template
     *
     * @param array $data Email data
     * @return string Email HTML content
     * @since 1.0.0
     */
    public function get_supplier_email_template($data) {
        $urgency_colors = array(
            'critical' => '#dc3545',
            'warning' => '#ffc107',
            'mild' => '#17a2b8',
            'none' => '#6c757d'
        );

        $color = $urgency_colors[$data['urgency']] ?? '#6c757d';

        // Allow filtering of template variables
        $template_vars = apply_filters('wisestock_supplier_email_template_vars', array(
            'urgency_colors' => $urgency_colors,
            'color' => $color,
            'data' => $data
        ), $data);

        // Extract filtered variables
        extract($template_vars);

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title><?php esc_html_e('Restock Request', 'wisestock'); ?></title>
        </head>
        <body class="wisestock-email-body" style="--email-color: <?php echo esc_attr($color); ?>;">
            <div class="wisestock-email-container">
                <h2 class="wisestock-email-header">
                    <?php
                    /* translators: %s: Inventory urgency level. */
                    printf(esc_html__('Restock Request - %s Priority', 'wisestock'), esc_html(ucfirst($data['urgency'])));
                    ?>
                </h2>

                <p>
                    <?php
                    /* translators: %s: Supplier name. */
                    printf(esc_html__('Dear %s,', 'wisestock'), esc_html($data['supplier_name']));
                    ?>
                </p>

                <p><?php esc_html_e('We need to restock the following product based on our inventory analysis:', 'wisestock'); ?></p>

                <div class="wisestock-email-alert-box">
                    <h3><?php esc_html_e('Product Information', 'wisestock'); ?></h3>
                    <?php
                    $product_name = isset($data['product_name']) ? $data['product_name'] : '';
                    $sku = !empty($data['product_sku']) ? $data['product_sku'] : esc_html__('N/A', 'wisestock');
                    $current_stock = isset($data['current_stock']) ? $data['current_stock'] : 0;
                    $recommended_quantity = isset($data['recommended_quantity']) ? $data['recommended_quantity'] : 0;
                    ?>
                    <p><strong><?php esc_html_e('Product Name:', 'wisestock'); ?></strong> <?php echo esc_html($product_name); ?></p>
                    <p><strong><?php esc_html_e('SKU:', 'wisestock'); ?></strong> <?php echo esc_html($sku); ?></p>
                    <p><strong><?php esc_html_e('Current Stock:', 'wisestock'); ?></strong> <?php echo esc_html($current_stock); ?> <?php esc_html_e('units', 'wisestock'); ?></p>
                    <p><strong><?php esc_html_e('Recommended:', 'wisestock'); ?></strong> <strong style="color: var(--email-color);"><?php echo esc_html($recommended_quantity); ?> <?php esc_html_e('units', 'wisestock'); ?></strong></p>
                </div>

                <div class="wisestock-email-analysis-details">
                    <h3><?php esc_html_e('Analysis Details', 'wisestock'); ?></h3>
                    <?php
                    $days_to_stockout = isset($data['days_to_stockout']) ? $data['days_to_stockout'] : 0;
                    $daily_velocity = isset($data['daily_velocity']) ? $data['daily_velocity'] : 0;
                    $lead_time = isset($data['lead_time']) ? $data['lead_time'] : 0;
                    $reasoning = isset($data['reasoning']) ? $data['reasoning'] : '';
                    ?>
                    <p><strong><?php esc_html_e('Days to Stockout:', 'wisestock'); ?></strong> <?php echo esc_html($days_to_stockout); ?> <?php esc_html_e('days', 'wisestock'); ?></p>
                    <p><strong><?php esc_html_e('Daily Sales Velocity:', 'wisestock'); ?></strong> <?php echo esc_html($daily_velocity); ?> <?php esc_html_e('units/day', 'wisestock'); ?></p>
                    <p><strong><?php esc_html_e('Lead Time:', 'wisestock'); ?></strong> <?php echo esc_html($lead_time); ?> <?php esc_html_e('days', 'wisestock'); ?></p>
                    <p><strong><?php esc_html_e('Reasoning:', 'wisestock'); ?></strong> <?php echo esc_html($reasoning); ?></p>
                </div>

                <div class="wisestock-email-next-steps">
                    <h3><?php esc_html_e('Next Steps', 'wisestock'); ?></h3>
                    <p><?php esc_html_e('Please confirm if you can fulfill this order and provide an estimated delivery date. If you have any questions or need additional information, please contact us at:', 'wisestock'); ?></p>
                    <p>
                        <strong><?php esc_html_e('Email:', 'wisestock'); ?></strong> <?php echo esc_html($data['admin_email']); ?>
                    </p>
                </div>

                <hr class="wisestock-email-footer">
                <p class="wisestock-email-footer-text">
                    <?php
                    /* translators: %s: Site name. */
                    printf(esc_html__('This request was automatically generated by Wise Stock inventory management system for %s.', 'wisestock'), esc_html($data['site_name']));
                    ?>
                </p>
            </div>
        </body>
        </html>
        <?php
        $template_html = ob_get_clean();
        
        // Allow complete template replacement
        return apply_filters('wisestock_supplier_email_template_html', $template_html, $template_vars, $data);
    }

    /**
     * Clear supplier caches
     *
     * @since 1.0.0
     */
    private function clear_supplier_caches() {
        // Clear all supplier-related transients
        delete_transient('wisestock_all_suppliers');
    }

    /**
     * Get suppliers for a specific product
     *
     * @param int $product_id Product ID
     * @return array Array of supplier objects
     * @since 1.0.0
     */
    public function get_product_suppliers($product_id) {
        global $wpdb;
        $table_name = esc_sql($wpdb->prefix . 'wisestock_suppliers');

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $suppliers = $wpdb->get_results($wpdb->prepare('SELECT id, name, email, product_ids, lead_time_days, created_at FROM `' . $table_name . '` WHERE %d = %d', 1, 1));
        
        $product_suppliers = array();
        foreach ($suppliers as $supplier) {
            $product_ids = json_decode($supplier->product_ids, true);
            if (is_array($product_ids) && in_array($product_id, $product_ids)) {
                $supplier->product_ids = $product_ids;
                
                // Check if this is the default supplier for this product
                $supplier->is_default = ($this->get_default_supplier_for_product($product_id) == $supplier->id);
                
                $product_suppliers[] = $supplier;
            }
        }
        
        return $product_suppliers;
    }

    /**
     * Get default supplier for a product
     *
     * @param int $product_id Product ID
     * @return int|false Default supplier ID or false
     * @since 1.0.0
     */
    public function get_default_supplier_for_product($product_id) {
        return get_post_meta($product_id, '_wisestock_default_supplier', true);
    }

    /**
     * Set default supplier for a product
     *
     * @param int $product_id Product ID
     * @param int $supplier_id Supplier ID
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function set_default_supplier_for_product($product_id, $supplier_id) {
        // Verify the supplier exists and is assigned to this product
        $supplier = $this->get_supplier($supplier_id);
        if (!$supplier) {
            return false;
        }
        
        if (!in_array($product_id, $supplier->product_ids)) {
            return false;
        }
        
        return update_post_meta($product_id, '_wisestock_default_supplier', $supplier_id);
    }

    /**
     * Get supplier statistics
     *
     * @return array Statistics array
     * @since 1.0.0
     */
    public function get_supplier_stats() {
        $suppliers = $this->get_all_suppliers();
        
        $total_suppliers = count($suppliers);
        $total_products_assigned = 0;
        
        foreach ($suppliers as $supplier) {
            if (is_array($supplier->product_ids)) {
                $total_products_assigned += count($supplier->product_ids);
            }
        }
        
        return array(
            'total_suppliers' => $total_suppliers,
            'total_products_assigned' => $total_products_assigned
        );
    }

    /**
     * Notify multiple suppliers for a product
     *
     * @param int $product_id Product ID
     * @param array $supplier_ids Array of supplier IDs
     * @return array Results array with success/failure for each supplier
     * @since 1.0.0
     */
    public function notify_suppliers($product_id, $supplier_ids) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return array('success' => false, 'message' => __('Product not found', 'wisestock'));
        }
        
        // Get velocity data for the product
        $velocity_calculator = new WISESTOCK_Velocity_Calculator();
        $velocity_data = $velocity_calculator->get_product_velocity($product_id);
        
        // Prepare recommendation data
        $recommendation = array(
            'recommended_quantity' => max(10, ceil($velocity_data['daily_velocity'] * 30)), // 30 days supply
            'urgency' => 'mild',
            'days_to_stockout' => ceil($product->get_stock_quantity() / max(1, $velocity_data['daily_velocity'])),
            'daily_velocity' => $velocity_data['daily_velocity'],
            'reasoning' => __('Restock request based on sales velocity analysis', 'wisestock')
        );
        
        $results = array();
        $success_count = 0;
        
        foreach ($supplier_ids as $supplier_id) {
            $result = $this->send_supplier_notification($product_id, $recommendation, $supplier_id);
            $results[$supplier_id] = $result;
            if ($result) {
                $success_count++;
            }
        }
        
        $total_suppliers = count($supplier_ids);

        if ($success_count === $total_suppliers) {
            /* translators: %1$d: Number of suppliers notified. */
            $success_message = sprintf(_n('%1$d supplier notified successfully', '%1$d suppliers notified successfully', $success_count, 'wisestock'), $success_count);

            return array(
                'success' => true,
                'message' => $success_message
            );
        } elseif ($success_count > 0) {
            /* translators: 1: Successful supplier count. 2: Total supplier count. */
            $partial_message = sprintf(__('%1$d of %2$d suppliers notified successfully', 'wisestock'), $success_count, $total_suppliers);

            return array(
                'success' => true,
                'message' => $partial_message
            );
        } else {
            return array(
                'success' => false,
                'message' => __('Failed to notify suppliers', 'wisestock')
            );
        }
    }
}


