<?php
/**
 * Plugin Updater Class
 *
 * Handles version-specific updates when the plugin is updated.
 *
 * @package WiseStock
 * @since 1.1.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin Updater Class
 */
class WISESTOCK_Plugin_Updater {

	/**
	 * Option name for storing installed plugin version
	 *
	 * @var string
	 */
	const VERSION_OPTION = 'wisestock_plugin_version';

	/**
	 * Updater instance
	 *
	 * @var WISESTOCK_Plugin_Updater
	 */
	private static $instance = null;

	/**
	 * Get updater instance
	 *
	 * @return WISESTOCK_Plugin_Updater
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'plugins_loaded', array( $this, 'check_for_updates' ), 5 );
	}

	/**
	 * Check for plugin updates on every page load
	 *
	 * @since 1.1.0
	 */
	public function check_for_updates() {
		$installed_version = get_option( self::VERSION_OPTION, false );
		$current_version = WISESTOCK_VERSION;

		// First install: just record the version
		if ( false === $installed_version ) {
			update_option( self::VERSION_OPTION, $current_version );
			return;
		}

		// Future versions: check for updates and run migration methods
		if ( version_compare( $installed_version, $current_version, '<' ) ) {
			$this->run_updates( $installed_version, $current_version );
		}
	}

	/**
	 * Run version-specific updates
	 *
	 * @param string $from_version Previous version
	 * @param string $to_version Current version  
	 * @since 1.1.0
	 */
	private function run_updates( $from_version, $to_version ) {
		// Example for v1.2.0:
		// if ( version_compare( $from_version, '1.2.0', '<' ) && 
		//      version_compare( $to_version, '1.2.0', '>=' ) ) {
		//     $this->update_1_2_0();
		// }
		
		// Update stored version
		update_option( self::VERSION_OPTION, $to_version );
		
		// Trigger action for other components
		do_action( 'wisestock_plugin_updated', $from_version, $to_version );
	}

	/**
	 * Get current installed version
	 *
	 * @return string Installed version
	 * @since 1.1.0
	 */
	public function get_installed_version() {
		return get_option( self::VERSION_OPTION, WISESTOCK_VERSION );
	}

	/**
	 * Get current plugin version
	 *
	 * @return string Current version
	 * @since 1.1.0
	 */
	public function get_current_version() {
		return WISESTOCK_VERSION;
	}
}