<?php
/**
 * Notification Manager Class
 *
 * This class handles the creation, management, and sending of inventory notifications.
 * It manages three types of notifications: Critical (1-3 days), Warning (4-7 days), and Mild (8-14 days).
 * Supports dashboard notifications, email notifications, and future mobile push notifications.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Notification Manager Class
 */
class WISESTOCK_Notification_Manager {

    /**
     * Velocity calculator instance
     *
     * @var WISESTOCK_Velocity_Calculator
     */
    private $velocity_calculator;

    /**
     * Notification thresholds
     *
     * @var array
     */
    private $notification_thresholds;

    /**
     * Constructor
     */
    public function __construct() {
        $this->velocity_calculator = new WISESTOCK_Velocity_Calculator();

        // Get notification thresholds from settings
        $options = get_option('wisestock_options', array());
        $defaults = WISESTOCK_Core::get_default_options();
        $default_thresholds = $defaults['notification_thresholds'];
        
        $this->notification_thresholds = isset($options['notification_thresholds']) ? $options['notification_thresholds'] : $default_thresholds;

        // Allow thresholds to be filtered
        $this->notification_thresholds = apply_filters('wisestock_notification_thresholds', $this->notification_thresholds);
    }

    /**
     * Check all products for notifications
     *
     * @since 1.0.0
     */
    public function check_all_notifications() {
        $products = wc_get_products(array(
            'limit' => -1,
            'status' => 'publish',
            'manage_stock' => true
        ));

        foreach ($products as $product) {
            // Double-check stock management is enabled and product is not digital
            if (!$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
                continue;
            }
            $this->check_product_notifications($product->get_id());
        }
    }

    /**
     * Check notifications for a specific product
     *
     * @param int $product_id Product ID
     * @since 1.0.0
     */
    public function check_product_notifications($product_id) {
        $product = wc_get_product($product_id);
        if (!$product || !$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
            return;
        }

        $current_stock = $product->get_stock_quantity();
        if (0 >= $current_stock) {
			$this->create_notification($product_id, 'critical', __('Product is out of stock', 'wisestock'));
			return;
		}

        $days_to_stockout = $this->velocity_calculator->predict_stockout($product_id);
        if (0 >= $days_to_stockout) {
			return; // No stockout prediction or increasing stock
		}

        $notification_type = $this->determine_notification_type($days_to_stockout);
        if ($notification_type) {
            $message = $this->generate_notification_message($product, $days_to_stockout, $notification_type);
            $this->create_notification($product_id, $notification_type, $message);
        }
    }

    /**
     * Create a notification
     *
     * @param int $product_id Product ID
     * @param string $notification_type Notification type (critical, warning, mild)
     * @param string $message Notification message
     * @return int|false Notification ID on success, false on failure
     * @since 1.0.0
     */
    public function create_notification($product_id, $notification_type, $message) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_notifications';

        // Check if similar notification already exists
        $existing_notification = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM `{$table_name}` WHERE product_id = %d AND notification_type = %s AND status = %s",
                $product_id,
                $notification_type,
                'active'
            )
        );

        if ($existing_notification) {
            // Update existing notification
            $result = $wpdb->update(
                $table_name,
                array(
                    'message' => $message,
                    'created_at' => current_time('mysql')
                ),
                array('id' => $existing_notification),
                array('%s', '%s'),
                array('%d')
            );

            return $existing_notification;
        } else {
            // Create new notification
            $result = $wpdb->insert(
                $table_name,
                array(
                    'product_id' => $product_id,
                    'notification_type' => $notification_type,
                    'message' => $message,
                    'status' => 'active'
                ),
                array('%d', '%s', '%s', '%s')
            );

            if ($result) {
                $notification_id = $wpdb->insert_id;

                // Clear notification caches
                $this->clear_notification_caches();

                // Send notifications
                $this->send_notification_emails($product_id, $notification_type, $message);

                // Trigger notification action
                do_action('wisestock_notification_triggered', $product_id, $notification_type, $message);

                return $notification_id;
            }
        }

        return false;
    }

    /**
     * Get active notifications for a product
     *
     * @param int $product_id Product ID
     * @return array Array of notification objects
     * @since 1.0.0
     */
    public function get_product_notifications($product_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_notifications';

        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, product_id, notification_type, message, status, created_at FROM `{$table_name}`
                 WHERE product_id = %d AND status = %s
                 ORDER BY created_at DESC",
                $product_id,
                'active'
            )
        );
    }

    /**
     * Get all active notifications
     *
     * @param string $notification_type Optional notification type filter
     * @return array Array of notification objects
     * @since 1.0.0
     */
    public function get_all_notifications($notification_type = null) {
        // Check cache first for performance
        $cache_key = 'wisestock_notifications_' . ($notification_type ? $notification_type : 'all');
        $cached_result = get_transient($cache_key);
        
        if ($cached_result !== false) {
            return $cached_result;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_notifications';

        if ($notification_type) {
            $sql = $wpdb->prepare(
                "SELECT id, product_id, notification_type, message, status, created_at FROM `{$table_name}`
                 WHERE status = %s AND notification_type = %s
                 ORDER BY created_at DESC",
                'active',
                $notification_type
            );
        } else {
            $sql = $wpdb->prepare(
                "SELECT id, product_id, notification_type, message, status, created_at FROM `{$table_name}`
                 WHERE status = %s
                 ORDER BY created_at DESC",
                'active'
            );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- The variable sql is prepared already.
        $result = $wpdb->get_results($sql);
        
        // Cache result for 15 minutes to improve performance
        set_transient($cache_key, $result, 15 * MINUTE_IN_SECONDS);
        
        return $result;
    }

    /**
     * Resolve a notification
     *
     * @param int $notification_id Notification ID
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function resolve_notification($notification_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_notifications';

        $result = $wpdb->update(
            $table_name,
            array('status' => 'resolved'),
            array('id' => $notification_id),
            array('%s'),
            array('%d')
        );

        if ($result !== false) {
            // Clear notification caches
            $this->clear_notification_caches();
        }

        return $result !== false;
    }

    /**
     * Dismiss a notification
     *
     * @param int $notification_id Notification ID
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function dismiss_notification($notification_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_notifications';

        $result = $wpdb->update(
            $table_name,
            array('status' => 'dismissed'),
            array('id' => $notification_id),
            array('%s'),
            array('%d')
        );

        if ($result !== false) {
            // Clear notification caches
            $this->clear_notification_caches();
        }

        return $result !== false;
    }

    /**
     * Determine notification type based on days to stockout
     *
     * @param int $days_to_stockout Days until stockout
     * @return string|false Notification type or false if no notification needed
     * @since 1.0.0
     */
    private function determine_notification_type($days_to_stockout) {
        if ($this->notification_thresholds['critical'] >= $days_to_stockout) {
			return 'critical';
		} elseif ($this->notification_thresholds['warning'] >= $days_to_stockout) {
			return 'warning';
		} elseif ($this->notification_thresholds['mild'] >= $days_to_stockout) {
			return 'mild';
		}

		return false;
	}

    /**
     * Generate notification message
     *
     * @param WC_Product $product Product object
     * @param int $days_to_stockout Days until stockout
     * @param string $notification_type Notification type
     * @return string Notification message
     * @since 1.0.0
     */
    private function generate_notification_message($product, $days_to_stockout, $notification_type) {
        $product_name = sanitize_text_field($product->get_name());
        $current_stock = (int) $product->get_stock_quantity();
        $sku = sanitize_text_field($product->get_sku());

        $urgency_text = '';
        switch ($notification_type) {
			case 'critical':
				$urgency_text = __('URGENT', 'wisestock');
				break;
			case 'warning':
				$urgency_text = __('WARNING', 'wisestock');
				break;
			case 'mild':
				$urgency_text = __('MILD', 'wisestock');
				break;
		}

        $product_identifier = $sku
            ? sprintf(
                /* translators: %s: Product SKU. */
                __('SKU: %s', 'wisestock'),
                $sku
            )
            : sprintf(
                /* translators: %d: Product ID. */
                __('ID: %d', 'wisestock'),
                $product->get_id()
            );

        /* translators: 1: urgency label, 2: product name, 3: product identifier, 4: days to stockout, 5: current stock quantity. */
        $message = sprintf('[%1$s] %2$s - %3$s will stockout in %4$d days. Current stock: %5$d units.', $urgency_text, $product_name, $product_identifier, $days_to_stockout, $current_stock);

        return sanitize_text_field($message);
    }

    /**
     * Send notification emails
     *
     * @param int $product_id Product ID
     * @param string $notification_type Notification type
     * @param string $message Notification message
     * @since 1.0.0
     */
    private function send_notification_emails($product_id, $notification_type, $message) {
        $defaults = WISESTOCK_Core::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);

        // Check if email notifications are enabled
        if (!$options['email_notifications']) {
            return;
        }

        // Check if this notification type should trigger emails
        $enabled_types = $options['email_notification_types'] ?? array('critical', 'warning', 'mild');
        if (!in_array($notification_type, $enabled_types)) {
            return;
        }

        // Send email notification
        $this->send_email_notification($product_id, $notification_type, $message);

        // Future: Add mobile push notifications here
        // $this->send_push_notification($product_id, $notification_type, $message);
    }

    /**
     * Send email notification
     *
     * @param int $product_id Product ID
     * @param string $notification_type Notification type
     * @param string $message Notification message
     * @since 1.0.0
     */
    private function send_email_notification($product_id, $notification_type, $message) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return;
        }

        // Get custom admin email or fallback to site admin email
        $defaults = WISESTOCK_Core::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        $admin_email = !empty($options['admin_email_address']) ? $options['admin_email_address'] : get_option('admin_email');
        $site_name = get_bloginfo('name');

        /* translators: 1: Site name. 2: Notification type label. */
        $subject = sprintf('[%1$s] Wise Stock Notification - %2$s', $site_name, ucfirst($notification_type));

        // Allow filtering of email subject
        $subject = apply_filters('wisestock_notification_email_subject', $subject, $product, $notification_type, $message);

        $email_body = $this->get_email_template($product, $notification_type, $message);

        // Allow filtering of email body
        $email_body = apply_filters('wisestock_notification_email_body', $email_body, $product, $notification_type, $message);

        $headers = array('Content-Type: text/html; charset=UTF-8');

        // Allow filtering of email headers
        $headers = apply_filters('wisestock_notification_email_headers', $headers, $product, $notification_type, $message);

        wp_mail($admin_email, $subject, $email_body, $headers);
    }

    /**
     * Get email template for notifications
     *
     * @param WC_Product $product Product object
     * @param string $notification_type Notification type
     * @param string $message Notification message
     * @return string Email HTML content
     * @since 1.0.0
     */
    private function get_email_template($product, $notification_type, $message) {
        // Sanitize input data
        $product_name = sanitize_text_field($product->get_name());
        $product_url = admin_url('post.php?post=' . intval($product->get_id()) . '&action=edit');
        $dashboard_url = admin_url('admin.php?page=wisestock-dashboard');
        $notification_type = sanitize_text_field($notification_type);
        $message = sanitize_text_field($message);

        $color = '';
        switch ($notification_type) {
            case 'critical':
                $color = '#dc3545';
                break;
            case 'warning':
                $color = '#ffc107';
                break;
            case 'mild':
                $color = '#17a2b8';
                break;
        }

        // Allow filtering of template variables
        $template_vars = apply_filters('wisestock_email_template_vars', array(
            'product_name' => $product_name,
            'product_url' => $product_url,
            'dashboard_url' => $dashboard_url,
            'color' => $color,
            'notification_type' => $notification_type,
            'message' => $message,
            'product' => $product
        ), $product, $notification_type, $message);

        // Extract filtered variables
        extract($template_vars);

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title><?php echo esc_html__('WiseStock Alert', 'wisestock'); ?></title>
        </head>
        <body class="wisestock-email-body" style="--email-color: <?php echo esc_attr($color); ?>;">
            <div class="wisestock-email-container">
                <h2 class="wisestock-email-header">
                    <?php
                    /* translators: %s: Notification type label. */
                    $notification_heading = sprintf(__('WiseStock Notification - %s', 'wisestock'), ucfirst($notification_type));
                    echo esc_html($notification_heading);
                    ?>
                </h2>

                <div class="wisestock-email-alert-box">
                    <p class="wisestock-email-alert-message"><strong><?php echo esc_html($message); ?></strong></p>
                </div>

                <div class="wisestock-email-product-details">
                    <h3><?php echo esc_html__('Product Details', 'wisestock'); ?></h3>
                    <p><strong><?php echo esc_html__('Product:', 'wisestock'); ?></strong> <?php echo esc_html($product_name); ?></p>
                    <p><strong><?php echo esc_html__('SKU:', 'wisestock'); ?></strong> <?php echo esc_html($product->get_sku() ?: __('N/A', 'wisestock')); ?></p>
                    <p><strong><?php echo esc_html__('Current Stock:', 'wisestock'); ?></strong> <?php echo esc_html($product->get_stock_quantity()); ?> <?php echo esc_html__('units', 'wisestock'); ?></p>
                    <p><strong><?php echo esc_html__('Manage Stock:', 'wisestock'); ?></strong> <?php echo $product->managing_stock() ? esc_html__('Yes', 'wisestock') : esc_html__('No', 'wisestock'); ?></p>
                </div>

                <div class="wisestock-email-actions">
                    <?php if (!empty($product_url)): ?>
                    <a href="<?php echo esc_url($product_url); ?>" class="wisestock-email-btn wisestock-email-btn-primary">
                        <?php echo esc_html__('Edit Product', 'wisestock'); ?>
                    </a>
                    <?php endif; ?>
                    <?php if (!empty($dashboard_url)): ?>
                    <a href="<?php echo esc_url($dashboard_url); ?>" class="wisestock-email-btn wisestock-email-btn-success">
                        <?php echo esc_html__('View Dashboard', 'wisestock'); ?>
                    </a>
                    <?php endif; ?>
                </div>

                <hr class="wisestock-email-footer">
                <p class="wisestock-email-footer-text">
                    <?php echo esc_html__('This notification was generated by Wise Stock. You can manage your notification preferences in the plugin settings.', 'wisestock'); ?>
                </p>
            </div>
        </body>
        </html>
        <?php
        $template_html = ob_get_clean();
        
        // Allow complete template replacement
        return apply_filters('wisestock_email_template_html', $template_html, $template_vars, $product, $notification_type, $message);
    }

    /**
     * Get email template for test notifications
     *
     * @param object $mock_product Mock product object
     * @param string $notification_type Notification type
     * @param string $message Notification message
     * @return string Email HTML content
     * @since 1.0.0
     */
    public function get_test_email_template($mock_product, $notification_type, $message) {
        return $this->get_email_template($mock_product, $notification_type, $message);
    }

    /**
     * Get notification statistics
     *
     * @return array Notification statistics
     * @since 1.0.0
     */
    public function get_notification_statistics() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wisestock_notifications';
        $sanitized_table = esc_sql($table_name);

        $stats = array();

        // Count by type
        $type_counts = $wpdb->get_results(
            "SELECT notification_type, COUNT(*) as count
             FROM `{$sanitized_table}`
             WHERE status = 'active'
             GROUP BY notification_type"
        );

        foreach ($type_counts as $count) {
            $stats[$count->notification_type] = (int) $count->count;
        }

        // Total active notifications
        $stats['total'] = array_sum($stats);

        return $stats;
    }

    /**
     * Clear notification caches
     *
     * @since 1.0.0
     */
    private function clear_notification_caches() {
        // Clear all notification-related transients
        delete_transient('wisestock_notifications_all');
        delete_transient('wisestock_notifications_critical');
        delete_transient('wisestock_notifications_warning');
        delete_transient('wisestock_notifications_mild');
    }
}
