<?php
/**
 * Demo Data Manager Class
 *
 * Handles versioned demo data management with migration support for algorithm changes.
 * This replaces the hardcoded demo data approach with a flexible, JSON-based system
 * that can easily adapt to future algorithm enhancements without breaking changes.
 *
 * @package WiseStock
 * @since 1.1.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Demo Data Manager Class
 */
class WISESTOCK_Demo_Data_Manager {

	/**
	 * Current demo data version
	 *
	 * @var string
	 */
	const DEMO_DATA_VERSION = '1.1.0';

	/**
	 * Demo data manager instance
	 *
	 * @var WISESTOCK_Demo_Data_Manager
	 */
	private static $instance = null;

	/**
	 * Get manager instance
	 *
	 * @return WISESTOCK_Demo_Data_Manager
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Private constructor for singleton
	}

	/**
	 * Initialize demo data system with version tracking
	 *
	 * @return bool Success status
	 * @since 1.1.0
	 */
	public function initialize_demo_data() {
		$current_version = get_option( 'wisestock_demo_data_version', '0.0.0' );
		
		// Check if migration is needed
		if ( version_compare( $current_version, self::DEMO_DATA_VERSION, '<' ) ) {
			$success = $this->migrate_demo_data( $current_version, self::DEMO_DATA_VERSION );
			if ( ! $success ) {
				return false;
			}
		}
		
		// Ensure demo data exists and is current
		return $this->ensure_demo_data_exists();
	}

	/**
	 * Create or update demo products table with new schema
	 *
	 * @return bool Success status
	 * @since 1.1.0
	 */
	public function create_demo_products_table() {
		global $wpdb;

		// Skip capability check during plugin updates and installation
		// The plugin updater and activation process handle permission validation  
		$skip_permission_check = (
			defined( 'WP_INSTALLING' ) ||
			did_action( 'upgrader_process_complete' ) ||
			did_action( 'activate_plugin' ) ||
			current_filter() === 'plugins_loaded'
		);
		
		if ( ! $skip_permission_check ) {
			if ( ! current_user_can( 'manage_options' ) && ! current_user_can( 'activate_plugins' ) ) {
				return false;
			}
		}

		$table_name = $wpdb->prefix . 'wisestock_demo_products';
		$charset_collate = $wpdb->get_charset_collate();

		// Use proper dbDelta() with WordPress standards - exact formatting required
		$sql = "CREATE TABLE $table_name (
		id mediumint(9) NOT NULL AUTO_INCREMENT,
		product_name varchar(255) NOT NULL,
		sku varchar(100) DEFAULT '',
		current_stock int(11) DEFAULT 0,
		daily_velocity decimal(10,2) DEFAULT 0.00,
		category varchar(100) DEFAULT '',
		price decimal(10,2) DEFAULT 0.00,
		supplier varchar(100) DEFAULT '',
		location varchar(100) DEFAULT 'Main Warehouse',
		algorithm_data longtext DEFAULT NULL,
		data_version varchar(10) DEFAULT '1.1.0',
		last_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
		PRIMARY KEY  (id),
		UNIQUE KEY idx_sku_unique (sku),
		KEY idx_product_name (product_name),
		KEY idx_category (category),
		KEY idx_data_version (data_version),
		KEY idx_daily_velocity (daily_velocity)
		) $charset_collate;";

		try {
			if ( ! function_exists( 'dbDelta' ) ) {
				require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
			}

			// Use WordPress standard dbDelta() function
			$result = dbDelta( $sql );
			
			if ( empty( $result ) ) {
				return false;
			}

			return true;

		} catch ( Exception $e ) {
			return false;
		}
	}

	/**
	 * Handle demo data migrations between versions
	 *
	 * @param string $from_version Current version
	 * @param string $to_version Target version
	 * @return bool Success status
	 * @since 1.1.0
	 */
	private function migrate_demo_data( $from_version, $to_version ) {
		try {			
			// Ensure table exists with latest schema
			if ( ! $this->create_demo_products_table() ) {
				throw new Exception( 'Failed to create/update demo products table' );
			}

			// Version-specific migrations
			if ( version_compare( $from_version, '1.0.0', '<' ) ) {
				$this->migrate_to_v1_0_0();
			}

			if ( version_compare( $from_version, '1.1.0', '<' ) ) {
				$this->migrate_to_v1_1_0();
			}

			// Update version after successful migration
			update_option( 'wisestock_demo_data_version', $to_version );
			
			return true;

		} catch ( Exception $e ) {
			return false;
		}
	}

	/**
	 * Migration to version 1.0.0 - Initial demo data setup
	 *
	 * @throws Exception If migration fails
	 * @since 1.1.0
	 */
	private function migrate_to_v1_0_0() {
		// This handles the initial setup from no demo data to v1.0.0
		$this->populate_demo_products_table();
	}

	/**
	 * Migration to version 1.1.0 - Drop and recreate demo products table
	 *
	 * @throws Exception If migration fails
	 * @since 1.1.0
	 */
	private function migrate_to_v1_1_0() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'wisestock_demo_products';
		$sanitized_table = esc_sql( $table_name );

		// Drop existing demo products table completely (if it exists)
		// This avoids any schema conflicts with the old structure from 1.0.1
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
		
		if ( $table_exists ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
			$result = $wpdb->query( "DROP TABLE IF EXISTS `$sanitized_table`" );
			if ( false === $result ) {
				throw new Exception( 'Failed to drop existing demo products table' );
			}
		}

		// Create fresh table with new 1.1.0 structure
		if ( ! $this->create_demo_products_table() ) {
			throw new Exception( 'Failed to create new demo products table structure' );
		}

		// Verify table was created with correct schema
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
		if ( ! $table_exists ) {
			throw new Exception( 'Demo products table was not created successfully' );
		}

		// Check if required columns exist
		$columns = $wpdb->get_results( "DESCRIBE `$sanitized_table`" );
		$column_names = wp_list_pluck( $columns, 'Field' );
		$has_algorithm_data = in_array( 'algorithm_data', $column_names, true );
		$has_data_version = in_array( 'data_version', $column_names, true );
		
		if ( ! $has_algorithm_data || ! $has_data_version ) {
			throw new Exception( 'Demo products table missing required columns: algorithm_data=' . ( $has_algorithm_data ? 'yes' : 'no' ) . ', data_version=' . ( $has_data_version ? 'yes' : 'no' ) );
		}

		// Populate with new versioned demo data
		$this->populate_demo_products_table();
	}

	/**
	 * Populate demo products table with versioned, JSON-based data
	 *
	 * @since 1.1.0
	 */
	private function populate_demo_products_table() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'wisestock_demo_products';
		$sanitized_table = esc_sql( $table_name );

		// Verify table exists and has the right structure before populating
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
		if ( ! $table_exists ) {
			return;
		}

		// Check table structure
		$columns = $wpdb->get_results( "DESCRIBE `$sanitized_table`" );
		$column_names = wp_list_pluck( $columns, 'Field' );
		
		$required_columns = array( 'algorithm_data', 'data_version' );
		$missing_columns = array_diff( $required_columns, $column_names );
		
		if ( ! empty( $missing_columns ) ) {
			return;
		}

		$demo_products = $this->get_versioned_demo_products();
		$success_count = 0;
		$error_count = 0;

		foreach ( $demo_products as $product ) {
			$result = $wpdb->insert( $table_name, $product );
			if ( false === $result ) {
				$error_count++;
			} else {
				$success_count++;
			}
		}
	}

	/**
	 * Get versioned demo products with JSON algorithm data
	 *
	 * @return array Demo products with flexible algorithm data
	 * @since 1.1.0
	 */
	private function get_versioned_demo_products() {
		return array(
			// High velocity products - Critical/Warning scenarios
			array(
				'product_name' => 'Energy Drink Pack (12)',
				'sku' => 'ED-001',
				'current_stock' => 156,
				'daily_velocity' => 8.5,
				'category' => 'Food & Beverage',
				'price' => 24.99,
				'supplier' => 'Beverage Direct',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'increasing',
						'volatility' => 0.18,
						'market_sensitivity' => 0.72,
						'confidence_decay_rate' => 0.12,
						'seasonal_factor' => 1.2,
						'prediction_window' => 14
					),
					'metadata' => array(
						'scenario_type' => 'high_velocity_growth',
						'demo_category' => 'fast_moving',
						'urgency_level' => 'info'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'Hand Sanitizer 8oz',
				'sku' => 'HS-001',
				'current_stock' => 234,
				'daily_velocity' => 5.8,
				'category' => 'Health & Safety',
				'price' => 8.99,
				'supplier' => 'Safety First',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'stable',
						'volatility' => 0.08,
						'market_sensitivity' => 0.45,
						'confidence_decay_rate' => 0.08,
						'seasonal_factor' => 0.9,
						'prediction_window' => 21
					),
					'metadata' => array(
						'scenario_type' => 'stable_high_volume',
						'demo_category' => 'consistent',
						'urgency_level' => 'none'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'Wireless Bluetooth Headphones',
				'sku' => 'WBH-001',
				'current_stock' => 6,
				'daily_velocity' => 3.2,
				'category' => 'Electronics',
				'price' => 89.99,
				'supplier' => 'TechSupply Co',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'decreasing',
						'volatility' => 0.25,
						'market_sensitivity' => 0.68,
						'confidence_decay_rate' => 0.15,
						'seasonal_factor' => 1.1,
						'prediction_window' => 7
					),
					'metadata' => array(
						'scenario_type' => 'critical_stockout',
						'demo_category' => 'urgent',
						'urgency_level' => 'critical'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'USB-C Cable 6ft',
				'sku' => 'USB-001',
				'current_stock' => 7,
				'daily_velocity' => 2.5,
				'category' => 'Electronics',
				'price' => 12.99,
				'supplier' => 'TechSupply Co',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'stable',
						'volatility' => 0.12,
						'market_sensitivity' => 0.52,
						'confidence_decay_rate' => 0.10,
						'seasonal_factor' => 1.0,
						'prediction_window' => 14
					),
					'metadata' => array(
						'scenario_type' => 'warning_level',
						'demo_category' => 'moderate',
						'urgency_level' => 'warning'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'Essential Oil Lavender',
				'sku' => 'EO-001',
				'current_stock' => 3,
				'daily_velocity' => 1.2,
				'category' => 'Wellness',
				'price' => 18.99,
				'supplier' => 'Aromatherapy Plus',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'increasing',
						'volatility' => 0.22,
						'market_sensitivity' => 0.58,
						'confidence_decay_rate' => 0.14,
						'seasonal_factor' => 1.3,
						'prediction_window' => 10
					),
					'metadata' => array(
						'scenario_type' => 'critical_low_stock',
						'demo_category' => 'urgent',
						'urgency_level' => 'critical'
					)
				) ),
				'data_version' => '1.1.0'
			),

			// Medium velocity products - Info/Warning scenarios
			array(
				'product_name' => 'Yoga Mat Premium',
				'sku' => 'YM-001',
				'current_stock' => 25,
				'daily_velocity' => 2.8,
				'category' => 'Fitness',
				'price' => 45.00,
				'supplier' => 'Fitness Plus',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'stable',
						'volatility' => 0.09,
						'market_sensitivity' => 0.41,
						'confidence_decay_rate' => 0.09,
						'seasonal_factor' => 1.1,
						'prediction_window' => 21
					),
					'metadata' => array(
						'scenario_type' => 'moderate_stock',
						'demo_category' => 'stable',
						'urgency_level' => 'info'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'LED Desk Lamp',
				'sku' => 'LED-001',
				'current_stock' => 9,
				'daily_velocity' => 1.8,
				'category' => 'Home & Office',
				'price' => 29.99,
				'supplier' => 'Lighting Solutions',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'decreasing',
						'volatility' => 0.16,
						'market_sensitivity' => 0.48,
						'confidence_decay_rate' => 0.11,
						'seasonal_factor' => 0.95,
						'prediction_window' => 14
					),
					'metadata' => array(
						'scenario_type' => 'declining_demand',
						'demo_category' => 'moderate',
						'urgency_level' => 'warning'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'Backpack Travel',
				'sku' => 'BT-001',
				'current_stock' => 8,
				'daily_velocity' => 1.4,
				'category' => 'Travel',
				'price' => 55.99,
				'supplier' => 'Travel Gear Co',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'increasing',
						'volatility' => 0.19,
						'market_sensitivity' => 0.62,
						'confidence_decay_rate' => 0.13,
						'seasonal_factor' => 1.4,
						'prediction_window' => 10
					),
					'metadata' => array(
						'scenario_type' => 'seasonal_increase',
						'demo_category' => 'growing',
						'urgency_level' => 'warning'
					)
				) ),
				'data_version' => '1.1.0'
			),

			// Low velocity products - Stable scenarios
			array(
				'product_name' => 'Bluetooth Speaker Portable',
				'sku' => 'BSP-001',
				'current_stock' => 34,
				'daily_velocity' => 0.9,
				'category' => 'Electronics',
				'price' => 59.99,
				'supplier' => 'TechSupply Co',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'stable',
						'volatility' => 0.05,
						'market_sensitivity' => 0.32,
						'confidence_decay_rate' => 0.06,
						'seasonal_factor' => 1.0,
						'prediction_window' => 45
					),
					'metadata' => array(
						'scenario_type' => 'healthy_stock',
						'demo_category' => 'stable',
						'urgency_level' => 'none'
					)
				) ),
				'data_version' => '1.1.0'
			),
			array(
				'product_name' => 'Notebook Set Premium',
				'sku' => 'NS-001',
				'current_stock' => 67,
				'daily_velocity' => 0.6,
				'category' => 'Office Supplies',
				'price' => 12.99,
				'supplier' => 'Paper Works',
				'location' => 'Main Warehouse',
				'algorithm_data' => wp_json_encode( array(
					'timebound' => array(
						'trend_direction' => 'stable',
						'volatility' => 0.04,
						'market_sensitivity' => 0.28,
						'confidence_decay_rate' => 0.05,
						'seasonal_factor' => 0.9,
						'prediction_window' => 60
					),
					'metadata' => array(
						'scenario_type' => 'overstocked',
						'demo_category' => 'slow_moving',
						'urgency_level' => 'none'
					)
				) ),
				'data_version' => '1.1.0'
			)
		);
	}

	/**
	 * Ensure demo data exists and is up to date
	 *
	 * @return bool Success status
	 * @since 1.1.0
	 */
	private function ensure_demo_data_exists() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'wisestock_demo_products';
		$sanitized_table = esc_sql( $table_name );

		// Check if table exists
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
		if ( ! $table_exists ) {
			return false;
		}

		// Check if data_version column exists (for backward compatibility)
		$columns = $wpdb->get_results( "DESCRIBE `$sanitized_table`" );
		$column_names = wp_list_pluck( $columns, 'Field' );
		$has_data_version = in_array( 'data_version', $column_names, true );

		if ( $has_data_version ) {
			// Check if table has current version data
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
			$current_version_count = $wpdb->get_var( $wpdb->prepare(
				"SELECT COUNT(*) FROM `$sanitized_table` WHERE data_version = %s",
				self::DEMO_DATA_VERSION
			) );
		} else {
			// Old table structure - check total count
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
			$current_version_count = $wpdb->get_var( "SELECT COUNT(*) FROM `$sanitized_table`" );
		}

		if ( $current_version_count < 5 ) { // Minimum expected products
			// Repopulate with current version data
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
			$wpdb->query( $wpdb->prepare( "DELETE FROM `$sanitized_table` WHERE %d = %d", 1, 1 ) );
			$this->populate_demo_products_table();
		}

		return true;
	}

	/**
	 * Get demo products with algorithm data
	 *
	 * @return array Demo products with parsed JSON data
	 * @since 1.1.0
	 */
	public function get_demo_products() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'wisestock_demo_products';
		$sanitized_table = esc_sql( $table_name );

		// Check if table exists first
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
		if ( ! $table_exists ) {
			return array();
		}

		// Check if data_version column exists (for backward compatibility)
		$columns = $wpdb->get_results( "DESCRIBE `$sanitized_table`" );
		$column_names = wp_list_pluck( $columns, 'Field' );
		$has_data_version = in_array( 'data_version', $column_names, true );

		if ( $has_data_version ) {
			// Use data_version filter for new table structure
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
			$products = $wpdb->get_results( $wpdb->prepare(
				"SELECT * FROM `$sanitized_table` WHERE data_version = %s ORDER BY daily_velocity DESC",
				self::DEMO_DATA_VERSION
			) );
		} else {
			// Old table structure - get all products
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
			$products = $wpdb->get_results( "SELECT * FROM `$sanitized_table` ORDER BY daily_velocity DESC" );
		}

		// Parse JSON algorithm data if it exists
		foreach ( $products as $product ) {
			if ( ! empty( $product->algorithm_data ) ) {
				$product->algorithm_data = json_decode( $product->algorithm_data, true );
			}
		}

		return $products;
	}

	/**
	 * Get current demo data version
	 *
	 * @return string Current version
	 * @since 1.1.0
	 */
	public function get_demo_data_version() {
		return get_option( 'wisestock_demo_data_version', '0.0.0' );
	}

	/**
	 * Check if demo data needs update
	 *
	 * @return bool True if update needed
	 * @since 1.1.0
	 */
	public function needs_update() {
		return version_compare( $this->get_demo_data_version(), self::DEMO_DATA_VERSION, '<' );
	}
}