<?php
/**
 * Dashboard Widget Class
 *
 * This class handles the Wise Stock dashboard widget that displays on the
 * WordPress admin dashboard. It shows critical notifications, velocity indicators,
 * and restock recommendations in a compact, informative format.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Dashboard Widget Class
 */
class WISESTOCK_Dashboard_Widget {

    /**
     * Notification manager instance
     *
     * @var WISESTOCK_Notification_Manager
     */
    private $notification_manager;

    /**
     * Velocity calculator instance
     *
     * @var WISESTOCK_Velocity_Calculator
     */
    private $velocity_calculator;

    /**
     * Constructor
     */
    public function __construct() {
        $this->notification_manager = new WISESTOCK_Notification_Manager();
        $this->velocity_calculator = new WISESTOCK_Velocity_Calculator();
    }

    /**
     * Add dashboard widget
     *
     * @since 1.0.0
     */
    public function add_widget() {
        wp_add_dashboard_widget(
            'wisestock_dashboard_widget',
            __('WiseStock', 'wisestock'),
            array($this, 'render_widget')
        );
        
        // Enqueue dashboard widget CSS only on dashboard screen
        add_action('admin_enqueue_scripts', array($this, 'enqueue_dashboard_styles'));
    }
    
    /**
     * Enqueue dashboard widget styles
     *
     * @param string $hook Current admin page hook
     * @since 1.0.0
     */
    public function enqueue_dashboard_styles($hook) {
        // Only load on dashboard screen
        if ($hook !== 'index.php') {
            return;
        }
        
        wp_enqueue_style(
            'wisestock-dashboard-widget',
            WISESTOCK_PLUGIN_URL . 'admin/css/wisestock-dashboard-widget.css',
            array(),
            WISESTOCK_VERSION
        );
    }

    /**
     * Render dashboard widget content
     *
     * @since 1.0.0
     */
    public function render_widget() {
        // Check user capabilities
        if (!current_user_can('manage_woocommerce')) {
            echo '<p>' . esc_html__('You do not have permission to view inventory data.', 'wisestock') . '</p>';
            return;
        }

        // Get widget data
        $widget_data = $this->get_widget_data();

        // Render widget HTML
        $this->render_widget_html($widget_data);
    }

    /**
     * Get widget data
     *
     * @return array Widget data
     * @since 1.0.0
     */
    private function get_widget_data() {
        // Hook to enhance widget data
        $widget_data = apply_filters('wisestock_dashboard_widget_data', null, 'main');
        if ($widget_data !== null) {
            return $widget_data;
        }
        
        $defaults = WISESTOCK_Core::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        $demo_stats_enabled = $options['demo_stats'];

        if ($demo_stats_enabled) {
            // Use demo data from historical orders
            $api = WISESTOCK_API::get_instance();
            $dashboard_data = $api->get_demo_dashboard_data();

            return array(
                'notifications' => $dashboard_data['notifications'],
                'notification_stats' => $dashboard_data['notification_stats'],
                'attention_products' => $dashboard_data['attention_products'],
                'velocity_summary' => $dashboard_data['velocity_summary'],
                'timebound_info' => $this->get_timebound_info(),
                'demo_mode' => true
            );
        }

        return array(
            'notifications' => $this->notification_manager->get_all_notifications(),
            'notification_stats' => $this->notification_manager->get_notification_statistics(),
            'attention_products' => $this->get_attention_products(),
            'velocity_summary' => $this->get_velocity_summary(),
            'timebound_info' => $this->get_timebound_info(),
            'demo_mode' => false
        );
    }

    /**
     * Get top attention products
     *
     * @param int $limit Number of products to return
     * @return array Attention products
     * @since 1.0.0
     */
    private function get_attention_products($limit = 10) {
        $critical_notifications = $this->notification_manager->get_all_notifications('critical');
        $warning_notifications = $this->notification_manager->get_all_notifications('warning');

        $attention_products = array();

        // Process critical notifications first
        foreach ($critical_notifications as $notification) {
            $product = wc_get_product($notification->product_id);
            if ($product) {
                $attention_products[] = array(
                    'product' => $product,
                    'notification' => $notification,
                    'priority' => 'critical'
                );
            }
        }

        // Add warning notifications if we have space
        $remaining_slots = $limit - count($attention_products);
        if ($remaining_slots > 0) {
            foreach (array_slice($warning_notifications, 0, $remaining_slots) as $notification) {
                $product = wc_get_product($notification->product_id);
                if ($product) {
                    $attention_products[] = array(
                        'product' => $product,
                        'notification' => $notification,
                        'priority' => 'warning'
                    );
                }
            }
        }

        return array_slice($attention_products, 0, $limit);
    }

    /**
     * Get velocity summary
     *
     * @return array Velocity summary data
     * @since 1.0.0
     */
    private function get_velocity_summary() {
        $products = wc_get_products(array(
            'limit' => 5,
            'status' => 'publish',
            'manage_stock' => true,
            'orderby' => 'meta_value_num',
            'meta_key' => '_stock'
        ));

        $velocity_data = array();
        foreach ($products as $product) {
            // Double-check stock management is enabled and product is not digital
            if (!$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
                continue;
            }

            $velocity = $this->velocity_calculator->calculate_weighted_velocity($product->get_id());
            if ($velocity['weighted_velocity'] > 0) {
                $velocity_data[] = array(
                    'product' => $product,
                    'velocity' => $velocity['weighted_velocity'],
                    'confidence' => $velocity['period_velocities']['7_day']['confidence'] ?? 0
                );
            }
        }

        return $velocity_data;
    }

    /**
     * Get timebound information for the dashboard
     *
     * @return array Timebound information
     * @since 1.0.1
     */
    private function get_timebound_info() {
        $timebound_settings_instance = WISESTOCK_Timebound_Settings::get_instance();
        $settings = $timebound_settings_instance->get_settings();

        return array(
            'store_type' => $settings['store_type_preset'] ?? 'retail_store',
            'prediction_window' => $settings['default_prediction_window'],
            'confidence_threshold' => $settings['min_confidence_threshold'],
            'confidence_decay_enabled' => $settings['confidence_decay_enabled'],
            'auto_window_adjustment' => $settings['auto_window_adjustment'],
            'volatility_threshold' => $settings['volatility_threshold']
        );
    }

    /**
     * Render widget HTML
     *
     * @param array $data Widget data
     * @since 1.0.0
     */
    private function render_widget_html($data) {
        ?>
        <div class="wisestock-widget">
            <?php if (isset($data['demo_mode']) && $data['demo_mode']): ?>
                <div class="wisestock-warning-notice">
                    <strong><?php esc_html_e('Demo Mode:', 'wisestock'); ?></strong> <?php esc_html_e('Showing historical data for demonstration purposes', 'wisestock'); ?>
                </div>
            <?php endif; ?>
            
            <div class="wisestock-widget-content">
                <?php $this->render_notification_summary($data['notification_stats']); ?>
                <?php $this->render_attention_products($data['attention_products']); ?>
                <?php $this->render_velocity_summary($data['velocity_summary']); ?>
            </div>
            
            <?php $this->render_quick_actions(); ?>
        </div>

        <?php
    }

    /**
     * Render notification summary
     *
     * @param array $notification_stats Notification statistics
     * @since 1.0.0
     */
    private function render_notification_summary($notification_stats) {
        $critical_count = $notification_stats['critical'] ?? 0;
        $warning_count = $notification_stats['warning'] ?? 0;
        $mild_count = $notification_stats['mild'] ?? 0;
        $total_count = $notification_stats['total'] ?? 0;
        ?>
        <div class="wisestock-section">
            <h4><?php esc_html_e('Inventory Alerts', 'wisestock'); ?></h4>
            <div class="wisestock-notification-summary">
                <div class="wisestock-notification-card critical">
                    <div class="wisestock-notification-count"><?php echo esc_html($critical_count); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Critical', 'wisestock'); ?></div>
                </div>
                <div class="wisestock-notification-card warning">
                    <div class="wisestock-notification-count"><?php echo esc_html($warning_count); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Warning', 'wisestock'); ?></div>
                </div>
                <div class="wisestock-notification-card mild">
                    <div class="wisestock-notification-count"><?php echo esc_html($mild_count); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Mild', 'wisestock'); ?></div>
                </div>
                <div class="wisestock-notification-card">
                    <div class="wisestock-notification-count"><?php echo esc_html($total_count); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Total', 'wisestock'); ?></div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render attention products
     *
     * @param array $attention_products Attention products
     * @since 1.0.0
     */
    private function render_attention_products($attention_products) {
        if (empty($attention_products)) {
            ?>
            <div class="wisestock-section">
                <h4><?php esc_html_e('Products Needing Attention', 'wisestock'); ?></h4>
                <div style="padding: 20px; text-align: center; color: #6c757d;">
                    <p><?php esc_html_e('No products currently need attention. Great job!', 'wisestock'); ?></p>
                </div>
            </div>
            <?php
            return;
        }
        ?>
        <div class="wisestock-section">
            <h4><?php esc_html_e('Products Needing Attention', 'wisestock'); ?></h4>
            <ul class="wisestock-product-list">
                <?php foreach (array_slice($attention_products, 0, 5) as $item): ?>
                    <?php $product = $item['product']; ?>
                    <li class="wisestock-product-item">
                        <div>
                            <div class="wisestock-product-name">
                                <a href="<?php echo esc_url(admin_url('post.php?post=' . $product->get_id() . '&action=edit')); ?>">
                                    <?php echo esc_html($product->get_name()); ?>
                                </a>
                            </div>
                            <div class="wisestock-product-stock">
                                <?php esc_html_e('Stock:', 'wisestock'); ?> <strong><?php echo esc_html($product->get_stock_quantity()); ?></strong> <?php esc_html_e('units', 'wisestock'); ?>
                            </div>
                        </div>
                        <span class="wisestock-priority-badge <?php echo esc_attr($item['priority']); ?>">
                            <?php echo esc_html(ucfirst($item['priority'])); ?>
                        </span>
                    </li>
                <?php endforeach; ?>
            </ul>
            <?php if (count($attention_products) > 5): ?>
                <div style="padding: 12px 20px; text-align: center; border-top: 1px solid #f1f3f4;">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications')); ?>" style="color: #007cba; text-decoration: none; font-size: 13px;">
                        <?php
                        /* translators: %d: Number of products requiring attention. */
                        printf(esc_html__('View all %d products needing attention', 'wisestock'), count($attention_products));
                        ?>
                    </a>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render velocity summary
     *
     * @param array $velocity_summary Velocity summary data
     * @since 1.0.0
     */
    private function render_velocity_summary($velocity_summary) {
        if (empty($velocity_summary)) {
            ?>
            <div class="wisestock-section">
                <h4><?php esc_html_e('Top Selling Products', 'wisestock'); ?></h4>
                <div style="padding: 20px; text-align: center; color: #6c757d;">
                    <p><?php esc_html_e('No velocity data available yet. Sales data will appear here as orders are processed.', 'wisestock'); ?></p>
                </div>
            </div>
            <?php
            return;
        }
        ?>
        <div class="wisestock-section">
            <h4><?php esc_html_e('Top Selling Products', 'wisestock'); ?></h4>
            <ul class="wisestock-product-list">
                <?php foreach (array_slice($velocity_summary, 0, 5) as $item): ?>
                    <?php $product = $item['product']; ?>
                    <li class="wisestock-velocity-item">
                        <div class="wisestock-product-name">
                            <a href="<?php echo esc_url(admin_url('post.php?post=' . $product->get_id() . '&action=edit')); ?>">
                                <?php echo esc_html($product->get_name()); ?>
                            </a>
                        </div>
                        <div class="wisestock-velocity-value">
                            <strong><?php echo '~ ' . esc_html(number_format($item['velocity'], 1)); ?></strong> units / day
                            <small class="wisestock-confidence-text">(<?php echo esc_html(round($item['confidence'] * 100)); ?>% <?php esc_html_e('confidence', 'wisestock'); ?>)</small>
                        </div>
                    </li>
                <?php endforeach; ?>
            </ul>
            <?php if (count($velocity_summary) > 5): ?>
                <div style="padding: 12px 20px; text-align: center; border-top: 1px solid #f1f3f4;">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-dashboard')); ?>" style="color: #007cba; text-decoration: none; font-size: 13px;">
                        <?php
                        /* translators: %d: Number of products with velocity data. */
                        printf(esc_html__('View all %d products with velocity data', 'wisestock'), count($velocity_summary));
                        ?>
                    </a>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render quick actions
     *
     * @since 1.0.0
     */
    private function render_quick_actions() {
        ?>
        <div class="wisestock-quick-actions">
            <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-dashboard')); ?>" class="wisestock-btn">
                <?php esc_html_e('Dashboard', 'wisestock'); ?>
            </a>
            <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications')); ?>" class="wisestock-btn">
                <?php esc_html_e('Notifications', 'wisestock'); ?>
            </a>
            <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-settings')); ?>" class="wisestock-btn">
                <?php esc_html_e('Settings', 'wisestock'); ?>
            </a>
        </div>
        <?php
    }
}

