<?php
/**
 * Wise Stock Core Class
 *
 * This is the main core class that handles plugin initialization,
 * database setup, cron jobs, and coordinates all other components.
 * It manages the plugin lifecycle and provides the central API.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Wise Stock Core Class
 */
class WISESTOCK_Core {

    /**
     * Plugin instance
     *
     * @var WISESTOCK_Core
     */
    private static $instance = null;

    /**
     * Velocity calculator instance (includes timebound features)
     *
     * @var WISESTOCK_Velocity_Calculator
     */
    private $velocity_calculator;

    /**
     * Timebound settings instance
     *
     * @var WISESTOCK_Timebound_Settings
     */
    private $timebound_settings;



    /**
     * Notification manager instance
     *
     * @var WISESTOCK_Notification_Manager
     */
    private $notification_manager;

    /**
     * Supplier manager instance
     *
     * @var WISESTOCK_Supplier_Manager
     */
    private $supplier_manager;

    /**
     * Dashboard widget instance
     *
     * @var WISESTOCK_Dashboard_Widget
     */
    private $dashboard_widget;

    /**
     * Historical data migrator instance
     *
     * @var WISESTOCK_Historical_Data_Migrator
     */
    private $data_migrator;

    /**
     * Get plugin instance
     *
     * @return WISESTOCK_Core
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init();
    }

    /**
     * Initialize the core functionality
     *
     * @since 1.0.0
     */
    public function init() {
        // Initialize components
        $this->init_components();

        // Register hooks
        $this->register_hooks();

        // Initialize admin
        if (is_admin()) {
            $this->init_admin();
        }
    }

    /**
     * Initialize plugin components
     *
     * @since 1.0.0
     */
    private function init_components() {
        // Initialize timebound settings (always loaded as it's the only algorithm)
        $this->timebound_settings = WISESTOCK_Timebound_Settings::get_instance();
        
        // Initialize unified velocity calculator (now includes all timebound features)
        $this->velocity_calculator = new WISESTOCK_Velocity_Calculator();
        
        $this->notification_manager = new WISESTOCK_Notification_Manager();
        $this->supplier_manager = new WISESTOCK_Supplier_Manager();
        $this->dashboard_widget = new WISESTOCK_Dashboard_Widget();
        
        // Always initialize timebound admin in admin context
        if ( is_admin() && class_exists( 'WISESTOCK_Timebound_Admin' ) ) {
            new WISESTOCK_Timebound_Admin();
        }
        
        // Hook to extend components
        do_action('wisestock_init_components', $this);
    }

    /**
     * Register WordPress hooks
     *
     * @since 1.0.0
     */
    private function register_hooks() {
        // WooCommerce hooks
        add_action('woocommerce_reduce_order_stock', array($this, 'update_velocity_data'));
        add_action('woocommerce_restore_order_stock', array($this, 'update_velocity_data'));
        add_filter('woocommerce_admin_reports_path', array($this, 'add_wisestock_reports'));
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));

        // Cron jobs
        add_action('wisestock_daily_calculations', array($this, 'run_velocity_updates'));
        add_action('wisestock_notification_check', array($this, 'check_and_send_notifications'));

        // Custom actions
        add_action('wisestock_velocity_updated', array($this, 'handle_velocity_updated'), 10, 2);
        add_action('wisestock_notification_triggered', array($this, 'handle_notification_triggered'), 10, 3);
        add_action('wisestock_supplier_notified', array($this, 'handle_supplier_notified'), 10, 3);
        
        // Dashboard enhancements - add timebound prediction data to dashboard/widget displays
        add_filter('wisestock_enhance_demo_velocity_data', array($this, 'enhance_velocity_with_timebound'), 10, 3);
        add_filter('wisestock_get_enhanced_velocity_data', array($this, 'get_timebound_velocity_data'), 10, 3);
        add_filter('wisestock_dashboard_widget_data', array($this, 'enhance_widget_with_timebound'), 10, 2);
    }

    /**
     * Initialize admin functionality
     *
     * @since 1.0.0
     */
    private function init_admin() {
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Set parent file for installation page to highlight main menu
        add_filter('parent_file', array($this, 'set_installation_parent_file'));

        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // Register AJAX handlers
        $this->register_ajax_handlers();
    }

    /**
     * Plugin activation
     *
     * @since 1.0.0
     * @return bool True on success, false on failure
     */
    public function activate() {
        try {
            // Create database tables with error handling
            $tables_created = $this->create_database_tables();

            if (!$tables_created) {
                return false;
            }            

            // Initialize first-time setup components
            $this->initialize_first_time_components();

            // Schedule cron jobs
            $this->schedule_cron_jobs();

            // Flush rewrite rules
            flush_rewrite_rules();

            // Mark activation as successful
            update_option('wisestock_activation_status', 'success');
            update_option('wisestock_activation_date', current_time('mysql'));

            return true;

        } catch (Exception $e) {
            update_option('wisestock_activation_status', 'failed');
            update_option('wisestock_activation_error', $e->getMessage());
            return false;
        }
    }

    /**
     * Get default plugin options
     *
     * @since 1.1.0
     * @return array Default options array
     */
    public static function get_default_options() {
        return array(
            // Notification settings
            'notification_thresholds' => array(
                'critical' => 3,
                'warning' => 7,
                'mild' => 14
            ),
            'email_notifications' => false,
            'email_notification_types' => array('critical', 'warning', 'mild'),
            'notification_frequency' => 'hourly',
            'admin_email_address' => get_option('admin_email'),
            
            // Algorithm settings
            'store_type' => 'retail_store', // Default store type preset (was 'default', now using actual preset key)
            
            // Timebound prediction settings
            'timebound_settings' => array(
                'enabled' => true,
                'default_prediction_window' => 14,
                'max_prediction_window' => 90,
                'min_confidence_threshold' => 0.5,
                'volatility_threshold' => 0.25, // Balanced (25%)
                'confidence_decay_enabled' => true,
                'auto_window_adjustment' => true,
                'product_type_decay_rates' => array(
                    'fashion' => 0.15,
                    'electronics' => 0.08,
                    'health' => 0.05,
                    'food' => 0.12,
                    'general' => 0.10
                ),
                'store_type_preset' => 'retail_store',
                'prediction_window_days' => 14,
                'confidence_threshold' => 0.7,
                'trend_sensitivity' => 0.5,
                'volatility_adjustment' => true,
                'seasonal_factor_enabled' => true,
                'market_sensitivity_enabled' => true,
            ),
            
            // Supplier settings
            'supplier_email_subject' => '[{site_name}] {urgency} - Restock Request for {product_name}',
            'supplier_from_email' => get_option('admin_email'),
            
            // UI settings
            'dashboard_widget' => false, // Disabled by default for performance
            'demo_stats' => false,
            
            // System settings
            'delete_data_on_uninstall' => false
        );
    }

    /**
     * Initialize components needed for first-time setup
     *
     * @since 1.1.0
     */
    private function initialize_first_time_components() {
        // Set default options using single source of truth
        add_option('wisestock_options', self::get_default_options());
    }
    
    /**
     * Get timebound settings from consolidated options
     * Helper method for easy access to timebound settings
     * 
     * @since 1.1.0
     * @return array Timebound settings
     */
    public static function get_timebound_settings_data() {
        $options = get_option('wisestock_options', array());
        if (isset($options['timebound_settings'])) {
            return $options['timebound_settings'];
        }
        
        // Return defaults if nothing exists
        $defaults = self::get_default_options();
        return $defaults['timebound_settings'];
    }
    
    /**
     * Update timebound settings in consolidated options
     * Helper method for easy updating of timebound settings
     * 
     * @since 1.1.0
     * @param array $new_settings New timebound settings
     * @return bool Success status
     */
    public static function update_timebound_settings_data($new_settings) {
        $defaults = self::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        $current_timebound = self::get_timebound_settings_data();
        $options['timebound_settings'] = wp_parse_args($new_settings, $current_timebound);
        
        return update_option('wisestock_options', $options);
    }

    /**
     * Plugin deactivation
     *
     * @since 1.0.0
     */
    public function deactivate() {
        // Clear scheduled cron jobs
        $this->clear_cron_jobs();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Create database tables
     *
     * @since 1.0.0
     * @return bool True on success, false on failure
     */
    public function create_database_tables() {
        global $wpdb;

        // Check user capabilities for database operations
        if (!current_user_can('manage_options') && !current_user_can('activate_plugins')) {
            return false;
        }

        // Start transaction for atomic operations using WordPress method
        $wpdb->query('START TRANSACTION');

        try {
            $charset_collate = $wpdb->get_charset_collate();
            $created_tables = array();
            $errors = array();

            // Define table schemas with proper WordPress naming conventions
            $tables = array(
                'velocity' => array(
                    'name' => $wpdb->prefix . 'wisestock_velocity',
                    'sql' => "CREATE TABLE {$wpdb->prefix}wisestock_velocity (
                        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                        product_id bigint(20) unsigned NOT NULL,
                        date date NOT NULL,
                        quantity_sold int(11) DEFAULT 0,
                        running_average decimal(10,2) DEFAULT NULL,
                        created_at datetime DEFAULT CURRENT_TIMESTAMP,
                        PRIMARY KEY (id),
                        UNIQUE KEY idx_product_date_unique (product_id, date),
                        KEY idx_product_id (product_id),
                        KEY idx_date (date),
                        KEY idx_quantity_sold (quantity_sold),
                        KEY idx_running_average (running_average),
                        KEY idx_created_at (created_at)
                    ) $charset_collate;"
                ),
                'notifications' => array(
                    'name' => $wpdb->prefix . 'wisestock_notifications',
                    'sql' => "CREATE TABLE {$wpdb->prefix}wisestock_notifications (
                        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                        product_id bigint(20) unsigned NOT NULL,
                        notification_type enum('critical','warning','mild') NOT NULL DEFAULT 'mild',
                        message text,
                        status enum('active','resolved','dismissed') NOT NULL DEFAULT 'active',
                        created_at datetime DEFAULT CURRENT_TIMESTAMP,
                        PRIMARY KEY (id),
                        KEY idx_product_id (product_id),
                        KEY idx_status (status),
                        KEY idx_notification_type (notification_type),
                        KEY idx_product_status (product_id, status),
                        KEY idx_type_status (notification_type, status),
                        KEY idx_created_at (created_at)
                    ) $charset_collate;"
                ),
                'suppliers' => array(
                    'name' => $wpdb->prefix . 'wisestock_suppliers',
                    'sql' => "CREATE TABLE {$wpdb->prefix}wisestock_suppliers (
                        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                        name varchar(255) NOT NULL,
                        email varchar(255) DEFAULT NULL,
                        product_ids longtext,
                        lead_time_days int(11) DEFAULT 7,
                        created_at datetime DEFAULT CURRENT_TIMESTAMP,
                        PRIMARY KEY (id),
                        UNIQUE KEY idx_name_unique (name),
                        UNIQUE KEY idx_email_unique (email),
                        KEY idx_lead_time (lead_time_days),
                        KEY idx_created_at (created_at)
                    ) $charset_collate;"
                )
            );

            // Include WordPress upgrade functions
            if (!function_exists('dbDelta')) {
                require_once ABSPATH . 'wp-admin/includes/upgrade.php';
            }

            // Create each table with existence check
            foreach ($tables as $table_key => $table_info) {
                $table_name = $table_info['name'];

                // Check if table already exists using prepared statement
                $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));

                if ($table_exists) {
                    // Table exists, check if it needs migration
                    $this->migrate_existing_table($table_name, $table_key);
                    $created_tables[] = $table_name;
                    continue;
                }

                // Create new table using dbDelta (WordPress standard)
                $result = dbDelta($table_info['sql']);

                if (empty($result)) {
                    $errors[] = sprintf('Failed to create table: %s', $table_name);
                    continue;
                }

                // Verify table was created successfully
                $verify_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));
                if (!$verify_exists) {
                    $errors[] = sprintf('Table creation verification failed: %s', $table_name);
                    continue;
                }

                $created_tables[] = $table_name;
            }

            // If there were errors, rollback and return false
            if (!empty($errors)) {
                $wpdb->query('ROLLBACK');
                return false;
            }

            // Commit transaction
            $wpdb->query('COMMIT');

            // Update database version for future migrations
            update_option('wisestock_db_version', WISESTOCK_VERSION);

            // Run post-creation validation
            $this->validate_database_tables();

            return true;

        } catch (Exception $e) {
            // Rollback on any exception
            $wpdb->query('ROLLBACK');
            return false;
        }
    }

    /**
     * Migrate existing table if needed
     *
     * @param string $table_name Table name
     * @param string $table_key Table key identifier
     * @since 1.0.0
     */
    private function migrate_existing_table($table_name, $table_key) {
        global $wpdb;

        // Check database version for migration decisions
        $current_db_version = get_option('wisestock_db_version', '0.0.0');
        $plugin_version = WISESTOCK_VERSION;

        if (version_compare($current_db_version, $plugin_version, '>=')) {
            // Database is up to date
            return;
        }

        // Get current table structure using prepared statement
        $columns = $wpdb->get_results($wpdb->prepare("DESCRIBE `%s`", $table_name));
        $existing_columns = array();

        foreach ($columns as $column) {
            $existing_columns[$column->Field] = $column;
        }

        // Define expected columns for each table
        $expected_columns = array(
            'velocity' => array('id', 'product_id', 'date', 'quantity_sold', 'running_average', 'created_at'),
            'notifications' => array('id', 'product_id', 'notification_type', 'message', 'status', 'created_at'),
            'suppliers' => array('id', 'name', 'email', 'product_ids', 'lead_time_days', 'created_at')
        );

        if (!isset($expected_columns[$table_key])) {
            return;
        }

        $missing_columns = array_diff($expected_columns[$table_key], array_keys($existing_columns));

        if (!empty($missing_columns)) {
            // In a real implementation, you would add the missing columns here using ALTER TABLE
            // For now, we'll just log the issue for manual migration
        }
    }

    /**
     * Validate database tables after creation
     *
     * @since 1.0.0
     * @return bool True if all tables are valid, false otherwise
     */
    private function validate_database_tables() {
        global $wpdb;

        $required_tables = array(
            $wpdb->prefix . 'wisestock_velocity',
            $wpdb->prefix . 'wisestock_notifications',
            $wpdb->prefix . 'wisestock_suppliers'
        );

        $missing_tables = array();

        foreach ($required_tables as $table) {
            $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
            if (!$exists) {
                $missing_tables[] = $table;
            }
        }

        if (!empty($missing_tables)) {
            return false;
        }

        // Test basic operations on each table
        foreach ($required_tables as $table) {
            try {
                // Test SELECT operation - use proper table name without prepare for table names
                $wpdb->get_var("SELECT COUNT(*) FROM `{$table}`");

                // Test table structure - use proper table name without prepare for table names
                $columns = $wpdb->get_results("DESCRIBE `{$table}`");
                if (empty($columns)) {
                    return false;
                }
            } catch (Exception $e) {
                return false;
            }
        }

        return true;
    }

    /**
     * Schedule cron jobs
     *
     * @since 1.0.0
     */
    private function schedule_cron_jobs() {
        if (!wp_next_scheduled('wisestock_daily_calculations')) {
            wp_schedule_event(time(), 'daily', 'wisestock_daily_calculations');
        }

        if (!wp_next_scheduled('wisestock_notification_check')) {
            // Get saved notification frequency or default to hourly
            $defaults = self::get_default_options();
            $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
            $frequency = $options['notification_frequency'];
            
            wp_schedule_event(time(), $frequency, 'wisestock_notification_check');
        }
    }

    /**
     * Clear scheduled cron jobs
     *
     * @since 1.0.0
     */
    private function clear_cron_jobs() {
        wp_clear_scheduled_hook('wisestock_daily_calculations');
        wp_clear_scheduled_hook('wisestock_notification_check');
    }

    /**
     * Reschedule notification cron job with new frequency
     *
     * @param string $frequency The new frequency (hourly, twicedaily, daily)
     * @since 1.1.0
     */
    private function reschedule_notification_cron($frequency) {
        // Clear existing notification cron
        wp_clear_scheduled_hook('wisestock_notification_check');
        
        // Schedule with new frequency
        if (!wp_next_scheduled('wisestock_notification_check')) {
            wp_schedule_event(time(), $frequency, 'wisestock_notification_check');
        }
    }

    /**
     * Update velocity data when stock is reduced
     *
     * @param WC_Order $order The order object
     * @since 1.0.0
     */
    public function update_velocity_data($order) {
        if (!$this->velocity_calculator) {
            return;
        }

        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $quantity = $item->get_quantity();

            $this->velocity_calculator->record_sale($product_id, $quantity);
        }
    }

    /**
     * Run daily velocity updates
     *
     * @since 1.0.0
     */
    public function run_velocity_updates() {
        if (!$this->velocity_calculator) {
            return;
        }

        $this->velocity_calculator->update_all_velocities();
    }

    /**
     * Get all products with stock management enabled
     *
     * @return array Product IDs
     * @since 1.0.2
     */
    private function get_managed_products() {
        $args = array(
            'post_type'      => 'product',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'meta_query'     => array(
                array(
                    'key'     => '_manage_stock',
                    'value'   => 'yes',
                    'compare' => '=',
                ),
            ),
        );

        $query = new WP_Query( $args );
        return $query->posts;
    }

    /**
     * Check and send notifications
     *
     * @since 1.0.0
     */
    public function check_and_send_notifications() {
        if (!$this->notification_manager) {
            return;
        }

        $this->notification_manager->check_all_notifications();
    }

    /**
     * Add Wise Stock reports to WooCommerce
     *
     * @param string $path The reports path
     * @return string Modified path
     * @since 1.0.0
     */
    public function add_wisestock_reports($path) {
        // Implementation for custom reports
        return $path;
    }

    /**
     * Add dashboard widget
     *
     * @since 1.0.0
     */
    public function add_dashboard_widget() {
        if (!$this->dashboard_widget) {
            return;
        }

        // Check if dashboard widget is enabled
        $defaults = self::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        if (!$options['dashboard_widget']) {
            return;
        }

        $this->dashboard_widget->add_widget();
    }

    /**
     * Add admin menu
     *
     * @since 1.0.0
     */
    public function add_admin_menu() {
        // Add top-level Wise Stock menu with dashicons-chart-line icon
        add_menu_page(
            __('WiseStock', 'wisestock'),           // Page title
            __('WiseStock', 'wisestock'),               // Menu title
            'manage_woocommerce',                             // Capability
            'wisestock-dashboard',                           // Menu slug
            array($this, 'admin_dashboard_page'),             // Callback function
            'dashicons-chart-line',                           // Icon (stock chart icon)
            56                                                // Position (after WooCommerce at 55)
        );

        // Add Dashboard submenu (duplicate of parent to rename first item)
        add_submenu_page(
            'wisestock-dashboard',                           // Parent slug
            __('Dashboard', 'wisestock'),                // Page title
            __('Dashboard', 'wisestock'),                // Menu title
            'manage_woocommerce',                             // Capability
            'wisestock-dashboard',                           // Menu slug (same as parent)
            array($this, 'admin_dashboard_page')              // Callback function
        );

        // Add Notifications submenu
        add_submenu_page(
            'wisestock-dashboard',                           // Parent slug
            __('Notifications', 'wisestock'),            // Page title
            __('Notifications', 'wisestock'),            // Menu title
            'manage_woocommerce',                             // Capability
            'wisestock-notifications',                       // Menu slug
            array($this, 'admin_notifications_page')          // Callback function
        );

        // Add Suppliers submenu
        add_submenu_page(
            'wisestock-dashboard',                           // Parent slug
            __('Suppliers', 'wisestock'),                // Page title
            __('Suppliers', 'wisestock'),                // Menu title
            'manage_woocommerce',                             // Capability
            'wisestock-suppliers',                           // Menu slug
            array($this, 'admin_suppliers_page')              // Callback function
        );

        // Add Settings submenu
        add_submenu_page(
            'wisestock-dashboard',                           // Parent slug
            __('Settings', 'wisestock'),                 // Page title
            __('Settings', 'wisestock'),                 // Menu title
            'manage_woocommerce',                             // Capability
            'wisestock-settings',                            // Menu slug
            array($this, 'admin_settings_page')               // Callback function
        );

        // Add Installation Wizard page (hidden from sidebar)
        add_submenu_page(
            '',                                           // Parent slug (empty for hidden from menu)
            __('WiseStock Installation', 'wisestock'),     // Page title
            __('WiseStock Installation', 'wisestock'),     // Menu title
            'manage_woocommerce',                          // Capability
            'wisestock-installation',                      // Menu slug
            array($this, 'admin_installation_page')       // Callback function
        );
    }

    /**
     * Set parent file for installation page to highlight main menu
     *
     * @param string $parent_file Current parent file
     * @return string Modified parent file
     * @since 1.0.1
     */
    public function set_installation_parent_file($parent_file) {
        global $pagenow;
        
        if ('admin.php' === $pagenow && isset($_GET['page']) && 'wisestock-installation' === $_GET['page']) {
            $parent_file = 'wisestock-dashboard';
        }
        
        return $parent_file;
    }

    /**
     * Admin dashboard page
     *
     * @since 1.0.0
     */
    public function admin_dashboard_page() {
        // Check if installation is completed
        $installation_completed = get_option('wisestock_installation_completed', false);

        if (!$installation_completed) {
            // Redirect to installation wizard
            wp_redirect(admin_url('admin.php?page=wisestock-installation'));
            exit;
        }

        include WISESTOCK_PLUGIN_DIR . 'admin/partials/dashboard.php';
    }

    /**
     * Admin notifications page
     *
     * @since 1.0.0
     */
    public function admin_notifications_page() {
        // Check if installation is completed
        $installation_completed = get_option('wisestock_installation_completed', false);

        if (!$installation_completed) {
            // Redirect to installation wizard
            wp_redirect(admin_url('admin.php?page=wisestock-installation'));
            exit;
        }

        include WISESTOCK_PLUGIN_DIR . 'admin/partials/notifications.php';
    }

    /**
     * Admin settings page
     *
     * @since 1.0.0
     */
    public function admin_settings_page() {
        // Check if installation is completed
        $installation_completed = get_option('wisestock_installation_completed', false);

        if (!$installation_completed) {
            // Redirect to installation wizard
            wp_redirect(admin_url('admin.php?page=wisestock-installation'));
            exit;
        }

        include WISESTOCK_PLUGIN_DIR . 'admin/partials/settings.php';
    }

    /**
     * Admin suppliers page
     *
     * @since 1.0.0
     */
    public function admin_suppliers_page() {
        // Check if installation is completed
        $installation_completed = get_option('wisestock_installation_completed', false);

        if (!$installation_completed) {
            // Redirect to installation wizard
            wp_redirect(admin_url('admin.php?page=wisestock-installation'));
            exit;
        }

        include WISESTOCK_PLUGIN_DIR . 'admin/partials/suppliers.php';
    }

    /**
     * Enqueue admin assets
     *
     * @param string $hook Current admin page hook
     * @since 1.0.0
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'wisestock') === false) {
            return;
        }

        wp_enqueue_style(
            'wisestock-admin',
            WISESTOCK_PLUGIN_URL . 'admin/css/wisestock-admin.css',
            array(),
            WISESTOCK_VERSION
        );

        // Determine dependencies
        $dependencies = array('jquery');

        wp_enqueue_script(
            'wisestock-admin',
            WISESTOCK_PLUGIN_URL . 'admin/js/wisestock-admin.js',
            $dependencies,
            WISESTOCK_VERSION,
            true
        );

        // Determine page types
        $is_settings_page = (strpos($hook, 'wisestock-settings') !== false);
        $is_suppliers_page = (strpos($hook, 'wisestock-suppliers') !== false);
        $is_dashboard_page = (strpos($hook, 'wisestock-dashboard') !== false);
        
        // Enqueue settings-specific JavaScript if on settings page
        if ($is_settings_page) {
            wp_enqueue_script(
                'wisestock-settings',
                WISESTOCK_PLUGIN_URL . 'admin/js/wisestock-settings.js',
                array('jquery'),
                WISESTOCK_VERSION,
                true
            );
            
            // Localize settings script with required strings
            wp_localize_script(
                'wisestock-settings',
                'wisestock_admin',
                array(
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('wisestock_admin_nonce'),
                    // Provide installation URL for reset-installation flow
                    'installation_url' => admin_url('admin.php?page=wisestock-installation'),
                    'strings' => array(
                        'saving' => __('Saving...', 'wisestock'),
                        'saved' => __('Saved', 'wisestock'),
                        'save_error' => __('Save Error', 'wisestock'),
                    )
                )
            );
        }
        
        // Enqueue suppliers-specific JavaScript if on suppliers page
        if ($is_suppliers_page) {
            wp_enqueue_script(
                'wisestock-suppliers',
                WISESTOCK_PLUGIN_URL . 'admin/js/wisestock-suppliers.js',
                array('jquery'),
                WISESTOCK_VERSION,
                true
            );
            
            // Localize suppliers script with required strings
            wp_localize_script(
                'wisestock-suppliers',
                'wisestock_suppliers',
                array(
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('wisestock_admin_nonce'),
                    'strings' => array(
                        'saving' => __('Saving...', 'wisestock'),
                        'saved' => __('Saved', 'wisestock'),
                        'error' => __('Error', 'wisestock'),
                        'ready' => __('Ready', 'wisestock'),
                        'save_settings' => __('Save Settings', 'wisestock'),
                        'email_subject_required' => __('Email subject line is required', 'wisestock'),
                        'email_subject_min_length' => __('Email subject must be at least 5 characters', 'wisestock'),
                        'from_email_required' => __('From email address is required', 'wisestock'),
                        'valid_email_required' => __('Please enter a valid email address', 'wisestock'),
                        'supplier_settings_saved' => __('Supplier settings saved successfully', 'wisestock'),
                        'failed_save_supplier_settings' => __('Failed to save supplier settings', 'wisestock'),
                        'network_error' => __('Network error occurred', 'wisestock'),
                        'failed_load_supplier_data' => __('Failed to load supplier data', 'wisestock'),
                        'delete_supplier_confirm' => __('Are you sure you want to delete this supplier?', 'wisestock'),
                        'supplier_deleted' => __('Supplier deleted successfully', 'wisestock'),
                        'failed_delete_supplier' => __('Failed to delete supplier', 'wisestock'),
                        'test_email_required' => __('Please enter a test email address', 'wisestock'),
                        'sending' => __('Sending...', 'wisestock'),
                        'test_email_sent' => __('Test email sent successfully', 'wisestock'),
                        'failed_send_test_email' => __('Failed to send test email', 'wisestock'),
                        'supplier_name_required' => __('Supplier name is required', 'wisestock'),
                        'supplier_name_min_length' => __('Supplier name must be at least 2 characters long', 'wisestock'),
                        'duplicate_supplier_name' => __('A supplier with this name already exists', 'wisestock'),
                        'supplier_email_required' => __('Supplier email is required', 'wisestock'),
                        'duplicate_supplier_email' => __('A supplier with this email already exists', 'wisestock'),
                        'lead_time_range' => __('Lead time must be between 0 and 365 days', 'wisestock'),
                        'select_at_least_one_product' => __('Please select at least one product', 'wisestock'),
                        'updating' => __('Updating...', 'wisestock'),
                        'supplier_updated' => __('Supplier updated successfully', 'wisestock'),
                        'failed_update_supplier' => __('Failed to update supplier', 'wisestock'),
                        'lead_time_min' => __('Lead time must be at least 1 day', 'wisestock'),
                        'lead_time_max' => __('Lead time cannot exceed 365 days', 'wisestock'),
                        'adding' => __('Adding...', 'wisestock'),
                        'supplier_added' => __('Supplier added successfully', 'wisestock'),
                        'failed_add_supplier' => __('Failed to add supplier', 'wisestock'),
                        'default_supplier_updated' => __('Default supplier updated successfully', 'wisestock'),
                    ),
                    'templates' => array(
                        'selected_item' => '<span class="wisestock-selected-item">{name} {sku}<span class="remove">&times;</span></span>',
                        'notice' => '<div class="notice notice-{type} is-dismissible"><p>{message}</p></div>',
                    )
                )
            );
        }

        // Localize script so config is available on all Wise Stock admin screens
        wp_localize_script(
            'wisestock-admin',
            'wisestock_admin',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('wisestock_admin_nonce'),
                'product_edit_url' => admin_url('post.php?post=%d&action=edit'),
                'installation_url' => admin_url('admin.php?page=wisestock-installation'),
                'is_dashboard_page' => $is_dashboard_page,
                'strings' => array(
                    'notification_resolved' => __('Notification resolved successfully', 'wisestock'),
                    'notification_dismissed' => __('Notification dismissed', 'wisestock'),
                    'settings_saved' => __('Settings saved successfully', 'wisestock'),
                    'supplier_deleted' => __('Supplier deleted successfully', 'wisestock'),
                    'supplier_updated' => __('Supplier updated successfully', 'wisestock'),
                    'default_supplier_updated' => __('Default supplier updated successfully', 'wisestock'),
                    'suppliers_notified' => __('Suppliers notified successfully', 'wisestock'),
                    'failed_resolve_notification' => __('Failed to resolve notification', 'wisestock'),
                    'failed_dismiss_notification' => __('Failed to dismiss notification', 'wisestock'),
                    'failed_save_settings' => __('Failed to save settings', 'wisestock'),
                    'failed_delete_supplier' => __('Failed to delete supplier', 'wisestock'),
                    'failed_update_supplier' => __('Failed to update supplier', 'wisestock'),
                    'failed_load_supplier_data' => __('Failed to load supplier data', 'wisestock'),
                    'failed_notify_suppliers' => __('Failed to notify suppliers', 'wisestock'),
                    'network_error' => __('Network error occurred', 'wisestock'),
                    'no_suppliers_found' => __('No suppliers found for this product', 'wisestock'),
                    'select_suppliers' => __('Please select at least one supplier', 'wisestock'),
                    'supplier_name_required' => __('Supplier name is required', 'wisestock'),
                    'valid_email_required' => __('Please enter a valid email address', 'wisestock'),
                    'positive_lead_time' => __('Lead time must be a positive number', 'wisestock'),
                    'no_email' => __('No email', 'wisestock'),
                    'default_supplier' => __('Default', 'wisestock'),
                    'save_settings' => __('Save Settings', 'wisestock'),
                    'failed_update_supplier' => __('Failed to update supplier', 'wisestock'),
                    'supplier_added' => __('Supplier added successfully', 'wisestock'),
                    'failed_add_supplier' => __('Failed to add supplier', 'wisestock'),
                    'sales_velocity' => __('Sales Velocity', 'wisestock'),
                    'critical' => __('Critical', 'wisestock'),
                    'warning' => __('Warning', 'wisestock'),
                    'mild' => __('Mild', 'wisestock'),
                    'active_alerts' => __('Active Alerts', 'wisestock'),
                    'velocity_weight_warning' => __('Weights should add up to 1.0 for optimal results. Current total: %total%', 'wisestock'),
                    'confidence_warning' => __('Confidence values should be in ascending order: min < sparse < good', 'wisestock'),
                    'reset_confirmation' => __('Are you sure you want to reset the installation? This will clear all settings and data.', 'wisestock'),
                    'delete_confirmation_text' => __('DELETE DATA', 'wisestock'),
                    'deleting' => __('Deleting...', 'wisestock'),
                    'data_deleted_success' => __('All data has been successfully deleted.', 'wisestock'),
                    'delete_failed' => __('Failed to delete data. Please try again.', 'wisestock'),
                    'server_error' => __('Server error occurred. Please try again.', 'wisestock'),
                    'no_notifications' => __('No notifications at this time', 'wisestock'),
                    'resolve' => __('Resolve', 'wisestock'),
                    'dismiss' => __('Dismiss', 'wisestock'),
                    'select_suppliers_to_notify' => __('Select Suppliers to Notify', 'wisestock'),
                    'multiple_suppliers_text' => __('This product has multiple suppliers. Select which suppliers to notify:', 'wisestock'),
                    'cancel' => __('Cancel', 'wisestock'),
                    'notify_selected' => __('Notify Selected', 'wisestock'),
                    'no_velocity_data' => __('No velocity data available yet', 'wisestock'),
                    'sales_velocity_by_product' => __('Sales Velocity by Product', 'wisestock'),
                    'no_alerts_available' => __('No alerts available', 'wisestock'),
                    'alert_distribution' => __('Alert Distribution', 'wisestock'),
                    'permission_denied' => __('Permission denied. You do not have access to perform this action.', 'wisestock'),
                    'setup_complete' => __('Setup Complete!', 'wisestock'),
                    'configured_ready' => __('Your Wise Stock is now configured and ready to use.', 'wisestock'),
                    'redirecting_dashboard' => __('Redirecting to dashboard...', 'wisestock'),
                    'setup_complete_title' => __('Setup Complete!', 'wisestock'),
                    'configured_ready_text' => __('Your Wise Stock is now configured and ready to use.', 'wisestock'),
                    'not_found' => __('Requested resource not found.', 'wisestock'),
                    'saving' => __('Saving...', 'wisestock'),
                    'saved' => __('Saved', 'wisestock'),
                    'save_error' => __('Save Error', 'wisestock'),
                    'notifying' => __('Notifying...', 'wisestock'),
                    'updating' => __('Updating...', 'wisestock'),
                    'adding' => __('Adding...', 'wisestock'),
                    'regenerating' => __('Regenerating...', 'wisestock'),
                    'name' => __('Name', 'wisestock'),
                    'email' => __('Email', 'wisestock'),
                    'lead_time' => __('Lead Time', 'wisestock'),
                    'products' => __('Products', 'wisestock'),
                    'actions' => __('Actions', 'wisestock'),
                    'no_email' => __('No email', 'wisestock'),
                    'edit' => __('Edit', 'wisestock'),
                    'delete' => __('Delete', 'wisestock'),
                    'sku' => __('SKU', 'wisestock'),
                    'suppliers' => __('Suppliers', 'wisestock'),
                    'default_supplier' => __('Default Supplier', 'wisestock'),
                    'select_default_supplier' => __('Select Default Supplier', 'wisestock'),
                    'default_supplier_help' => __('This supplier will be pre-selected when using Notify Supplier', 'wisestock'),
                    'save_settings' => __('Save Settings', 'wisestock'),
                    'deleting' => __('Deleting...', 'wisestock'),
                    'network_error_occurred' => __('Network error occurred while deleting data', 'wisestock'),
                    'connection_timeout' => __('Connection timeout or network error. Please check your internet connection and try again.', 'wisestock'),
                    'server_error_occurred' => __('Server error occurred. Please check the error logs and try again.', 'wisestock'),
                    'permission_denied_refresh' => __('Permission denied. Please refresh the page and try again.', 'wisestock'),
                    'request_not_found' => __('Request not found. Please refresh the page and try again.', 'wisestock'),
                    'analytics_regenerated_success' => __('Analytics regenerated successfully! Dashboard data should now be available.', 'wisestock'),
                    'error_unknown' => __('Unknown error', 'wisestock'),
                    'failed_regenerate_analytics' => __('Failed to regenerate analytics', 'wisestock'),
                    'error_prefix' => __('Error: ', 'wisestock'),
                    'saved' => __('Saved!', 'wisestock'),
                    'error_try_again' => __('Error - try again', 'wisestock'),
                    'redirecting' => __('Redirecting...', 'wisestock'),
                    'completing_setup' => __('Completing Setup...', 'wisestock'),
                    'please_select_option' => __('Please select an option to continue', 'wisestock'),
                    'installation_failed' => __('Installation failed', 'wisestock'),
                    'network_error_occurred' => __('Network error occurred', 'wisestock'),
                    'digital_products' => __('Digital Products', 'wisestock'),
                    'mixed_store' => __('Mixed Store', 'wisestock'),
                    'classic_predictions' => __('Classic Predictions', 'wisestock'),
                    'timebound_predictions' => __('Time-Bound Predictions', 'wisestock'),
                    'yes' => __('Yes', 'wisestock'),
                    'no' => __('No', 'wisestock'),
                    'not_to_use' => __('not to use', 'wisestock'),
                    'stock_management_explanation' => __('You\'ll only see sales velocity and basic analytics.', 'wisestock'),
                    'retail_store' => __('Retail Store', 'wisestock'),
                    'dropshipping' => __('Dropshipping', 'wisestock'),
                ),
                'templates' => array(
                    'no_notifications' => '<div class="wisestock-no-notifications">' . __('No notifications at this time', 'wisestock') . '</div>',
                    'notification_item' => '<div class="wisestock-notification-item {type}"></div>',
                    'notification_content' => '<div class="wisestock-notification-content"><h4 class="wisestock-notification-title">{product_name}</h4><p class="wisestock-notification-message">{message}</p><div class="wisestock-notification-time">{time_ago}</div></div><div class="wisestock-notification-actions"><button class="wisestock-btn small wisestock-notification-resolve" data-notification-id="{id}">' . __('Resolve', 'wisestock') . '</button><button class="wisestock-btn small secondary wisestock-notification-dismiss" data-notification-id="{id}">' . __('Dismiss', 'wisestock') . '</button></div>',
                    'selected_item' => '<span class="wisestock-selected-item">{name} {sku}<span class="remove">&times;</span></span>',
                    'supplier_modal' => '<div id="wisestock-supplier-selection-modal" class="wisestock-modal" style="display: block;"><div class="wisestock-modal-content"><div class="wisestock-modal-header"><h3 class="wisestock-modal-title">' . __('Select Suppliers to Notify', 'wisestock') . '</h3><button type="button" class="wisestock-modal-close">&times;</button></div><div class="wisestock-modal-body"><p>' . __('This product has multiple suppliers. Select which suppliers to notify:', 'wisestock') . '</p><div class="wisestock-supplier-list">{supplier_list}</div></div><div class="wisestock-modal-footer"><button type="button" class="wisestock-btn secondary wisestock-modal-cancel">' . __('Cancel', 'wisestock') . '</button><button type="button" class="wisestock-btn wisestock-notify-selected-suppliers" data-product-id="{product_id}">' . __('Notify Selected', 'wisestock') . '</button></div></div></div>',
                    'supplier_item' => '<label class="wisestock-supplier-item"><input type="checkbox" value="{id}" {checked}><span class="supplier-name">{name}{default_text}</span><span class="supplier-email">({email})</span></label>',
                    'preset_card' => '<div class="wisestock-preset-card{active}" data-value="{key}"><h3 class="wisestock-preset-title">{name}</h3><p class="wisestock-preset-description">{description}</p></div>',
                    'feature_item' => '<div class="wisestock-feature-item">{feature}</div>',
                    'success_message' => '<div class="wisestock-success-message"><div class="wisestock-success-icon"><span class="dashicons dashicons-yes"></span></div><div class="wisestock-success-content"><h3>' . __('Setup Complete!', 'wisestock') . '</h3><p>' . __('Your Wise Stock is now configured and ready to use.', 'wisestock') . '</p><p>' . __('Redirecting to dashboard...', 'wisestock') . '</p></div></div>',
                    'installation_notice' => '<div class="wisestock-notice wisestock-notice-{type}"><span class="dashicons dashicons-{icon}"></span><span class="wisestock-notice-message">{message}</span><button type="button" class="wisestock-notice-dismiss"><span class="dashicons dashicons-dismiss"></span></button></div>'
                )
            )
        );
    }

    /**
     * Handle velocity updated action
     *
     * @param int $product_id Product ID
     * @param array $velocity_data Velocity data
     * @since 1.0.0
     */
    public function handle_velocity_updated($product_id, $velocity_data) {
        // Custom action handling
        do_action('wisestock_velocity_updated_custom', $product_id, $velocity_data);
    }

    /**
     * Handle notification triggered action
     *
     * @param int $product_id Product ID
     * @param string $notification_type Notification type
     * @param string $message Notification message
     * @since 1.0.0
     */
    public function handle_notification_triggered($product_id, $notification_type, $message) {
        // Custom action handling
        do_action('wisestock_notification_triggered_custom', $product_id, $notification_type, $message);
    }

    /**
     * Handle supplier notified action
     *
     * @param int $product_id Product ID
     * @param int $supplier_id Supplier ID
     * @param array $email_data Email data
     * @since 1.0.0
     */
    public function handle_supplier_notified($product_id, $supplier_id, $email_data) {
        // Custom action handling
        do_action('wisestock_supplier_notified_custom', $product_id, $supplier_id, $email_data);
    }

    /**
     * Register AJAX handlers
     *
     * @since 1.0.0
     */
    private function register_ajax_handlers() {
        add_action('wp_ajax_wisestock_get_notifications', array($this, 'ajax_get_notifications'));
        add_action('wp_ajax_wisestock_resolve_notification', array($this, 'ajax_resolve_notification'));
        add_action('wp_ajax_wisestock_dismiss_notification', array($this, 'ajax_dismiss_notification'));
        add_action('wp_ajax_wisestock_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_wisestock_add_supplier', array($this, 'ajax_add_supplier'));
        add_action('wp_ajax_wisestock_delete_supplier', array($this, 'ajax_delete_supplier'));
        add_action('wp_ajax_wisestock_save_supplier_settings', array($this, 'ajax_save_supplier_settings'));
        add_action('wp_ajax_wisestock_assign_products_to_supplier', array($this, 'ajax_assign_products_to_supplier'));
        add_action('wp_ajax_wisestock_get_supplier_products', array($this, 'ajax_get_supplier_products'));
        add_action('wp_ajax_wisestock_notify_supplier', array($this, 'ajax_notify_supplier'));
        add_action('wp_ajax_wisestock_get_product_suppliers', array($this, 'ajax_get_product_suppliers'));
        add_action('wp_ajax_wisestock_set_default_supplier', array($this, 'ajax_set_default_supplier'));
        add_action('wp_ajax_wisestock_get_supplier', array($this, 'ajax_get_supplier'));
        add_action('wp_ajax_wisestock_update_supplier', array($this, 'ajax_update_supplier'));
        add_action('wp_ajax_wisestock_get_product_data', array($this, 'ajax_get_product_data'));
        add_action('wp_ajax_wisestock_send_test_email', array($this, 'ajax_send_test_email'));
        add_action('wp_ajax_wisestock_send_test_notification_email', array($this, 'ajax_send_test_notification_email'));
        
        // Dynamic sections AJAX handlers
        add_action('wp_ajax_wisestock_get_supplier_stats', array($this, 'ajax_get_supplier_stats'));
        add_action('wp_ajax_wisestock_get_suppliers_table', array($this, 'ajax_get_suppliers_table'));
        add_action('wp_ajax_wisestock_get_product_supplier_overview', array($this, 'ajax_get_product_supplier_overview'));
        add_action('wp_ajax_wisestock_get_multi_supplier_products', array($this, 'ajax_get_multi_supplier_products'));

        // Installation wizard AJAX handlers
        add_action('wp_ajax_wisestock_complete_installation', array($this, 'ajax_complete_installation'));
        add_action('wp_ajax_wisestock_save_demo_stats', array($this, 'ajax_save_demo_stats'));

        // Timebound AJAX handlers (delegated to timebound admin)
        add_action('wp_ajax_wisestock_get_timebound_settings', array($this, 'ajax_get_timebound_settings'));

    }

    /**
     * AJAX handler for getting notifications
     *
     * @since 1.0.0
     */
    public function ajax_get_notifications() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        try {
            $api = WISESTOCK_API::get_instance();

            // Check if demo stats are enabled
            $defaults = self::get_default_options();
            $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
            $demo_stats_enabled = $options['demo_stats'];

            if ($demo_stats_enabled) {
                // Return demo notifications
                $dashboard_data = $api->get_demo_dashboard_data();
                $notifications = $dashboard_data['notifications'];
            } else {
                // Return real notifications
                $notifications = $api->get_all_notifications();
            }

            $formatted_notifications = $this->format_notifications($notifications);

            wp_send_json_success($formatted_notifications);
        } catch (Exception $e) {
            wp_send_json_error('Failed to load notifications: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for resolving notifications
     *
     * @since 1.0.0
     */
    public function ajax_resolve_notification() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $notification_id = absint($_POST['notification_id']);
        if (!$notification_id) {
            wp_send_json_error('Invalid notification ID');
        }

        try {
            $api = WISESTOCK_API::get_instance();
            $result = $api->resolve_notification($notification_id);

            if ($result) {
                wp_send_json_success('Notification resolved successfully');
            } else {
                wp_send_json_error('Failed to resolve notification');
            }
        } catch (Exception $e) {
            wp_send_json_error('Failed to resolve notification: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for dismissing notifications
     *
     * @since 1.0.0
     */
    public function ajax_dismiss_notification() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $notification_id = absint($_POST['notification_id']);
        if (!$notification_id) {
            wp_send_json_error('Invalid notification ID');
        }

        try {
            $api = WISESTOCK_API::get_instance();
            $result = $api->dismiss_notification($notification_id);

            if ($result) {
                wp_send_json_success('Notification dismissed successfully');
            } else {
                wp_send_json_error('Failed to dismiss notification');
            }
        } catch (Exception $e) {
            wp_send_json_error('Failed to dismiss notification: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for saving settings
     *
     * @since 1.0.0
     */
    public function ajax_save_settings() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        try {
            // Get current options
            $defaults = self::get_default_options();
            $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);

            // Update notification thresholds
            if (isset($_POST['notification_thresholds']) && is_array($_POST['notification_thresholds'])) {
                $options['notification_thresholds'] = array_map('absint', $_POST['notification_thresholds']);
                // Ensure minimum values for thresholds
                foreach ($options['notification_thresholds'] as $key => $value) {
                    if ($value < 1) {
                        $options['notification_thresholds'][$key] = 1;
                    }
                }
            }

            // Update email notifications (handle checkbox - they don't get submitted when unchecked)
            $options['email_notifications'] = isset($_POST['email_notifications']) ? (bool) $_POST['email_notifications'] : false;

            // Update email notification types
            if (isset($_POST['email_notification_types']) && is_array($_POST['email_notification_types'])) {
                $options['email_notification_types'] = array_map('sanitize_text_field', $_POST['email_notification_types']);
            } else {
                // Default to all types if none selected
                $options['email_notification_types'] = array('critical', 'warning', 'mild');
            }

            // Update notification frequency
            if (isset($_POST['notification_frequency'])) {
                $frequency = sanitize_text_field($_POST['notification_frequency']);
                if (in_array($frequency, array('hourly', 'twicedaily', 'daily'))) {
                    $options['notification_frequency'] = $frequency;
                    
                    // Update cron schedule when frequency changes
                    $this->reschedule_notification_cron($frequency);
                }
            }

            // Update admin email address for notifications
            if (isset($_POST['admin_email_address'])) {
                $admin_email = sanitize_email($_POST['admin_email_address']);
                if (is_email($admin_email)) {
                    $options['admin_email_address'] = $admin_email;
                } elseif (empty($admin_email)) {
                    // Allow empty email to use default
                    $options['admin_email_address'] = '';
                }
            }

            // Handle checkbox fields (they don't get submitted when unchecked)
            $options['dashboard_widget'] = isset($_POST['dashboard_widget']) ? (bool) $_POST['dashboard_widget'] : false;
            $options['demo_stats'] = isset($_POST['demo_stats']) ? (bool) $_POST['demo_stats'] : false;
            $options['delete_data_on_uninstall'] = isset($_POST['delete_data_on_uninstall']) ? (bool) $_POST['delete_data_on_uninstall'] : false;

            // Save options
            update_option('wisestock_options', $options);

            // Hook for saving additional settings (like timebound)
            do_action('wisestock_save_settings');

            wp_send_json_success('Settings saved successfully');
        } catch (Exception $e) {
            wp_send_json_error('Failed to save settings: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for adding suppliers
     *
     * @since 1.0.0
     */
    public function ajax_add_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Get and sanitize form data
        $product_ids = array();
        if (isset($_POST['supplier_products']) && !empty($_POST['supplier_products'])) {
            if (is_array($_POST['supplier_products'])) {
                $product_ids = array_map('absint', $_POST['supplier_products']);
                // Remove any zero values (invalid IDs)
                $product_ids = array_filter($product_ids);
            }
        }

        $supplier_data = array(
            'name' => sanitize_text_field($_POST['supplier_name'] ?? ''),
            'email' => sanitize_email($_POST['supplier_email'] ?? ''),
            'lead_time_days' => absint($_POST['supplier_lead_time'] ?? 7),
            'product_ids' => $product_ids
        );

        // Validate required fields
        if (empty($supplier_data['name'])) {
            wp_send_json_error('Supplier name is required');
        }

        if (strlen($supplier_data['name']) < 2) {
            wp_send_json_error('Supplier name must be at least 2 characters');
        }

        if (empty($supplier_data['email'])) {
            wp_send_json_error('Email address is required');
        }

        if (!is_email($supplier_data['email'])) {
            wp_send_json_error('Invalid email address');
        }

        if ($supplier_data['lead_time_days'] < 1) {
            wp_send_json_error('Lead time must be at least 1 day');
        }

        if ($supplier_data['lead_time_days'] > 365) {
            wp_send_json_error('Lead time cannot exceed 365 days');
        }

        if (empty($product_ids)) {
            wp_send_json_error('Please select at least one product');
        }

        // Check for uniqueness
        $supplier_manager = new WISESTOCK_Supplier_Manager();
        $existing_suppliers = $supplier_manager->get_all_suppliers();
        
        foreach ($existing_suppliers as $existing_supplier) {
            if (strtolower($existing_supplier->name) === strtolower($supplier_data['name'])) {
                wp_send_json_error('A supplier with this name already exists');
            }
            
            if (strtolower($existing_supplier->email) === strtolower($supplier_data['email'])) {
                wp_send_json_error('A supplier with this email already exists');
            }
        }

        try {
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $supplier_id = $supplier_manager->add_supplier($supplier_data);

            if ($supplier_id) {
                // Get the complete supplier data for DOM update
                $supplier_manager = new WISESTOCK_Supplier_Manager();
                $supplier = $supplier_manager->get_supplier($supplier_id);
                
                wp_send_json_success(array(
                    'message' => 'Supplier added successfully',
                    'supplier_id' => $supplier_id,
                    'supplier' => $supplier
                ));
            } else {
                wp_send_json_error('Failed to add supplier');
            }
        } catch (Exception $e) {
            wp_send_json_error('Failed to add supplier: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for getting product data
     *
     * @since 1.0.0
     */
    public function ajax_get_product_data() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $product_id = absint($_POST['product_id'] ?? 0);
        
        if (!$product_id) {
            wp_send_json_error('Product ID is required');
        }

        $product = wc_get_product($product_id);
        
        if (!$product) {
            wp_send_json_error('Product not found');
        }

        $product_data = array(
            'id' => $product->get_id(),
            'name' => $product->get_name(),
            'sku' => $product->get_sku(),
            'edit_url' => admin_url('post.php?post=' . $product->get_id() . '&action=edit')
        );

        wp_send_json_success($product_data);
    }

    /**
     * AJAX handler for deleting suppliers
     *
     * @since 1.0.0
     */
    public function ajax_delete_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_id = absint($_POST['supplier_id']);
        if (!$supplier_id) {
            wp_send_json_error('Invalid supplier ID');
        }

        // Check user capabilities for supplier deletion
        if (!current_user_can('manage_options') && !current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions to delete supplier');
            return;
        }

        try {
            global $wpdb;
            $table_name = $wpdb->prefix . 'wisestock_suppliers';

            // Verify supplier exists before deletion
            $supplier_exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE id = %d",
                $supplier_id
            ));

            if (!$supplier_exists) {
                wp_send_json_error('Supplier not found');
                return;
            }

            $result = $wpdb->delete($table_name, array('id' => $supplier_id), array('%d'));

            if ($result !== false) {
                // Clear supplier caches
                delete_transient('wisestock_all_suppliers');
                
                wp_send_json_success('Supplier deleted successfully');
            } else {
                wp_send_json_error('Failed to delete supplier');
            }
        } catch (Exception $e) {
            wp_send_json_error('Failed to delete supplier: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for saving supplier settings
     *
     * @since 1.0.0
     */
    public function ajax_save_supplier_settings() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        try {
            // Get current options
            $defaults = self::get_default_options();
            $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);

            // Update supplier-related options (auto notifications removed, manual notifications still available)


            // Validate and sanitize email subject
            if (isset($_POST['supplier_email_subject'])) {
                $email_subject = sanitize_text_field($_POST['supplier_email_subject']);
                
                if (empty($email_subject)) {
                    wp_send_json_error('Email subject line is required');
                }
                
                if (strlen($email_subject) < 5) {
                    wp_send_json_error('Email subject must be at least 5 characters');
                }
                
                $options['supplier_email_subject'] = $email_subject;
            }

            // Validate and sanitize from email
            if (isset($_POST['supplier_from_email'])) {
                $from_email = sanitize_email($_POST['supplier_from_email']);
                
                if (empty($from_email)) {
                    wp_send_json_error('From email address is required');
                }
                
                if (!is_email($from_email)) {
                    wp_send_json_error('Please enter a valid email address');
                }
                
                $options['supplier_from_email'] = $from_email;
            }

            // Save options
            update_option('wisestock_options', $options);

            wp_send_json_success('Supplier settings saved successfully');
        } catch (Exception $e) {
            wp_send_json_error('Failed to save supplier settings: ' . $e->getMessage());
        }
    }

    /**
     * Format notifications for frontend
     *
     * @param array $notifications Raw notifications data
     * @return array Formatted notifications
     * @since 1.0.0
     */
    private function format_notifications($notifications) {
        $formatted = array();

        foreach ($notifications as $notification) {
            // Handle demo notifications differently
            if (isset($notification->is_demo) && $notification->is_demo) {
                $formatted[] = array(
                    'id' => $notification->id,
                    'product_id' => $notification->product_id,
                    'product_name' => '[DEMO] ' . $notification->product_name,
                    'notification_type' => $notification->notification_type,
                    'message' => $notification->message,
                    'status' => 'active',
                    'created_at' => $notification->created_at,
                    'time_ago' => human_time_diff(strtotime($notification->created_at), current_time('timestamp')) . ' ago'
                );
            } else {
                // Handle real notifications
                $product = wc_get_product($notification->product_id);

            $formatted[] = array(
                    'id' => $notification->id,
                    'product_id' => $notification->product_id,
                    'product_name' => $product ? $product->get_name() : 'Unknown Product',
                    'notification_type' => $notification->notification_type,
                    'message' => $notification->message,
                    'status' => $notification->status,
                    'created_at' => $notification->created_at,
                    'time_ago' => human_time_diff(strtotime($notification->created_at), current_time('timestamp')) . ' ago'
                );
            }
        }

        return $formatted;
    }

    /**
     * AJAX handler for assigning products to supplier
     *
     * @since 1.0.0
     */
    public function ajax_assign_products_to_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_id = absint($_POST['supplier_id']);
        $product_ids = array();

        if (isset($_POST['product_ids']) && is_array($_POST['product_ids'])) {
            $product_ids = array_map('absint', $_POST['product_ids']);
            // Remove any zero values (invalid IDs)
            $product_ids = array_filter($product_ids);
        }

        if (!$supplier_id) {
            wp_send_json_error('Invalid supplier ID');
        }

        try {
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $result = $supplier_manager->update_supplier($supplier_id, array('product_ids' => $product_ids));

            if ($result) {
                wp_send_json_success('Products assigned successfully');
            } else {
                wp_send_json_error('Failed to assign products');
            }
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for getting supplier products
     *
     * @since 1.0.0
     */
    public function ajax_get_supplier_products() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_id = absint($_POST['supplier_id']);
        if (!$supplier_id) {
            wp_send_json_error('Invalid supplier ID');
        }

        try {
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $supplier = $supplier_manager->get_supplier($supplier_id);

            if (!$supplier) {
                wp_send_json_error('Supplier not found');
            }

            $products = array();
            if (!empty($supplier->product_ids)) {
                foreach ($supplier->product_ids as $product_id) {
                    $product = wc_get_product($product_id);
                    if ($product) {
                        $products[] = array(
                            'id' => $product->get_id(),
                            'name' => $product->get_name(),
                            'sku' => $product->get_sku() ?: 'No SKU'
                        );
                    }
                }
            }

            wp_send_json_success($products);
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for notifying supplier
     *
     * @since 1.0.0
     */
    public function ajax_notify_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $product_id = absint($_POST['product_id']);
        $supplier_ids = array();

        if (isset($_POST['supplier_ids']) && is_array($_POST['supplier_ids'])) {
            $supplier_ids = array_map('absint', $_POST['supplier_ids']);
            // Remove any zero values (invalid IDs)
            $supplier_ids = array_filter($supplier_ids);
        }

        if (!$product_id) {
            wp_send_json_error('Invalid product ID');
        }

        try {
            // Delegate to WISESTOCK_Supplier_Manager
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $result = $supplier_manager->notify_suppliers($product_id, $supplier_ids);

            if ($result['success']) {
                wp_send_json_success($result['message']);
            } else {
                wp_send_json_error($result['message']);
            }
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for sending test email
     *
     * @since 1.0.0
     */
    public function ajax_send_test_email() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $test_email = sanitize_email($_POST['test_email']);
        $email_subject = sanitize_text_field($_POST['email_subject']);
        $from_email = sanitize_email($_POST['from_email']);

        if (!$test_email) {
            wp_send_json_error('Please enter a valid email address');
        }

        try {
            // Create a test supplier object
            $test_supplier = (object) array(
                'name' => 'Test Supplier',
                'email' => $test_email,
                'lead_time_days' => 7
            );

            // Create test product data
            $test_product = (object) array(
                'name' => 'Test Product',
                'sku' => 'TEST-001',
                'stock_quantity' => 5
            );

            // Create test recommendation data
            $test_recommendation = array(
                'recommended_quantity' => 25,
                'urgency' => 'warning',
                'days_to_stockout' => 3,
                'daily_velocity' => 1.5,
                'reasoning' => 'This is a test email to verify the email template functionality.'
            );

            // Use the supplier manager to send the test email
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            
            // Prepare email data
            $email_data = array(
                'supplier_name' => $test_supplier->name,
                'supplier_email' => $test_supplier->email,
                'product_name' => $test_product->name,
                'product_sku' => $test_product->sku,
                'current_stock' => $test_product->stock_quantity,
                'recommended_quantity' => $test_recommendation['recommended_quantity'],
                'urgency' => $test_recommendation['urgency'],
                'days_to_stockout' => $test_recommendation['days_to_stockout'],
                'daily_velocity' => $test_recommendation['daily_velocity'],
                'lead_time' => $test_supplier->lead_time_days,
                'reasoning' => $test_recommendation['reasoning'],
                'site_name' => get_bloginfo('name'),
                'admin_email' => get_option('admin_email')
            );

            // Use the supplier template method with demo data
            $body = $supplier_manager->get_supplier_email_template($email_data);

            // Use custom subject if provided, otherwise use default
            if (!empty($email_subject)) {
                $subject = $email_subject;
                $subject = str_replace('{site_name}', $email_data['site_name'], $subject);
                $subject = str_replace('{urgency}', strtoupper($email_data['urgency']), $subject);
                $subject = str_replace('{product_name}', $email_data['product_name'], $subject);
                $subject = 'TEST - ' . $subject;
            } else {
                $subject = sprintf('[%s] TEST - Restock Request for %s', $email_data['site_name'], $email_data['product_name']);
            }

            // Set headers with custom from email if provided
            $headers = array('Content-Type: text/html; charset=UTF-8');
            if (!empty($from_email)) {
                $headers[] = 'From: ' . $from_email;
            }

            $result = wp_mail($test_email, $subject, $body, $headers);

            if ($result) {
                wp_send_json_success('Test email sent successfully to ' . $test_email);
            } else {
                wp_send_json_error('Failed to send test email. Please check your WordPress email configuration.');
            }
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for sending test notification email
     */
    public function ajax_send_test_notification_email() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        try {
            // Use WordPress admin email for test
            $admin_email = get_option('admin_email');

            if (!$admin_email || !is_email($admin_email)) {
                wp_send_json_error(__('Please configure a valid admin email address first', 'wisestock'));
            }

            // Create test notification data (simulating real product data)
            $test_data = array(
                'site_name' => get_bloginfo('name'),
                'product_name' => __('Test Product - Wireless Headphones', 'wisestock'),
                'product_sku' => 'TEST-WH-001',
                'product_id' => 99999, // Test product ID
                'current_stock' => 3,
                'days_to_stockout' => 2,
                'notification_type' => 'warning',
                'urgency_text' => __('WARNING', 'wisestock'),
                'admin_email' => $admin_email,
                'timestamp' => current_time('Y-m-d H:i:s')
            );

            // Generate test notification message (same format as real notifications)
            /* translators: 1: Urgency text. 2: Product name. 3: Product SKU. 4: Days to stockout. 5: Current stock quantity. */
            $test_message = sprintf(__('[%1$s] %2$s - SKU: %3$s will stockout in %4$d days. Current stock: %5$d units.', 'wisestock'), $test_data['urgency_text'], $test_data['product_name'], $test_data['product_sku'], $test_data['days_to_stockout'], $test_data['current_stock']);

            // Create email subject (same format as real notifications)
            /* translators: 1: Site name. 2: Notification type label. */
            $subject = sprintf(__('[%1$s] TEST - Wise Stock Notification - %2$s', 'wisestock'), $test_data['site_name'], ucfirst($test_data['notification_type']));

            // Allow filtering of test email subject
            $subject = apply_filters('wisestock_test_email_subject', $subject, $test_data);

            // Create a proper mock product class for testing
            $mock_product = new class($test_data) {
                private $data;
                
                public function __construct($data) {
                    $this->data = $data;
                }
                
                public function get_name() {
                    return $this->data['product_name'];
                }
                
                public function get_id() {
                    return $this->data['product_id'];
                }
                
                public function get_sku() {
                    return $this->data['product_sku'];
                }
                
                public function get_stock_quantity() {
                    return $this->data['current_stock'];
                }
                
                public function managing_stock() {
                    return true;
                }
            };

            // Use the actual email template method from notification manager
            $notification_manager = new WISESTOCK_Notification_Manager();
            
            // Add filter to modify URLs for test emails
            add_filter('wisestock_email_template_vars', function($template_vars, $product, $notification_type, $message) {
                // Set empty URLs for test emails
                $template_vars['product_url'] = '';
                $template_vars['dashboard_url'] = '';
                return $template_vars;
            }, 10, 4);
            
            $body = $notification_manager->get_test_email_template($mock_product, $test_data['notification_type'], $test_message);
            
            // Remove the filter after use
            remove_all_filters('wisestock_email_template_vars');

            // Allow filtering of test email body
            $body = apply_filters('wisestock_test_email_body', $body, $test_data);

            // Set headers
            $headers = array('Content-Type: text/html; charset=UTF-8');

            // Allow filtering of test email headers
            $headers = apply_filters('wisestock_test_email_headers', $headers, $test_data);

            // Send the test email
            $result = wp_mail($admin_email, $subject, $body, $headers);

            if ($result) {
                /* translators: %s: Admin email address. */
                wp_send_json_success(sprintf(esc_html__('Test notification email sent successfully to %s', 'wisestock'), $admin_email));
            } else {
                wp_send_json_error(esc_html__('Failed to send test notification email. Please check your WordPress email configuration.', 'wisestock'));
            }
        } catch (Exception $e) {
            /* translators: %s: Error message. */
            wp_send_json_error(sprintf(esc_html__('Error: %s', 'wisestock'), $e->getMessage()));
        }
    }

    /**
     * AJAX handler for getting product suppliers
     *
     * @since 1.0.0
     */
    public function ajax_get_product_suppliers() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $product_id = absint($_POST['product_id']);
        if (!$product_id) {
            wp_send_json_error('Invalid product ID');
        }

        try {
            // Delegate to WISESTOCK_Supplier_Manager
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $product_suppliers = $supplier_manager->get_product_suppliers($product_id);

            wp_send_json_success($product_suppliers);
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for setting default supplier
     *
     * @since 1.0.0
     */
    public function ajax_set_default_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $product_id = absint($_POST['product_id']);
        $supplier_id = absint($_POST['supplier_id']);

        if (!$product_id) {
            wp_send_json_error('Invalid product ID');
        }

        try {
            // Delegate to WISESTOCK_Supplier_Manager
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            
            if ($supplier_id > 0) {
                $result = $supplier_manager->set_default_supplier_for_product($product_id, $supplier_id);
                if ($result) {
                    wp_send_json_success('Default supplier set successfully');
                } else {
                    wp_send_json_error('Failed to set default supplier');
                }
            } else {
                // Remove default supplier
                delete_post_meta($product_id, '_wisestock_default_supplier');
                wp_send_json_success('Default supplier removed successfully');
            }
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for getting single supplier
     *
     * @since 1.0.0
     */
    public function ajax_get_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_id = absint($_POST['supplier_id']);
        if (!$supplier_id) {
            wp_send_json_error('Invalid supplier ID');
        }

        try {
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $supplier = $supplier_manager->get_supplier($supplier_id);

            if (!$supplier) {
                wp_send_json_error('Supplier not found');
            }

            wp_send_json_success($supplier);
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for updating supplier
     *
     * @since 1.0.0
     */
    public function ajax_update_supplier() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_id = absint($_POST['supplier_id']);
        if (!$supplier_id) {
            wp_send_json_error('Invalid supplier ID');
        }

        // Get and sanitize form data
        $product_ids = array();
        if (isset($_POST['supplier_products']) && !empty($_POST['supplier_products'])) {
            if (is_array($_POST['supplier_products'])) {
                $product_ids = array_map('absint', $_POST['supplier_products']);
                // Remove any zero values (invalid IDs)
                $product_ids = array_filter($product_ids);
            }
        }

        $supplier_data = array(
            'name' => sanitize_text_field($_POST['supplier_name'] ?? ''),
            'email' => sanitize_email($_POST['supplier_email'] ?? ''),
            'lead_time_days' => absint($_POST['supplier_lead_time'] ?? 7),
            'product_ids' => $product_ids
        );

        // Validate required fields
        if (empty($supplier_data['name'])) {
            wp_send_json_error('Supplier name is required');
        }

        if (!empty($supplier_data['email']) && !is_email($supplier_data['email'])) {
            wp_send_json_error('Invalid email address');
        }

        if ($supplier_data['lead_time_days'] < 0) {
            wp_send_json_error('Lead time must be a positive number');
        }

        try {
            $supplier_manager = new WISESTOCK_Supplier_Manager();
            $result = $supplier_manager->update_supplier($supplier_id, $supplier_data);

            if ($result) {
                wp_send_json_success(array(
                    'message' => 'Supplier updated successfully',
                    'supplier_id' => $supplier_id
                ));
            } else {
                wp_send_json_error('Failed to update supplier');
            }
        } catch (Exception $e) {
            wp_send_json_error('Failed to update supplier: ' . $e->getMessage());
        }
    }

    /**
     * Admin installation page
     *
     * @since 1.0.0
     */
    public function admin_installation_page() {
        // Enqueue installation scripts and styles
        wp_enqueue_script('wisestock-installation', WISESTOCK_PLUGIN_URL . 'admin/js/wisestock-installation.js', array('jquery'), WISESTOCK_VERSION, true);
        
        // Also enqueue wisestock-admin.js to provide templates and localization
        wp_enqueue_script('wisestock-admin', WISESTOCK_PLUGIN_URL . 'admin/js/wisestock-admin.js', array('jquery'), WISESTOCK_VERSION, true);

        // Get existing settings
        $defaults = self::get_default_options();
        $existing_options = wp_parse_args(get_option('wisestock_options', array()), $defaults);

        // Get timebound preset data (timebound is now the only algorithm)
        $timebound_settings = WISESTOCK_Timebound_Settings::get_instance();
        $timebound_store_type_presets = $timebound_settings->get_store_type_presets();

        // Localize script with AJAX data and existing settings
        wp_localize_script('wisestock-installation', 'wisestock_admin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wisestock_admin_nonce'),
            // Ensure consistent URLs for navigation
            'installation_url' => admin_url('admin.php?page=wisestock-installation'),
            'dashboard_url' => admin_url('admin.php?page=wisestock-dashboard'),
            'settings_url' => admin_url('admin.php?page=wisestock-settings'),
            'existing_settings' => array(
                'store_preset' => isset($existing_options['store_type']) ? $existing_options['store_type'] : 'default',
                'demo_stats' => isset($existing_options['demo_stats']) ? $existing_options['demo_stats'] : false
            ),
            'presets' => array(
                'timebound' => $timebound_store_type_presets,
            ),
            'strings' => array(
                'retail_store' => __('Retail Store', 'wisestock'),
                'dropshipping' => __('Dropshipping', 'wisestock'),
                'digital_products' => __('Digital Products', 'wisestock'),
                'mixed_store' => __('Mixed Store', 'wisestock'),
                'classic_predictions' => __('Classic Predictions', 'wisestock'),
                'timebound_predictions' => __('Time-Bound Predictions', 'wisestock'),
                'yes' => __('Yes', 'wisestock'),
                'no' => __('No', 'wisestock'),
                'saving' => __('Saving...', 'wisestock'),
                'saved' => __('Saved!', 'wisestock'),
                'error_try_again' => __('Error - try again', 'wisestock'),
                'redirecting' => __('Redirecting...', 'wisestock'),
                'completing_setup' => __('Completing Setup...', 'wisestock'),
                'please_select_option' => __('Please select an option to continue', 'wisestock'),
                'installation_failed' => __('Installation failed', 'wisestock'),
                'network_error_occurred' => __('Network error occurred', 'wisestock')
            ),
            'templates' => array(
                'no_notifications' => '<div class="wisestock-no-notifications">' . __('No notifications at this time', 'wisestock') . '</div>',
                'notification_item' => '<div class="wisestock-notification-item {type}"></div>',
                'notification_content' => '<div class="wisestock-notification-content"><h4 class="wisestock-notification-title">{product_name}</h4><p class="wisestock-notification-message">{message}</p><div class="wisestock-notification-time">{time_ago}</div></div><div class="wisestock-notification-actions"><button class="wisestock-btn small wisestock-notification-resolve" data-notification-id="{id}">' . __('Resolve', 'wisestock') . '</button><button class="wisestock-btn small secondary wisestock-notification-dismiss" data-notification-id="{id}">' . __('Dismiss', 'wisestock') . '</button></div>',
                'preset_card' => '<div class="wisestock-preset-card{active}" data-value="{key}"><h3 class="wisestock-preset-title">{name}</h3><p class="wisestock-preset-description">{description}</p></div>',
                'feature_item' => '<div class="wisestock-feature-item">{feature}</div>',
                'success_message' => '<div class="wisestock-success-message"><div class="wisestock-success-icon"><span class="dashicons dashicons-yes"></span></div><div class="wisestock-success-content"><h3>' . __('Setup Complete!', 'wisestock') . '</h3><p>' . __('Your Wise Stock is now configured and ready to use.', 'wisestock') . '</p><p>' . __('Redirecting to dashboard...', 'wisestock') . '</p></div></div>',
                'installation_notice' => '<div class="wisestock-notice wisestock-notice-{type}"><span class="dashicons dashicons-{icon}"></span><span class="wisestock-notice-message">{message}</span><button type="button" class="wisestock-notice-dismiss"><span class="dashicons dashicons-dismiss"></span></button></div>'
            )
        ));

        include WISESTOCK_PLUGIN_DIR . 'admin/partials/installation.php';
    }

    /**
     * AJAX handler for completing installation
     *
     * @since 1.0.0
     */
    public function ajax_complete_installation() {

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check user capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Get installation data
        $installation_data = isset($_POST['installation_data']) ? $_POST['installation_data'] : array();

        if (empty($installation_data)) {
            wp_send_json_error('No installation data provided');
        }

        try {
            // Get current options
            $defaults = self::get_default_options();
            $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);

            // Update store_type with the selected preset (timebound is now the only algorithm)
            $store_preset = sanitize_text_field($installation_data['store_preset']);
            $options['store_type'] = $store_preset;

            // Apply the timebound store type preset
            if ($this->timebound_settings) {
                $this->timebound_settings->apply_store_type_preset($store_preset);
            }

            // Save updated options
            update_option('wisestock_options', $options);

            // Mark installation as completed
            update_option('wisestock_installation_completed', true);

            // Set installation timestamp
            update_option('wisestock_installation_date', current_time('mysql'));

            // Prepare response data
            $response_data = array(
                'message' => 'Installation completed successfully',
                'redirect_url' => admin_url('admin.php?page=wisestock-dashboard')
            );

            wp_send_json_success($response_data);

        } catch (Exception $e) {
            wp_send_json_error('Installation failed: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for saving demo stats setting
     *
     * @since 1.0.0
     */
    public function ajax_save_demo_stats() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check user capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Get demo stats value
        if (isset($_POST['demo_stats'])) {
            $demo_stats = ($_POST['demo_stats'] == 'true') ? true : false;
        } else {
            $demo_stats = false;
        }

        // Get current options and update demo_stats
        $defaults = self::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        $options['demo_stats'] = $demo_stats;
        update_option('wisestock_options', $options);


        // Send success response
        wp_send_json_success(array(
            'message' => 'Demo stats setting saved successfully',
            'demo_stats' => $demo_stats
        ));
    }

    /**
     * Admin test page
     *
     * @since 1.0.1
     */

    /**
     * Get store type presets (now returns timebound presets as they're the only algorithm)
     *
     * @return array
     * @since 1.0.0
     * @deprecated 1.0.1 Use WISESTOCK_Timebound_Settings::get_store_type_presets() instead
     */
    public static function get_classic_store_type_presets() {
        // Return timebound presets for backward compatibility
        $timebound_settings = WISESTOCK_Timebound_Settings::get_instance();
        $timebound_presets = $timebound_settings->get_store_type_presets();
        
        // Convert to classic format for backward compatibility
        $classic_format = array();
        foreach ( $timebound_presets as $key => $preset ) {
            $classic_format[$key] = array(
                'name' => $preset['name'],
                'description' => $preset['description'],
                'velocity_weights' => array(
                    '7_day' => 0.5,
                    '14_day' => 0.3,
                    '30_day' => 0.2
                ),
                'confidence_settings' => array(
                    'min_confidence' => $preset['settings']['min_confidence_threshold'],
                    'sparse_data_confidence' => max(0.2, $preset['settings']['min_confidence_threshold'] - 0.1),
                    'good_data_min_confidence' => min(0.9, $preset['settings']['min_confidence_threshold'] + 0.1)
                )
            );
        }
        
        return $classic_format;
    }

    /**
     * Get velocity weights and confidence settings based on store type
     *
     * This middleware method retrieves the appropriate velocity weights and confidence settings
     * from the store type presets based on the currently selected store type in options.
     *
     * @param string $store_type Optional. Store type to get settings for. If not provided, uses current option.
     * @return array Array containing 'velocity_weights' and 'confidence_settings'
     * @since 1.0.0
     */
    public static function get_store_type_settings($store_type = null) {
        // Get store type from options if not provided
        if (empty($store_type)) {
            $defaults = self::get_default_options();
            $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
            $store_type = $options['store_type'];
        }

        // Get timebound settings instance
        $timebound_settings = WISESTOCK_Timebound_Settings::get_instance();
        $presets = $timebound_settings->get_store_type_presets();

        // Return default if store type not found
        if (!isset($presets[$store_type])) {
            $store_type = 'default'; // fallback to default
        }

        $preset = $presets[$store_type];

        // Convert timebound settings to classic format for backward compatibility
        return array(
            'velocity_weights' => array(
                '7_day' => 0.5,
                '14_day' => 0.3,
                '30_day' => 0.2
            ),
            'confidence_settings' => array(
                'min_confidence' => $preset['settings']['min_confidence_threshold'],
                'sparse_data_confidence' => $preset['settings']['min_confidence_threshold'] - 0.1,
                'good_data_min_confidence' => $preset['settings']['min_confidence_threshold'] + 0.1
            )
        );
    }

    /**
     * Get velocity weights based on store type
     *
     * @param string $store_type Optional. Store type to get settings for. If not provided, uses current option.
     * @return array Velocity weights array
     * @since 1.0.0
     */
    public static function get_velocity_weights_by_store_type($store_type = null) {
        $settings = self::get_store_type_settings($store_type);
        return $settings['velocity_weights'];
    }

    /**
     * Get confidence settings based on store type
     *
     * @param string $store_type Optional. Store type to get settings for. If not provided, uses current option.
     * @return array Confidence settings array
     * @since 1.0.0
     */
    public static function get_confidence_settings_by_store_type($store_type = null) {
        $settings = self::get_store_type_settings($store_type);
        return $settings['confidence_settings'];
    }

    /**
     * AJAX handler for getting supplier statistics
     *
     * @since 1.0.0
     */
    public function ajax_get_supplier_stats() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check capability
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Delegate to WISESTOCK_Supplier_Manager
        $supplier_manager = new WISESTOCK_Supplier_Manager();
        $stats = $supplier_manager->get_supplier_stats();

        wp_send_json_success($stats);
    }

    /**
     * AJAX handler for getting suppliers table
     *
     * @since 1.0.0
     */
    public function ajax_get_suppliers_table() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check capability
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_manager = new WISESTOCK_Supplier_Manager();
        $suppliers = $supplier_manager->get_all_suppliers();

        if (empty($suppliers)) {
            wp_send_json_success('<div class="wisestock-no-data">' . __('No suppliers found. Add your first supplier to get started.', 'wisestock') . '</div>');
            return;
        }

        $table_html = '<div class="wisestock-table-container">';
        $table_html .= '<table class="wisestock-table">';
        $table_html .= '<thead>';
        $table_html .= '<tr>';
        $table_html .= '<th>' . __('Name', 'wisestock') . '</th>';
        $table_html .= '<th>' . __('Email', 'wisestock') . '</th>';
        $table_html .= '<th>' . __('Lead Time', 'wisestock') . '</th>';
        $table_html .= '<th>' . __('Products', 'wisestock') . '</th>';
        $table_html .= '<th>' . __('Actions', 'wisestock') . '</th>';
        $table_html .= '</tr>';
        $table_html .= '</thead>';
        $table_html .= '<tbody>';

        foreach ($suppliers as $supplier) {
            $product_ids = maybe_unserialize($supplier->product_ids);
            $product_count = is_array($product_ids) ? count($product_ids) : 0;
            
            $email_html = !empty($supplier->email) ? 
                '<a href="mailto:' . esc_attr($supplier->email) . '">' . esc_html($supplier->email) . '</a>' :
                '<span class="wisestock-no-data">' . __('No email', 'wisestock') . '</span>';

            $table_html .= '<tr>';
            $table_html .= '<td><strong>' . esc_html($supplier->name) . '</strong></td>';
            $table_html .= '<td>' . $email_html . '</td>';
            $table_html .= '<td>' . intval($supplier->lead_time_days) . ' days</td>';
            $table_html .= '<td><span class="wisestock-product-count">' . $product_count . '</span></td>';
            $table_html .= '<td>';
            $table_html .= '<div class="wisestock-btn-group">';
            $table_html .= '<button type="button" class="wisestock-btn small wisestock-edit-supplier" data-supplier-id="' . intval($supplier->id) . '">' . __('Edit', 'wisestock') . '</button>';
            $table_html .= '<button type="button" class="wisestock-btn small danger wisestock-delete-supplier" data-supplier-id="' . intval($supplier->id) . '">' . __('Delete', 'wisestock') . '</button>';
            $table_html .= '</div>';
            $table_html .= '</td>';
            $table_html .= '</tr>';
        }

        $table_html .= '</tbody>';
        $table_html .= '</table>';
        $table_html .= '</div>';

        wp_send_json_success($table_html);
    }

    /**
     * AJAX handler for getting product-supplier overview
     *
     * @since 1.0.0
     */
    public function ajax_get_product_supplier_overview() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check capability
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_manager = new WISESTOCK_Supplier_Manager();
        $suppliers = $supplier_manager->get_all_suppliers();

        if (empty($suppliers)) {
            wp_send_json_success('<div class="wisestock-no-data">' . __('No suppliers found. Add suppliers and assign products to see the overview.', 'wisestock') . '</div>');
            return;
        }

        // Build product-supplier mapping
        $product_suppliers = array();
        foreach ($suppliers as $supplier) {
            $product_ids = maybe_unserialize($supplier->product_ids);
            if (is_array($product_ids)) {
                foreach ($product_ids as $product_id) {
                    if (!isset($product_suppliers[$product_id])) {
                        $product_suppliers[$product_id] = array();
                    }
                    $product_suppliers[$product_id][] = $supplier;
                }
            }
        }

        if (empty($product_suppliers)) {
            wp_send_json_success('<div class="wisestock-no-data">' . __('No products assigned to suppliers yet.', 'wisestock') . '</div>');
            return;
        }

        $overview_html = '<div class="wisestock-product-supplier-grid">';
        
        foreach ($product_suppliers as $product_id => $suppliers_for_product) {
            $product = wc_get_product($product_id);
            if (!$product) {
                continue;
            }

            $overview_html .= '<div class="wisestock-product-supplier-item" data-product-id="' . intval($product_id) . '">';
            $overview_html .= '<div class="wisestock-product-info">';
            $overview_html .= '<h4 class="wisestock-product-name">';
            $overview_html .= '<a href="' . esc_url(get_edit_post_link($product_id)) . '">' . esc_html($product->get_name()) . '</a>';
            $overview_html .= '</h4>';
            $overview_html .= '<p class="wisestock-product-sku">' . __('SKU', 'wisestock') . ': ' . esc_html($product->get_sku() ?: 'N/A') . '</p>';
            $overview_html .= '</div>';
            $overview_html .= '<div class="wisestock-suppliers-list">';
            $overview_html .= '<h5>' . __('Suppliers', 'wisestock') . ':</h5>';
            $overview_html .= '<ul>';

            foreach ($suppliers_for_product as $supplier) {
                $overview_html .= '<li class="wisestock-supplier-badge">';
                $overview_html .= '<span class="supplier-name">' . esc_html($supplier->name) . '</span>';
                if (!empty($supplier->email)) {
                    $overview_html .= '<span class="supplier-contact">';
                    $overview_html .= '<a href="mailto:' . esc_attr($supplier->email) . '">' . esc_html($supplier->email) . '</a>';
                    $overview_html .= '</span>';
                }
                $overview_html .= '</li>';
            }

            $overview_html .= '</ul>';
            $overview_html .= '</div>';
            $overview_html .= '</div>';
        }

        $overview_html .= '</div>';

        wp_send_json_success($overview_html);
    }

    /**
     * AJAX handler for getting multi-supplier products
     *
     * @since 1.0.0
     */
    public function ajax_get_multi_supplier_products() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check capability
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $supplier_manager = new WISESTOCK_Supplier_Manager();
        $suppliers = $supplier_manager->get_all_suppliers();

        // Build product-supplier mapping
        $product_suppliers = array();
        foreach ($suppliers as $supplier) {
            $product_ids = maybe_unserialize($supplier->product_ids);
            if (is_array($product_ids)) {
                foreach ($product_ids as $product_id) {
                    if (!isset($product_suppliers[$product_id])) {
                        $product_suppliers[$product_id] = array();
                    }
                    $product_suppliers[$product_id][] = $supplier;
                }
            }
        }

        // Filter products with multiple suppliers
        $multi_supplier_products = array_filter($product_suppliers, function($suppliers) {
            return count($suppliers) > 1;
        });

        if (empty($multi_supplier_products)) {
            wp_send_json_success('<div class="wisestock-no-data">' . __('No products have multiple suppliers assigned.', 'wisestock') . '</div>');
            return;
        }

        $products_html = '<div class="wisestock-multi-supplier-list">';
        
        foreach ($multi_supplier_products as $product_id => $suppliers_for_product) {
            $product = wc_get_product($product_id);
            if (!$product) {
                continue;
            }

            // Get the current default supplier for this product
            $default_supplier_id = get_post_meta($product_id, '_wisestock_default_supplier', true);

            $products_html .= '<div class="wisestock-multi-supplier-item">';
            $products_html .= '<div class="wisestock-product-info">';
            $products_html .= '<h4 class="wisestock-product-name">';
            $products_html .= '<a href="' . esc_url(get_edit_post_link($product_id)) . '">' . esc_html($product->get_name()) . '</a>';
            $products_html .= '</h4>';
            $products_html .= '<p class="wisestock-product-sku">' . __('SKU', 'wisestock') . ': ' . esc_html($product->get_sku() ?: 'N/A') . '</p>';
            $products_html .= '</div>';
            $products_html .= '<div class="wisestock-default-supplier">';
            $products_html .= '<label for="default_supplier_' . intval($product_id) . '">';
            $products_html .= __('Default Supplier', 'wisestock') . ':';
            $products_html .= '</label>';
            $products_html .= '<select id="default_supplier_' . intval($product_id) . '"';
            $products_html .= ' class="wisestock-form-select wisestock-default-supplier-select"';
            $products_html .= ' data-product-id="' . intval($product_id) . '">';
            $products_html .= '<option value="">' . __('Select Default Supplier', 'wisestock') . '</option>';

            foreach ($suppliers_for_product as $supplier) {
                $selected = selected($default_supplier_id, $supplier->id, false);
                $products_html .= '<option value="' . intval($supplier->id) . '"' . $selected . '>';
                $products_html .= esc_html($supplier->name);
                $products_html .= '</option>';
            }

            $products_html .= '</select>';
            $products_html .= '<div class="wisestock-form-help">';
            $products_html .= __('This supplier will be pre-selected when using Notify Supplier', 'wisestock');
            $products_html .= '</div>';
            $products_html .= '</div>';
            $products_html .= '</div>';
        }

        $products_html .= '</div>';

        wp_send_json_success($products_html);
    }

    /**
     * AJAX handler for getting timebound settings
     *
     * @since 1.0.1
     */
    public function ajax_get_timebound_settings() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wisestock_timebound_nonce')) {
            wp_send_json_error('Security check failed');
        }

        // Check capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        if ($this->timebound_settings) {
            wp_send_json_success($this->timebound_settings->get_settings());
        } else {
            wp_send_json_error('Timebound settings not available');
        }
    }



    /**
     * Get the velocity calculator (unified, includes timebound features)
     *
     * @return WISESTOCK_Velocity_Calculator
     * @since 1.0.0
     */
    public function get_velocity_calculator() {
        return $this->velocity_calculator;
    }

    /**
     * Get the standard velocity calculator (deprecated - returns unified calculator)
     * 
     * @deprecated 1.0.2 Use get_velocity_calculator() instead
     * @return WISESTOCK_Velocity_Calculator
     * @since 1.0.1
     */
    public function get_standard_velocity_calculator() {
        return $this->velocity_calculator;
    }



    /**
     * Get the timebound velocity calculator (deprecated - use get_velocity_calculator)
     *
     * @deprecated 1.0.2 Use get_velocity_calculator() instead
     * @return WISESTOCK_Velocity_Calculator|null
     * @since 1.0.1
     */
    public function get_timebound_velocity_calculator() {
        return $this->velocity_calculator;
    }

    /**
     * Get timebound settings instance
     *
     * @return WISESTOCK_Timebound_Settings
     * @since 1.0.1
     */
    public function get_timebound_settings() {
        return $this->timebound_settings;
    }

    /**
     * Check if timebound features are enabled
     * 
     * Timebound predictions are now the default and only prediction method
     *
     * @return bool Always true since timebound is the only algorithm
     * @since 1.0.1
     */
    public function is_timebound_enabled() {
        // Timebound predictions are now the default and only option
        return true;
    }

    /**
     * Enable timebound features
     *
     * @since 1.0.1
     */
    public function enable_timebound() {
        if ( $this->timebound_settings ) {
            $this->timebound_settings->enable();
        }
    }

    /**
     * Disable timebound features
     *
     * @since 1.0.1
     */
    public function disable_timebound() {
        if ( $this->timebound_settings ) {
            $this->timebound_settings->disable();
        }
    }

    /**
     * Get timebound velocity data for a product
     *
     * @param int $product_id Product ID
     * @param int $prediction_window_days Optional prediction window
     * @return array|false Timebound velocity data or false if not available
     * @since 1.0.1
     */
    public function get_timebound_velocity( $product_id, $prediction_window_days = null ) {
        if ( ! $this->velocity_calculator ) {
            return false;
        }
        
        return $this->velocity_calculator->get_timebound_velocity( $product_id, $prediction_window_days );
    }

    /**
     * Get enhanced restock recommendation using timebound analysis
     *
     * @param int $product_id Product ID
     * @param int $preferred_days Optional preferred prediction window
     * @return array Enhanced recommendation or standard recommendation
     * @since 1.0.1
     */
    public function get_enhanced_restock_recommendation( $product_id, $preferred_days = null ) {
        if ( $this->velocity_calculator ) {
            return $this->velocity_calculator->get_smart_stock_recommendation( $product_id, $preferred_days );
        }
        
        // Fallback to base recommendation
        return $this->velocity_calculator->get_restock_recommendation( $product_id );
    }



    /**
     * Get intelligent prediction windows for a product
     *
     * @param int $product_id Product ID
     * @return array|false Prediction windows or false if not available
     * @since 1.0.1
     */
    public function get_intelligent_prediction_windows( $product_id ) {
        if ( ! $this->velocity_calculator ) {
            return false;
        }
        
        return $this->velocity_calculator->get_intelligent_prediction_windows( $product_id );
    }

    /**
     * Enhance velocity data with timebound predictions, ensuring realistic correlations
     * 
     * This method generates correlated volatility, trend direction, and market sensitivity
     * to avoid unrealistic combinations like "5% volatility + 40% market change".
     * 
     * Correlation logic:
     * - Stable trends: Low volatility (2-10%), low market sensitivity (15-45%)
     * - Changing trends: Higher volatility (5-35%), higher market sensitivity (30-85%)
     * - Velocity influences trend probability and sensitivity ranges
     * - Confidence decay correlates directly with volatility
     * 
     * @param array $velocity_item Base velocity data
     * @param object $product Product object for storing trend data
     * @param bool $enhanced_enabled Whether timebound features are enabled
     * @return array Enhanced velocity data with realistic correlations
     * @since 1.0.1
     */
    public function enhance_velocity_with_timebound( $velocity_item, $product, $enhanced_enabled ) {
        if ( ! $this->velocity_calculator ) {
            return $velocity_item;
        }

        // Get current settings to derive realistic values
        $options = get_option( 'wisestock_options', array() );
        $defaults = self::get_default_options();
        $store_type = $options['store_type'] ?? $defaults['store_type'];
        
        // Realistic prediction windows based on velocity and category
        $velocity = $velocity_item['velocity'];
        $prediction_windows = array(
            'high' => array( 7, 10, 14 ),    // High velocity: shorter windows
            'medium' => array( 14, 21, 30 ), // Medium velocity: medium windows
            'low' => array( 30, 45 )         // Low velocity: longer windows
        );
        
        if ( $velocity >= 4.0 ) {
            $window_options = $prediction_windows['high'];
            $velocity_category = 'high';
        } elseif ( $velocity >= 1.5 ) {
            $window_options = $prediction_windows['medium'];
            $velocity_category = 'medium';
        } else {
            $window_options = $prediction_windows['low'];
            $velocity_category = 'low';
        }
        
        // Randomly select from appropriate range
        $prediction_window = $window_options[ array_rand( $window_options ) ];
        
        // Determine trend direction based on velocity (assign to product for market change correlation)
        $trend_options = array( 'increasing', 'stable', 'decreasing' );
        if ( $velocity >= 3.0 ) {
            $trend_weights = array( 40, 40, 20 ); // High velocity: likely growing
        } else {
            $trend_weights = array( 20, 50, 30 ); // Low velocity: likely stable
        }
        
        $rand = mt_rand( 1, 100 );
        if ( $rand <= $trend_weights[0] ) {
            $trend_direction = 'increasing';
        } elseif ( $rand <= $trend_weights[0] + $trend_weights[1] ) {
            $trend_direction = 'stable';
        } else {
            $trend_direction = 'decreasing';
        }
        
        // Store trend on product object for market change correlation
        $product->trend_direction = $trend_direction;
        
        // Calculate volatility based on trend AND velocity with realistic correlations
        // Higher volatility when trend is changing (increasing/decreasing) vs stable
        if ( 'stable' === $trend_direction ) {
            // Low volatility for stable trends - tighter ranges for realism
            if ( 'high' === $velocity_category ) {
                $base_volatility = 0.08; // Even high velocity products can be stable
                $random_adjustment = mt_rand( -1, 2 ) / 100; // ±1-2%
            } elseif ( 'medium' === $velocity_category ) {
                $base_volatility = 0.06;
                $random_adjustment = mt_rand( -1, 1 ) / 100; // ±1%
            } else {
                $base_volatility = 0.04; // Low velocity + stable = very low volatility 
                $random_adjustment = mt_rand( 0, 1 ) / 100; // 0-1%
            }
            $volatility = round( max( 0.02, $base_volatility + $random_adjustment ), 3 );
        } else {
            // Higher volatility for changing trends - but realistic maximums
            if ( 'high' === $velocity_category ) {
                // High velocity + changing trend = significant volatility but capped
                $base_volatility = $trend_direction === 'increasing' ? 0.22 : 0.28; // Decreasing slightly higher
                $random_adjustment = mt_rand( -2, 3 ) / 100; // ±2-3%
            } elseif ( 'medium' === $velocity_category ) {
                $base_volatility = $trend_direction === 'increasing' ? 0.15 : 0.18;
                $random_adjustment = mt_rand( -2, 2 ) / 100; // ±2%
            } else {
                // Low velocity products rarely have extreme volatility
                $base_volatility = $trend_direction === 'increasing' ? 0.10 : 0.12;
                $random_adjustment = mt_rand( -1, 2 ) / 100; // ±1-2%
            }
            $volatility = round( min( 0.35, max( 0.05, $base_volatility + $random_adjustment ) ), 3 );
        }
        
        // Store volatility on product for market change correlation
        $product->volatility = $volatility;
        
        // Generate realistic market sensitivity based on velocity and volatility with proper correlation
        // Market sensitivity should correlate with volatility but within realistic bounds
        if ( $volatility <= 0.08 ) {
            // Low volatility products = low market sensitivity
            $base_sensitivity = 0.25 + ( $velocity * 0.02 ); // Velocity adds minor influence
            $random_adjustment = mt_rand( -3, 3 ) / 100;
        } elseif ( $volatility <= 0.18 ) {
            // Medium volatility = medium market sensitivity
            $base_sensitivity = 0.40 + ( $volatility * 1.2 ) + ( $velocity * 0.03 );
            $random_adjustment = mt_rand( -4, 4 ) / 100;
        } else {
            // High volatility = high market sensitivity but capped realistically
            $base_sensitivity = 0.55 + ( $volatility * 1.0 ) + ( $velocity * 0.04 );
            $random_adjustment = mt_rand( -5, 3 ) / 100; // Slight bias toward lower end for realism
        }
        $market_sensitivity = round( min( 0.85, max( 0.15, $base_sensitivity + $random_adjustment ) ), 2 );
        
        // Confidence decay rate correlates directly with volatility for realistic behavior
        if ( $volatility > 0.25 ) {
            $base_decay = 0.12 + ( $volatility * 0.2 ); // Higher volatility = faster decay
        } elseif ( $volatility > 0.12 ) {
            $base_decay = 0.08 + ( $volatility * 0.3 ); // Medium correlation
        } else {
            $base_decay = 0.04 + ( $volatility * 0.4 ); // Low volatility = slow decay
        }
        // Add small random variation but keep within reasonable bounds
        $random_adjustment = mt_rand( -1, 1 ) / 100;
        $confidence_decay = round( min( 0.20, max( 0.03, $base_decay + $random_adjustment ) ), 3 );
        
        // Add timebound prediction data
        $velocity_item['prediction_window'] = $prediction_window;
        $velocity_item['confidence_decay'] = $confidence_decay;
        $velocity_item['market_sensitivity'] = $market_sensitivity;
        $velocity_item['trend_direction'] = $trend_direction;
        $velocity_item['volatility'] = $volatility;
        
        return $velocity_item;
    }

    /**
     * Get timebound velocity data for real products
     *
     * @param mixed $default Default value (null)
     * @param int $product_id Product ID
     * @param array $item Base velocity item
     * @return array|null Enhanced velocity data or null
     * @since 1.0.1
     */
    public function get_timebound_velocity_data( $default, $product_id, $item ) {
        if ( ! $this->velocity_calculator ) {
            return $default;
        }

        $timebound_data = $this->velocity_calculator->get_timebound_velocity( $product_id, 14 );
        
        if ( ! $timebound_data || empty( $timebound_data['velocity'] ) ) {
            return $default;
        }

        return array(
            'prediction_window' => $timebound_data['prediction_window'] ?? 14,
            'confidence' => $timebound_data['confidence_score'] ?? $item['confidence'],
            'confidence_decay' => $timebound_data['confidence_decay_rate'] ?? 0.12,
            'trend_direction' => $timebound_data['trend_direction'] ?? 'stable',
            'volatility' => $timebound_data['volatility'] ?? 0,
        );
    }

    /**
     * Enhance dashboard widget data with timebound information
     *
     * @param mixed $default Default widget data (null)
     * @param string $context Widget context
     * @return array|null Enhanced widget data or null
     * @since 1.0.1
     */
    public function enhance_widget_with_timebound( $default, $context ) {
        // Let the normal widget data processing happen
        return $default;
    }


}
