<?php
/**
 * Wise Stock API Class
 *
 * This class provides a public API for external access to Wise Stock functionality.
 * It exposes key methods for velocity calculations, stockout predictions, and notifications.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Wise Stock API Class
 */
class WISESTOCK_API {

    /**
     * API instance
     *
     * @var WISESTOCK_API
     */
    private static $instance = null;

    /**
     * Velocity calculator instance
     *
     * @var WISESTOCK_Velocity_Calculator
     */
    private $velocity_calculator;

    /**
     * Timebound velocity calculator instance
     *
     * @var WISESTOCK_Timebound_Velocity_Calculator
     */
    private $timebound_velocity_calculator;

    /**
     * Notification manager instance
     *
     * @var WISESTOCK_Notification_Manager
     */
    private $notification_manager;

    /**
     * Get API instance
     *
     * @return WISESTOCK_API
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    public function __construct() {
        $this->velocity_calculator = new WISESTOCK_Velocity_Calculator();
        $this->notification_manager = new WISESTOCK_Notification_Manager();
        
        // Get timebound calculator from core instance
        $core = WISESTOCK_Core::get_instance();
        if ($core) {
            $this->timebound_velocity_calculator = $core->get_timebound_velocity_calculator();
        }
    }

    /**
     * Get velocity for a product over specified days
     *
     * @param int $product_id Product ID
     * @param int $days Number of days to calculate (default: 7)
     * @return array Velocity data with confidence score
     * @since 1.0.0
     */
    public function get_velocity($product_id, $days = 7) {
        return $this->velocity_calculator->calculate_velocity($product_id, $days);
    }

    /**
     * Predict days until stockout for a product
     *
     * @param int $product_id Product ID
     * @return int Days until stockout (0 if no stock, -1 if increasing stock)
     * @since 1.0.0
     */
    public function predict_stockout($product_id) {
        return $this->velocity_calculator->predict_stockout($product_id);
    }

    /**
     * Check notifications for a specific product
     *
     * @param int $product_id Product ID
     * @return array Array of active notifications
     * @since 1.0.0
     */
    public function check_notifications($product_id) {
        return $this->notification_manager->get_product_notifications($product_id);
    }

    /**
     * Get restock recommendation for a product
     *
     * @param int $product_id Product ID
     * @return array Restock recommendation data
     * @since 1.0.0
     */
    public function get_restock_recommendation($product_id) {
        return $this->velocity_calculator->get_restock_recommendation($product_id);
    }

    /**
     * Get weighted velocity for a product
     *
     * @param int $product_id Product ID
     * @return array Weighted velocity data
     * @since 1.0.0
     */
    public function get_weighted_velocity($product_id) {
        return $this->velocity_calculator->calculate_weighted_velocity($product_id);
    }

    /**
     * Get notifications (alias for backward compatibility)
     *
     * @param array $args Optional arguments for filtering notifications
     * @return array Array of notification objects
     * @since 1.0.0
     */
    public function get_notifications( $args = array() ) {
        $notification_type = isset( $args['type'] ) ? $args['type'] : null;
        
        // Note: The notification manager currently only supports active status
        // This method maintains backward compatibility
        return $this->notification_manager->get_all_notifications( $notification_type );
    }

    /**
     * Get all active notifications
     *
     * @param string $notification_type Optional notification type filter
     * @return array Array of notification objects
     * @since 1.0.0
     */
    public function get_all_notifications($notification_type = null) {
        return $this->notification_manager->get_all_notifications($notification_type);
    }

    /**
     * Get notification statistics
     *
     * @return array Notification statistics
     * @since 1.0.0
     */
    public function get_notification_statistics() {
        return $this->notification_manager->get_notification_statistics();
    }

    /**
     * Create a custom notification
     *
     * @param int $product_id Product ID
     * @param string $notification_type Notification type (critical, warning, mild)
     * @param string $message Notification message
     * @return int|false Notification ID on success, false on failure
     * @since 1.0.0
     */
    public function create_notification($product_id, $notification_type, $message) {
        return $this->notification_manager->create_notification($product_id, $notification_type, $message);
    }

    /**
     * Resolve a notification
     *
     * @param int $notification_id Notification ID
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function resolve_notification($notification_id) {
        return $this->notification_manager->resolve_notification($notification_id);
    }

    /**
     * Dismiss a notification
     *
     * @param int $notification_id Notification ID
     * @return bool True on success, false on failure
     * @since 1.0.0
     */
    public function dismiss_notification($notification_id) {
        return $this->notification_manager->dismiss_notification($notification_id);
    }

    /**
     * Get product stock information
     *
     * @param int $product_id Product ID
     * @return array Stock information
     * @since 1.0.0
     */
    public function get_product_stock_info($product_id) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return false;
        }

        $velocity = $this->get_weighted_velocity($product_id);
        $stockout_days = $this->predict_stockout($product_id);
        $recommendation = $this->get_restock_recommendation($product_id);
        $notifications = $this->check_notifications($product_id);

        return array(
            'product_id' => $product_id,
            'product_name' => $product->get_name(),
            'current_stock' => $product->get_stock_quantity(),
            'managing_stock' => $product->managing_stock(),
            'stock_status' => $product->get_stock_status(),
            'velocity' => $velocity,
            'days_to_stockout' => $stockout_days,
            'restock_recommendation' => $recommendation,
            'active_notifications' => $notifications
        );
    }

    /**
     * Get dashboard data (now always uses timebound predictions)
     *
     * @return array Dashboard data with timebound enhancements
     * @since 1.0.0
     */
    public function get_dashboard_data() {
        $defaults = WISESTOCK_Core::get_default_options();
        $options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
        $demo_stats_enabled = $options['demo_stats'];

        if ($demo_stats_enabled) {
            return $this->get_demo_dashboard_data();
        }

        $notification_stats = $this->get_notification_statistics();
        $all_notifications = $this->get_all_notifications();

        // Get top attention products
        $attention_products = array();
        $critical_notifications = $this->get_all_notifications('critical');
        $warning_notifications = $this->get_all_notifications('warning');

        foreach (array_merge($critical_notifications, array_slice($warning_notifications, 0, 5)) as $notification) {
            $product = wc_get_product($notification->product_id);
            if ($product) {
                $attention_products[] = array(
                    'product' => $product,
                    'notification' => $notification,
                    'priority' => $notification->notification_type
                );
            }
        }

        // Get velocity summary - find products that have velocity data AND stock management enabled AND are not digital
        global $wpdb;
        $velocity_table = esc_sql($wpdb->prefix . 'wisestock_velocity');
        $postmeta_table = esc_sql($wpdb->postmeta);

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is already escaped.
        $product_ids_with_velocity = $wpdb->get_col($wpdb->prepare("SELECT DISTINCT v.product_id FROM `{$velocity_table}` v INNER JOIN `{$postmeta_table}` pm ON v.product_id = pm.post_id INNER JOIN `{$postmeta_table}` pm2 ON v.product_id = pm2.post_id WHERE v.quantity_sold > %d AND pm.meta_key = %s AND pm.meta_value = %s AND pm2.meta_key = %s AND pm2.meta_value != %s ORDER BY v.product_id LIMIT %d", 0, '_manage_stock', 'yes', '_virtual', 'yes', 10));

        $velocity_summary = array();
        foreach ($product_ids_with_velocity as $product_id) {
            $product = wc_get_product($product_id);
            if (!$product || $product->get_status() !== 'publish' || !$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
                continue;
            }

            $velocity = $this->get_weighted_velocity($product_id);
            if ($velocity['weighted_velocity'] > 0) {
                $velocity_summary[] = array(
                    'product' => $product,
                    'velocity' => $velocity['weighted_velocity'],
                    'confidence' => $velocity['confidence'] ?? ($velocity['period_velocities']['7_day']['confidence'] ?? 0)
                );
            }
        }

        // Sort by velocity descending
        usort($velocity_summary, function($a, $b) {
            return $b['velocity'] <=> $a['velocity'];
        });

        // Limit to top 10
        $velocity_summary = array_slice($velocity_summary, 0, 10);

        // Get restock recommendations - use products with velocity data AND stock management enabled AND are not digital
        $recommendations = array();
        foreach ($product_ids_with_velocity as $product_id) {
            $product = wc_get_product($product_id);
            if (!$product || $product->get_status() !== 'publish' || !$product->managing_stock() || $product->is_virtual() || $product->is_downloadable()) {
                continue;
            }

            $recommendation = $this->get_restock_recommendation($product_id);
            if ($recommendation['recommended_quantity'] > 0) {
                $recommendations[] = array_merge($recommendation, array(
                    'product' => $product,
                    'product_name' => $product->get_name(),
                    'product_id' => $product->get_id()
                ));
            }
        }

        // Sort by urgency and recommended quantity
        usort($recommendations, function($a, $b) {
            $urgency_order = array('critical' => 3, 'warning' => 2, 'mild' => 1, 'none' => 0);
            $a_urgency = $urgency_order[$a['urgency']] ?? 0;
            $b_urgency = $urgency_order[$b['urgency']] ?? 0;

            if ($a_urgency !== $b_urgency) {
                return $b_urgency <=> $a_urgency;
            }

            return $b['recommended_quantity'] <=> $a['recommended_quantity'];
        });

        // Limit to top 10
        $recommendations = array_slice($recommendations, 0, 10);

        // No market change detection - removed for improved UX

        return array(
            'notification_stats' => $notification_stats,
            'notifications' => array_slice($all_notifications, 0, 10),
            'attention_products' => $attention_products,
            'velocity_summary' => $velocity_summary,
            'recommendations' => $recommendations
        );
    }

    /**
     * Get demo dashboard data using new versioned demo data manager
     * 
     * This method now uses WISESTOCK_Demo_Data_Manager for:
     * - Versioned demo data with JSON algorithm flexibility
     * - Automatic migration between algorithm versions  
     * - Separate demo data independent of WooCommerce products
     * 
     * @return array Demo dashboard data
     * @since 1.0.0
     * @since 1.1.0 Updated to use WISESTOCK_Demo_Data_Manager
     */
    public function get_demo_dashboard_data() {

        // Get demo data (should already be initialized by plugin updater)
        $demo_manager = WISESTOCK_Demo_Data_Manager::get_instance();
        
        // Get demo products for velocity summary
        $demo_products = $demo_manager->get_demo_products();
        
        // If no demo products exist, ensure they're initialized (fallback)
        if ( empty( $demo_products ) ) {
            $demo_manager->initialize_demo_data();
            $demo_products = $demo_manager->get_demo_products();
        }
        
        // Enhanced predictions (timebound) are now always enabled as the only algorithm
        $enhanced_predictions_enabled = true;

        // Get velocity summary from demo products (top 10 by velocity)
        $velocity_summary = array();
        foreach (array_slice($demo_products, 0, 10) as $index => $product) {
            if ($product->daily_velocity > 0) {
                // Create more varied and realistic confidence values
                $confidence_variations = array(0.75, 0.82, 0.68, 0.91, 0.59, 0.87, 0.73, 0.65, 0.88, 0.71);
                $base_confidence = $confidence_variations[$index % count($confidence_variations)];

                // Base velocity data
                $velocity_item = array(
                    'product' => new WISESTOCK_Demo_Product($product),
                    'velocity' => round($product->daily_velocity, 2),
                    'confidence' => round($base_confidence, 3),
                );

                // Hook to enhance velocity data with advanced predictions
                $velocity_item = apply_filters('wisestock_enhance_demo_velocity_data', $velocity_item, $product, $enhanced_predictions_enabled);
                
                $velocity_summary[] = $velocity_item;
            }
        }

        // Sort by velocity descending
        usort($velocity_summary, function($a, $b) {
            return $b['velocity'] <=> $a['velocity'];
        });

        // Limit to top 10
        $velocity_summary = array_slice($velocity_summary, 0, 10);

        // Generate restock recommendations from ALL demo products (not just top 10)
        $recommendations = array();
        foreach ($demo_products as $index => $product) {
            $current_stock = $product->current_stock;
            $daily_velocity = $product->daily_velocity;

            if ($daily_velocity > 0) {
                // Base recommendation data
                $base_recommendation = array(
                    'product_id' => 'demo_' . $product->id,
                    'current_stock' => $current_stock,
                    'daily_velocity' => $daily_velocity
                );
                
                // Hook to provide enhanced recommendation data
                $enhanced_recommendation = apply_filters('wisestock_get_enhanced_restock_recommendation', null, $base_recommendation, $enhanced_predictions_enabled);
                
                if ($enhanced_recommendation) {
                    $recommended_quantity = $enhanced_recommendation['recommended_quantity'];
                    $days_to_stockout = $enhanced_recommendation['days_to_stockout'];
                    $confidence_factor = $enhanced_recommendation['confidence'] ?? 0.75;
                    $prediction_window = $enhanced_recommendation['prediction_window'] ?? 14;
                } else {
                    // Fallback to simple calculation
                    $recommended_quantity = max(0, ($daily_velocity * 14) - $current_stock);
                    $days_to_stockout = $current_stock > 0 ? ceil($current_stock / $daily_velocity) : 0;
                    $confidence_factor = 0.75;
                    $prediction_window = 14;
                }

                if ($recommended_quantity > 0 || $days_to_stockout <= 14) {
                    $urgency = 'none';
                    if ($days_to_stockout <= 3) {
                        $urgency = 'critical';
                    } elseif ($days_to_stockout <= 7) {
                        $urgency = 'warning';
                    } elseif ($days_to_stockout <= 14) {
                        $urgency = 'mild';
                    }

                    // Base recommendation array
                    $recommendation = array(
                        'product' => new WISESTOCK_Demo_Product($product),
                        'product_name' => '[DEMO] ' . $product->product_name,
                        'product_id' => 'demo_' . $product->id,
                        'recommended_quantity' => (int) $recommended_quantity,
                        'days_to_stockout' => $days_to_stockout,
                        'daily_velocity' => round($daily_velocity, 2),
                        'urgency' => $urgency,
                        'reasoning' => $this->generate_demo_reasoning($days_to_stockout, $daily_velocity)
                    );

                    // Hook to enhance recommendation data
                    $recommendation = apply_filters('wisestock_enhance_demo_recommendation', $recommendation, $enhanced_recommendation, $base_recommendation);
                    
                    $recommendations[] = $recommendation;
                }
            }
        }

        // Sort by urgency and recommended quantity
        usort($recommendations, function($a, $b) {
            $urgency_order = array('critical' => 3, 'warning' => 2, 'mild' => 1, 'none' => 0);
            $a_urgency = $urgency_order[$a['urgency']] ?? 0;
            $b_urgency = $urgency_order[$b['urgency']] ?? 0;

            if ($a_urgency !== $b_urgency) {
                return $b_urgency <=> $a_urgency;
            }

            return $b['recommended_quantity'] <=> $a['recommended_quantity'];
        });

        // Limit to top 10 recommendations
        $recommendations = array_slice($recommendations, 0, 10);

        // Generate notifications ONLY for products in top recommendations (correlated)
        $demo_notifications = array();
        $attention_products = array();
        
        foreach ($recommendations as $recommendation) {
            // Find the demo product for this recommendation
            $demo_product = null;
            foreach ($demo_products as $product) {
                if ('demo_' . $product->id === $recommendation['product_id']) {
                    $demo_product = $product;
                    break;
                }
            }

            if ($demo_product && $recommendation['urgency'] !== 'none') {
                $notification_type = $recommendation['urgency'];
                $days_to_stockout = $recommendation['days_to_stockout'];
                $current_stock = $demo_product->current_stock;

                $demo_message = $this->get_demo_notification_message($notification_type, $current_stock, $days_to_stockout);

                $notification = (object) array(
                    'id' => 'demo_' . $demo_product->id,
                    'product_id' => 'demo_' . $demo_product->id,
                    'product_name' => $demo_product->product_name,
                    'notification_type' => $notification_type,
                    'message' => $demo_message,
                    'created_at' => current_time('mysql'),
                    'is_demo' => true
                );

                $demo_notifications[] = $notification;

                $attention_products[] = array(
                    'product' => new WISESTOCK_Demo_Product($demo_product),
                    'notification' => $notification,
                    'priority' => $notification_type
                );
            }
        }
        
        // Calculate notification stats from generated notifications
        $notification_stats = $this->calculate_demo_notification_stats($demo_notifications);

        // No market change detection - removed for improved UX

        // No market change detection - removed for improved UX

        return array(
            'notification_stats' => $notification_stats,
            'notifications' => array_slice($demo_notifications, 0, 10),
            'attention_products' => $attention_products,
            'velocity_summary' => $velocity_summary,
            'recommendations' => $recommendations
        );
    }

    /**
     * Generate demo notifications based on historical sales data
     *
     * @param array $products_with_sales Products with sales data
     * @return array Demo notifications
     * @since 1.0.0
     */
    private function generate_demo_notifications($products_with_sales) {
        $notifications = array();

        foreach ($products_with_sales as $product_data) {
            $product = wc_get_product($product_data->product_id);
            if (!$product || $product->get_status() !== 'publish') {
                continue;
            }

            $current_stock = $product->get_stock_quantity();
            $daily_velocity = $product_data->avg_velocity ?: 0; // Handle null values

            if ($daily_velocity > 0) {
                $days_to_stockout = $current_stock > 0 ? ceil($current_stock / $daily_velocity) : 0;

                $notification_type = null;
                if ($current_stock <= 0) {
                    $notification_type = 'critical';
                } elseif ($days_to_stockout <= 3) {
                    $notification_type = 'critical';
                } elseif ($days_to_stockout <= 7) {
                    $notification_type = 'warning';
                } elseif ($days_to_stockout <= 14) {
                    $notification_type = 'info';
                }

                if ($notification_type) {
                    $notifications[] = array(
                        'id' => 'demo_' . $product_data->product_id,
                        'product_id' => $product_data->product_id,
                        'notification_type' => $notification_type,
                        /* translators: 1: Notification type label. 2: Product name. 3: Days to stockout. 4: Current stock quantity. */
                        'message' => sprintf(__('[DEMO] %1$s - %2$s will stockout in %3$d days. Current stock: %4$d units.', 'wisestock'), strtoupper($notification_type), $product->get_name(), $days_to_stockout, $current_stock),
                        'status' => 'active',
                        'created_at' => current_time('mysql')
                    );
                }
            }
        }

        return $notifications;
    }

    /**
     * Calculate demo notification statistics
     *
     * @param array $notifications Demo notifications
     * @return array Notification statistics
     * @since 1.0.0
     */
    private function calculate_demo_notification_stats($notifications) {
        $stats = array('critical' => 0, 'warning' => 0, 'mild' => 0, 'total' => 0);

        foreach ($notifications as $notification) {
            $stats[$notification->notification_type]++;
            $stats['total']++;
        }

        return $stats;
    }

    /**
     * Generate demo reasoning for recommendations (matches live data sophistication)
     *
     * @param int $days_to_stockout Days until stockout
     * @param float $daily_velocity Daily sales velocity
     * @return string Reasoning text
     * @since 1.0.0
     */
    private function generate_demo_reasoning($days_to_stockout, $daily_velocity) {
        // Simulate confidence and volatility for demo consistency
        $demo_confidence = 0.75; // 75% confidence for demo
        $demo_volatility = 0.2;  // Moderate volatility for demo
        
        $confidence_text = round($demo_confidence * 100, 1) . '%';
        
        // Base urgency message (matching live data patterns)
        if ($days_to_stockout <= 0) {
            $urgency_message = __('CRITICAL: Out of stock! ', 'wisestock');
        } elseif ($days_to_stockout <= 3) {
            $urgency_message = sprintf(
                /* translators: %d: days to stockout */
                __('CRITICAL: Stock runs out in %d days. ', 'wisestock'),
                $days_to_stockout
            );
        } elseif ($days_to_stockout <= 7) {
            $urgency_message = sprintf(
                /* translators: %d: days to stockout */
                __('WARNING: Stock runs out in %d days. ', 'wisestock'),
                $days_to_stockout
            );
        } else {
            $urgency_message = '';
        }
        
        // Add confidence message (matching live data)
        $confidence_message = sprintf(
            /* translators: %s: confidence percentage */
            __('Prediction confidence: %s. ', 'wisestock'),
            $confidence_text
        );
        
        // Add volatility context for moderate volatility demo
        $volatility_message = sprintf(
            /* translators: %s: volatility percentage */
            __('Moderate volatility (%.1f%%). ', 'wisestock'),
            $demo_volatility * 100
        );
        
        return $urgency_message . $confidence_message . $volatility_message;
    }

    /**
     * Generate demo time-bound reasoning for recommendations
     *
     * @param int $days_to_stockout Days until stockout
     * @param float $daily_velocity Daily sales velocity
     * @param int $prediction_window_days Prediction window in days
     * @param float $confidence Final confidence score
     * @return string Time-bound reasoning text
     * @since 1.0.0
     */
    private function generate_demo_timebound_reasoning($days_to_stockout, $daily_velocity, $prediction_window_days, $confidence) {
        $confidence_percent = round($confidence * 100, 1);

        if ($days_to_stockout <= 0) {
            /* translators: 1: Daily velocity. 2: Prediction window days. 3: Confidence percentage. */
            return sprintf(esc_html__('[DEMO] Product is out of stock and needs immediate restocking. Time-bound prediction: %1$.1f units/day for next %2$d days (%3$s%% confidence).', 'wisestock'), $daily_velocity, $prediction_window_days, $confidence_percent);
        }

        if ($days_to_stockout <= 7) {
            /* translators: 1: Days to stockout. 2: Daily velocity. 3: Prediction window days. 4: Confidence percentage. */
            return sprintf(esc_html__('[DEMO] Product will stockout in %1$d days. Time-bound prediction: %2$.1f units/day for next %3$d days (%4$s%% confidence). Immediate action required.', 'wisestock'), $days_to_stockout, $daily_velocity, $prediction_window_days, $confidence_percent);
        }

        if ($days_to_stockout <= 14) {
            /* translators: 1: Days to stockout. 2: Daily velocity. 3: Prediction window days. 4: Confidence percentage. */
            return sprintf(esc_html__('[DEMO] Product will stockout in %1$d days. Time-bound prediction: %2$.1f units/day for next %3$d days (%4$s%% confidence). Consider restocking soon.', 'wisestock'), $days_to_stockout, $daily_velocity, $prediction_window_days, $confidence_percent);
        }

        /* translators: 1: Prediction window days. 2: Daily velocity. 3: Confidence percentage. */
        return sprintf(esc_html__('[DEMO] Product has good stock levels. Time-bound prediction: %1$.1f units/day for next %2$d days (%3$s%% confidence). Consider restocking for optimal inventory management.', 'wisestock'), $daily_velocity, $prediction_window_days, $confidence_percent);
    }

    /**
     * Calculate demo risk level based on confidence
     *
     * @param float $confidence Confidence score
     * @return string Risk level
     * @since 1.0.0
     */
    private function calculate_demo_risk_level($confidence) {
        if ($confidence >= 0.8) {
            return 'low';
        } elseif ($confidence >= 0.6) {
            return 'medium';
        } elseif ($confidence >= 0.4) {
            return 'high';
        } else {
            return 'very_high';
        }
    }

    /**
     * Generate demo data for existing products when no sales data is available
     *
     * @param array $products Products with velocity data
     * @return array Demo dashboard data
     * @since 1.0.0
     */
    private function generate_demo_data_for_existing_products($products) {
        $notifications = array();
        $attention_products = array();
        $velocity_summary = array();
        $recommendations = array();

        foreach ($products as $product_data) {
            $product = wc_get_product($product_data->product_id);
            if (!$product || $product->get_status() !== 'publish') {
                continue;
            }

            $current_stock = $product->get_stock_quantity();

            // Generate realistic demo data based on product
            $demo_velocity = wp_rand(1, 10) / 2; // Random velocity between 0.5 and 5
            $days_to_stockout = $current_stock > 0 ? ceil($current_stock / $demo_velocity) : 0;

            // Create demo notification
            $notification_type = null;
            if ($current_stock <= 0) {
                $notification_type = 'critical';
            } elseif ($days_to_stockout <= 3) {
                $notification_type = 'critical';
            } elseif ($days_to_stockout <= 7) {
                $notification_type = 'warning';
            } elseif ($days_to_stockout <= 14) {
                $notification_type = 'info';
            }

            if ($notification_type) {
                /* translators: 1: Notification type label. 2: Product name. 3: Days to stockout. 4: Current stock quantity. */
                $notification_message = sprintf(__('[DEMO] %1$s - %2$s will stockout in %3$d days. Current stock: %4$d units.', 'wisestock'), strtoupper($notification_type), $product->get_name(), $days_to_stockout, $current_stock);

                $notification = array(
                    'id' => 'demo_' . $product->get_id(),
                    'product_id' => $product->get_id(),
                    'notification_type' => $notification_type,
                    'message' => $notification_message,
                    'status' => 'active',
                    'created_at' => current_time('mysql')
                );

                $notifications[] = $notification;
                $attention_products[] = array(
                    'product' => $product,
                    'notification' => (object) $notification,
                    'priority' => $notification_type
                );
            }

            // Add to velocity summary
            $velocity_summary[] = array(
                'product' => $product,
                'velocity' => $demo_velocity,
                'confidence' => 0.7 // Demo confidence
            );

            // Add to recommendations
            if ($demo_velocity > 0) {
                $recommended_quantity = max(0, ($demo_velocity * 14) - $current_stock);
                if ($recommended_quantity > 0 || $days_to_stockout <= 7) {
                    $urgency = 'none';
                    if ($days_to_stockout <= 3) {
                        $urgency = 'critical';
                    } elseif ($days_to_stockout <= 7) {
                        $urgency = 'warning';
                    } elseif ($days_to_stockout <= 14) {
                        $urgency = 'mild';
                    }

                    $recommendations[] = array(
                        'product' => $product,
                        'product_name' => $product->get_name(),
                        'product_id' => $product->get_id(),
                        'recommended_quantity' => (int) $recommended_quantity,
                        'days_to_stockout' => $days_to_stockout,
                        'daily_velocity' => round($demo_velocity, 2),
                        'urgency' => $urgency,
                        'reasoning' => $this->generate_demo_reasoning($days_to_stockout, $demo_velocity)
                    );
                }
            }
        }

        // Calculate notification stats
        $notification_stats = array('critical' => 0, 'warning' => 0, 'mild' => 0, 'total' => 0);
        foreach ($notifications as $notification) {
            $notification_stats[$notification['notification_type']]++;
            $notification_stats['total']++;
        }

        // Sort velocity summary by velocity descending
        usort($velocity_summary, function($a, $b) {
            return $b['velocity'] <=> $a['velocity'];
        });

        // Sort recommendations by urgency
        usort($recommendations, function($a, $b) {
            $urgency_order = array('critical' => 3, 'warning' => 2, 'mild' => 1, 'none' => 0);
            $a_urgency = $urgency_order[$a['urgency']] ?? 0;
            $b_urgency = $urgency_order[$b['urgency']] ?? 0;

            if ($a_urgency !== $b_urgency) {
                return $b_urgency <=> $a_urgency;
            }

            return $b['recommended_quantity'] <=> $a['recommended_quantity'];
        });

        return array(
            'notification_stats' => $notification_stats,
            'notifications' => array_slice($notifications, 0, 10),
            'attention_products' => array_slice($attention_products, 0, 10),
            'velocity_summary' => array_slice($velocity_summary, 0, 5),
            'recommendations' => array_slice($recommendations, 0, 5)
        );
    }

    /**
     * Generate demo notifications from demo products
     *
     * @param array $demo_products Demo products data
     * @return array Demo notifications
     * @since 1.0.0
     */
    private function generate_demo_notifications_from_demo_products($demo_products) {
        $notifications = array();

        foreach ($demo_products as $product) {
            $current_stock = $product->current_stock;
            $daily_velocity = $product->daily_velocity;

            if ($daily_velocity > 0) {
                $days_to_stockout = $current_stock > 0 ? ceil($current_stock / $daily_velocity) : 0;

                $notification_type = null;
                if ($current_stock <= 0) {
                    $notification_type = 'critical';
                } elseif ($days_to_stockout <= 3) {
                    $notification_type = 'critical';
                } elseif ($days_to_stockout <= 7) {
                    $notification_type = 'warning';
                } elseif ($days_to_stockout <= 14) {
                    $notification_type = 'mild';
                }

                if ($notification_type) {
                    $notifications[] = (object) array(
                        'id' => 'demo_' . $product->id,
                        'product_id' => 'demo_' . $product->id,
                        'product_name' => $product->product_name,
                        'notification_type' => $notification_type,
                        'message' => $this->get_demo_notification_message($notification_type, $current_stock, $days_to_stockout),
                        'created_at' => current_time('mysql'),
                        'is_demo' => true
                    );
                }
            }
        }

        return $notifications;
    }

    /**
     * Get demo notification message based on notification type
     *
     * @param string $notification_type Notification type
     * @param int $current_stock Current stock
     * @param int $days_to_stockout Days to stockout
     * @return string Notification message
     * @since 1.0.0
     */
    private function get_demo_notification_message($notification_type, $current_stock, $days_to_stockout) {
        switch ($notification_type) {
            case 'critical':
                if ($current_stock <= 0) {
                    return esc_html__('Product is out of stock!', 'wisestock');
                }

                /* translators: %d: Days to stockout. */
                return sprintf(esc_html__('Critical: Only %d days of stock remaining', 'wisestock'), $days_to_stockout);
            case 'warning':
                /* translators: %d: Days to stockout. */
                return sprintf(esc_html__('Warning: %d days of stock remaining', 'wisestock'), $days_to_stockout);
            case 'mild':
                /* translators: %d: Days to stockout. */
                return sprintf(esc_html__('Mild: %d days of stock remaining', 'wisestock'), $days_to_stockout);
            default:
                return esc_html__('Stock level notification', 'wisestock');
        }
    }

    // ======================================================================
    // TIMEBOUND API METHODS
    // ======================================================================

    /**
     * Check if timebound features are enabled
     *
     * @return bool Always true since timebound is the only algorithm
     * @since 1.0.1
     */
    public function is_timebound_enabled() {
        // Timebound is now the only algorithm, so always return true
        return true;
    }

    /**
     * Get timebound velocity for a product
     *
     * @param int $product_id Product ID
     * @param int $prediction_window_days Optional prediction window
     * @return array|false Timebound velocity data or false if not available
     * @since 1.0.1
     */
    public function get_timebound_velocity( $product_id, $prediction_window_days = null ) {
        $core = WISESTOCK_Core::get_instance();
        return $core->get_timebound_velocity( $product_id, $prediction_window_days );
    }

    /**
     * Get enhanced restock recommendation using timebound analysis
     *
     * @param int $product_id Product ID
     * @param int $preferred_days Optional preferred prediction window
     * @return array Enhanced recommendation or standard recommendation
     * @since 1.0.1
     */
    public function get_enhanced_restock_recommendation( $product_id, $preferred_days = null ) {
        $core = WISESTOCK_Core::get_instance();
        return $core->get_enhanced_restock_recommendation( $product_id, $preferred_days );
    }



    /**
     * Get intelligent prediction windows for a product
     *
     * @param int $product_id Product ID
     * @return array|false Prediction windows or false if not available
     * @since 1.0.1
     */
    public function get_intelligent_prediction_windows( $product_id ) {
        $core = WISESTOCK_Core::get_instance();
        return $core->get_intelligent_prediction_windows( $product_id );
    }

    /**
     * Get optimal prediction window for a product
     *
     * @param int $product_id Product ID
     * @return int|false Optimal prediction window or false if not available
     * @since 1.0.1
     */
    public function get_optimal_prediction_window( $product_id ) {
        if ( ! $this->is_timebound_enabled() ) {
            return false;
        }

        $core = WISESTOCK_Core::get_instance();
        $timebound_calculator = $core->get_timebound_velocity_calculator();
        
        if ( null === $timebound_calculator ) {
            return false;
        }

        return $timebound_calculator->get_optimal_prediction_window( $product_id );
    }



    /**
     * Enable timebound features
     *
     * @since 1.0.1
     */
    public function enable_timebound() {
        $core = WISESTOCK_Core::get_instance();
        $core->enable_timebound();
    }

    /**
     * Disable timebound features
     *
     * @since 1.0.1
     */
    public function disable_timebound() {
        $core = WISESTOCK_Core::get_instance();
        $core->disable_timebound();
    }

    /**
     * Get timebound settings
     *
     * @return array Timebound settings
     * @since 1.0.1
     */
    public function get_timebound_settings() {
        $core = WISESTOCK_Core::get_instance();
        $timebound_settings = $core->get_timebound_settings();
        
        if ( null === $timebound_settings ) {
            return array();
        }

        return $timebound_settings->get_settings();
    }

    /**
     * Update timebound settings
     *
     * @param array $settings New timebound settings
     * @return bool Success status
     * @since 1.0.1
     */
    public function update_timebound_settings( $settings ) {
        $core = WISESTOCK_Core::get_instance();
        $timebound_settings = $core->get_timebound_settings();
        
        if ( null === $timebound_settings ) {
            return false;
        }

        return $timebound_settings->update_settings( $settings );
    }

    /**
     * Get store type presets for timebound
     *
     * @return array Store type presets
     * @since 1.0.1
     */
    public function get_timebound_store_type_presets() {
        $core = WISESTOCK_Core::get_instance();
        $timebound_settings = $core->get_timebound_settings();
        
        if ( null === $timebound_settings ) {
            return array();
        }

        return $timebound_settings->get_store_type_presets();
    }

    /**
     * Apply store type preset to timebound settings
     *
     * @param string $preset_key Store type preset key
     * @return bool Success status
     * @since 1.0.1
     */
    public function apply_timebound_store_type_preset( $preset_key ) {
        $core = WISESTOCK_Core::get_instance();
        $timebound_settings = $core->get_timebound_settings();
        
        if ( null === $timebound_settings ) {
            return false;
        }

        return $timebound_settings->apply_store_type_preset( $preset_key );
    }

    /**
     * Get comprehensive product analysis with timebound features
     *
     * @param int $product_id Product ID
     * @return array Complete product analysis
     * @since 1.0.1
     */
    public function get_comprehensive_product_analysis( $product_id ) {
        $base_analysis = $this->get_product_stock_info( $product_id );
        
        if ( ! $base_analysis || ! $this->is_timebound_enabled() ) {
            return $base_analysis;
        }

        // Add timebound analysis
        $timebound_velocity = $this->get_timebound_velocity( $product_id );
        $prediction_windows = $this->get_intelligent_prediction_windows( $product_id );
        $enhanced_recommendation = $this->get_enhanced_restock_recommendation( $product_id );

        $base_analysis['timebound_analysis'] = array(
            'velocity_data' => $timebound_velocity,
            'prediction_windows' => $prediction_windows,
            'enhanced_recommendation' => $enhanced_recommendation,
            'timebound_enabled' => true
        );

        return $base_analysis;
    }
}

/**
 * Demo Product class to simulate WooCommerce product behavior
 *
 * @since 1.0.0
 */
class WISESTOCK_Demo_Product {
    private $demo_data;

    public function __construct($demo_data) {
        $this->demo_data = $demo_data;
    }

    public function get_id() {
        return 'demo_' . $this->demo_data->id;
    }

    public function get_name() {
        return '[DEMO] ' . $this->demo_data->product_name;
    }

    public function get_status() {
        return 'publish';
    }

    public function get_stock_quantity() {
        return $this->demo_data->current_stock;
    }

    public function get_price() {
        return $this->demo_data->price;
    }

    public function get_sku() {
        return $this->demo_data->sku;
    }

    public function get_category() {
        return $this->demo_data->category;
    }

    public function get_supplier() {
        return $this->demo_data->supplier;
    }

    public function managing_stock() {
        // All demo products manage stock (no digital products)
        return true;
    }

    public function is_virtual() {
        return false;
    }

    public function is_downloadable() {
        return false;
    }
}
