<?php
/**
 * Wise Stock Notifications Page Template
 *
 * This template displays all notifications in a detailed table format.
 * It provides comprehensive notification management with filtering and actions.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Check if demo stats are enabled with proper defaults
$defaults = WISESTOCK_Core::get_default_options();
$options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
$demo_stats_enabled = $options['demo_stats'];

// Get notification manager
$notification_manager = new WISESTOCK_Notification_Manager();

// Get notifications and statistics based on demo mode
if ($demo_stats_enabled) {
    // Use demo data from API
    $api = WISESTOCK_API::get_instance();
    $dashboard_data = $api->get_demo_dashboard_data();

    $all_notifications = $dashboard_data['notifications'] ?? array();
    $notification_stats = $dashboard_data['notification_stats'] ?? array();
} else {
    // Use real data
    $all_notifications = $notification_manager->get_all_notifications();
    $notification_stats = $notification_manager->get_notification_statistics();
}

// Get attention products (products with notifications)
$attention_products = array();
if (!empty($all_notifications)) {
    foreach ($all_notifications as $notification) {
        $product_id = $notification->product_id;
        $product = null;

        // Handle demo products
        if (strpos($product_id, 'demo_') === 0) {
            // Create a mock product for demo data
            $product = new stdClass();
            $product->id = $product_id;
            $product->name = 'Demo Product ' . substr($product_id, 5);
            $product->stock_quantity = wp_rand(1, 10);
            $product->sku = 'DEMO-' . substr($product_id, 5);
            $is_demo = true;
        } else {
            $product = wc_get_product($product_id);
            $is_demo = false;
        }

        if ($product) {
            $priority = $notification->notification_type;
            $attention_products[] = array(
                'product' => $product,
                'notification' => $notification,
                'priority' => $priority,
                'is_demo' => $is_demo ?? false
            );
        }
    }
}

// Sort by priority (critical first, then warning, then mild)
usort($attention_products, function($a, $b) {
    $priority_order = array('critical' => 1, 'warning' => 2, 'mild' => 3);
    $a_priority = $priority_order[$a['priority']] ?? 4;
    $b_priority = $priority_order[$b['priority']] ?? 4;
    return $a_priority - $b_priority;
});

// Get current filter
$current_filter = isset($_GET['filter']) ? sanitize_key(wp_unslash($_GET['filter'])) : 'all';

// Filter notifications based on current filter
$filtered_notifications = $all_notifications;
if ($current_filter !== 'all') {
    $filtered_notifications = array_filter($all_notifications, function($notification) use ($current_filter) {
        return $notification->notification_type === $current_filter;
    });
}

// Get current page for pagination
$current_page = isset($_GET['paged']) ? max(1, absint(wp_unslash($_GET['paged']))) : 1;
$per_page = 20;
$total_notifications = count($filtered_notifications);
$total_pages = ceil($total_notifications / $per_page);
$offset = ($current_page - 1) * $per_page;

// Get paginated notifications
$paginated_notifications = array_slice($filtered_notifications, $offset, $per_page);
?>

<div class="wisestock-admin">
    <?php include WISESTOCK_PLUGIN_DIR . 'admin/partials/navigation.php'; ?>

    <?php if ($demo_stats_enabled): ?>
        <div class="wisestock-demo-notice">
            <div class="wisestock-notice-content">
                <div class="wisestock-notice-icon">
                    <span class="dashicons dashicons-chart-line"></span>
                </div>
                <div class="wisestock-notice-text">
                    <h3><?php esc_html_e('Demo Stats Mode Active', 'wisestock'); ?></h3>
                    <p><?php esc_html_e('Notifications page is showing demo statistics. Disable in Settings to use real-time data.', 'wisestock'); ?></p>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <div class="wisestock-dashboard">
        <!-- Notifications Header -->
        <div class="wisestock-dashboard-header">
            <h1 class="wisestock-dashboard-title">
                <?php esc_html_e('Notifications', 'wisestock'); ?>
            </h1>
            <p class="wisestock-dashboard-subtitle">
                <?php esc_html_e('Manage all notifications and alerts', 'wisestock'); ?>
            </p>
        </div>

        <!-- Filter Tabs -->
        <div class="wisestock-filter-tabs">
        <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications')); ?>"
           class="wisestock-filter-tab <?php echo ($current_filter === 'all') ? 'active' : ''; ?>">
            <?php esc_html_e('All', 'wisestock'); ?> (<?php echo esc_html($notification_stats['total'] ?? 0); ?>)
        </a>
        <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications&filter=critical')); ?>"
           class="wisestock-filter-tab <?php echo ($current_filter === 'critical') ? 'active' : ''; ?>">
            <?php esc_html_e('Critical', 'wisestock'); ?> (<?php echo esc_html($notification_stats['critical'] ?? 0); ?>)
        </a>
        <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications&filter=warning')); ?>"
           class="wisestock-filter-tab <?php echo ($current_filter === 'warning') ? 'active' : ''; ?>">
            <?php esc_html_e('Warning', 'wisestock'); ?> (<?php echo esc_html($notification_stats['warning'] ?? 0); ?>)
        </a>
        <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications&filter=mild')); ?>"
           class="wisestock-filter-tab <?php echo ($current_filter === 'mild') ? 'active' : ''; ?>">
            <?php esc_html_e('Mild', 'wisestock'); ?> (<?php echo esc_html($notification_stats['mild'] ?? 0); ?>)
        </a>
    </div>

    <!-- Notifications Table -->
    <div class="wisestock-card">
        <div class="wisestock-card-header">
            <h3 class="wisestock-card-title">
                <?php
                if ($current_filter === 'all') {
                    esc_html_e('All Notifications', 'wisestock');
                } else {
                    /* translators: %s: Current notification filter label. */
                    printf(esc_html__('%s Notifications', 'wisestock'), esc_html(ucfirst($current_filter)));
                }
                ?>
            </h3>
            <p class="wisestock-card-subtitle">
                <?php
                $displayed_notifications = absint(count($paginated_notifications));
                $total_notifications_count = absint($total_notifications);

                /* translators: 1: Number of notifications shown. 2: Total notifications. */
                printf(esc_html__('Showing %1$d of %2$d notifications', 'wisestock'), esc_html($displayed_notifications), esc_html($total_notifications_count));
                ?>
            </p>
        </div>

        <div class="wisestock-notifications-table">
            <?php if (empty($paginated_notifications)): ?>
                <div class="wisestock-no-data">
                    <div class="wisestock-no-data-icon">
                        <span class="dashicons dashicons-bell"></span>
                    </div>
                    <h3><?php esc_html_e('No notifications found', 'wisestock'); ?></h3>
                    <p>
                        <?php if ($current_filter === 'all'): ?>
                            <?php esc_html_e('No notifications are currently active. Great job managing your inventory!', 'wisestock'); ?>
                        <?php else: ?>
                            <?php
                            /* translators: %s: Current notification filter label. */
                            printf(esc_html__('No %s notifications found.', 'wisestock'), esc_html($current_filter));
                            ?>
                        <?php endif; ?>
                    </p>
                </div>
            <?php else: ?>
                <div class="wisestock-notifications-header">
                    <div class="wisestock-notification-col product"><?php esc_html_e('Product', 'wisestock'); ?></div>
                    <div class="wisestock-notification-col type"><?php esc_html_e('Type', 'wisestock'); ?></div>
                    <div class="wisestock-notification-col message"><?php esc_html_e('Message', 'wisestock'); ?></div>
                    <div class="wisestock-notification-col stock"><?php esc_html_e('Stock', 'wisestock'); ?></div>
                    <div class="wisestock-notification-col date"><?php esc_html_e('Date', 'wisestock'); ?></div>
                    <div class="wisestock-notification-col actions"><?php esc_html_e('Actions', 'wisestock'); ?></div>
                </div>

                <?php foreach ($paginated_notifications as $notification): ?>
                    <?php
                    $product_id = $notification->product_id;
                    $product = null;
                    $is_demo = false;

                    // Handle demo products
                    if (strpos($product_id, 'demo_') === 0) {
                        // Create a mock product for demo data
                        $product = new stdClass();
                        $product->id = $product_id;
                        $product->name = 'Demo Product ' . substr($product_id, 5);
                        $product->stock_quantity = wp_rand(1, 10);
                        $product->sku = 'DEMO-' . substr($product_id, 5);
                        $is_demo = true;
                    } else {
                        $product = wc_get_product($product_id);
                        $is_demo = false;
                    }
                    ?>

                    <?php if ($product): ?>
                        <div class="wisestock-notification-row">
                            <div class="wisestock-notification-col product">
                                <div class="wisestock-product-info">
                                    <div class="wisestock-product-name">
                                        <?php if ($is_demo): ?>
                                            <span>[DEMO] <?php echo esc_html($product->name); ?></span>
                                        <?php else: ?>
                                            <a href="<?php echo esc_url(admin_url('post.php?post=' . $product->get_id() . '&action=edit')); ?>">
                                                <?php echo esc_html($product->get_name()); ?>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                    <div class="wisestock-product-sku">
                                        <?php esc_html_e('SKU:', 'wisestock'); ?>
                                        <?php echo esc_html($is_demo ? $product->sku : ($product->get_sku() ?: 'N/A')); ?>
                                    </div>
                                </div>
                            </div>

                            <div class="wisestock-notification-col type">
                                <span class="wisestock-priority-badge <?php echo esc_attr($notification->notification_type); ?>">
                                    <?php echo esc_html(ucfirst($notification->notification_type)); ?>
                                </span>
                            </div>

                            <div class="wisestock-notification-col message">
                                <div class="wisestock-notification-message">
                                    <?php echo esc_html($notification->message); ?>
                                </div>
                            </div>

                            <div class="wisestock-notification-col stock">
                                <div class="wisestock-stock-info">
                                    <?php echo esc_html($is_demo ? $product->stock_quantity : $product->get_stock_quantity()); ?>
                                </div>
                            </div>

                            <div class="wisestock-notification-col date">
                                <div class="wisestock-notification-date">
                                    <?php echo esc_html(gmdate('M j, Y', strtotime($notification->created_at . ' UTC'))); ?>
                                </div>
                            </div>

                            <div class="wisestock-notification-col actions">
                                <div class="wisestock-notification-actions">
                                    <button class="wisestock-btn small wisestock-notification-resolve"
                                            data-notification-id="<?php echo esc_attr($notification->id); ?>">
                                        <?php esc_html_e('Resolve', 'wisestock'); ?>
                                    </button>
                                    <button class="wisestock-btn small secondary wisestock-notification-dismiss"
                                            data-notification-id="<?php echo esc_attr($notification->id); ?>">
                                        <?php esc_html_e('Dismiss', 'wisestock'); ?>
                                    </button>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="wisestock-pagination">
                <?php
                $base_url = admin_url('admin.php?page=wisestock-notifications');
                if ($current_filter !== 'all') {
                    $base_url .= '&filter=' . urlencode($current_filter);
                }

                // Previous page
                if ($current_page > 1):
                    $prev_url = $base_url . '&paged=' . ($current_page - 1);
                ?>
                    <a href="<?php echo esc_url($prev_url); ?>" class="wisestock-pagination-btn">
                        <?php esc_html_e('← Previous', 'wisestock'); ?>
                    </a>
                <?php endif; ?>

                <!-- Page numbers -->
                <?php
                $start_page = max(1, $current_page - 2);
                $end_page = min($total_pages, $current_page + 2);

                for ($i = $start_page; $i <= $end_page; $i++):
                    $page_url = $base_url . '&paged=' . $i;
                    $is_current = ($i === $current_page);
                ?>
                    <a href="<?php echo esc_url($page_url); ?>"
                       class="wisestock-pagination-btn <?php echo $is_current ? 'active' : ''; ?>">
                        <?php echo esc_html($i); ?>
                    </a>
                <?php endfor; ?>

                <!-- Next page -->
                <?php if ($current_page < $total_pages):
                    $next_url = $base_url . '&paged=' . ($current_page + 1);
                ?>
                    <a href="<?php echo esc_url($next_url); ?>" class="wisestock-pagination-btn">
                        <?php esc_html_e('Next →', 'wisestock'); ?>
                    </a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        </div> <!-- End wisestock-notifications-table -->
    </div> <!-- End wisestock-dashboard -->
</div> <!-- End wisestock-admin -->
