<?php
/**
 * Wise Stock Dashboard Template
 *
 * This template displays the main Wise Stock dashboard with alerts,
 * velocity indicators, and restock recommendations.
 *
 * @package WiseStock
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Use centralized API to get dashboard data (following DRY principle)
$api = WISESTOCK_API::get_instance();
$dashboard_data = $api->get_dashboard_data();

// Extract data from centralized API response
$notifications = $dashboard_data['notifications'];
$notification_stats = $dashboard_data['notification_stats'];
$attention_products = $dashboard_data['attention_products'];
$velocity_summary = $dashboard_data['velocity_summary'];
$recommendations = $dashboard_data['recommendations'];

// Initialize managers for any additional operations needed
$supplier_manager = new WISESTOCK_Supplier_Manager();

// Check if demo stats are enabled for UI display with proper defaults
$defaults = WISESTOCK_Core::get_default_options();
$options = wp_parse_args(get_option('wisestock_options', array()), $defaults);
$demo_stats_enabled = $options['demo_stats'];
?>

<div class="wisestock-admin">
    <?php include WISESTOCK_PLUGIN_DIR . 'admin/partials/navigation.php'; ?>

    <?php if ($demo_stats_enabled): ?>
        <div class="wisestock-demo-notice">
            <div class="wisestock-notice-content">
                <div class="wisestock-notice-icon">
                    <span class="dashicons dashicons-chart-line"></span>
                </div>
                <div class="wisestock-notice-text">
                    <h3><?php esc_html_e('Demo Stats Mode Active', 'wisestock'); ?></h3>
                    <p><?php esc_html_e('Dashboard is showing demo statistics. Disable in Settings to use real-time data.', 'wisestock'); ?></p>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <div class="wisestock-dashboard">
        <!-- Dashboard Header -->
        <div class="wisestock-dashboard-header">
            <div class="wisestock-header-content">
                <div class="wisestock-header-text">
                    <h1 class="wisestock-dashboard-title">
                        <?php esc_html_e('Dashboard', 'wisestock'); ?>
                    </h1>
                    <p class="wisestock-dashboard-subtitle">
                        <?php
                        $default_subtitle = esc_html__('Intelligent sales forecast and restock recommendations', 'wisestock');
                        $subtitle = apply_filters('wisestock_dashboard_subtitle', $default_subtitle);
                        echo esc_html($subtitle);
                        ?>
                    </p>
                </div>
                <div class="wisestock-header-actions">
                    <!-- Notification Bell -->
                    <div class="wisestock-notification-bell">
                        <button class="wisestock-bell-btn" id="wisestock-notification-toggle">
                            <span class="dashicons dashicons-bell"></span>
                            <span class="wisestock-notification-count" id="wisestock-notification-count">
                                <?php echo esc_html($notification_stats['total'] ?? 0); ?>
                            </span>
                        </button>
                        <div class="wisestock-notification-dropdown" id="wisestock-notification-dropdown">
                            <div class="wisestock-notification-header">
                                <h3><?php esc_html_e('Notifications', 'wisestock'); ?></h3>
                                <button class="wisestock-close-dropdown" id="wisestock-close-notifications">
                                    <span class="dashicons dashicons-no-alt"></span>
                                </button>
                            </div>
                            <div class="wisestock-notification-list" id="wisestock-notification-list">
                                <!-- Content will be dynamically loaded via JavaScript -->
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Notification Summary -->
        <div class="wisestock-card">
            <div class="wisestock-card-header">
                <h2 class="wisestock-card-title">
                    <?php esc_html_e('Stockout Notifications', 'wisestock'); ?>
                    <span class="wisestock-help-icon" data-tooltip="predicted-stockout-info">
                        <span class="dashicons dashicons-editor-help"></span>
                    </span>
                </h2>
                <p class="wisestock-card-subtitle"><?php esc_html_e('Stockout notifications requiring attention', 'wisestock'); ?></p>
            </div>
            <div class="wisestock-notification-summary">
                <div class="wisestock-notification-card critical">
                    <div class="wisestock-notification-count critical"><?php echo esc_html($notification_stats['critical'] ?? 0); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Critical', 'wisestock'); ?></div>
                </div>
                <div class="wisestock-notification-card warning">
                    <div class="wisestock-notification-count warning"><?php echo esc_html($notification_stats['warning'] ?? 0); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Warning', 'wisestock'); ?></div>
                </div>
                <div class="wisestock-notification-card mild">
                    <div class="wisestock-notification-count mild"><?php echo esc_html($notification_stats['mild'] ?? 0); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Mild', 'wisestock'); ?></div>
                </div>
                <div class="wisestock-notification-card">
                    <div class="wisestock-notification-count"><?php echo esc_html($notification_stats['total'] ?? 0); ?></div>
                    <div class="wisestock-notification-label"><?php esc_html_e('Total', 'wisestock'); ?></div>
                </div>
            </div>
            <div class="wisestock-notification-actions">
                <a href="<?php echo esc_url(admin_url('admin.php?page=wisestock-notifications')); ?>" class="wisestock-btn primary">
                    <?php esc_html_e('Show All Notifications', 'wisestock'); ?>
                </a>
            </div>
        </div>

        <!-- Sales Velocity -->
        <div class="wisestock-card">
            <div class="wisestock-card-header">
                <h3 class="wisestock-card-title"><?php esc_html_e('Products by Sales Velocity', 'wisestock'); ?></h3>
                <p class="wisestock-card-subtitle">
                    <?php
                    $default_velocity_subtitle = esc_html__('Products with predicted sales and confidence', 'wisestock');
                    $velocity_subtitle = apply_filters('wisestock_velocity_section_subtitle', $default_velocity_subtitle);
                    echo esc_html($velocity_subtitle);
                    ?>
                </p>
            </div>
            <div class="wisestock-velocity-summary">
                <?php if (empty($velocity_summary)): ?>
                    <div class="wisestock-no-data">
                        <?php esc_html_e('No velocity data available yet', 'wisestock'); ?>
                    </div>
                <?php else: ?>
                    <div class="wisestock-velocity-table">
                        <div class="wisestock-velocity-header">
                            <div class="wisestock-velocity-col product-name"><?php esc_html_e('Product Name', 'wisestock'); ?></div>
                            <div class="wisestock-velocity-col sales-prediction">
                                <?php
                                $default_prediction_header = esc_html__('Sales Prediction', 'wisestock');
                                $prediction_header = apply_filters('wisestock_velocity_prediction_header', $default_prediction_header);
                                echo esc_html($prediction_header);
                                ?>
                            </div>
                            <div class="wisestock-velocity-col prediction-window"><?php esc_html_e('Valid For Next', 'wisestock'); ?></div>
                            <div class="wisestock-velocity-col confidence"><?php esc_html_e('Confidence', 'wisestock'); ?></div>
                        </div>
                        <?php foreach ($velocity_summary as $item): ?>
                            <?php
                            $product = $item['product'];
                            $product_id = $product->get_id();

                            // Hook to provide enhanced velocity data
                            $enhanced_velocity_data = apply_filters('wisestock_get_enhanced_velocity_data', null, $product_id, $item);
                            ?>
                            <div class="wisestock-velocity-row">
                                <div class="wisestock-velocity-col product-name">
                                    <?php if (strpos($product_id, 'demo_') === 0): ?>
                                        <span class="wisestock-demo-product-name"><?php echo esc_html($product->get_name()); ?></span>
                                    <?php else: ?>
                                        <a href="<?php echo esc_url(admin_url('post.php?post=' . $product_id . '&action=edit')); ?>">
                                            <?php echo esc_html($product->get_name()); ?>
                                        </a>
                                    <?php endif; ?>
                                </div>
                                <div class="wisestock-velocity-col sales-prediction">
                                    <?php
                                    // Default velocity display
                                    $velocity_display = '<span class="wisestock-velocity-value">~ ' . esc_html(round($item['velocity'])) . ' units / day</span>';
                                    
                                    // Hook to customize velocity display
                                    $velocity_display = apply_filters('wisestock_velocity_display', $velocity_display, $item, $enhanced_velocity_data);
                                    echo wp_kses_post($velocity_display);
                                    ?>
                                </div>
                                <div class="wisestock-velocity-col prediction-window">
                                    <?php
                                    $prediction_window = $enhanced_velocity_data['prediction_window'] ?? $item['prediction_window'] ?? null;
                                    if ($prediction_window) {
                                        echo esc_html($prediction_window . ' days');
                                    } else {
                                        echo '<span class="wisestock-no-data">—</span>';
                                    }
                                    ?>
                                </div>
                                <div class="wisestock-velocity-col confidence">
                                    <?php
                                    // Default confidence display
                                    $confidence_display = '<span class="wisestock-velocity-confidence">' . esc_html(round($item['confidence'] * 100)) . '%</span>';
                                    
                                    // Hook to customize confidence display
                                    $confidence_display = apply_filters('wisestock_confidence_display', $confidence_display, $item, $enhanced_velocity_data);
                                    echo wp_kses_post($confidence_display);
                                    ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>



        <!-- Restock Recommendations -->
        <div class="wisestock-card">
            <div class="wisestock-card-header">
                <h3 class="wisestock-card-title"><?php esc_html_e('Products Restock Recommendations', 'wisestock'); ?></h3>
                <p class="wisestock-card-subtitle">
                    <?php
                    $default_restock_subtitle = esc_html__('Automated restock suggestions based on sales velocity', 'wisestock');
                    $restock_subtitle = apply_filters('wisestock_restock_section_subtitle', $default_restock_subtitle);
                    echo esc_html($restock_subtitle);
                    ?>
                </p>
            </div>
            <div class="wisestock-recommendations">
                <?php if (empty($recommendations)): ?>
                    <div class="wisestock-no-data">
                        <?php esc_html_e('No restock recommendations at this time', 'wisestock'); ?>
                    </div>
                <?php else: ?>
                    <?php foreach ($recommendations as $rec): ?>
                        <div class="wisestock-recommendation">
                            <div class="wisestock-recommendation-header">
                                <h4 class="wisestock-recommendation-title">
                                    <?php echo esc_html($rec['product_name']); ?>

                                </h4>
                                <div class="wisestock-recommendation-badges">
                                    <span class="wisestock-priority-badge <?php echo esc_attr($rec['urgency']); ?>">
                                        <?php echo esc_html(ucfirst($rec['urgency'])); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="wisestock-recommendation-details">
                                <div class="wisestock-recommendation-detail">
                                    <span class="wisestock-recommendation-detail-label"><?php esc_html_e('Recommended:', 'wisestock'); ?></span>
                                    <span class="wisestock-recommendation-detail-value"><?php echo esc_html($rec['recommended_quantity']); ?> units</span>
                                </div>
                                <div class="wisestock-recommendation-detail">
                                    <span class="wisestock-recommendation-detail-label"><?php esc_html_e('Days to Stockout:', 'wisestock'); ?></span>
                                    <span class="wisestock-recommendation-detail-value"><?php echo esc_html($rec['days_to_stockout']); ?> days</span>
                                </div>
                                <div class="wisestock-recommendation-detail">
                                    <span class="wisestock-recommendation-detail-label"><?php esc_html_e('Daily Velocity:', 'wisestock'); ?></span>
                                    <span class="wisestock-recommendation-detail-value"><?php echo '~ ' . esc_html(round($rec['daily_velocity'])); ?> units / day</span>
                                </div>
                                <?php
                                // Hook to add additional recommendation details
                                do_action('wisestock_restock_recommendation_details', $rec);
                                ?>
                            </div>
                            <div class="wisestock-recommendation-reasoning"><?php echo esc_html($rec['reasoning']); ?></div>
                            <div class="wisestock-btn-group">
                                <?php if (strpos($rec['product_id'], 'demo_') === 0): ?>
                                    <button class="wisestock-btn small" disabled title="<?php esc_attr_e('Demo product - view not available', 'wisestock'); ?>">
                                        <?php esc_html_e('View Product', 'wisestock'); ?>
                                    </button>
                                    <button class="wisestock-btn small secondary" disabled title="<?php esc_attr_e('Demo mode - supplier notifications disabled', 'wisestock'); ?>">
                                        <?php esc_html_e('Notify Supplier', 'wisestock'); ?>
                                    </button>
                                <?php else: ?>
                                    <button class="wisestock-btn small wisestock-product-action" data-action="view" data-product-id="<?php echo esc_attr($rec['product_id']); ?>">
                                        <?php esc_html_e('View Product', 'wisestock'); ?>
                                    </button>
                                    <button class="wisestock-btn small secondary wisestock-notify-supplier" data-product-id="<?php echo esc_attr($rec['product_id']); ?>">
                                        <?php esc_html_e('Notify Supplier', 'wisestock'); ?>
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <?php
        // Hook to render advanced analytics sections
        do_action('wisestock_dashboard_advanced_analytics', $dashboard_data, $velocity_summary, $demo_stats_enabled);
        ?>
    </div>
</div>

<!-- Tooltip Content -->
<div id="predicted-stockout-info" class="wisestock-tooltip-content" style="display: none;">
    <div class="wisestock-tooltip-section">
        <h4><?php esc_html_e('What are these notifications', 'wisestock'); ?></h4>
        <ul class="wisestock-tooltip-list">
            <li>✅ <?php esc_html_e('Stockout alerts based on sales velocity', 'wisestock'); ?></li>
            <li>✅ <?php esc_html_e('Designed to notify about future stockouts', 'wisestock'); ?></li>
        </ul>
    </div>
</div>


