/**
 * Wise Stock Timebound Admin JavaScript
 *
 * This file handles the timebound-specific admin functionality including
 * settings management, store type presets, and real-time validation.
 *
 * @package WiseStock
 * @since 1.0.1
 */

(function($) {
    'use strict';

    /**
     * Timebound Admin Handler
     */
    const WiseStockTimeboundAdmin = {

        /**
         * Initialize timebound admin functionality
         */
        init: function() {
            this.bindEvents();
            this.initPresetHandling();
            this.initVolatilityPresetHandling();
            this.initConditionalDisplay();
            this.initValidation();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Store type preset change (updated selector for unified settings)
            $(document).on('change', '#store_type_preset', this.handlePresetChange);
            
            // Volatility threshold preset change
            $(document).on('change', '#volatility_threshold_preset', this.handleVolatilityPresetChange);
            
            // Validation on input changes
            $(document).on('input', '#default_prediction_window, #max_prediction_window', this.validatePredictionWindows);
            
            // Confidence threshold preset handling
            $(document).on('change', '#min_confidence_threshold', this.updateConfidenceThresholdDescription);
            

        },

        /**
         * Initialize store type preset handling
         */
        initPresetHandling: function() {
            // Update description when preset changes
            this.updatePresetDescription();
        },

        /**
         * Initialize volatility threshold preset handling
         */
        initVolatilityPresetHandling: function() {
            // Initialize preset descriptions on page load
            this.handleVolatilityPresetChange.call($('#volatility_threshold_preset'));
        },

        /**
         * Initialize conditional display logic
         */
        initConditionalDisplay: function() {
            // Timebound is now always enabled, so always show advanced settings
            $('#prediction-advanced-settings').show();
        },

        /**
         * Initialize form validation
         */
        initValidation: function() {
            this.validatePredictionWindows();
            this.validateThresholds();
        },

        /**
         * Toggle timebound settings visibility (deprecated - timebound always enabled)
         */
        toggleTimeboundSettings: function() {
            // Timebound is now always enabled, no need to toggle
            return;
        },

        /**
         * Handle store type preset change
         */
        handlePresetChange: function() {
            const selectedOption = $(this).find(':selected');
            const description = selectedOption.data('description');

            // Update description (use correct ID from HTML)
            $('#store-type-preset-description').text(description);


        },

        /**
         * Handle volatility threshold preset change
         */
        handleVolatilityPresetChange: function() {
            const selectedValue = $(this).val();
            const $descriptionDiv = $('#volatility_description');
            
            // Update description based on preset selection
            const presetDescriptions = {
                'conservative': 'Low volatility tolerance - only flag products with very stable sales patterns as low-volatility. Recommended for stores with steady, predictable demand.',
                'balanced': 'Moderate volatility tolerance - balanced approach suitable for most stores. Flags products with reasonable sales consistency as reliable.',
                'aggressive': 'High volatility tolerance - accepts more variation in sales patterns. Good for trend-driven stores where some fluctuation is normal.'
            };
            
            const description = presetDescriptions[selectedValue] || 'Please select a volatility threshold preset.';
            $descriptionDiv.text(description);
            
            // Update hidden threshold value based on preset
            const presetValues = {
                'conservative': '0.15',
                'balanced': '0.25',
                'aggressive': '0.40'
            };
            
            if (presetValues[selectedValue]) {
                $('#volatility_threshold').val(presetValues[selectedValue]);
            }
        },

        /**
         * Update confidence threshold description
         */
        updateConfidenceThresholdDescription: function() {
            const select = $(this);
            const selectedOption = select.find('option:selected');
            const description = selectedOption.attr('data-description');
            const $descriptionDiv = $('#confidence-threshold-description');
            
            if (description && $descriptionDiv.length) {
                $descriptionDiv.text(description);
            }
        },

        /**
         * Update store type preset description
         */
        updatePresetDescription: function() {
            const selectedOption = $('#store_type_preset').find(':selected');
            const description = selectedOption.data('description');
            if (description) {
                $('#store-type-preset-description').text(description);
            }
        },

        /**
         * Validate prediction windows
         */
        validatePredictionWindows: function() {
            const defaultWindow = parseInt($('#default_prediction_window').val()) || 14;
            const maxWindow = parseInt($('#max_prediction_window').val()) || 90;
            
            // Clear previous validation messages
            $('.wisestock-validation-error').remove();
            
            // Validate default window
            if (defaultWindow < 1 || defaultWindow > 365) {
                WiseStockTimeboundAdmin.addValidationError('#default_prediction_window', 'Prediction window must be between 1 and 365 days');
                return false;
            }
            
            // Validate max window
            if (maxWindow < 1 || maxWindow > 365) {
                WiseStockTimeboundAdmin.addValidationError('#max_prediction_window', 'Maximum window must be between 1 and 365 days');
                return false;
            }
            
            // Validate that max is greater than or equal to default
            if (maxWindow < defaultWindow) {
                WiseStockTimeboundAdmin.addValidationError('#max_prediction_window', 'Maximum window must be greater than or equal to default window');
                return false;
            }
            
            return true;
        },

        /**
         * Validate threshold settings
         */
        validateThresholds: function() {
            const minConfidence = parseInt($('#min_confidence_threshold').val()) || 70;
            const volatilityThreshold = parseFloat($('#volatility_threshold').val()) || 0.25;
            
            // Clear previous validation messages
            $('.wisestock-threshold-validation-error').remove();
            
            // Validate confidence threshold (dropdown values)
            const allowedConfidenceValues = [40, 50, 60, 70, 80, 90];
            if (!allowedConfidenceValues.includes(minConfidence)) {
                WiseStockTimeboundAdmin.addValidationError('#min_confidence_threshold', 'Please select a valid confidence threshold', 'wisestock-threshold-validation-error');
                return false;
            }
            
            // Validate volatility threshold
            if (volatilityThreshold < 0.1 || volatilityThreshold > 1.0) {
                WiseStockTimeboundAdmin.addValidationError('#volatility_threshold_preset', 'Volatility threshold must be between 0.1 and 1.0', 'wisestock-threshold-validation-error');
                return false;
            }
            
            return true;
        },

        /**
         * Add validation error message
         */
        addValidationError: function(fieldId, message, className) {
            className = className || 'wisestock-validation-error';
            const errorHtml = '<div class="' + className + '" style="color: #d63638; font-size: 13px; margin-top: 5px;">' + message + '</div>';
            $(fieldId).closest('.wisestock-form-group').append(errorHtml);
        },

        /**
         * Show preset application notification
         */
        showPresetNotification: function() {
            // Remove existing notification
            $('.wisestock-preset-notification').remove();
            
            // Add new notification
            const notification = $('<div class="wisestock-preset-notification" style="background: #d7f1d7; border: 1px solid #6ba76b; color: #155724; padding: 10px; border-radius: 4px; margin: 10px 0;">' +
                '<strong>Store Type Applied:</strong> The prediction settings have been updated based on your selected store type.' +
                '</div>');
            
            $('#store_type_preset').closest('.wisestock-form-group').after(notification);
            
            // Auto-hide notification after 5 seconds
            setTimeout(function() {
                $('.wisestock-preset-notification').fadeOut(500, function() {
                    $(this).remove();
                });
            }, 5000);
        },



        /**
         * Show success message
         */
        showSuccessMessage: function(message) {
            const notification = $('<div class="notice notice-success is-dismissible" style="margin: 15px 0;"><p>' + message + '</p></div>');
            $('.wisestock-settings-form').prepend(notification);
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                notification.fadeOut();
            }, 5000);
        },

        /**
         * Show error message
         */
        showErrorMessage: function(message) {
            const notification = $('<div class="notice notice-error is-dismissible" style="margin: 15px 0;"><p>' + message + '</p></div>');
            $('.wisestock-settings-form').prepend(notification);
            
            // Auto-hide after 8 seconds
            setTimeout(function() {
                notification.fadeOut();
            }, 8000);
        }
    };

    /**
     * Integration with main settings form
     */
    const WiseStockTimeboundSettingsIntegration = {
        
        /**
         * Initialize integration with main settings form
         */
        init: function() {
            // Hook into main settings validation
            if (typeof WiseStockSettings !== 'undefined') {
                // Add timebound validation to main settings validation
                const originalValidateForm = WiseStockSettings.validateForm;
                WiseStockSettings.validateForm = function() {
                    let isValid = originalValidateForm.call(this);
                    
                    // Add timebound validation (timebound is now always enabled)
                    isValid = WiseStockTimeboundAdmin.validatePredictionWindows() && isValid;
                    isValid = WiseStockTimeboundAdmin.validateThresholds() && isValid;
                    
                    return isValid;
                };
            }
        }
    };

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        WiseStockTimeboundAdmin.init();
        WiseStockTimeboundSettingsIntegration.init();
    });

    // Make timebound admin available globally
    window.WiseStockTimeboundAdmin = WiseStockTimeboundAdmin;

})(jQuery);