/**
 * WiseStock - Suppliers Page JavaScript
 *
 * This file contains all the JavaScript functionality for the WiseStock Suppliers page.
 * It handles supplier management, product assignments, and supplier notifications.
 *
 * @package WiseStock
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * WiseStock Suppliers Object
     */
    window.WiseStockSuppliers = {
        
        // Auto-save variables for supplier form
        supplierAutoSaveTimeout: null,
        isSupplierSaving: false,

        /**
         * Initialize the suppliers interface
         */
        init: function() {
            this.bindEvents();
            this.initSupplierAutoSave();

            // Load dynamic sections on suppliers page
            if (window.location.href.indexOf('wisestock-suppliers') !== -1) {
                this.refreshSupplierSections();
            }
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Supplier settings form
            $(document).on('submit', '.wisestock-supplier-form', this.saveSupplierSettings);

            // Supplier management
            $(document).on('click', '.wisestock-add-supplier', this.showAddSupplierModal);
            $(document).on('click', '.wisestock-edit-supplier', this.editSupplier);
            $(document).on('click', '.wisestock-delete-supplier', this.deleteSupplier);
            $(document).on('click', '.wisestock-assign-products', this.showAssignProductsModal);
            $(document).on('click', '.wisestock-send-test-email', this.sendTestEmail);

            // Product selection
            $(document).on('change', '.wisestock-product-checkbox', this.handleProductSelection);
            $(document).on('input', '#product_search', this.filterProducts);
            $(document).on('click', '.wisestock-selected-item .remove', this.removeSelectedProduct);

            // Default supplier selection
            $(document).on('change', '.wisestock-default-supplier-select', this.handleDefaultSupplierChange);

            // Modal controls
            $(document).on('click', '.wisestock-modal-close, .wisestock-modal-cancel', this.closeModal);
            $(document).on('click', '.wisestock-modal-save', this.saveSupplier);
            $(document).on('click', '.wisestock-modal-save-edit', this.saveEditSupplier);
            $(document).on('click', '.wisestock-modal', this.closeModalOnBackdrop);
            $(document).on('click', '.wisestock-modal-content', this.preventModalClose);

            // Edit modal product selection
            $(document).on('change', '.edit-product-checkbox', this.handleEditProductSelection);
            $(document).on('input', '#edit_product_search', this.filterEditProducts);

            // Keyboard support for modals
            $(document).on('keydown', this.handleModalKeydown);

            // Real-time validation for add supplier form
            $(document).on('input', '#supplier_name', this.validateSupplierName);
            $(document).on('input', '#supplier_email', this.validateSupplierEmail);
            $(document).on('input', '#supplier_lead_time', this.validateLeadTime);
            $(document).on('change', '.wisestock-product-checkbox', this.validateProductSelection);

            // Real-time validation for supplier settings form
            $(document).on('input', '#supplier_email_subject', this.validateEmailSubject);
            $(document).on('input', '#supplier_from_email', this.validateFromEmail);

            // Auto-save event handlers for supplier form
            $(document).on('change input', '.wisestock-supplier-form input, .wisestock-supplier-form select, .wisestock-supplier-form textarea', this.handleSupplierAutoSave);
        },



        /**
         * Initialize supplier auto-save functionality
         */
        initSupplierAutoSave: function() {
            // Initialize the auto-save status indicator
            this.updateSupplierAutoSaveStatus('ready');
        },

        /**
         * Handle supplier auto-save trigger
         */
        handleSupplierAutoSave: function(event) {
            const $target = $(event.target);
            
            // Skip auto-save for certain elements (exclude add new supplier modal)
            if ($target.closest('#wisestock-add-supplier-modal').length > 0 ||
                $target.closest('#wisestock-edit-supplier-modal').length > 0 ||
                $target.is('#test_email_address') ||
                $target.is('.wisestock-modal-close') ||
                $target.is('.wisestock-modal-cancel') ||
                $target.is('.wisestock-modal-save') ||
                $target.is('.wisestock-modal-save-edit') ||
                $target.is('.wisestock-product-checkbox') ||
                $target.is('#product_search') ||
                $target.is('#edit_product_search')) {
                return;
            }

            // Clear existing timeout
            if (WiseStockSuppliers.supplierAutoSaveTimeout) {
                clearTimeout(WiseStockSuppliers.supplierAutoSaveTimeout);
            }

            // Set new timeout for auto-save
            WiseStockSuppliers.supplierAutoSaveTimeout = setTimeout(function() {
                WiseStockSuppliers.performSupplierAutoSave();
            }, 1000); // 1 second delay
        },

        /**
         * Perform supplier auto-save by triggering the existing save functionality
         */
        performSupplierAutoSave: function() {
            if (this.isSupplierSaving) {
                return; // Prevent multiple simultaneous saves
            }

            const $form = $('.wisestock-supplier-form');
            if ($form.length === 0) {
                return;
            }

            this.isSupplierSaving = true;
            this.updateSupplierAutoSaveStatus('saving');

            // Directly call the AJAX save without triggering form submit
            var formData = $form.serialize();
            
            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: formData + '&action=wisestock_save_supplier_settings&nonce=' + wisestock_suppliers.nonce,
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.updateSupplierAutoSaveStatus('saved');
                        setTimeout(function() {
                            WiseStockSuppliers.hideSupplierAutoSaveStatus();
                        }, 2000);
                    } else {
                        WiseStockSuppliers.updateSupplierAutoSaveStatus('error');
                        setTimeout(function() {
                            WiseStockSuppliers.hideSupplierAutoSaveStatus();
                        }, 3000);
                    }
                },
                error: function() {
                    WiseStockSuppliers.updateSupplierAutoSaveStatus('error');
                    setTimeout(function() {
                        WiseStockSuppliers.hideSupplierAutoSaveStatus();
                    }, 3000);
                },
                complete: function() {
                    WiseStockSuppliers.isSupplierSaving = false;
                }
            });
        },

        /**
         * Update supplier auto-save status indicator
         */
        updateSupplierAutoSaveStatus: function(status) {
            const $indicator = $('#wisestock-autosave-status');
            const $text = $indicator.find('.wisestock-autosave-text');
            
            if ($indicator.length === 0) {
                return;
            }
            
            // Remove existing status classes
            $indicator.removeClass('saving saved error ready show');
            
            // Show the indicator and add status class
            if (status !== 'ready') {
                $indicator.css('display', 'flex').addClass(status).addClass('show');
            }
            
            // Update text based on status
            switch(status) {
                case 'saving':
                    $text.text(wisestock_suppliers.strings.saving || 'Saving...');
                    break;
                case 'saved':
                    $text.text(wisestock_suppliers.strings.saved || 'Saved');
                    break;
                case 'error':
                    $text.text(wisestock_suppliers.strings.error || 'Error');
                    break;
                case 'ready':
                    $text.text(wisestock_suppliers.strings.ready || 'Ready');
                    break;
            }
        },

        /**
         * Hide supplier auto-save status indicator
         */
        hideSupplierAutoSaveStatus: function() {
            const $indicator = $('#wisestock-autosave-status');
            if ($indicator.length > 0) {
                $indicator.removeClass('show').css('display', 'none');
            }
        },

        /**
         * Save supplier settings
         */
        saveSupplierSettings: function(e) {
            e.preventDefault();
            
            // Clear previous validation errors
            WiseStockSuppliers.clearValidationErrors();
            
            var $form = $(this);
            var hasErrors = false;
            
            // Validate required fields
            var emailSubject = $('#supplier_email_subject').val().trim();
            var fromEmail = $('#supplier_from_email').val().trim();
            
            // Validate email subject
            if (!emailSubject) {
                WiseStockSuppliers.showFieldError('supplier_email_subject', wisestock_suppliers.strings.email_subject_required || 'Email subject line is required');
                hasErrors = true;
            } else if (emailSubject.length < 5) {
                WiseStockSuppliers.showFieldError('supplier_email_subject', wisestock_suppliers.strings.email_subject_min_length || 'Email subject must be at least 5 characters');
                hasErrors = true;
            }
            
            // Validate from email
            if (!fromEmail) {
                WiseStockSuppliers.showFieldError('supplier_from_email', wisestock_suppliers.strings.from_email_required || 'From email address is required');
                hasErrors = true;
            } else if (!WiseStockSuppliers.isValidEmail(fromEmail)) {
                WiseStockSuppliers.showFieldError('supplier_from_email', wisestock_suppliers.strings.valid_email_required || 'Please enter a valid email address');
                hasErrors = true;
            }
            
            if (hasErrors) {
                return;
            }
            
            var formData = $form.serialize();
            
            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: formData + '&action=wisestock_save_supplier_settings&nonce=' + wisestock_suppliers.nonce,
                beforeSend: function() {
                    $form.find('button[type="submit"]').prop('disabled', true).text(wisestock_suppliers.strings.saving || 'Saving...');
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.showNotice('success', response.data || wisestock_suppliers.strings.supplier_settings_saved || 'Supplier settings saved successfully');
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_save_supplier_settings || 'Failed to save supplier settings');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                },
                complete: function() {
                    $form.find('button[type="submit"]').prop('disabled', false).text(wisestock_suppliers.strings.save_settings || 'Save Settings');
                }
            });
        },

        /**
         * Show add supplier modal
         */
        showAddSupplierModal: function(e) {
            e.preventDefault();
            const $modal = $('#wisestock-add-supplier-modal');
            const $form = $('#wisestock-add-supplier-form');

            // Clear validation errors
            WiseStockSuppliers.clearValidationErrors();

            // Remove the display-none class and show modal with proper flex display to maintain centering
            $modal.removeClass('wisestock-dynamic-display-none').css({
                'display': 'flex',
                'opacity': '0'
            }).animate({
                'opacity': '1'
            }, 300);

            $('#supplier_name').focus();
        },

        /**
         * Edit supplier
         */
        editSupplier: function(e) {
            e.preventDefault();
            const supplierId = $(this).data('supplier-id');

            // Get supplier data
            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_supplier',
                    supplier_id: supplierId,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.populateEditModal(response.data);
                        // Show edit modal with proper flex display to maintain centering
                        $('#wisestock-edit-supplier-modal').removeClass('wisestock-dynamic-display-none').css({
                            'display': 'flex',
                            'opacity': '0'
                        }).animate({
                            'opacity': '1'
                        }, 300);
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_load_supplier_data || 'Failed to load supplier data');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                }
            });
        },

        /**
         * Delete supplier
         */
        deleteSupplier: function(e) {
            e.preventDefault();

            if (!confirm(wisestock_suppliers.strings.delete_supplier_confirm || 'Are you sure you want to delete this supplier?')) {
                return;
            }

            const supplierId = $(this).data('supplier-id');
            const $row = $(this).closest('tr');

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_delete_supplier',
                    supplier_id: supplierId,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $row.fadeOut(300, function() {
                            $(this).remove();
                        });
                        WiseStockSuppliers.showNotice('success', wisestock_suppliers.strings.supplier_deleted || 'Supplier deleted successfully');
                        
                        // Refresh all dynamic sections
                        WiseStockSuppliers.refreshSupplierSections();
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_delete_supplier || 'Failed to delete supplier');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                }
            });
        },

        /**
         * Send test email
         */
        sendTestEmail: function(e) {
            e.preventDefault();

            var testEmail = $('#test_email_address').val().trim();
            var emailSubject = $('#supplier_email_subject').val().trim();
            var fromEmail = $('#supplier_from_email').val().trim();

            // Validation
            if (!testEmail) {
                WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.test_email_required || 'Please enter a test email address');
                $('#test_email_address').focus();
                return;
            }

            if (!WiseStockSuppliers.isValidEmail(testEmail)) {
                WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.valid_email_required || 'Please enter a valid email address');
                $('#test_email_address').focus();
                return;
            }

            var $btn = $(this);
            var originalText = $btn.text();
            $btn.prop('disabled', true).text(wisestock_suppliers.strings.sending || 'Sending...');

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_send_test_email',
                    test_email: testEmail,
                    email_subject: emailSubject,
                    from_email: fromEmail,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.showNotice('success', wisestock_suppliers.strings.test_email_sent || 'Test email sent successfully to ' + testEmail);
                        $('#test_email_address').val(''); // Clear the field
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_send_test_email || 'Failed to send test email');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                },
                complete: function() {
                    $btn.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Close modal
         */
        closeModal: function(e) {
            e.preventDefault();

            // Close modal with proper animation
            $('.wisestock-modal').animate({
                'opacity': '0'
            }, 300, function() {
                $(this).addClass('wisestock-dynamic-display-none').css('display', 'none');
            });

            var $form = $('#wisestock-add-supplier-form');
            if ($form.length > 0) {
                $form[0].reset();
                // Clear product selections
                $('.wisestock-product-checkbox').prop('checked', false);
                $('.wisestock-selected-list').empty();
            }
        },

        /**
         * Handle product selection
         */
        handleProductSelection: function() {
            var $checkbox = $(this);
            var productId = $checkbox.val();
            var productName = $checkbox.siblings('.wisestock-product-name').text();
            var productSku = $checkbox.siblings('.wisestock-product-sku').text();

            if ($checkbox.is(':checked')) {
                // Add to selected list - find the correct container
                var $selectedList;
                if ($checkbox.hasClass('edit-product-checkbox')) {
                    // Edit modal
                    $selectedList = $('#edit_selected_products .wisestock-selected-list');
                } else {
                    // Add modal
                    $selectedList = $('#selected_products .wisestock-selected-list');
                }

                var $item = $(wisestock_suppliers.templates.selected_item
                    .replace('{name}', productName)
                    .replace('{sku}', productSku));
                $item.data('product-id', productId);
                $selectedList.append($item);
            } else {
                // Remove from selected list
                $('.wisestock-selected-item').each(function() {
                    if ($(this).data('product-id') === productId) {
                        $(this).remove();
                    }
                });
            }
        },

        /**
         * Filter products based on search
         */
        filterProducts: function() {
            var searchTerm = $(this).val().toLowerCase();
            $('.wisestock-product-item').each(function() {
                var productName = $(this).find('.wisestock-product-name').text().toLowerCase();
                var productSku = $(this).find('.wisestock-product-sku').text().toLowerCase();

                if (productName.includes(searchTerm) || productSku.includes(searchTerm)) {
                    $(this).show();
                } else {
                    $(this).hide();
                }
            });
        },

        /**
         * Remove selected product
         */
        removeSelectedProduct: function(e) {
            e.preventDefault();
            var $item = $(this).closest('.wisestock-selected-item');
            var productId = $item.data('product-id');

            // Uncheck the corresponding checkbox
            $('.wisestock-product-checkbox[value="' + productId + '"]').prop('checked', false);

            // Remove from selected list
            $item.remove();
        },

        /**
         * Show assign products modal for existing supplier
         */
        showAssignProductsModal: function(e) {
            e.preventDefault();
            const supplierId = $(this).data('supplier-id');

            // Get supplier data and populate edit modal
            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_supplier',
                    supplier_id: supplierId,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.populateEditModal(response.data);
                        // Show edit modal with proper flex display to maintain centering
                        $('#wisestock-edit-supplier-modal').css({
                            'display': 'flex',
                            'opacity': '0'
                        }).animate({
                            'opacity': '1'
                        }, 300);
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_load_supplier_data || 'Failed to load supplier data');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                }
            });
        },

        /**
         * Handle default supplier change
         */
        handleDefaultSupplierChange: function() {
            var $select = $(this);
            var productId = $select.data('product-id');
            var supplierId = $select.val();

            if (!productId) {
                return;
            }

            // Show saving status
            WiseStockSuppliers.updateSupplierAutoSaveStatus('saving');

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_set_default_supplier',
                    product_id: productId,
                    supplier_id: supplierId,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.updateSupplierAutoSaveStatus('saved');
                        setTimeout(function() {
                            WiseStockSuppliers.hideSupplierAutoSaveStatus();
                        }, 2000);
                    } else {
                        WiseStockSuppliers.updateSupplierAutoSaveStatus('error');
                        setTimeout(function() {
                            WiseStockSuppliers.hideSupplierAutoSaveStatus();
                        }, 3000);
                    }
                },
                error: function() {
                    WiseStockSuppliers.updateSupplierAutoSaveStatus('error');
                    setTimeout(function() {
                        WiseStockSuppliers.hideSupplierAutoSaveStatus();
                    }, 3000);
                }
            });
        },

        /**
         * Populate edit modal with supplier data
         */
        populateEditModal: function(supplier) {
            $('#edit_supplier_id').val(supplier.id);
            $('#edit_supplier_name').val(supplier.name);
            $('#edit_supplier_email').val(supplier.email);
            $('#edit_supplier_lead_time').val(supplier.lead_time_days);

            // Clear existing selections
            $('.edit-product-checkbox').prop('checked', false);
            $('#edit_selected_products .wisestock-selected-list').empty();

            // Select products if any
            if (supplier.product_ids && supplier.product_ids.length > 0) {
                supplier.product_ids.forEach(function(productId) {
                    $('.edit-product-checkbox[value="' + productId + '"]').prop('checked', true);
                    WiseStockSuppliers.addToEditSelectedList(productId);
                });
            }
        },

        /**
         * Add product to edit selected list
         */
        addToEditSelectedList: function(productId) {
            var $checkbox = $('.edit-product-checkbox[value="' + productId + '"]');
            var productName = $checkbox.siblings('.wisestock-product-name').text();
            var productSku = $checkbox.siblings('.wisestock-product-sku').text();

            var $selectedList = $('#edit_selected_products .wisestock-selected-list');
            var $item = $(wisestock_suppliers.templates.selected_item
                .replace('{name}', productName)
                .replace('{sku}', productSku));
            $item.data('product-id', productId);
            $selectedList.append($item);
        },

        /**
         * Handle edit product selection
         */
        handleEditProductSelection: function() {
            var $checkbox = $(this);
            var productId = $checkbox.val();

            if ($checkbox.is(':checked')) {
                WiseStockSuppliers.addToEditSelectedList(productId);
            } else {
                // Remove from selected list
                $('#edit_selected_products .wisestock-selected-item').each(function() {
                    if ($(this).data('product-id') === productId) {
                        $(this).remove();
                    }
                });
            }
        },

        /**
         * Filter edit products
         */
        filterEditProducts: function() {
            var searchTerm = $(this).val().toLowerCase();
            $('#edit_available_products .wisestock-product-item').each(function() {
                var productName = $(this).find('.wisestock-product-name').text().toLowerCase();
                var productSku = $(this).find('.wisestock-product-sku').text().toLowerCase();

                if (productName.includes(searchTerm) || productSku.includes(searchTerm)) {
                    $(this).show();
                } else {
                    $(this).hide();
                }
            });
        },

        /**
         * Save edit supplier
         */
        saveEditSupplier: function(e) {
            e.preventDefault();

            var supplierId = $('#edit_supplier_id').val();
            var supplierName = $('#edit_supplier_name').val().trim();
            var supplierEmail = $('#edit_supplier_email').val().trim();
            var leadTime = parseInt($('#edit_supplier_lead_time').val()) || 0;

            // Get selected products
            var selectedProducts = [];
            $('.edit-product-checkbox:checked').each(function() {
                selectedProducts.push($(this).val());
            });

            // Clear any previous validation errors
            WiseStockSuppliers.clearValidationErrors();

            // Validation
            var hasErrors = false;

            if (!supplierName) {
                WiseStockSuppliers.showFieldError('edit_supplier_name', wisestock_suppliers.strings.supplier_name_required || 'Supplier name is required');
                $('#edit_supplier_name').addClass('error');
                hasErrors = true;
            } else if (supplierName.length < 2) {
                WiseStockSuppliers.showFieldError('edit_supplier_name', wisestock_suppliers.strings.supplier_name_min_length || 'Supplier name must be at least 2 characters long');
                $('#edit_supplier_name').addClass('error');
                hasErrors = true;
            } else if (WiseStockSuppliers.checkDuplicateSupplierName(supplierName, supplierId)) {
                WiseStockSuppliers.showFieldError('edit_supplier_name', wisestock_suppliers.strings.duplicate_supplier_name || 'A supplier with this name already exists');
                $('#edit_supplier_name').addClass('error');
                hasErrors = true;
            }

            if (!supplierEmail) {
                WiseStockSuppliers.showFieldError('edit_supplier_email', wisestock_suppliers.strings.supplier_email_required || 'Supplier email is required');
                $('#edit_supplier_email').addClass('error');
                hasErrors = true;
            } else if (!WiseStockSuppliers.isValidEmail(supplierEmail)) {
                WiseStockSuppliers.showFieldError('edit_supplier_email', wisestock_suppliers.strings.valid_email_required || 'Please enter a valid email address');
                $('#edit_supplier_email').addClass('error');
                hasErrors = true;
            } else if (WiseStockSuppliers.checkDuplicateSupplierEmail(supplierEmail, supplierId)) {
                WiseStockSuppliers.showFieldError('edit_supplier_email', wisestock_suppliers.strings.duplicate_supplier_email || 'A supplier with this email already exists');
                $('#edit_supplier_email').addClass('error');
                hasErrors = true;
            }

            if (leadTime < 0 || leadTime > 365) {
                WiseStockSuppliers.showFieldError('edit_supplier_lead_time', wisestock_suppliers.strings.lead_time_range || 'Lead time must be between 0 and 365 days');
                $('#edit_supplier_lead_time').addClass('error');
                hasErrors = true;
            }

            if (selectedProducts.length === 0) {
                WiseStockSuppliers.showFieldError('edit_supplier_products', wisestock_suppliers.strings.select_at_least_one_product || 'Please select at least one product');
                hasErrors = true;
            }

            if (hasErrors) {
                return;
            }

            var $saveBtn = $(this);
            var originalText = $saveBtn.text();
            $saveBtn.prop('disabled', true).text(wisestock_suppliers.strings.updating || 'Updating...');

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_update_supplier',
                    supplier_id: supplierId,
                    supplier_name: supplierName,
                    supplier_email: supplierEmail,
                    supplier_lead_time: leadTime,
                    supplier_products: selectedProducts,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.closeModal(e);
                        WiseStockSuppliers.showNotice('success', wisestock_suppliers.strings.supplier_updated || 'Supplier updated successfully');
                        
                        // Refresh all dynamic sections
                        WiseStockSuppliers.refreshSupplierSections();
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_update_supplier || 'Failed to update supplier');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                },
                complete: function() {
                    $saveBtn.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Close modal when clicking backdrop
         */
        closeModalOnBackdrop: function(e) {
            if (e.target === this) {
                WiseStockSuppliers.closeModal(e);
            }
        },

        /**
         * Prevent modal from closing when clicking inside content
         */
        preventModalClose: function(e) {
            e.stopPropagation();
        },

        /**
         * Handle keyboard events for modals
         */
        handleModalKeydown: function(e) {
            if (e.key === 'Escape' && $('.wisestock-modal:visible').length > 0) {
                WiseStockSuppliers.closeModal(e);
            }
        },

        /**
         * Validate email address
         */
        isValidEmail: function(email) {
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        },

        /**
         * Clear all validation errors
         */
        clearValidationErrors: function() {
            $('.wisestock-validation-error').hide();
            $('.wisestock-form-group').removeClass('error');
            $('.wisestock-form-input').removeClass('error');
        },

        /**
         * Show field validation error
         */
        showFieldError: function(fieldId, message) {
            $('#' + fieldId + '_error').text(message).show();
            $('#' + fieldId).closest('.wisestock-form-group').addClass('error');
            $('#' + fieldId).addClass('error');
        },

        /**
         * Check for duplicate supplier name
         */
        checkDuplicateSupplierName: function(name, excludeId) {
            var isDuplicate = false;
            $('.wisestock-table tbody tr').each(function() {
                var $row = $(this);
                var existingName = $row.find('td:first strong').text().trim().toLowerCase();
                var existingId = $row.find('.wisestock-edit-supplier').data('supplier-id');
                
                if (existingName === name.toLowerCase() && existingId != excludeId) {
                    isDuplicate = true;
                    return false; // break
                }
            });
            return isDuplicate;
        },

        /**
         * Check for duplicate supplier email
         */
        checkDuplicateSupplierEmail: function(email, excludeId) {
            var isDuplicate = false;
            $('.wisestock-table tbody tr').each(function() {
                var $row = $(this);
                var existingEmail = $row.find('td:nth-child(2) a').text().trim().toLowerCase();
                var existingId = $row.find('.wisestock-edit-supplier').data('supplier-id');
                
                if (existingEmail === email.toLowerCase() && existingId != excludeId) {
                    isDuplicate = true;
                    return false; // break
                }
            });
            return isDuplicate;
        },

        /**
         * Real-time validation for supplier name
         */
        validateSupplierName: function() {
            var name = $(this).val().trim();
            var $field = $(this);
            var $errorDiv = $('#supplier_name_error');
            var $formGroup = $field.closest('.wisestock-form-group');

            // Clear previous error
            $errorDiv.hide();
            $formGroup.removeClass('error');
            $field.removeClass('error');

            if (name.length === 0) {
                WiseStockSuppliers.showFieldError('supplier_name', wisestock_suppliers.strings.supplier_name_required || 'Supplier name is required');
            } else if (name.length < 2) {
                WiseStockSuppliers.showFieldError('supplier_name', wisestock_suppliers.strings.supplier_name_min_length || 'Supplier name must be at least 2 characters');
            } else if (WiseStockSuppliers.checkDuplicateSupplierName(name)) {
                WiseStockSuppliers.showFieldError('supplier_name', wisestock_suppliers.strings.duplicate_supplier_name || 'A supplier with this name already exists');
            }
        },

        /**
         * Real-time validation for supplier email
         */
        validateSupplierEmail: function() {
            var email = $(this).val().trim();
            var $field = $(this);
            var $errorDiv = $('#supplier_email_error');
            var $formGroup = $field.closest('.wisestock-form-group');

            // Clear previous error
            $errorDiv.hide();
            $formGroup.removeClass('error');
            $field.removeClass('error');

            if (email.length === 0) {
                WiseStockSuppliers.showFieldError('supplier_email', wisestock_suppliers.strings.supplier_email_required || 'Email address is required');
            } else if (!WiseStockSuppliers.isValidEmail(email)) {
                WiseStockSuppliers.showFieldError('supplier_email', wisestock_suppliers.strings.valid_email_required || 'Please enter a valid email address');
            } else if (WiseStockSuppliers.checkDuplicateSupplierEmail(email)) {
                WiseStockSuppliers.showFieldError('supplier_email', wisestock_suppliers.strings.duplicate_supplier_email || 'A supplier with this email already exists');
            }
        },

        /**
         * Real-time validation for lead time
         */
        validateLeadTime: function() {
            var leadTime = parseInt($(this).val()) || 0;
            var $field = $(this);
            var $errorDiv = $('#supplier_lead_time_error');
            var $formGroup = $field.closest('.wisestock-form-group');

            // Clear previous error
            $errorDiv.hide();
            $formGroup.removeClass('error');
            $field.removeClass('error');

            if (leadTime < 1) {
                WiseStockSuppliers.showFieldError('supplier_lead_time', wisestock_suppliers.strings.lead_time_min || 'Lead time must be at least 1 day');
            } else if (leadTime > 365) {
                WiseStockSuppliers.showFieldError('supplier_lead_time', wisestock_suppliers.strings.lead_time_max || 'Lead time cannot exceed 365 days');
            }
        },

        /**
         * Real-time validation for product selection
         */
        validateProductSelection: function() {
            var selectedProducts = [];
            $('.wisestock-product-checkbox:checked').each(function() {
                selectedProducts.push($(this).val());
            });

            var selectedFromList = [];
            $('#selected_products .wisestock-selected-item').each(function() {
                selectedFromList.push($(this).data('product-id'));
            });

            if (selectedProducts.length === 0 && selectedFromList.length > 0) {
                selectedProducts = selectedFromList;
            }

            var $errorDiv = $('#supplier_products_error');
            var $formGroup = $('.wisestock-form-group').has('#supplier_products_error');

            // Clear previous error
            $errorDiv.hide();
            $formGroup.removeClass('error');

            if (selectedProducts.length === 0) {
                WiseStockSuppliers.showFieldError('supplier_products', wisestock_suppliers.strings.select_at_least_one_product || 'Please select at least one product');
            }
        },



        /**
         * Real-time validation for email subject
         */
        validateEmailSubject: function() {
            var subject = $(this).val().trim();
            var $field = $(this);
            var $errorDiv = $('#supplier_email_subject_error');
            var $formGroup = $field.closest('.wisestock-form-group');

            // Clear previous error
            $errorDiv.hide();
            $formGroup.removeClass('error');
            $field.removeClass('error');

            if (subject.length === 0) {
                WiseStockSuppliers.showFieldError('supplier_email_subject', wisestock_suppliers.strings.email_subject_required || 'Email subject line is required');
            } else if (subject.length < 5) {
                WiseStockSuppliers.showFieldError('supplier_email_subject', wisestock_suppliers.strings.email_subject_min_length || 'Email subject must be at least 5 characters');
            }
        },

        /**
         * Real-time validation for from email
         */
        validateFromEmail: function() {
            var email = $(this).val().trim();
            var $field = $(this);
            var $errorDiv = $('#supplier_from_email_error');
            var $formGroup = $field.closest('.wisestock-form-group');

            // Clear previous error
            $errorDiv.hide();
            $formGroup.removeClass('error');
            $field.removeClass('error');

            if (email.length === 0) {
                WiseStockSuppliers.showFieldError('supplier_from_email', wisestock_suppliers.strings.from_email_required || 'From email address is required');
            } else if (!WiseStockSuppliers.isValidEmail(email)) {
                WiseStockSuppliers.showFieldError('supplier_from_email', wisestock_suppliers.strings.valid_email_required || 'Please enter a valid email address');
            }
        },

        /**
         * Save supplier
         */
        saveSupplier: function(e) {
            e.preventDefault();

            // Clear previous validation errors
            WiseStockSuppliers.clearValidationErrors();

            // Get values directly from input fields (more reliable than form serialization)
            var supplierName = $('#supplier_name').val() || '';
            var supplierEmail = $('#supplier_email').val() || '';
            var leadTime = parseInt($('#supplier_lead_time').val()) || 0;

            // Get selected products
            var selectedProducts = [];
            $('.wisestock-product-checkbox:checked').each(function() {
                selectedProducts.push($(this).val());
            });

            // Debug: Check what's in the selected products list
            var selectedFromList = [];
            $('#selected_products .wisestock-selected-item').each(function() {
                selectedFromList.push($(this).data('product-id'));
            });

            // Fallback: If no checkboxes are checked but we have items in the selected list, use those
            if (selectedProducts.length === 0 && selectedFromList.length > 0) {
                selectedProducts = selectedFromList;
            }

            // Trim the values
            supplierName = supplierName.trim();
            supplierEmail = supplierEmail.trim();

            // Validation
            var hasErrors = false;

            // Validate supplier name
            if (!supplierName) {
                WiseStockSuppliers.showFieldError('supplier_name', wisestock_suppliers.strings.supplier_name_required || 'Supplier name is required');
                hasErrors = true;
            } else if (supplierName.length < 2) {
                WiseStockSuppliers.showFieldError('supplier_name', wisestock_suppliers.strings.supplier_name_min_length || 'Supplier name must be at least 2 characters');
                hasErrors = true;
            }

            // Validate email
            if (!supplierEmail) {
                WiseStockSuppliers.showFieldError('supplier_email', wisestock_suppliers.strings.supplier_email_required || 'Email address is required');
                hasErrors = true;
            } else if (!WiseStockSuppliers.isValidEmail(supplierEmail)) {
                WiseStockSuppliers.showFieldError('supplier_email', wisestock_suppliers.strings.valid_email_required || 'Please enter a valid email address');
                hasErrors = true;
            }

            // Validate lead time
            if (!leadTime || leadTime < 1) {
                WiseStockSuppliers.showFieldError('supplier_lead_time', wisestock_suppliers.strings.lead_time_min || 'Lead time must be at least 1 day');
                hasErrors = true;
            } else if (leadTime > 365) {
                WiseStockSuppliers.showFieldError('supplier_lead_time', wisestock_suppliers.strings.lead_time_max || 'Lead time cannot exceed 365 days');
                hasErrors = true;
            }

            // Validate product selection
            if (selectedProducts.length === 0) {
                WiseStockSuppliers.showFieldError('supplier_products', wisestock_suppliers.strings.select_at_least_one_product || 'Please select at least one product');
                hasErrors = true;
            }

            // Check for duplicates
            if (supplierName && WiseStockSuppliers.checkDuplicateSupplierName(supplierName)) {
                WiseStockSuppliers.showFieldError('supplier_name', wisestock_suppliers.strings.duplicate_supplier_name || 'A supplier with this name already exists');
                hasErrors = true;
            }

            if (supplierEmail && WiseStockSuppliers.checkDuplicateSupplierEmail(supplierEmail)) {
                WiseStockSuppliers.showFieldError('supplier_email', wisestock_suppliers.strings.duplicate_supplier_email || 'A supplier with this email already exists');
                hasErrors = true;
            }

            if (hasErrors) {
                return;
            }

            var $saveBtn = $(this);
            var originalText = $saveBtn.text();

            $saveBtn.prop('disabled', true).text(wisestock_suppliers.strings.adding || 'Adding...');

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_add_supplier',
                    supplier_name: supplierName,
                    supplier_email: supplierEmail,
                    supplier_lead_time: leadTime,
                    supplier_products: selectedProducts,
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSuppliers.closeModal(e);
                        WiseStockSuppliers.showNotice('success', response.data.message || wisestock_suppliers.strings.supplier_added || 'Supplier added successfully');
                        
                        // Refresh dynamic sections via AJAX
                        WiseStockSuppliers.refreshSupplierSections();
                    } else {
                        WiseStockSuppliers.showNotice('error', response.data || wisestock_suppliers.strings.failed_add_supplier || 'Failed to add supplier');
                    }
                },
                error: function() {
                    WiseStockSuppliers.showNotice('error', wisestock_suppliers.strings.network_error || 'Network error occurred');
                },
                complete: function() {
                    $saveBtn.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Refresh all supplier sections via AJAX
         */
        refreshSupplierSections: function() {
            // Check if we're on the suppliers page
            if (window.location.href.indexOf('wisestock-suppliers') === -1) {
                return;
            }

            this.loadSupplierStats();
            this.loadSuppliersTable();
            this.loadProductSupplierOverview();
            this.loadMultiSupplierProducts();
        },

        /**
         * Load supplier statistics via AJAX
         */
        loadSupplierStats: function() {
            var $statsContainer = $('.wisestock-supplier-stats');
            if ($statsContainer.length === 0) {
                return;
            }

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_supplier_stats',
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var stats = response.data;
                        $statsContainer.find('.wisestock-stat-item').eq(0).find('.wisestock-stat-value').text(stats.total_suppliers);
                        $statsContainer.find('.wisestock-stat-item').eq(1).find('.wisestock-stat-value').text(stats.total_products_assigned);
                    }
                }
            });
        },

        /**
         * Load suppliers table via AJAX
         */
        loadSuppliersTable: function() {
            var $tableContainer = $('.wisestock-current-suppliers .wisestock-card-body');
            if ($tableContainer.length === 0) {
                return;
            }

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_suppliers_table',
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $tableContainer.html(response.data);
                    }
                }
            });
        },

        /**
         * Load product-supplier overview via AJAX
         */
        loadProductSupplierOverview: function() {
            var $overviewContainer = $('.wisestock-product-supplier-overview .wisestock-card-body');
            if ($overviewContainer.length === 0) {
                return;
            }

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_product_supplier_overview',
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $overviewContainer.html(response.data);
                    }
                }
            });
        },

        /**
         * Load multi-supplier products via AJAX
         */
        loadMultiSupplierProducts: function() {
            var $productsContainer = $('.wisestock-multi-supplier-products .wisestock-card-body');
            if ($productsContainer.length === 0) {
                return;
            }

            $.ajax({
                url: wisestock_suppliers.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_multi_supplier_products',
                    nonce: wisestock_suppliers.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $productsContainer.html(response.data);
                    }
                }
            });
        },

        /**
         * Show notice
         */
        showNotice: function(type, message) {
            var noticeTemplate = wisestock_suppliers.templates.notice || '<div class="notice notice-{type} is-dismissible"><p>{message}</p></div>';
            var $notice = $(noticeTemplate
                .replace('{type}', type)
                .replace('{message}', message));
            $('.wisestock-admin .wrap').prepend($notice);

            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        // Check if we're on the suppliers page before initializing
        if (typeof wisestock_suppliers !== 'undefined') {
            WiseStockSuppliers.init();
        }
    });

})(jQuery);
