/**
 * WiseStock - Settings JavaScript
 *
 * This file handles the settings page functionality including
 * prediction mode toggling, preset handlers, and form validation.
 *
 * @package WiseStock
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * WiseStock Settings Object
     */
    window.WiseStockSettings = {
        
        // Auto-save variables
        autoSaveTimeout: null,
        isSaving: false,
        
        /**
         * Initialize settings functionality
         */
        init: function() {
            this.bindEvents();
            this.initializeAutoSave();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Store type preset handlers
            $(document).on('change', '#timebound_store_type_preset', this.handleTimeboundPreset);
            
            // Advanced settings toggle handlers
            $(document).on('change', '#timebound_show_advanced_settings', this.toggleTimeboundAdvancedSettings);
            
            // Form validation
            $(document).on('input', '.wisestock-velocity-input', this.validateVelocityWeights);
            $(document).on('input', '.wisestock-confidence-input', this.validateConfidenceValues);
            
            // Auto-save event handlers
            $(document).on('change input', 'input, select, textarea', this.handleAutoSave);
            
            // Email notifications conditional visibility
            $(document).on('change', '#email_notifications', this.toggleEmailNotificationsFields);
            
            // Test notification email
            $(document).on('click', '.wisestock-send-test-notification-email', this.sendTestNotificationEmail);
            
            // Reset installation function
            $(document).on('click', '#reset-installation-btn', this.resetInstallation);
            
            // Delete data toggle confirmation
            $(document).on('change', '#delete_data_on_uninstall', this.handleDeleteDataToggle);
            $(document).on('click', '.wisestock-popup-close, #cancel-delete-btn, #cancel-delete-btn-2', this.closeDeleteDataConfirmation);
            $(document).on('click', '#confirm-delete-btn', this.confirmDeleteDataSetting);
            $(document).on('input', '#delete-confirmation-text', this.toggleDeleteButton);
        },

        /**
         * Initialize auto-save functionality
         */
        initializeAutoSave: function() {
            // Initialize the auto-save status indicator
            this.updateAutoSaveStatus('ready');
        },

        /**
         * Handle auto-save trigger
         */
        handleAutoSave: function(event) {
            const $target = $(event.target);
            
            // Skip auto-save for certain elements
            if ($target.is('#reset-installation-btn') || 

                $target.is('#timebound_show_advanced_settings') ||
                $target.is('#delete_data_on_uninstall') ||
                $target.is('#delete-confirmation-text')) {
                return;
            }

            // Clear existing timeout
            if (WiseStockSettings.autoSaveTimeout) {
                clearTimeout(WiseStockSettings.autoSaveTimeout);
            }

            // Set new timeout for auto-save
            WiseStockSettings.autoSaveTimeout = setTimeout(function() {
                WiseStockSettings.performAutoSave();
            }, 100); // 1 second delay
        },

        /**
         * Perform auto-save by triggering the existing save functionality
         */
        performAutoSave: function() {
            if (this.isSaving) {
                return; // Prevent multiple simultaneous saves
            }

            this.isSaving = true;
            this.updateAutoSaveStatus('saving');

            // Create a hidden submit button and trigger the form submission
            const $form = $('.wisestock-settings-form');
            if ($form.length === 0) {
                this.isSaving = false;
                return;
            }

            // Create a temporary submit button
            const $tempButton = $('<button type="submit" style="display:none;"></button>');
            $form.append($tempButton);

            // Trigger the form submission
            $tempButton.trigger('click');

            // Remove the temporary button
            $tempButton.remove();

            // Reset the saving flag after a short delay
            setTimeout(() => {
                this.isSaving = false;
            }, 1000);
        },

        /**
         * Update auto-save status indicator
         */
        updateAutoSaveStatus: function(status) {
            const $indicator = $('#wisestock-autosave-status');
            const $text = $indicator.find('.wisestock-autosave-text');
            
            if ($indicator.length === 0) {
                return;
            }
            
            // Remove existing status classes
            $indicator.removeClass('saving saved error ready show');
            
            // Show the indicator and add status class
            if (status !== 'ready') {
                $indicator.css('display', 'flex').addClass(status).addClass('show');
            }
            
            // Update text based on status
            switch(status) {
                case 'saving':
                    $text.text(wisestock_admin.strings.saving);
                    break;
                case 'saved':
                    $text.text(wisestock_admin.strings.saved);
                    break;
                case 'error':
                    $text.text(wisestock_admin.strings.save_error);
                    break;
                case 'ready':
                    $text.text('');

                    break;
            }
        },

        /**
         * Hide auto-save status indicator
         */
        hideAutoSaveStatus: function() {
            const $indicator = $('#wisestock-autosave-status');
            $indicator.removeClass('show');
            
            // Hide the indicator after animation
            setTimeout(function() {
                $indicator.css('display', 'none').removeClass('saving saved error ready');
            }, 300);
        },



        /**
         * Handle timebound preset selection
         */
        handleTimeboundPreset: function() {
            const select = document.getElementById('timebound_store_type_preset');
            const option = select.options[select.selectedIndex];
            const description = document.getElementById('timebound-preset-description');
            
            if (description) {
                description.textContent = option.getAttribute('data-description');
            }

            // Update form fields with preset values
            document.getElementById('timebound_velocity_7_day').value = option.getAttribute('data-velocity-7');
            document.getElementById('timebound_velocity_14_day').value = option.getAttribute('data-velocity-14');
            document.getElementById('timebound_velocity_30_day').value = option.getAttribute('data-velocity-30');
            document.getElementById('timebound_min_confidence').value = option.getAttribute('data-confidence-min');
            document.getElementById('timebound_sparse_data_confidence').value = option.getAttribute('data-confidence-sparse');
            document.getElementById('timebound_good_data_min_confidence').value = option.getAttribute('data-confidence-good');


        },



        /**
         * Toggle timebound advanced settings
         */
        toggleTimeboundAdvancedSettings: function() {
            const checkbox = document.getElementById('timebound_show_advanced_settings');
            const velocityCard = document.getElementById('timebound-velocity-settings-card');
            const confidenceCard = document.getElementById('timebound-confidence-settings-card');
            const predictionCard = document.getElementById('timebound-prediction-settings-card');
            
            if (checkbox.checked) {
                velocityCard.style.display = 'block';
                velocityCard.classList.remove('wisestock-dynamic-display-none');
                confidenceCard.style.display = 'block';
                confidenceCard.classList.remove('wisestock-dynamic-display-none');
                predictionCard.style.display = 'block';
                predictionCard.classList.remove('wisestock-dynamic-display-none');
            } else {
                velocityCard.style.display = 'none';
                velocityCard.classList.add('wisestock-dynamic-display-none');
                confidenceCard.style.display = 'none';
                confidenceCard.classList.add('wisestock-dynamic-display-none');
                predictionCard.style.display = 'none';
                predictionCard.classList.add('wisestock-dynamic-display-none');
            }
        },

        /**
         * Validate velocity weights
         */
        validateVelocityWeights: function() {
            const inputs = document.querySelectorAll('.wisestock-velocity-input');
            let total = 0;
            
            inputs.forEach(function(input) {
                total += parseFloat(input.value) || 0;
            });

            const warning = document.getElementById('velocity-weight-warning');
            if (warning) {
                if (Math.abs(total - 1.0) > 0.01) {
                    warning.style.display = 'block';
                    warning.textContent = wisestock_admin.strings.velocity_weight_warning.replace('%total%', total.toFixed(2));
                } else {
                    warning.style.display = 'none';
                }
            }
        },

        /**
         * Validate confidence values
         */
        validateConfidenceValues: function() {
            const minConfidence = parseFloat(document.getElementById('classic_min_confidence').value) || 0;
            const sparseConfidence = parseFloat(document.getElementById('classic_sparse_data_confidence').value) || 0;
            const goodConfidence = parseFloat(document.getElementById('classic_good_data_min_confidence').value) || 0;

            const warning = document.getElementById('confidence-warning');
            if (warning) {
                if (sparseConfidence < minConfidence || goodConfidence < sparseConfidence) {
                    warning.style.display = 'block';
                    warning.textContent = wisestock_admin.strings.confidence_warning;
                } else {
                    warning.style.display = 'none';
                }
            }
        },

        /**
         * Reset installation
         */
        resetInstallation: function() {
            window.location.href = wisestock_admin.installation_url;
        },


        /**
         * Toggle email notifications fields visibility
         */
        toggleEmailNotificationsFields: function() {
            var isChecked = $(this).is(':checked');
            var $fields = $('#email_notifications_fields');
            var $notificationTypes = $('#notification_types_fields');
            
            if (isChecked) {
                $fields.slideDown(300);
                $notificationTypes.slideDown(300);
            } else {
                $fields.slideUp(300);
                $notificationTypes.slideUp(300);
            }
        },

        /**
         * Send test notification email
         */
        sendTestNotificationEmail: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var originalText = $button.text();
            
            // Disable button and show loading state
            $button.prop('disabled', true).text('Sending...');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_send_test_notification_email',
                    nonce: wisestock_admin.nonce
                },
                success: function(response) {
                    alert(response.data);
                },
                complete: function() {
                    // Re-enable button
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle delete data toggle with confirmation
         */
        handleDeleteDataToggle: function(event) {
            const $checkbox = $(this);
            const isChecked = $checkbox.is(':checked');
            
            if (isChecked) {
                // Prevent the checkbox from appearing checked until confirmed
                event.preventDefault();
                $checkbox.prop('checked', false);
                
                // Show confirmation popup when enabling
                document.getElementById('wisestock-delete-confirmation-popup').style.display = 'flex';
                // Store reference to checkbox for later use
                WiseStockSettings.pendingDeleteToggle = $checkbox;
            } else {
                // Allow unchecking without confirmation - save immediately
                WiseStockSettings.saveDeleteDataSetting(false);
            }
        },

        /**
         * Close delete data confirmation popup
         */
        closeDeleteDataConfirmation: function() {
            document.getElementById('wisestock-delete-confirmation-popup').style.display = 'none';
            document.getElementById('delete-confirmation-text').value = '';
            document.getElementById('confirm-delete-btn').disabled = true;
            document.getElementById('delete-confirmation-error').style.display = 'none';
            
            // Uncheck the checkbox if user cancels
            if (WiseStockSettings.pendingDeleteToggle) {
                WiseStockSettings.pendingDeleteToggle.prop('checked', false);
                WiseStockSettings.pendingDeleteToggle = null;
            }
        },

        /**
         * Toggle delete button based on confirmation text
         */
        toggleDeleteButton: function() {
            const confirmationText = document.getElementById('delete-confirmation-text').value;
            const deleteBtn = document.getElementById('confirm-delete-btn');
            
            deleteBtn.disabled = confirmationText !== 'DELETE DATA';
        },

        /**
         * Confirm and save delete data setting
         */
        confirmDeleteDataSetting: function() {
            const confirmationText = document.getElementById('delete-confirmation-text').value;
            
            if (confirmationText !== 'DELETE DATA') {
                document.getElementById('delete-confirmation-error').style.display = 'block';
                return;
            }

            // Clear error message and hide the popup
            document.getElementById('delete-confirmation-error').style.display = 'none';
            document.getElementById('wisestock-delete-confirmation-popup').style.display = 'none';
            document.getElementById('delete-confirmation-text').value = '';
            document.getElementById('confirm-delete-btn').disabled = true;
            
            // Save the setting as enabled (don't clear pendingDeleteToggle until after save)
            WiseStockSettings.saveDeleteDataSetting(true);
        },

        /**
         * Save delete data on uninstall setting
         */
        saveDeleteDataSetting: function(enabled) {
            WiseStockSettings.updateAutoSaveStatus('saving');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_save_settings',
                    nonce: wisestock_admin.nonce,
                    delete_data_on_uninstall: enabled ? '1' : '0'
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockSettings.updateAutoSaveStatus('saved');
                        
                        // Update checkbox to reflect saved state
                        if (enabled && WiseStockSettings.pendingDeleteToggle) {
                            WiseStockSettings.pendingDeleteToggle.prop('checked', true);
                        }
                        
                        // Clear the pending reference after successful save
                        WiseStockSettings.pendingDeleteToggle = null;
                        
                        // Hide status after delay
                        setTimeout(function() {
                            WiseStockSettings.hideAutoSaveStatus();
                        }, 2000);
                    } else {
                        WiseStockSettings.updateAutoSaveStatus('error');
                    }
                },
                error: function() {
                    WiseStockSettings.updateAutoSaveStatus('error');
                    
                    // Reset checkbox state on error
                    if (WiseStockSettings.pendingDeleteToggle) {
                        WiseStockSettings.pendingDeleteToggle.prop('checked', false);
                        WiseStockSettings.pendingDeleteToggle = null;
                    }
                }
            });
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        WiseStockSettings.init();
    });

})(jQuery);
