/**
 * WiseStock Installation Wizard
 *
 * @package WiseStock_Pro
 * @since 1.0.0
 */

(function($) {
    'use strict';

    const WiseStockInstallation = {

        currentStep: 1,
        totalSteps: 4,
        installationData: {
            store_preset: '',
            prediction_mode: '',
            stock_management: '',
            demo_stats: false
        },

        /**
         * Initialize installation wizard
         */
        init: function() {

            // Load existing settings if available
            this.loadExistingSettings();

            // Set default prediction mode if not available
            if (!this.installationData.prediction_mode) {
                // Default to classic when prediction selection step is not available
                this.installationData.prediction_mode = 'classic';
            }

            this.bindEvents();
            this.updateProgress();
        },

        /**
         * Load existing settings from WordPress options
         */
        loadExistingSettings: function() {
            // Check if wisestock_admin has existing settings
            if (typeof wisestock_admin !== 'undefined' && wisestock_admin.existing_settings) {
                const settings = wisestock_admin.existing_settings;

                // Load existing values
                if (settings.store_preset) {
                    this.installationData.store_preset = settings.store_preset;
                }
                if (settings.prediction_mode) {
                    this.installationData.prediction_mode = settings.prediction_mode;
                }
                if (typeof settings.demo_stats !== 'undefined') {
                    this.installationData.demo_stats = settings.demo_stats;

                    // Set checkbox state based on existing value
                    const checkbox = document.getElementById('enable-demo-stats');
                    if (checkbox) {
                        checkbox.checked = settings.demo_stats;
                    }
                }

                // Load presets on initialization since we start with store category
                this.loadPresetsForStoreType('retail'); // Default to retail presets

            }
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            const self = this;

            // Option card clicks (exclude acknowledgement cards)
            $(document).on('click', '.wisestock-option-card:not(.wisestock-acknowledgement-card)', function(e) {
                self.handleOptionClick.call(this, e);
            });
            $(document).on('click', '.wisestock-preset-card', function(e) {
                self.handleOptionClick.call(this, e);
            });

            // Navigation buttons
            $(document).on('click', '#wisestock-next-btn', function(e) {
                self.handleNext.call(this, e);
            });
            $(document).on('click', '#wisestock-prev-btn', function(e) {
                self.handlePrevious.call(this, e);
            });
            $(document).on('click', '#wisestock-complete-btn', function(e) {
                self.handleComplete.call(self, e);
            });

            // Demo stats toggle
            $(document).on('change', '#enable-demo-stats', function(e) {
                self.handleDemoToggle.call(self, e);
            });

            // Store type change - show digital products info
            $(document).on('change', '.wisestock-option-card[data-value="digital"], .wisestock-option-card[data-value="mixed"]', function(e) {
                self.showDigitalInfo.call(this, e);
            });
            $(document).on('change', '.wisestock-option-card[data-value="retail"], .wisestock-option-card[data-value="dropshipping"]', function(e) {
                self.hideDigitalInfo.call(this, e);
            });

            // Go to dashboard button
            $(document).on('click', '#wisestock-go-to-dashboard', function(e) {
                self.handleGoToDashboard.call(this, e);
            });

            // Exit installation button
            $(document).on('click', '#wisestock-exit-installation', function(e) {
                e.preventDefault();
                // Go back to settings page
                if (wisestock_admin.settings_url) {
                    window.location.href = wisestock_admin.settings_url;
                } else {
                    window.location.href = '/wp-admin/admin.php?page=wisestock-settings';
                }
            });

            // Help modal functions
            $(document).on('click', '.wisestock-help-button button', function(e) {
                e.preventDefault();
                self.showStockManagementHelp();
            });

            $(document).on('click', '.wisestock-modal-close, .wisestock-modal-footer button', function(e) {
                e.preventDefault();
                self.closeHelpModal();
            });

            // Close modal when clicking outside
            $(document).on('click', '#wisestock-help-modal', function(e) {
                if (e.target === this) {
                    self.closeHelpModal();
                }
            });

            // Close modal with Escape key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    const modal = document.getElementById('wisestock-help-modal');
                    if (modal && !modal.classList.contains('wisestock-dynamic-display-none')) {
                        self.closeHelpModal();
                    }
                }
            });
        },

        /**
         * Handle option card click
         */
        handleOptionClick: function(e) {
            e.preventDefault();

            const $card = $(this);
            const $step = $card.closest('.wisestock-installation-step');
            const value = $card.data('value');

            // Remove active class from all cards in this step
            $step.find('.wisestock-option-card, .wisestock-preset-card').removeClass('active');

            // Add active class to clicked card
            $card.addClass('active');

            // Store the value based on current step
            WiseStockInstallation.storeStepData(value);

            // Enable next button
            $('#wisestock-next-btn').prop('disabled', false);
        },

        /**
         * Store data for current step
         */
        storeStepData: function(value) {
            switch (this.currentStep) {
                case 1:
                    this.installationData.store_preset = value;
                    break;
                case 2:
                    // Step 2 is acknowledgement only - no selection recorded
                    break;
                case 3:
                    // Step 3 is prediction mode if available
                    this.installationData.prediction_mode = value;
                    // Reload presets with new prediction mode if we're going back to step 1
                    this.loadPresetsForStoreType('retail');
                    break;
            }
        },

        /**
         * Show digital products info
         */
        showDigitalInfo: function() {
            $('#digital-stats-info').show();
        },

        /**
         * Hide digital products info
         */
        hideDigitalInfo: function() {
            $('#digital-stats-info').hide();
        },

        /**
         * Handle demo stats toggle
         */
        handleDemoToggle: function() {
            const checkbox = document.getElementById('enable-demo-stats');
            const isEnabled = checkbox ? checkbox.checked : false;

            this.installationData.demo_stats = isEnabled;

            // Check if wisestock_admin is available
            if (typeof wisestock_admin === 'undefined') {
                return;
            }

            // Show loading state
            const $toggle = $('.wisestock-toggle');
            const $checkbox = $('#enable-demo-stats');
            const $label = $('.wisestock-toggle-label');
            const originalLabel = $label.text();

            $checkbox.prop('disabled', true);
            $label.text(wisestock_admin.strings.saving);
            $toggle.addClass('loading');

            // Save demo stats setting via AJAX
            $.ajax({
                url: wisestock_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'wisestock_save_demo_stats',
                    demo_stats: isEnabled,
                    nonce: wisestock_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $label.text(wisestock_admin.strings.saved);
                        setTimeout(function() {
                            $label.text(originalLabel);
                        }, 1000);
                    } else {
                        $label.text(wisestock_admin.strings.error_try_again);
                        setTimeout(function() {
                            $label.text(originalLabel);
                        }, 2000);
                    }
                },
                error: function(xhr, status, error) {
                    $label.text(wisestock_admin.strings.error_try_again);
                    setTimeout(function() {
                        $label.text(originalLabel);
                    }, 2000);
                },
                complete: function() {
                    // Re-enable checkbox and remove loading state
                    $checkbox.prop('disabled', false);
                    $toggle.removeClass('loading');
                }
            });
        },

        /**
         * Handle go to dashboard button click
         */
        handleGoToDashboard: function(e) {
            e.preventDefault();

            const $button = $(this);
            const originalText = $button.html();

            // Show loading state
            $button.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> ' + wisestock_admin.strings.redirecting);

            // Redirect to dashboard
            var targetUrl = WiseStockInstallation.dashboardUrl || wisestock_admin.dashboard_url || '/wp-admin/admin.php?page=wisestock-dashboard';
            window.location.href = targetUrl;
        },

        /**
         * Scroll to top of the page
         */
        scrollToTop: function() {
            $('html, body').animate({
                scrollTop: 0
            }, 300);
        },

        /**
         * Handle next button click
         */
        handleNext: function(e) {
            e.preventDefault();


            // Move to next step
            WiseStockInstallation.currentStep++;

            // Skip any missing steps
            while (!document.getElementById('step-' + WiseStockInstallation.currentStep) && WiseStockInstallation.currentStep <= WiseStockInstallation.totalSteps) {
                WiseStockInstallation.currentStep++;
            }

            // No need to skip steps since we removed digital store type logic

            if (WiseStockInstallation.currentStep > WiseStockInstallation.totalSteps) {
                WiseStockInstallation.currentStep = WiseStockInstallation.totalSteps;
            }

            WiseStockInstallation.showStep(WiseStockInstallation.currentStep);
            WiseStockInstallation.updateProgress();
            WiseStockInstallation.updateNavigation();

            // Scroll to top of the page
            WiseStockInstallation.scrollToTop();

            // Update summary in step 3
            if (WiseStockInstallation.currentStep === 3) {
                WiseStockInstallation.updateSummary();
            }
        },

        /**
         * Handle previous button click
         */
        handlePrevious: function(e) {
            e.preventDefault();

            WiseStockInstallation.currentStep--;

            // Skip any missing steps
            while (!document.getElementById('step-' + WiseStockInstallation.currentStep) && WiseStockInstallation.currentStep >= 1) {
                WiseStockInstallation.currentStep--;
            }

            // No need to skip steps since we removed digital store type logic

            if (WiseStockInstallation.currentStep < 1) {
                WiseStockInstallation.currentStep = 1;
            }

            WiseStockInstallation.showStep(WiseStockInstallation.currentStep);
            WiseStockInstallation.updateProgress();
            WiseStockInstallation.updateNavigation();

            // Scroll to top of the page
            WiseStockInstallation.scrollToTop();
        },

        /**
         * Handle complete button click
         */
        handleComplete: function(e) {
            e.preventDefault();

            const button = document.getElementById('wisestock-complete-btn');
            const originalText = button.innerHTML;

            // Show loading state
            button.disabled = true;
            button.innerHTML = '<span class="dashicons dashicons-update"></span> ' + wisestock_admin.strings.completing_setup;

            // Send installation data to server

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_complete_installation',
                    nonce: wisestock_admin.nonce,
                    installation_data: WiseStockInstallation.installationData
                },
                success: function(response) {
                    if (response.success) {
                        // Move to step 4 (completion page) instead of redirecting
                        WiseStockInstallation.currentStep = 4;
                        WiseStockInstallation.showStep(WiseStockInstallation.currentStep);
                        WiseStockInstallation.updateProgress();
                        WiseStockInstallation.updateNavigation();

                        // Scroll to top of the page
                        WiseStockInstallation.scrollToTop();

                        // Store the redirect URL for later use
                        WiseStockInstallation.dashboardUrl = response.data.redirect_url;
                    } else {
                        WiseStockInstallation.showErrorMessage(response.data || wisestock_admin.strings.installation_failed);
                        button.disabled = false;
                        button.innerHTML = originalText;
                    }
                },
                error: function(xhr, status, error) {
                    WiseStockInstallation.showErrorMessage(wisestock_admin.strings.network_error_occurred);
                    button.disabled = false;
                    button.innerHTML = originalText;
                }
            });
        },

        /**
         * Validate current step
         */
        validateCurrentStep: function() {
            const $currentStep = $('#step-' + this.currentStep);
            const $selectedCard = $currentStep.find('.wisestock-option-card.active, .wisestock-preset-card.active');

            if ($selectedCard.length === 0) {
                this.showValidationError(wisestock_admin.strings.please_select_option);
                return false;
            }

            return true;
        },

        /**
         * Show step
         */
        showStep: function(stepNumber) {
            // Hide all steps
            $('.wisestock-installation-step').removeClass('active');

            // Show current step
            $('#step-' + stepNumber).addClass('active');

            // Hide navigation on step 4 (completion)
            if (stepNumber === 4) {
                $('.wisestock-installation-navigation').hide();
            } else {
                $('.wisestock-installation-navigation').show();
            }

            // Load presets for step 1 if needed
            if (stepNumber === 1) {
                this.loadPresetsForStoreType('retail');
            }
        },

        /**
         * Update progress bar
         */
        updateProgress: function() {
            const progress = (this.currentStep / this.totalSteps) * 100;
            $('#wisestockgress-fill').css('width', progress + '%');

            // Update step indicators
            $('.wisestock-step').each(function(index) {
                const stepNumber = index + 1;
                const $step = $(this);

                if (stepNumber < WiseStockInstallation.currentStep) {
                    $step.removeClass('active').addClass('completed');
                } else if (stepNumber === WiseStockInstallation.currentStep) {
                    $step.removeClass('completed').addClass('active');
                } else {
                    $step.removeClass('active completed');
                }
            });
        },

        /**
         * Update navigation buttons
         */
        updateNavigation: function() {
            var $prevBtn = $('#wisestock-prev-btn');
            var $nextBtn = $('#wisestock-next-btn');
            var $completeBtn = $('#wisestock-complete-btn');

            // Show/hide previous button
            if (this.currentStep === 1) {
                $prevBtn.hide();
            } else {
                $prevBtn.show();
            }

            // Show/hide next/complete buttons
            if (this.currentStep === 3) {
                // Step 3 is the summary step - show Complete button
                $nextBtn.hide();
                $completeBtn.show();
            } else {
                $nextBtn.show();
                $completeBtn.hide();
            }

            // Handle step 2 specially (acknowledgement step)
            if (this.currentStep === 2) {
                // Step 2 is always enabled (acknowledgement step)
                $nextBtn.prop('disabled', false);
                // Change button text for step 2
                $nextBtn.html('I understand - Proceed <span class="dashicons dashicons-arrow-right-alt2"></span>');
            } else {
                // For all other steps, reset button text and handle selection logic
                $nextBtn.html('Next <span class="dashicons dashicons-arrow-right-alt2"></span>');
                
                // Enable/disable next button based on selection (except for summary step)
                if (this.currentStep < this.totalSteps) {
                    var $currentStep = $('#step-' + this.currentStep);
                    var hasSelection = $currentStep.find('.wisestock-option-card.active, .wisestock-preset-card.active').length > 0;
                    $nextBtn.prop('disabled', !hasSelection);
                }
            }
        },

        /**
         * Load presets for store type
         */
        loadPresetsForStoreType: function(storeType) {
            // This will be populated with AJAX call to get presets
            var presets = this.getPresetsForStoreType(storeType);
            this.renderPresets(presets);
        },

        /**
         * Get presets for store type
         */
        getPresetsForStoreType: function(storeType) {
            // Return timebound presets (timebound is now the only algorithm)
            if (typeof wisestock_admin !== 'undefined' && wisestock_admin.presets && wisestock_admin.presets.timebound) {
                return wisestock_admin.presets.timebound;
            }
            return {};
        },

        /**
         * Render presets in step 2
         */
        renderPresets: function(presets) {
            var container = $('#presets-container');
            container.empty();
            
            var self = this;
            $.each(presets, function(key, preset) {
                var isActive = (self.installationData.store_preset === key) ? ' active' : '';
                var presetCard = $('<div class="wisestock-preset-card' + isActive + '" data-value="' + key + '">' +
                    '<h3 class="wisestock-preset-title">' + preset.name + '</h3>' +
                    '<p class="wisestock-preset-description">' + preset.description + '</p>' +
                '</div>');
                container.append(presetCard);
            });
        },

        /**
         * Update summary in step 5
         */
        updateSummary: function() {
            var data = this.installationData;

            // Store preset
            if (data.store_preset) {
                $('#preset-summary').show();
                var presets = this.getPresetsForStoreType();              
                $('#summary-store-preset').text(presets[data.store_preset] ? presets[data.store_preset].name : data.store_preset);
            } else {
                $('#preset-summary').hide();
            }

            // Stock management
            $('#summary-stock-management').text(data.stock_management === 'yes' ? wisestock_admin.strings.yes : wisestock_admin.strings.no);

            // Update features preview
            this.updateFeaturesPreview();

            // Update stock management impact notice
            this.updateStockManagementImpact();
        },

        /**
         * Update features preview based on configuration
         */
        updateFeaturesPreview: function() {
            var data = this.installationData;
            var features = [];

            // Base features
            features.push('✓ Sales velocity tracking');
            features.push('✓ Basic analytics');

            // Physical product features (defaulted to retail)
            if (data.stock_management === 'yes') {
                features.push('✓ Stockout predictions');
                features.push('✓ Restock recommendations');
                features.push('✓ Inventory optimization');
                features.push('✓ Supplier management');
            } else {
                features.push('⚠️ Limited to sales velocity only');
            }

            // Algorithm specific features
            if (data.prediction_mode === 'timebound') {
                features.push('✓ Dynamic prediction windows');
                features.push('✓ Confidence decay analysis');
            } else {
                features.push('✓ Simple weighted averages');
            }

            var featuresHtml = features.map(function(feature) {
                return wisestock_admin.templates.feature_item.replace('{feature}', feature);
            }).join('');

            $('#features-preview').html(featuresHtml);
        },

        /**
         * Update stock management impact notice
         */
        updateStockManagementImpact: function() {
            var data = this.installationData;
            var impactNotice = $('#stock-management-impact');

            if (data.stock_management === 'no') {
                impactNotice.show();
                $('#stock-management-status').text(wisestock_admin.strings.not_to_use);
                $('#stock-management-explanation').text(wisestock_admin.strings.stock_management_explanation);
            } else {
                impactNotice.hide();
            }
        },

        /**
         * Show validation error
         */
        showValidationError: function(message) {
            this.showNotice('error', message);
        },

        /**
         * Show success message
         */
        showSuccessMessage: function() {
            var html = wisestock_admin.templates.success_message;
            $('.wisestock-installation-wizard').html(html);
        },

        /**
         * Show error message
         */
        showErrorMessage: function(message) {
            this.showNotice('error', message);
        },

        /**
         * Show notice
         */
        showNotice: function(type, message) {
            var icon = type === 'error' ? 'warning' : 'info';
            var noticeHtml = wisestock_admin.templates.installation_notice
                .replace('{type}', type)
                .replace('{icon}', icon)
                .replace('{message}', message);
            
            var $notice = $(noticeHtml);

            $('.wisestock-installation-container').prepend($notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $notice.remove();
                });
            }, 5000);

            // Manual dismiss
            $notice.find('.wisestock-notice-dismiss').on('click', function() {
                $notice.fadeOut(function() {
                    $notice.remove();
                });
            });
        },

        /**
         * Show stock management help modal
         */
        showStockManagementHelp: function() {
            const modal = document.getElementById('wisestock-help-modal');
            if (modal) {
                modal.classList.remove('wisestock-dynamic-display-none');
                modal.classList.add('wisestock-dynamic-display-block');
            }
        },

        /**
         * Close help modal
         */
        closeHelpModal: function() {
            const modal = document.getElementById('wisestock-help-modal');
            if (modal) {
                modal.classList.remove('wisestock-dynamic-display-block');
                modal.classList.add('wisestock-dynamic-display-none');
            }
        }
    };

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        WiseStockInstallation.init();
    });

})(jQuery);
