/**
 * WiseStock - Admin JavaScript
 *
 * This file contains all the JavaScript functionality for the WiseStock admin interface.
 * It handles dashboard interactions, AJAX requests, and dynamic UI updates.
 *
 * @package WiseStock
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * WiseStock Admin Object
     */
    window.WiseStockAdmin = {

        /**
         * Initialize the admin interface
         */
        init: function() {
            this.bindEvents();
            this.initDashboard();
            this.initNotifications();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Notification actions
            $(document).on('click', '.wisestock-notification-resolve', this.resolveNotification);
            $(document).on('click', '.wisestock-notification-dismiss', this.dismissNotification);

            // Notification bell
            $(document).on('click', '#wisestock-notification-toggle', this.toggleNotificationDropdown);
            $(document).on('click', '#wisestock-close-notifications', this.closeNotificationDropdown);
            $(document).on('click', document, this.closeNotificationDropdownOnClickOutside);

            // Product actions
            $(document).on('click', '.wisestock-product-action', this.handleProductAction);
            $(document).on('click', '.wisestock-notify-supplier', this.handleNotifySupplier);

            // Settings form
            $(document).on('submit', '.wisestock-settings-form', this.saveSettings);

            // Help icon tooltips
            $(document).on('mouseenter', '.wisestock-help-icon', this.showTooltip);
            $(document).on('mouseleave', '.wisestock-help-icon', this.hideTooltip);
        },

        /**
         * Initialize dashboard
         */
        initDashboard: function() {
            if ($('.wisestock-dashboard').length) {
                // Don't load data on page load since it's already rendered server-side
                // Only load if explicitly requested via refresh button
            }
        },

        /**
         * Initialize notifications system
         */
        initNotifications: function() {
            // Initialize notification bell functionality
            if ($("#wisestock-notification-list").length > 0) {
                this.loadNotifications();
            }
        },

        /**
         * Load notifications
         */
        loadNotifications: function() {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_notifications',
                    nonce: wisestock_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockAdmin.updateNotificationsList(response.data);
                    }
                }
            });
        },

        /**
         * Update notifications dropdown list
         */
        updateNotificationsList: function(notifications) {
            const $container = $('#wisestock-notification-list');
            $container.empty();

            if (notifications.length === 0) {
                $container.append(wisestock_admin.templates.no_notifications);
                return;
            }

            notifications.forEach(function(notification) {
                const $notification = WiseStockAdmin.createNotificationItem(notification);
                $container.append($notification);
            });
        },

        /**
         * Create notification item
         */
        createNotificationItem: function(notification) {
            const $item = $(wisestock_admin.templates.notification_item.replace('{type}', notification.notification_type));
            
            const contentHtml = wisestock_admin.templates.notification_content
                .replace('{product_name}', notification.product_name)
                .replace('{message}', notification.message)
                .replace('{time_ago}', notification.time_ago)
                .replace(/{id}/g, notification.id);

            $item.html(contentHtml);
            return $item;
        },

        /**
         * Refresh dashboard
         */

        /**
         * Resolve notification
         */
        resolveNotification: function(e) {
            e.preventDefault();

            const notificationId = $(this).data('notification-id');
            const $notification = $(this).closest('.wisestock-notification-item, .wisestock-notification-row');

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_resolve_notification',
                    notification_id: notificationId,
                    nonce: wisestock_admin.nonce
                },
                beforeSend: function() {
                    $notification.addClass('loading');
                },
                success: function(response) {
                    if (response.success) {
                        $notification.fadeOut(300, function() {
                            $(this).remove();
                        });
                        WiseStockAdmin.showNotice('success', wisestock_admin.strings.notification_resolved);
                        WiseStockAdmin.updateNotificationCount();

                        // Reload page if on notifications page to refresh counts
                        if (window.location.href.indexOf('wisestock-notifications') !== -1) {
                            setTimeout(function() {
                                window.location.reload();
                            }, 1000);
                        }
                    } else {
                        WiseStockAdmin.showNotice('error', response.data || wisestock_admin.strings.failed_resolve_notification);
                    }
                },
                error: function() {
                    WiseStockAdmin.showNotice('error', wisestock_admin.strings.network_error);
                },
                complete: function() {
                    $notification.removeClass('loading');
                }
            });
        },

        /**
         * Dismiss notification
         */
        dismissNotification: function(e) {
            e.preventDefault();

            const notificationId = $(this).data('notification-id');
            const $notification = $(this).closest('.wisestock-notification-item, .wisestock-notification-row');

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_dismiss_notification',
                    notification_id: notificationId,
                    nonce: wisestock_admin.nonce
                },
                beforeSend: function() {
                    $notification.addClass('loading');
                },
                success: function(response) {
                    if (response.success) {
                        $notification.fadeOut(300, function() {
                            $(this).remove();
                        });
                        WiseStockAdmin.showNotice('success', wisestock_admin.strings.notification_dismissed);
                        WiseStockAdmin.updateNotificationCount();

                        // Reload page if on notifications page to refresh counts
                        if (window.location.href.indexOf('wisestock-notifications') !== -1) {
                            setTimeout(function() {
                                window.location.reload();
                            }, 1000);
                        }
                    } else {
                        WiseStockAdmin.showNotice('error', response.data || wisestock_admin.strings.failed_dismiss_notification);
                    }
                },
                error: function() {
                    WiseStockAdmin.showNotice('error', wisestock_admin.strings.network_error);
                },
                complete: function() {
                    $notification.removeClass('loading');
                }
            });
        },

        /**
         * Handle product action
         */
        handleProductAction: function(e) {
            e.preventDefault();

            const action = $(this).data('action');
            const productId = $(this).data('product-id');

            switch (action) {
                case 'view':
                    window.open(wisestock_admin.product_edit_url.replace('%d', productId), '_blank');
                    break;
                case 'edit':
                    window.location.href = wisestock_admin.product_edit_url.replace('%d', productId);
                    break;
            }
        },

        /**
         * Save settings
         */
        saveSettings: function(e) {
            e.preventDefault();

            const $form = $(this);
            const formData = $form.serialize();

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: formData + '&action=wisestock_save_settings&nonce=' + wisestock_admin.nonce,
                beforeSend: function() {
                    $form.find('button[type="submit"]').prop('disabled', true).text(wisestock_admin.strings.saving);
                },
                success: function(response) {
                    if (response.success) {
                        // Check if we're on the settings page and auto-save is active
                        if (typeof WiseStockSettings !== 'undefined' && WiseStockSettings.updateAutoSaveStatus) {
                            WiseStockSettings.updateAutoSaveStatus('saved');
                            setTimeout(function() {
                                WiseStockSettings.hideAutoSaveStatus();
                            }, 2000);
                        } else {
                            WiseStockAdmin.showNotice('success', wisestock_admin.strings.settings_saved);
                        }
                    } else {
                        if (typeof WiseStockSettings !== 'undefined' && WiseStockSettings.updateAutoSaveStatus) {
                            WiseStockSettings.updateAutoSaveStatus('error');
                        } else {
                            WiseStockAdmin.showNotice('error', response.data || wisestock_admin.strings.failed_save_settings);
                        }
                    }
                },
                error: function() {
                    if (typeof WiseStockSettings !== 'undefined' && WiseStockSettings.updateAutoSaveStatus) {
                        WiseStockSettings.updateAutoSaveStatus('error');
                    } else {
                        WiseStockAdmin.showNotice('error', wisestock_admin.strings.network_error);
                    }
                },
                complete: function() {
                    $form.find('button[type="submit"]').prop('disabled', false).text(wisestock_admin.strings.save_settings);
                }
            });
        },

        /**
         * Handle notify supplier button click
         */
        handleNotifySupplier: function(e) {
            e.preventDefault();
            var productId = $(this).data('product-id');
            var $btn = $(this);

            // Get suppliers for this product
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_get_product_suppliers',
                    product_id: productId,
                    nonce: wisestock_admin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.length > 0) {
                        if (response.data.length === 1) {
                            // Single supplier - notify directly
                            WiseStockAdmin.notifySupplier(productId, [response.data[0].id], $btn);
                        } else {
                            // Multiple suppliers - check for default supplier
                            var defaultSupplierId = response.data.find(function(supplier) {
                                return supplier.is_default;
                            });

                            if (defaultSupplierId) {
                                // Use default supplier
                                WiseStockAdmin.notifySupplier(productId, [defaultSupplierId.id], $btn);
                            } else {
                                // Show selection modal
                                WiseStockAdmin.showSupplierSelectionModal(productId, response.data, $btn);
                            }
                        }
                    } else {
                        WiseStockAdmin.showNotice('warning', wisestock_admin.strings.no_suppliers_found);
                    }
                },
                error: function() {
                    WiseStockAdmin.showNotice('error', wisestock_admin.strings.failed_load_supplier_data);
                }
            });
        },

        /**
         * Show supplier selection modal
         */
        showSupplierSelectionModal: function(productId, suppliers, $btn) {
            var supplierList = '';

            suppliers.forEach(function(supplier) {
                var isDefaultText = supplier.is_default ? ' (' + wisestock_admin.strings.default_supplier + ')' : '';
                supplierList += wisestock_admin.templates.supplier_item
                    .replace('{id}', supplier.id)
                    .replace('{checked}', supplier.is_default ? 'checked' : '')
                    .replace('{name}', supplier.name)
                    .replace('{default_text}', isDefaultText)
                    .replace('{email}', supplier.email || wisestock_admin.strings.no_email);
            });

            var modalHtml = wisestock_admin.templates.supplier_modal
                .replace('{supplier_list}', supplierList)
                .replace('{product_id}', productId);

            $('body').append(modalHtml);

            // Bind events
            $('#wisestock-supplier-selection-modal .wisestock-modal-close, #wisestock-supplier-selection-modal .wisestock-modal-cancel').on('click', function() {
                $('#wisestock-supplier-selection-modal').remove();
            });

            $('#wisestock-supplier-selection-modal .wisestock-notify-selected-suppliers').on('click', function() {
                var selectedSuppliers = [];
                $('#wisestock-supplier-selection-modal input:checked').each(function() {
                    selectedSuppliers.push($(this).val());
                });

                if (selectedSuppliers.length > 0) {
                    WiseStockAdmin.notifySupplier(productId, selectedSuppliers, $btn);
                    $('#wisestock-supplier-selection-modal').remove();
                } else {
                    WiseStockAdmin.showNotice('warning', wisestock_admin.strings.select_suppliers);
                }
            });
        },

        /**
         * Notify selected suppliers
         */
        notifySupplier: function(productId, supplierIds, $btn) {
            var originalText = $btn.text();
            $btn.prop('disabled', true).text(wisestock_admin.strings.notifying || 'Notifying...');

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_notify_supplier',
                    product_id: productId,
                    supplier_ids: supplierIds,
                    nonce: wisestock_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockAdmin.showNotice('success', response.data);
                    } else {
                        WiseStockAdmin.showNotice('error', response.data || wisestock_admin.strings.failed_notify_suppliers);
                    }
                },
                error: function() {
                    WiseStockAdmin.showNotice('error', wisestock_admin.strings.network_error);
                },
                complete: function() {
                    $btn.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Toggle notification dropdown
         */
        toggleNotificationDropdown: function(e) {
            e.preventDefault();
            e.stopPropagation();

            var $dropdown = $('#wisestock-notification-dropdown');
            $dropdown.toggleClass('show');
        },

        /**
         * Close notification dropdown
         */
        closeNotificationDropdown: function(e) {
            if (e) {
            e.preventDefault();
                e.stopPropagation();
            }

            $('#wisestock-notification-dropdown').removeClass('show');
        },

        /**
         * Close notification dropdown when clicking outside
         */
        closeNotificationDropdownOnClickOutside: function(e) {
            if (!$(e.target).closest('.wisestock-notification-bell').length) {
                WiseStockAdmin.closeNotificationDropdown();
            }
        },

        /**
         * Update notification count
         */
        updateNotificationCount: function() {
            var count = $('.wisestock-notification-item').length;
            $('#wisestock-notification-count').text(count);

            if (count === 0) {
                $('#wisestock-notification-count').hide();
            } else {
                $('#wisestock-notification-count').show();
            }
        },

        /**
         * Show notice
         */
        showNotice: function(type, message) {
            if (typeof(wisestock_admin.templates.notice) === 'undefined') {
                return;
            }

            var $notice = $(wisestock_admin.templates.notice
                .replace('{type}', type)
                .replace('{message}', message));
            $('.wisestock-admin .wrap').prepend($notice);

            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        },

        /**
         * Show loading state
         */
        showLoading: function($element) {
            $element.addClass('loading');
        },

        /**
         * Hide loading state
         */
        hideLoading: function($element) {
            $element.removeClass('loading');
        },

        /**
         * Format date for display
         */
        formatDate: function(dateString) {
            var date = new Date(dateString);
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            });
        },

        /**
         * Format date and time for display
         */
        formatDateTime: function(dateString) {
            var date = new Date(dateString);
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'short',
                day: 'numeric',
                hour: 'numeric',
                minute: '2-digit'
            });
        },

        /**
         * Regenerate analytics
         */
        regenerateAnalytics: function(e) {
            e.preventDefault();

            var $button = $(e.target);
            var originalText = $button.text();

            $button.prop('disabled', true).text(wisestock_admin.strings.regenerating || 'Regenerating...');

            $.ajax({
                url: wisestock_admin.ajaxurl,
                type: 'POST',
                data: {
                    action: 'wisestock_regenerate_analytics',
                    nonce: wisestock_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        WiseStockAdmin.refreshMigrationStatus();
                        WiseStockAdmin.showMessage(wisestock_admin.strings.analytics_regenerated_success, 'success');
                    } else {
                        WiseStockAdmin.showMessage(wisestock_admin.strings.error_prefix + (response.data || wisestock_admin.strings.error_unknown), 'error');
                    }
                },
                error: function() {
                    WiseStockAdmin.showMessage(wisestock_admin.strings.failed_regenerate_analytics, 'error');
                },
                complete: function() {
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Show tooltip on help icon hover
         */
        showTooltip: function(e) {
            var $icon = $(this);
            var tooltipId = $icon.data('tooltip');
            var $tooltip = $('#' + tooltipId);

            if ($tooltip.length) {
                // Position tooltip
                var iconOffset = $icon.offset();
                var iconWidth = $icon.outerWidth();
                var iconHeight = $icon.outerHeight();

                $tooltip.css({
                    'position': 'absolute',
                    'top': (iconOffset.top + iconHeight + 10) + 'px',
                    'left': (iconOffset.left + iconWidth / 2) + 'px',
                    'transform': 'translateX(-50%)',
                    'display': 'block'
                });

                // Ensure tooltip is visible
                $tooltip.appendTo('body');
            }
        },

        /**
         * Hide tooltip when mouse leaves help icon
         */
        hideTooltip: function(e) {
            var $icon = $(this);
            var tooltipId = $icon.data('tooltip');
            var $tooltip = $('#' + tooltipId);

            if ($tooltip.length) {
                $tooltip.hide();
            }
        }
    };

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        WiseStockAdmin.init();
    });

})(jQuery);


