<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Elementor Widget for 3D Model Interactive
 */
class Wings3dpr_Elementor_Widget extends \Elementor\Widget_Base {

	protected static $shortcode_used = false;
	
	/**
	 * Get widget name.
	*
	* @return string Widget name.
	*/
	public function get_name() {
		return 'wings-3d-model-viewer';
	}

	/**
	 * Get widget title.
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( '3D Model Viewer', 'wingstech-3d-product-viewer-interactive' );
	}

	/**
	 * Get widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-gallery-grid';
	}

	/**
	 * Get widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'wings3dpr-widgets' );
	}

	/**
	 * Get widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return array( '3d', 'model', 'viewer', 'interactive', 'ar', 'glb', 'gltf' );
	}

	/**
	 * Get script dependencies.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'wings3dpr-model-viewer', 'wings3dpr-model-viewer-effects', 'wings3dpr-qrcode', 'wings3dpr-model-public-js' );
	}

	/**
	 * Get style dependencies.
	 *
	 * @return array Widget styles dependencies.
	 */
	public function get_style_depends() {
		return array( 'wings3dpr-model-public' );
	}

	/**
	 * Register widget controls.
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'content_section',
			array(
				'label' => __( '3D Model Settings', 'wingstech-3d-product-viewer-interactive' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		// Get all published 3D models
		// Get all published 3D models
		$models = get_posts(
			array(
				'post_type'      => 'wings3dpr_model',
				'post_status'    => 'publish',
				'posts_per_page' => -1,
				'orderby'        => 'title',
				'order'          => 'ASC',
			)
		);

		$model_options = array();
		if ( ! empty( $models ) ) {
			foreach ( $models as $model ) {
				$model_options[ $model->ID ] = $model->post_title;
			}
		}

		// Add model selection control
		$this->add_control(
			'model_id',
			array(
				'label'       => __( 'Select 3D Model', 'wingstech-3d-product-viewer-interactive' ),
				'type'        => \Elementor\Controls_Manager::SELECT,
				'options'     => $model_options,
				'default'     => ! empty( $model_options ) ? array_key_first( $model_options ) : '',
				'description' => __( 'Choose a 3D model to display', 'wingstech-3d-product-viewer-interactive' ),
			)
		);

		// Add help text if no models exist
		if ( empty( $model_options ) ) {
			$this->add_control(
				'no_models_notice',
				array(
					'type'            => \Elementor\Controls_Manager::RAW_HTML,
					'raw'             => sprintf(
						/* translators: %s: URL to create new model */
						__( 'No 3D models found. <a href="%s" target="_blank">Create your first 3D model</a>', 'wingstech-3d-product-viewer-interactive' ),
						admin_url( 'post-new.php?post_type=wings3dpr_model' )
					),
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				)
			);
		}

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend and editor.
	 */
	protected function render() {
		self::$shortcode_used = true;

		$settings = $this->get_settings_for_display();
		$model_id = ! empty( $settings['model_id'] ) ? intval( $settings['model_id'] ) : 0;

		if ( empty( $model_id ) ) {
			if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
				echo '<div style="padding: 20px; background: #f0f0f0; border: 1px dashed #ccc; text-align: center;">';
				echo esc_html__( 'Please select a 3D model from the widget settings.', 'wingstech-3d-product-viewer-interactive' );
				echo '</div>';
			}
			return;
		}

		// Check if post exists and is published
		if ( 'publish' !== get_post_status( $model_id ) ) {
			if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
				echo '<div style="padding: 20px; background: #fff3cd; border: 1px solid #ffc107; text-align: center;">';
				echo esc_html__( 'The selected 3D model is not published or does not exist.', 'wingstech-3d-product-viewer-interactive' );
				echo '</div>';
			}
			return;
		}

		// Add CSS to hide fullscreen and AR buttons in editor
		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			echo '<style>.tpv-fullscreen, .tpv-ar-btn { display: none !important; }</style>';
		}

		// Render the 3D model using existing shortcode functionality
		echo wp_kses( Wings3dpr_Shortcodes::tpv_3d_model_html( $model_id ), WINGS3DPR_Admin_Settings::get_allowed_html() );
	}
}
