<?php

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Checked if not exists Wings3dpr_Model_Meta_Box class.
 */
if ( !class_exists( 'Wings3dpr_Model_Meta_Box' ) ) {
    /**
     * Wings3dpr_Model_Meta_Box class file.
     */
    class Wings3dpr_Model_Meta_Box {
        /**
         * Wings3dpr_Model_Meta_Box constructor.
         */
        public function __construct() {
            // Model viewer meta field.
            add_action( 'add_meta_boxes', array($this, 'model_add_custom_meta') );
            // Side update meta field.
            add_action( 'add_meta_boxes', array($this, 'model_add_side_custom_meta') );
            // Save model viewer meta field.
            add_action( 'save_post', array($this, 'model_save_custom_meta_box') );
        }

        /**
         * Adds a custom meta box for file upload.
         *
         * This method adds a meta box to the 'wings3dpr_model' post type for uploading files.
         *
         * @return void
         */
        public function model_add_custom_meta() {
            add_meta_box(
                'model_viewer_custom_meta_box',
                // ID of the meta box.
                __( '3D Model Interactive', 'wingstech-3d-product-viewer-interactive' ),
                // Title of the meta box.
                array($this, 'model_render_custom_meta_box'),
                // Callback to render the meta box content.
                'wings3dpr_model',
                // Post type.
                'normal',
                // Context (normal, side, advanced).
                'high'
            );
        }

        /**
         * Renders the custom meta box for file upload.
         *
         * This method displays the content of the custom meta box, including a file upload input and a preview of the uploaded file.
         *
         * @param WP_Post $post The current post object.
         *
         * @return void
         */
        public function model_render_custom_meta_box( $post ) {
            wp_nonce_field( 'save_model_custom_meta_box', 'model_custom_meta_box_nonce' );
            $file_url = get_post_meta( $post->ID, '_tpv_file', true );
            ?>
			<div id="file-upload-container">
				<div id="file-upload-section" class="<?php 
            echo ( !empty( $file_url ) ? 'tpv-d-none' : '' );
            ?>">
					<input type="hidden" name="tpv_file" id="tpv_file" value="<?php 
            echo esc_attr( $file_url );
            ?>">
					<div id="tpv-upload">
						<h4><?php 
            esc_html_e( 'Upload or Select 3d object files', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
						<p>Supported file type: <strong>glb, glTF</strong> </p>
						<a href="javascript:void(0)" id="select-file">Select file</a>
					</div>
				</div>
				<div id="file-preview">
					<?php 
            if ( !empty( $file_url ) ) {
                echo wp_kses( Wings3dpr_Shortcodes::tpv_3d_model_html( $post->ID ), WINGS3DPR_Admin_Settings::get_allowed_html() );
            }
            ?>
				</div>
			</div>
			<?php 
        }

        /**
         * Adds a custom meta box for model viewer settings to the side of the post edit screen.
         *
         * @return void
         */
        public function model_add_side_custom_meta() {
            add_meta_box(
                'tpv_custom_side_meta_box',
                // ID of the meta box.
                __( '3D Viewer Settings', 'wingstech-3d-product-viewer-interactive' ),
                // Title of the meta box.
                array($this, 'tpv_render_custom_side_meta_box'),
                // Callback to render the meta box content.
                'wings3dpr_model',
                // Post type.
                'side',
                // Context (normal, side, advanced).
                'low'
            );
        }

        /**
         * Renders the custom meta box for model control.
         *
         * @param WP_Post $post The current post object.
         *
         * @return void
         */
        public function tpv_render_custom_side_meta_box( $post ) {
            wp_nonce_field( 'save_tpv_side_custom_meta_box', 'tpv_side_custom_meta_box_nonce' );
            $model_file_url = get_post_meta( $post->ID, '_tpv_file', true );
            $tpv_option_class = ( empty( $model_file_url ) ? 'tpv-disable-model-action' : '' );
            $auto_rotate = get_post_meta( $post->ID, '_tpv_auto_rotate', true );
            $tpv_ar_mode = get_post_meta( $post->ID, '_tpv_ar_mode', true );
            $tpv_place_on_wall = get_post_meta( $post->ID, '_tpv_place_on_wall', true );
            $background_color = get_post_meta( $post->ID, '_tpv_background_color', true );
            $background_color = ( $background_color !== '' ? $background_color : 'rgba(0,0,0,0)' );
            $moving_control = get_post_meta( $post->ID, '_tpv_moving_control', true );
            $loading_type = get_post_meta( $post->ID, '_tpv_loading_type', true );
            $loading_type = ( !empty( $loading_type ) ? $loading_type : 'auto' );
            $env_image_url = get_post_meta( $post->ID, '_tpv_environment_image_url', true );
            $env_skybox = get_post_meta( $post->ID, '_tpv_env_skybox', true );
            $skybox_image_url = get_post_meta( $post->ID, '_tpv_skybox_image_url', true );
            $tpv_zoom_disable = get_post_meta( $post->ID, '_tpv_zoom_disable', true );
            $tpv_fullscreen_mode = get_post_meta( $post->ID, '_tpv_fullscreen_mode', true );
            $poster_image_url = get_post_meta( $post->ID, '_tpv_poster_image_url', true );
            $exposure_value = get_post_meta( $post->ID, '_tpv_exposure_value', true );
            $shadow_intensity_value = get_post_meta( $post->ID, '_tpv_shadow_intensity_value', true );
            $shadow_softness_value = get_post_meta( $post->ID, '_tpv_shadow_softness_value', true );
            $tone_mapping_data = array(
                'neutral' => 'Neutral',
                'aces'    => 'ACES',
                'agx'     => 'AgX',
            );
            $tone_mapping_value = get_post_meta( $post->ID, '_tpv_tone_mapping_option', true );
            $environment_images = array(
                'None'       => '',
                'Neutral'    => WINGS3DPR_PLUGIN_URL . 'admin/assets/images/environment/neutral.jpg',
                'Legacy'     => WINGS3DPR_PLUGIN_URL . 'admin/assets/images/environment/legacy.jpg',
                'Music Hall' => WINGS3DPR_PLUGIN_URL . 'admin/assets/images/environment/music_hall.jpeg',
                'Pillars'    => WINGS3DPR_PLUGIN_URL . 'admin/assets/images/environment/pillars.jpg',
            );
            /********* Customize Limits Meta Attributes Start *********/
            $tpv_apply_yaw_limits_box = get_post_meta( $post->ID, '_tpv_apply_yaw_limits_box', true );
            $tpv_apply_pitch_limits_box = get_post_meta( $post->ID, '_tpv_apply_pitch_limits_box', true );
            $tpv_apply_minimum_zoom_box = get_post_meta( $post->ID, '_tpv_apply_minimum_zoom_box', true );
            /********* Customize Limits Meta Attributes End *********/
            /********* Minimum camera orbit attributes *********/
            $min_camera_orbit_x = get_post_meta( $post->ID, '_tpv_min_camera_orbit_x', true );
            $min_camera_orbit_x = ( !empty( $min_camera_orbit_x ) ? $min_camera_orbit_x : '-180' );
            $min_camera_orbit_num_x = ( is_numeric( $min_camera_orbit_x ) ? $min_camera_orbit_x : '-180' );
            $min_camera_orbit_y = get_post_meta( $post->ID, '_tpv_min_camera_orbit_y', true );
            $min_camera_orbit_y = ( !empty( $min_camera_orbit_y ) ? $min_camera_orbit_y : '0' );
            $min_camera_orbit_num_y = ( is_numeric( $min_camera_orbit_y ) ? $min_camera_orbit_y : '0' );
            $min_camera_orbit_z = get_post_meta( $post->ID, '_tpv_min_camera_orbit_z', true );
            $min_camera_orbit_z = ( !empty( $min_camera_orbit_z ) ? $min_camera_orbit_z : 'auto' );
            $min_field_of_view = get_post_meta( $post->ID, '_tpv_min_field_of_view', true );
            /********* Minimum camera orbit attributes *********/
            /********* Maximum camera orbit attributes *********/
            $max_camera_orbit_x = get_post_meta( $post->ID, '_tpv_max_camera_orbit_x', true );
            $max_camera_orbit_x = ( !empty( $max_camera_orbit_x ) ? $max_camera_orbit_x : '180' );
            $max_camera_orbit_num_x = ( is_numeric( $max_camera_orbit_x ) ? $max_camera_orbit_x : '180' );
            $max_camera_orbit_y = get_post_meta( $post->ID, '_tpv_max_camera_orbit_y', true );
            $max_camera_orbit_y = ( !empty( $max_camera_orbit_y ) ? $max_camera_orbit_y : '180' );
            $max_camera_orbit_num_y = ( is_numeric( $max_camera_orbit_y ) ? $max_camera_orbit_y : '180' );
            $max_camera_orbit_z = get_post_meta( $post->ID, '_tpv_max_camera_orbit_z', true );
            $max_camera_orbit_z = ( !empty( $max_camera_orbit_z ) ? $max_camera_orbit_z : 'auto' );
            $tpv_animation_autoplay = $tpv_animation_type = $tpv_animation_speed = '';
            $tpv_animation_tab_class = '';
            $tpv_animation_type_lists = array();
            /********* Animation Meta Fields *********/
            $effects = [
                'outline'     => 'Outline Effect',
                'ssao'        => 'SSAO Effect',
                'glitch'      => 'Glitch Effect',
                'smaa'        => 'SMAA Effect',
                'bloom'       => 'Bloom Effect',
                'pixelate'    => 'Pixelate Effect',
                'color-grade' => 'Color Grade Effect',
            ];
            /********* Animation Meta Fields *********/
            ?>
			<div id="tpv-tabs" class="tpv-metafield-tabs tpv-model-options m-0 <?php 
            echo esc_attr( $tpv_option_class );
            ?> <?php 
            echo ( WINGS3DPR_IS_PREMIUM ? '' : 'tpv-pro-items' );
            ?>">
				<ul id="tpv-metafield-tabs-nav">
					<li><a href="#tab1"><span class="dashicons dashicons-admin-generic"></span></a></li>
					<li><a href="#tab2"><span class="dashicons dashicons-admin-settings"></span></a></li>
					<li><a href="#tab3"><span class="dashicons dashicons-images-alt2"></span></a></li>
					<li><a href="#tab4"><span class="dashicons dashicons-admin-customizer"></span></a></li>
					<li class="tpv-animation-option <?php 
            echo esc_attr( $tpv_animation_tab_class );
            ?>">
						<?php 
            if ( !WINGS3DPR_IS_PREMIUM ) {
                ?>
							<span class="tpv-badge-pro">Pro</span>
						<?php 
            }
            ?>
						<a href="#tab5"><span class="dashicons dashicons-format-video"></span></a>
					</li>
                    <li>
						<?php 
            if ( !WINGS3DPR_IS_PREMIUM ) {
                ?>
							<span class="tpv-badge-pro">Pro</span>
						<?php 
            }
            ?>
						<a href="#tab6">
							<span class="dashicons">
								<img 
									src="<?php 
            echo esc_url( WINGS3DPR_PLUGIN_URL . 'admin/assets/images/magic-pen.svg' );
            ?>" 
									alt="<?php 
            esc_attr_e( 'Magic Pen Icon', 'wingstech-3d-product-viewer-interactive' );
            ?>" 
								/>
							</span>
						</a>
					</li>	
                </ul>
				<div id="tabs-content">
					<!-- Tab1 Content Start -->
					<div id="tab1" class="tab-content">
						<!-- Loading Type Area Start -->
						<div class="tpv-loading-type-radio-box tpv_box tpv-model-options p-12 m-0 border-bottom <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<label for="tpv_loading_type">
								<h4 class="tpv-normal-title m-0"><?php 
            echo esc_html__( 'Loading Type', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								<fieldset>
									<div class="emwvp-loading-type-input tpv-radio-flex">
										<input type="radio" class="emwvp-loading-type" id="auto" name="emwvp_loading_type"
											value="auto" <?php 
            checked( $loading_type, 'auto', true );
            ?> />
										<label for="auto"><?php 
            echo esc_html__( 'Auto', 'wingstech-3d-product-viewer-interactive' );
            ?></label>
									</div>
									<div class="emwvp-loading-type-input tpv-radio-flex">
										<input type="radio" class="emwvp-loading-type" id="lazy" name="emwvp_loading_type"
											value="lazy" <?php 
            checked( $loading_type, 'lazy', true );
            ?> />
										<label for="lazy"><?php 
            echo esc_html__( 'Lazy', 'wingstech-3d-product-viewer-interactive' );
            ?></label>
									</div>
									<div class="emwvp-loading-type-input tpv-radio-flex">
										<input type="radio" class="emwvp-loading-type" id="eager" name="emwvp_loading_type"
											value="eager" <?php 
            checked( $loading_type, 'eager', true );
            ?> />
										<label for="eager"><?php 
            echo esc_html__( 'Eager', 'wingstech-3d-product-viewer-interactive' );
            ?></label>
									</div>
								</fieldset>
							</label>
						</div>
						<!-- Loading Type Area End -->

						 <!-- Fullscreen Model Start -->
						<div class="tpv_switch_box tpv_box tpv-model-options border-bottom p-12 m-0 <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-fullscreen-mode-section">
								<label for="tpv_fullscreen_mode">
									<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Fullscreen Mode', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								</label>
								<input type="checkbox" id="tpv_fullscreen_mode" name="tpv_fullscreen_mode" class="tpv_switch"
									value="yes" <?php 
            checked( $tpv_fullscreen_mode, true );
            ?> />
							</div>
						</div>
						<!-- Fullscreen Model End -->

						<!-- Display Poster Upload Start -->
						<div class="tpv-poster-image-box tpv_box tpv-model-options border-bottom p-12 m-0 <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-environment-image-uploader p-0">
								<p class="tpv-normal-title m-0">
									<label for="tpv_poster_image">
										<strong><?php 
            echo esc_html__( 'Poster Image', 'wingstech-3d-product-viewer-interactive' );
            ?></strong>
									</label>
								</p>
								<div class="tpv-upload-poster-preview <?php 
            echo ( empty( $poster_image_url ) ? 'tpv-d-none' : '' );
            ?>">
									<img 
										id="tpvPosterImgPreview"
										src="<?php 
            echo esc_url( $poster_image_url );
            ?>" 
										alt="<?php 
            esc_attr_e( 'Poster Image Preview ', 'wingstech-3d-product-viewer-interactive' );
            ?>" 
										class="tpv-poster-img-preview" 
									/>
									<button type="button" id="tpvPosterRemoveImgBtn" class="tpv-poster-img-btn-rmv">
										<img 
											src="<?php 
            echo esc_url( WINGS3DPR_PLUGIN_URL . 'admin/assets/images/close.svg' );
            ?>" 
											alt="<?php 
            esc_attr_e( 'Close Image Icon ', 'wingstech-3d-product-viewer-interactive' );
            ?>" 
										/>
									</button>
								</div>
								<div class="tpv-upload-poster-btn-wrapper">
									<input type="hidden" id="tpv_poster_image_url" name="tpv_poster_image_url"
										value="<?php 
            echo esc_url( $poster_image_url );
            ?>" />
									<button class="btn tpv-poster-uploader"
										id="upload-poster-btn"><?php 
            echo esc_html__( 'Choose File', 'wingstech-3d-product-viewer-interactive' );
            ?></button>
								</div>
							</div>
						</div>
						<!-- Display Poster Upload End -->

						<!-- AR Mode Area Start -->
						<div class="tpv_switch_box tpv_box tpv-model-options border-bottom p-12 m-0 flex <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<label for="tpv_ar_mode">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'AR Mode', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
							</label>
							<input type="checkbox" id="tpv_ar_mode" name="tpv_ar_mode" class="tpv_switch" value="yes"
								<?php 
            checked( $tpv_ar_mode, true );
            ?> />
						</div>
						<!-- AR Mode Area End -->	

						<!-- AR Place on a Wall Start -->
						<div class="tpv_switch_box tpv_box tpv-model-options border-bottom p-12 m-0 flex <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<label for="tpv_place_on_wall">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Place on a Wall', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
							</label>
							<input type="checkbox" id="tpv_place_on_wall" name="tpv_place_on_wall" class="tpv_switch" value="yes"
								<?php 
            checked( $tpv_place_on_wall, true );
            ?> />
						</div>
						<!-- AR Place on a Wall Start End -->

						<!-- Background Color Start -->		
						<div class="tpv_box tpv-model-options p-12 m-0 <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-color-picker-title">
								<label for="model_background_color">
									<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Background Color', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								</label>
							</div>
							<div class="tpv-color-picker-options">
								<input type="hidden" class="tpv-color-picker" id="model_background_color" name="model_background_color" value="<?php 
            echo esc_attr( $background_color );
            ?>" />
							</div>
						</div>
						<!-- Background Color End -->	
					</div>
					<!-- Tab1 Content End -->

					<!-- Tab2 Content Start -->
					<div id="tab2" class="tab-content d-none">	
						<!-- Auto Rotate Area Start -->
						<div class="tpv_switch_box tpv_box tpv-model-options border-bottom p-12 m-0 flex <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<label for="tpv_auto_rotate">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Auto Rotate', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
							</label>
							<input type="checkbox" id="tpv_auto_rotate" name="tpv_auto_rotate" class="tpv_switch" value="yes"
								<?php 
            checked( $auto_rotate, true );
            ?> />
						</div>
						<!-- Auto Rotate Area End -->						

						<!-- Moving Control Area Start -->
						<div class="tpv_switch_box tpv_box tpv-model-options border-bottom p-12 m-0 flex <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<label for="tpv_moving_controls">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Moving Controls', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
							</label>
							<input type="checkbox" id="tpv_moving_controls" name="tpv_moving_controls" class="tpv_switch"
								value="yes" <?php 
            checked( $moving_control, true );
            ?> />
						</div>
						<!-- Moving Control Area End -->
						 					
						<!-- Customize Limits Start -->
						<div class="tpv-customize-limits-section tpv_box tpv-model-options m-0 p-0 <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-customize-limits-inner-section">
								<div class="m-0 p-12 tpv-customize-title">
									<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Customize Limits', 'wingstech-3d-product-viewer-interactive' );
            ?></hh>
								</div>
								<!-- Apply Yaw Limits Start -->
								<div class="tpv-customize-apply-yaw-limit-section border-bottom">
									<div class="tpv-apply-yaw-limits-box flex tpv-w-auto p-12">
										<label for="tpv_apply_yaw_limits_box">
											<strong><?php 
            echo esc_html__( 'Apply Yaw Limits', 'wingstech-3d-product-viewer-interactive' );
            ?></strong>
										</label>
										<input type="checkbox" id="tpv_apply_yaw_limits_box" name="tpv_apply_yaw_limits_box"
											class="tpv_switch" value="yes" <?php 
            checked( $tpv_apply_yaw_limits_box, true );
            ?> />
									</div>

									<div
										class="tpv-apply-yaw-limits-controls p-12 <?php 
            echo ( '1' !== $tpv_apply_yaw_limits_box ? 'tpv-d-none' : '' );
            ?>">
										<div class="tpv-apply-yaw-limits-inner-section">
											<!-- Counter-Clockwise Limit Start -->
											<div class="tpv-counter-clockwise-limit-inner-section">
												<div class="tpv-counter-clockwise-limit-lable-btn-section">
													<label for="tpv_set_counter_clockwise_limits">
														<h5 class="tpv-inner-title"><?php 
            echo esc_html__( 'Counter-Clockwise Limit', 'wingstech-3d-product-viewer-interactive' );
            ?></h5>
													</label>
												</div>
												<div class="tpv-counter-clockwise-limit-control-section">
													<input type="range" class="tpv-range-input"
														id="tpv_counter_clockwise_limit_range"
														name="tpv_counter_clockwise_limit_range"
														value="<?php 
            echo esc_attr( $min_camera_orbit_num_x );
            ?>" step="01" min="-180"
														max="180">
													<input type="number" class="tpv-range-input"
														id="tpv_counter_clockwise_limit_value"
														name="tpv_counter_clockwise_limit_value"
														value="<?php 
            echo esc_attr( $min_camera_orbit_num_x );
            ?>" step="01" min="-180"
														max="180">
												</div>
											</div>
											<!-- Counter-Clockwise Limit End -->

											<!-- Clockwise Limit Start -->
											<div class="tpv-clockwise-limit-inner-section">
												<div class="tpv-clockwise-limit-lable-btn-section">
													<label for="tpv_set_clockwise_limits">
														<h5 class="tpv-inner-title"><?php 
            echo esc_html__( 'Clockwise Limit', 'wingstech-3d-product-viewer-interactive' );
            ?></h5>
													</label>
												</div>

												<div class="tpv-clockwise-limit-control-section">
													<input type="range" class="tpv-range-input" id="tpv_clockwise_limit_range"
														name="tpv_clockwise_limit_range"
														value="<?php 
            echo esc_attr( $max_camera_orbit_num_x );
            ?>" step="-01"
														min="-180" max="180">
													<input type="number" class="tpv-range-input" id="tpv_clockwise_limit_value"
														name="tpv_clockwise_limit_value"
														value="<?php 
            echo esc_attr( $max_camera_orbit_num_x );
            ?>" step="-01"
														min="-180" max="180">
												</div>
											</div>
											<!-- Clockwise Limit End -->
										</div>
									</div>
								</div>
								<!-- Apply Yaw Limits End -->

								<!-- Apply Pitch Limits Start -->
								<div class="tpv-customize-apply-pitch-limit-section">
									<div class="tpv-apply-pitch-limits-box flex tpv-w-auto p-12 m-0">
										<label for="tpv_apply_pitch_limits_box">
											<strong><?php 
            echo esc_html__( 'Apply Pitch Limits', 'wingstech-3d-product-viewer-interactive' );
            ?></strong>
										</label>
										<input type="checkbox" id="tpv_apply_pitch_limits_box" name="tpv_apply_pitch_limits_box"
											class="tpv_switch" value="yes"
											<?php 
            checked( $tpv_apply_pitch_limits_box, true );
            ?> />
									</div>

									<div class="tpv-apply-pitch-limits-controls <?php 
            echo ( '1' !== $tpv_apply_pitch_limits_box ? 'tpv-d-none' : '' );
            ?>">
										<div class="tpv-apply-pitch-limits-inner-section">
											<!-- Top-down Limit Start -->
											<div class="tpv-pitch-top-down-inner-section p-12">
												<div class="tpv-pitch-top-down-limit-lable-btn-section">
													<label for="tpv_set_pitch_top_down_limits">
														<h5 class="tpv-inner-title"><?php 
            echo esc_html__( 'Top-down Limit', 'wingstech-3d-product-viewer-interactive' );
            ?></h5>
													</label>
												</div>
												<div class="tpv-pitch-top-down-limit-control-section">
													<input type="range" class="tpv-range-input"
														id="tpv_pitch_top_down_limit_range"
														name="tpv_pitch_top_down_limit_range"
														value="<?php 
            echo esc_attr( $min_camera_orbit_num_y );
            ?>" step="01" min="0"
														max="180">
													<input type="number" class="tpv-range-input"
														id="tpv_pitch_top_down_limit_value"
														name="tpv_pitch_top_down_limit_value"
														value="<?php 
            echo esc_attr( $min_camera_orbit_num_y );
            ?>" step="01" min="0"
														max="180">
												</div>
											</div>
											<!-- Top-down Limit End -->

											<!-- Bottom-up Limit Start -->
											<div class="tpv-pitch-bottom-up-inner-section p-12">
												<div class="tpv-pitch-bottom-up-limit-lable-btn-section">
													<label for="tpv_set_pitch-bottom-up_limits">
														<h5 class="tpv-inner-title"><?php 
            echo esc_html__( 'Bottom-up Limit', 'wingstech-3d-product-viewer-interactive' );
            ?></h5>
													</label>
												</div>

												<div class="tpv-pitch-bottom-up-limit-control-section">
													<input type="range" class="tpv-range-input"
														id="tpv_pitch_bottom_up_limit_range"
														name="tpv_pitch_bottom_up_limit_range"
														value="<?php 
            echo esc_attr( $max_camera_orbit_num_y );
            ?>" step="-01" min="0"
														max="180">
													<input type="number" class="tpv-range-input"
														id="tpv_pitch_bottom_up_limit_value"
														name="tpv_pitch_bottom_up_limit_value"
														value="<?php 
            echo esc_attr( $max_camera_orbit_num_y );
            ?>" step="-01" min="0"
														max="180">
												</div>
											</div>
											<!-- Bottom-up Limit End -->
										</div>
									</div>
								</div>							
								<!-- Apply Pitch Limits End -->

								<!-- Apply Minimum Zoom Start -->
								<div class="tpv-customize-apply-minimum-zoom-section">
									<div class="tpv-apply-minimum-zoom-box flex tpv-w-auto p-12">
										<label for="tpv_apply_minimum_zoom_box">
											<strong><?php 
            echo esc_html__( 'Apply Minimum Zoom', 'wingstech-3d-product-viewer-interactive' );
            ?></strong>
										</label>
										<input type="checkbox" id="tpv_apply_minimum_zoom_box" name="tpv_apply_minimum_zoom_box"
											class="tpv_switch" value="yes"
											<?php 
            checked( $tpv_apply_minimum_zoom_box, true );
            ?> />
									</div>

									<div
										class="tpv-apply-minimum-zoom-controls <?php 
            echo ( '1' !== $tpv_apply_minimum_zoom_box ? 'tpv-d-none' : '' );
            ?>">
										<div class="tpv-apply-minimum-zoom-inner-section">
											<button type="button" id="tpv_customize_set_min_zoom"
												<?php 
            echo ( 'auto' !== $min_camera_orbit_z ? 'disabled' : '' );
            ?> > <?php 
            echo esc_html__( 'Set Min', 'wingstech-3d-product-viewer-interactive' );
            ?></button>
											<button type="button"
												id="tpv_customize_reset_set_min_zoom" <?php 
            echo ( 'auto' === $min_camera_orbit_z ? 'disabled' : '' );
            ?>><?php 
            echo esc_html__( 'Reset Min', 'wingstech-3d-product-viewer-interactive' );
            ?></button>
										</div>
									</div>
								</div>
								<!-- Apply Minimum Zoom End -->
							</div>
						</div>
						<!-- Customize Limits End -->

						<!-- Min Camera Orbit Hidden Fields Start -->
						<input type="hidden" name="tpv_min_camera_orbit_x" id="tpv_min_camera_orbit_x"
							value="<?php 
            echo esc_attr( $min_camera_orbit_x );
            ?>">
						<input type="hidden" name="tpv_min_camera_orbit_y" id="tpv_min_camera_orbit_y"
							value="<?php 
            echo esc_attr( $min_camera_orbit_y );
            ?>" />
						<input type="hidden" name="tpv_min_camera_orbit_z" id="tpv_min_camera_orbit_z"
							value="<?php 
            echo esc_attr( $min_camera_orbit_z );
            ?>">
						<input type="hidden" name="tpv_min_field_of_view" id="tpv_min_field_of_view"
							value="<?php 
            echo esc_attr( $min_field_of_view );
            ?>">
						<!-- Min Camera Orbit Hidden Fields End -->

						<!-- Max Camera Orbit Hidden Fields Start -->
						<input type="hidden" name="tpv_max_camera_orbit_x" id="tpv_max_camera_orbit_x"
							value="<?php 
            echo esc_attr( $max_camera_orbit_x );
            ?>" />
						<input type="hidden" name="tpv_max_camera_orbit_y" id="tpv_max_camera_orbit_y"
							value="<?php 
            echo esc_attr( $max_camera_orbit_y );
            ?>" />
						<input type="hidden" name="tpv_max_camera_orbit_z" id="tpv_max_camera_orbit_z"
							value="<?php 
            echo esc_attr( $max_camera_orbit_z );
            ?>" />
						<!-- Max Camera Orbit Hidden Fields End -->
					</div>
					<!-- Tab2 Content End -->

					<!-- Tab3 Content Start -->
					<div id="tab3" class="tab-content d-none">
						<!-- Environment Image Area Start -->
						<?php 
            if ( !empty( $environment_images ) && is_array( $environment_images ) ) {
                ?>
						<div class="tpv-environment-image-box tpv-model-options border-bottom m-0 <?php 
                echo esc_attr( $tpv_option_class );
                ?>">
							<div class="tpv-environment-image-uploader">
								<h4 class="tpv-normal-title m-0"><?php 
                echo esc_html__( 'Environment Image', 'wingstech-3d-product-viewer-interactive' );
                ?></h4>
								<div class="tpv-environment-image-section">
									<div class="tpv-select-wrapper">
										<select name="tpv_environment_image" id="tpv_environment_image" class="tpv-environment-select">
											<?php 
                foreach ( $environment_images as $title => $url ) {
                    ?>
											<option value="<?php 
                    echo esc_url( $url );
                    ?>"
												<?php 
                    selected( $env_image_url, $url, true );
                    ?>><?php 
                    echo esc_html( $title );
                    ?>
											</option>
											<?php 
                }
                ?>
										</select>
									</div>
									<div class="tpv-environment-separator tpv-inner-title">OR</div>
									<div class="tpv-upload-hdr-preview <?php 
                echo ( empty( $env_image_url ) ? 'tpv-d-none' : '' );
                ?>">
										<img 
											id="tpvHDRImgPreview"
											src="<?php 
                echo esc_url( $env_image_url );
                ?>"
											alt="<?php 
                esc_attr_e( 'HDR Image Preview', 'wingstech-3d-product-viewer-interactive' );
                ?>" 
											class="tpv-hdr-img-preview" 
										/>
										<button type="button" id="tpvHDRRemoveImgBtn" class="tpv-hdr-img-btn-rmv">
											<img 
												src="<?php 
                echo esc_url( WINGS3DPR_PLUGIN_URL . 'admin/assets/images/close.svg' );
                ?>"
												alt="<?php 
                esc_attr_e( 'Close Image Preview Icon', 'wingstech-3d-product-viewer-interactive' );
                ?>" 
											/>
										</button>
									</div>
									<div class="tpv-upload-btn-wrapper">
										<button class="btn tpv-hdr-uploader"
											id="upload-hdr-btn"><?php 
                echo esc_html__( 'Choose file', 'wingstech-3d-product-viewer-interactive' );
                ?></button>
										<input type="hidden" name="tpv_skybox_image_url" id="tpv_skybox_image_url"
											value="<?php 
                echo esc_attr( $skybox_image_url );
                ?>">
									</div>
								</div>
							</div>
							<input type="hidden" name="tpv_environment_image_url" id="tpv_environment_image_url"
								value="<?php 
                echo esc_attr( $env_image_url );
                ?>">
						</div>
						<?php 
            }
            ?>
						<!-- Environment Image Area End -->

						<!-- Use Environment as Skybox Area Start -->
						<div class="tpv_switch_box tpv_box tpv-env-skybox-section tpv-model-options border-bottom p-12 m-0 flex skybox-area <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-env-skybox-section-box flex">
								<label for="tpv_env_skybox">
									<h4 class="tpv-title m-0"><?php 
            esc_html_e( 'Environment as Skybox', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								</label>
								<input type="checkbox" id="tpv_env_skybox" name="tpv_env_skybox" class="tpv_switch"
									value="yes" <?php 
            checked( $env_skybox, true );
            ?> />
							</div>							
							<span class="tpv-errors tpv-d-none"></span>							
						</div>
						<!-- Use Environment as Skybox Area End -->
						
						<!-- Exposure Start -->
						<div class="tpv-exposure-section tpv-model-options border-bottom <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-exposure-inner-section p-12">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Exposure', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								<div class="tpv-exposure-control-section">
									<input type="range" class="tpv-range-input" id="tpv_exposure_range" 
										name="tpv_exposure_range"
										value="<?php 
            echo ( !empty( $exposure_value ) ? esc_attr( $exposure_value ) : '0' );
            ?>"
										step="0.01" min="0" max="2">
									<input type="number" class="tpv-range-input" id="tpv_exposure_value"
										name="tpv_exposure_value"
										value="<?php 
            echo ( !empty( $exposure_value ) ? esc_attr( $exposure_value ) : '0' );
            ?>"
										step="0.01" min="0" max="2">
								</div>
							</div>
						</div>
						<!-- Exposure End -->

						<!-- Shadow Intensity Start -->
						<div class="tpv-shadow-intensity-section tpv-model-options border-bottom <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-shadow-intensity-inner-section p-12">								
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Set Shadow Intensity', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								<div class="tpv-shadow-intensity-section-control-section">
									<input type="range" class="tpv-range-input" id="tpv_shadow-intensity_range"
										name="tpv_shadow_intensity_range"
										value="<?php 
            echo ( !empty( $shadow_intensity_value ) ? esc_attr( $shadow_intensity_value ) : '0' );
            ?>"
										step="0.01" min="0" max="2">
									<input type="number" class="tpv-range-input" id="tpv_shadow-intensity_value"
										name="tpv_shadow_intensity_value"
										value="<?php 
            echo ( !empty( $shadow_intensity_value ) ? esc_attr( $shadow_intensity_value ) : '0' );
            ?>"
										step="0.01" min="0" max="2">
								</div>
							</div>
						</div>
						<!-- Shadow Intensity End -->

						<!-- Shadow Softness Start -->
						<div class="tpv-shadow-softness-section tpv-model-options <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-shadow-softness-inner-section p-12">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Set Shadow Softness', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								<div class="tpv-shadow-softness-section-control-section">
									<input type="range" class="tpv-range-input" id="tpv_shadow-softness_range"
										name="tpv_shadow_softness_range"
										value="<?php 
            echo ( !empty( $shadow_softness_value ) ? esc_attr( $shadow_softness_value ) : '0' );
            ?>"
										step="0.01" min="0" max="1">
									<input type="number" class="tpv-range-input" id="tpv_shadow-softness_value"
										name="tpv_shadow_softness_value"
										value="<?php 
            echo ( !empty( $shadow_softness_value ) ? esc_attr( $shadow_softness_value ) : '0' );
            ?>"
										step="0.01" min="0" max="1">
								</div>
							</div>
						</div>
						<!-- Shadow Softness End -->
					</div>
					<!-- Tab3 Content End -->

					<!-- Tab4 Content Start -->
					<div id="tab4" class="tab-content d-none">						
						<!-- Zoom Disable Start -->
						<div class="tpv_switch_box tpv_box tpv-zoom-enable-disable-section tpv-model-options border-bottom p-12 m-0 flex <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<label for="tpv_zoom_disable">
								<h4 class="tpv-title m-0"><?php 
            esc_html_e( 'Zoom Disable', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
							</label>
							<input type="checkbox" id="tpv_zoom_disable" name="tpv_zoom_disable" class="tpv_switch"
								value="yes" <?php 
            checked( $tpv_zoom_disable, true );
            ?> />
						</div>
						<!-- Zoom Disable End -->	

						<!-- Tone Mapping Start -->
						<?php 
            if ( !empty( $tone_mapping_data ) && is_array( $tone_mapping_data ) ) {
                ?>
						<div class="tpv-tone-mapping-box tpv-model-options p-12 m-0 <?php 
                echo esc_attr( $tpv_option_class );
                ?>">
							<div class="tpv-tone-mapping-box-inner">
								<h4 class="tpv-normal-title m-0"><?php 
                echo esc_html__( 'Tone Mapping', 'wingstech-3d-product-viewer-interactive' );
                ?></h4>
								<div class="tpv-tone-mapping-box-section">
									<div class="tpv-select-wrapper">
										<select name="tpv_tone_mapping_option" id="tpv_tone_mapping_option" class="tpv-tone-mapping-select">
											<?php 
                foreach ( $tone_mapping_data as $key => $value ) {
                    ?>
											<option value="<?php 
                    echo esc_attr( $key );
                    ?>"
												<?php 
                    selected( $tone_mapping_value, $key, true );
                    ?>>
												<?php 
                    echo esc_html( $value );
                    ?></option>
											<?php 
                }
                ?>
										</select>
									</div>
								</div>
							</div>
						</div>
						<?php 
            }
            ?>
						<!-- Tone Mapping End -->
					</div>
					<!-- Tab4 Content End -->

					<!-- Tab5 Content Start -->
					<div id="tab5" class="tab-content tpv-animation-option tpv-animation-tab d-none <?php 
            echo ( WINGS3DPR_IS_PREMIUM ? '' : 'tpv-pro-feature' );
            ?>">
						<div class="tpv-animation-inner-section">
							<div class="m-0 p-12 tpv-animation-title">
								<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Animation', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
							</div>
						
							<!-- Animation Autoplay Start -->
							<div class="tpv_switch_box tpv_box tpv-model-options border-bottom p-12 m-0 flex <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
								<label for="tpv_animation_autoplay">
									<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Play', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								</label>
								<input type="checkbox" id="tpv_animation_autoplay" name="tpv_animation_autoplay" class="tpv_switch" value="yes" <?php 
            checked( $tpv_animation_autoplay, 'yes' );
            ?> />
							</div>
							<!-- Animation Autoplay End -->

							<?php 
            ?>
						</div>
					</div>
					<!-- Tab5 Content End -->

                    <!-- Tab6 Content Start -->
					<div id="tab6" class="tab-content d-none <?php 
            echo ( WINGS3DPR_IS_PREMIUM ? '' : 'tpv-pro-feature' );
            ?>">
                        <!-- Effects Start -->						 
						<div class="tpv-effects-section tpv-model-options m-0 p-0 <?php 
            echo esc_attr( $tpv_option_class );
            ?>">
							<div class="tpv-effects-inner-section">
								<div class="m-0 p-12 tpv-effects-title">
									<h4 class="tpv-title m-0"><?php 
            echo esc_html__( 'Effects', 'wingstech-3d-product-viewer-interactive' );
            ?></h4>
								</div>							
								<?php 
            if ( !empty( $effects ) && is_array( $effects ) ) {
                foreach ( $effects as $effect => $label ) {
                    $value = get_post_meta( $post->ID, '_tpv_effect_' . $effect, true );
                    $checkbox_val = str_replace( '-', '_', $effect );
                    ?>
										<div class="tpv-effect-<?php 
                    echo esc_attr( $effect );
                    ?>-section border-bottom">
											<div class="tpv_switch_box tpv_effects_box flex tpv-w-auto p-12">
												<label for="tpv_effect_<?php 
                    echo esc_attr( $effect );
                    ?>">
													<h4 class="tpv-title m-0"><?php 
                    echo esc_html( $label );
                    ?></h4>
												</label>                                        
												<input type="checkbox" class="tpv_switch" id="tpv_effect_<?php 
                    echo esc_attr( $checkbox_val );
                    ?>" name="tpv_effect_<?php 
                    echo esc_attr( $checkbox_val );
                    ?>" value="yes" <?php 
                    checked( $value, 'yes' );
                    ?> />
											</div>
											
											<?php 
                    ?>
										</div>
										<?php 
                }
            }
            ?>
							</div>
						</div>
                        <!-- Effects End -->
                    </div>
					<!-- Tab6 Content End -->
				</div> <!-- End Tabs Content -->
			</div> <!-- End Tabs -->
			<?php 
        }

        /**
         * Save custom meta box callback.
         *
         * @param int $post_id The ID of the post being saved.
         *
         * @return void
         */
        public function model_save_custom_meta_box( $post_id ) {
            if ( !isset( $_POST['model_custom_meta_box_nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['model_custom_meta_box_nonce'] ) ), 'save_model_custom_meta_box' ) ) {
                return;
            }
            if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
                return;
            }
            if ( !current_user_can( 'edit_post', $post_id ) ) {
                return;
            }
            if ( isset( $_POST['tpv_file'] ) ) {
                update_post_meta( $post_id, '_tpv_file', esc_url_raw( wp_unslash( $_POST['tpv_file'] ) ) );
            }
            $tpv_auto_rotate_value = ( isset( $_POST['tpv_auto_rotate'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_auto_rotate', sanitize_text_field( $tpv_auto_rotate_value ) );
            $tpv_ar_mode_value = ( isset( $_POST['tpv_ar_mode'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_ar_mode', sanitize_text_field( $tpv_ar_mode_value ) );
            $tpv_place_on_wall = ( isset( $_POST['tpv_place_on_wall'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_place_on_wall', sanitize_text_field( $tpv_place_on_wall ) );
            if ( isset( $_POST['model_background_color'] ) ) {
                $model_background_color = sanitize_text_field( wp_unslash( $_POST['model_background_color'] ) );
                update_post_meta( $post_id, '_tpv_background_color', $model_background_color );
            }
            $tpv_moving_control_value = ( isset( $_POST['tpv_moving_controls'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_moving_control', sanitize_text_field( $tpv_moving_control_value ) );
            if ( isset( $_POST['emwvp_loading_type'] ) ) {
                $emwvp_loading_type = sanitize_text_field( wp_unslash( $_POST['emwvp_loading_type'] ) );
                update_post_meta( $post_id, '_tpv_loading_type', $emwvp_loading_type );
            }
            $tpv_env_skybox = ( isset( $_POST['tpv_env_skybox'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_env_skybox', sanitize_text_field( $tpv_env_skybox ) );
            $tpv_zoom_disable = ( isset( $_POST['tpv_zoom_disable'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_zoom_disable', sanitize_text_field( $tpv_zoom_disable ) );
            $tpv_fullscreen_mode = ( isset( $_POST['tpv_fullscreen_mode'] ) ? '1' : '0' );
            update_post_meta( $post_id, '_tpv_fullscreen_mode', sanitize_text_field( $tpv_fullscreen_mode ) );
            if ( isset( $_POST['tpv_skybox_image_url'] ) ) {
                update_post_meta( $post_id, '_tpv_skybox_image_url', esc_url_raw( wp_unslash( $_POST['tpv_skybox_image_url'] ) ) );
            }
            if ( isset( $_POST['tpv_environment_image_url'] ) ) {
                $env_image_url = ( !empty( $_POST['tpv_environment_image_url'] ) ? esc_url_raw( wp_unslash( $_POST['tpv_environment_image_url'] ) ) : '' );
                update_post_meta( $post_id, '_tpv_environment_image_url', $env_image_url );
            }
            if ( isset( $_POST['tpv_poster_image_url'] ) ) {
                update_post_meta( $post_id, '_tpv_poster_image_url', esc_url_raw( wp_unslash( $_POST['tpv_poster_image_url'] ) ) );
            }
            if ( isset( $_POST['tpv_exposure_value'] ) ) {
                $tpv_exposure_value = (float) $_POST['tpv_exposure_value'];
                update_post_meta( $post_id, '_tpv_exposure_value', sanitize_text_field( $tpv_exposure_value ) );
            }
            if ( isset( $_POST['tpv_shadow_intensity_value'] ) ) {
                $tpv_shadow_intensity_value = (float) $_POST['tpv_shadow_intensity_value'];
                update_post_meta( $post_id, '_tpv_shadow_intensity_value', sanitize_text_field( $tpv_shadow_intensity_value ) );
            }
            if ( isset( $_POST['tpv_shadow_softness_value'] ) ) {
                $tpv_shadow_softness_value = (float) $_POST['tpv_shadow_softness_value'];
                update_post_meta( $post_id, '_tpv_shadow_softness_value', sanitize_text_field( $tpv_shadow_softness_value ) );
            }
            if ( isset( $_POST['tpv_tone_mapping_option'] ) ) {
                $tpv_tone_mapping_option = sanitize_text_field( wp_unslash( $_POST['tpv_tone_mapping_option'] ) );
                update_post_meta( $post_id, '_tpv_tone_mapping_option', $tpv_tone_mapping_option );
            }
            /********* Apply Yaw Limits Value Save Start */
            $tpv_apply_yaw_limits_box = ( isset( $_POST['tpv_apply_yaw_limits_box'] ) ? '1' : '0' );
            if ( isset( $_POST['tpv_min_camera_orbit_x'] ) ) {
                $tpv_min_camera_orbit_x = sanitize_text_field( wp_unslash( $_POST['tpv_min_camera_orbit_x'] ) );
                update_post_meta( $post_id, '_tpv_min_camera_orbit_x', $tpv_min_camera_orbit_x );
            }
            if ( isset( $_POST['tpv_max_camera_orbit_x'] ) ) {
                $tpv_max_camera_orbit_x = sanitize_text_field( wp_unslash( $_POST['tpv_max_camera_orbit_x'] ) );
                update_post_meta( $post_id, '_tpv_max_camera_orbit_x', $tpv_max_camera_orbit_x );
            }
            update_post_meta( $post_id, '_tpv_apply_yaw_limits_box', sanitize_text_field( $tpv_apply_yaw_limits_box ) );
            /********* Apply Yaw Limits Value Save End */
            /********* Apply Pitch Limits Save Start */
            $tpv_apply_pitch_limits_box = ( isset( $_POST['tpv_apply_pitch_limits_box'] ) ? '1' : '0' );
            if ( isset( $_POST['tpv_min_camera_orbit_y'] ) ) {
                $tpv_min_camera_orbit_y = sanitize_text_field( wp_unslash( $_POST['tpv_min_camera_orbit_y'] ) );
                update_post_meta( $post_id, '_tpv_min_camera_orbit_y', $tpv_min_camera_orbit_y );
            }
            if ( isset( $_POST['tpv_max_camera_orbit_y'] ) ) {
                $tpv_max_camera_orbit_y = sanitize_text_field( wp_unslash( $_POST['tpv_max_camera_orbit_y'] ) );
                update_post_meta( $post_id, '_tpv_max_camera_orbit_y', $tpv_max_camera_orbit_y );
            }
            update_post_meta( $post_id, '_tpv_apply_pitch_limits_box', sanitize_text_field( $tpv_apply_pitch_limits_box ) );
            /********* Apply Pitch Limits Save End */
            /********* Apply Minimum Zoom Start */
            $tpv_apply_minimum_zoom_box = ( isset( $_POST['tpv_apply_minimum_zoom_box'] ) ? '1' : '0' );
            if ( isset( $_POST['tpv_min_camera_orbit_z'] ) ) {
                $tpv_min_camera_orbit_z = sanitize_text_field( wp_unslash( $_POST['tpv_min_camera_orbit_z'] ) );
                update_post_meta( $post_id, '_tpv_min_camera_orbit_z', $tpv_min_camera_orbit_z );
            }
            if ( isset( $_POST['tpv_max_camera_orbit_z'] ) ) {
                $tpv_max_camera_orbit_z = sanitize_text_field( wp_unslash( $_POST['tpv_max_camera_orbit_z'] ) );
                update_post_meta( $post_id, '_tpv_max_camera_orbit_z', $tpv_max_camera_orbit_z );
            }
            if ( isset( $_POST['tpv_min_field_of_view'] ) ) {
                $tpv_min_field_of_view = sanitize_text_field( wp_unslash( $_POST['tpv_min_field_of_view'] ) );
                update_post_meta( $post_id, '_tpv_min_field_of_view', $tpv_min_field_of_view );
            }
            update_post_meta( $post_id, '_tpv_apply_minimum_zoom_box', sanitize_text_field( $tpv_apply_minimum_zoom_box ) );
        }

    }

}