<?php
/*
 * Wetterwarner Admin Einstellungen
 * Author: Tim Knigge
 * https://it93.de/projekte/wetterwarner/
 */
defined( 'ABSPATH' ) or die( 'No script kiddies please!' );

// Admin-Menü & Scripts
add_action( 'admin_menu', 'wetterwarner_add_admin_menu' );
add_action( 'admin_init', 'wetterwarner_settings_init' );
add_action( 'admin_enqueue_scripts', 'wetterwarner_admin_scripts' );

/**
 * Menüpunkt hinzufügen
 */
function wetterwarner_add_admin_menu() {
    add_options_page(
        'Wetterwarner Settings',
        'Wetterwarner',
        'manage_options',
        'wetterwarner',
        'wetterwarner_options_page'
    );
}

/**
 * Sanitize-Callback für alle Einstellungen
 *
 * @param array $input Roh-Daten aus $_POST
 * @return array Gesäuberte Settings
 */
function wetterwarner_sanitize_settings( $input ) {
    $output = array();

    for ( $i = 1; $i <= 4; $i++ ) {
        $key = "ww_farbe_stufe{$i}";
        if ( ! empty( $input[ $key ] ) ) {
            // Erlaubt: rgba(r,g,b,a) oder #rrggbb
            if ( preg_match( '/^(rgba\(\s*\d{1,3},\s*\d{1,3},\s*\d{1,3},\s*0?\.\d+\)\s*|#?[A-Fa-f0-9]{6})$/', $input[ $key ] ) ) {
                $output[ $key ] = sanitize_text_field( $input[ $key ] );
            }
        }
    }

    /**
     * Filter: zusätzliche Sanitizer
     *
     * @param array $output bereits gesäuberte Settings
     * @param array $input  ursprüngliche Eingabe
     */
    return apply_filters( 'wetterwarner_sanitized_settings', $output, $input );
}

/**
 * Einstellungen initialisieren
 */
function wetterwarner_settings_init() {
    register_setting(
        'pluginPage',
        'wetterwarner_settings',
        WW_SETTINGS_ARGS
    );

    add_settings_section(
        'wetterwarner_pluginPage_section',
        __( 'Wetterwarner Settings', 'wetterwarner' ),
        'wetterwarner_settings_section_callback',
        'pluginPage'
    );

    // Farbfelder Stufen 1–4
    foreach ( range( 1, 4 ) as $stufe ) {
        add_settings_field(
            "ww_farbe_stufe{$stufe}",
            sprintf(
                /* translators: %d = Level number */
                __( 'Background Color level %d', 'wetterwarner' ),
                $stufe
            ),
            "ww_farbe_stufe{$stufe}_field",
            'pluginPage',
            'wetterwarner_pluginPage_section'
        );
    }
}

/**
 * WP Color Picker laden
 */
function wetterwarner_admin_scripts() {
    wp_enqueue_style( 'wp-color-picker' );
    wp_enqueue_script(
        'wp-color-picker-alpha',
        plugins_url( 'js/wp-color-picker-alpha.js', __FILE__ ),
        array( 'wp-color-picker', 'jquery' ),
        "3.0.3",
        true
    );
    wp_add_inline_script(
        'wp-color-picker-alpha',
        'jQuery(function($){ $(".color-picker").wpColorPicker(); });'
    );
}

/**
 * Feld-Renderer für jede Stufe
 */
foreach ( range( 1, 4 ) as $stufe ) {
    add_action( 'admin_init', function() use ( $stufe ) {
        $fn = "ww_farbe_stufe{$stufe}_field";
        if ( ! function_exists( $fn ) ) {
            function ww_farbe_stufe1_field() {}
        }
    } );
}

function ww_farbe_stufe1_field() { ww_farbe_stufeX_field( 1 ); }
function ww_farbe_stufe2_field() { ww_farbe_stufeX_field( 2 ); }
function ww_farbe_stufe3_field() { ww_farbe_stufeX_field( 3 ); }
function ww_farbe_stufe4_field() { ww_farbe_stufeX_field( 4 ); }

function ww_farbe_stufeX_field( $stufe ) {
    $options = get_option( 'wetterwarner_settings', array() );
    $key     = "ww_farbe_stufe{$stufe}";
    $default = array(
        1 => 'rgba(255,255,170,0.5)',
        2 => 'rgba(255,218,188,0.5)',
        3 => 'rgba(255,204,204,0.5)',
        4 => 'rgba(198,155,198,0.5)',
    );
    $value = isset( $options[ $key ] ) ? $options[ $key ] : $default[ $stufe ];

    printf(
        '<input type="text" class="color-picker" name="wetterwarner_settings[%1$s]" data-alpha-enabled="true" value="%2$s">',
        esc_attr( $key ),
        esc_attr( $value )
    );
}

/**
 * Sektionstitel und Beschreibung
 */
function wetterwarner_settings_section_callback() {
    echo '<div class="wrap">';
    echo esc_html__( 'The following settings are independent of the widget options.', 'wetterwarner' );
    echo '</div>';
}

/**
 * Optionsseite ausgeben
 */
function wetterwarner_options_page() {
    ?>
    <form action="options.php" method="post">
        <?php
        settings_fields( 'pluginPage' );
        do_settings_sections( 'pluginPage' );
        submit_button();
        ?>
    </form>
    <?php
}


