<?php

namespace WBSY\CF7\Utils;

class Parser {

    public function convert( $form_content ) {
        if ( ! class_exists('WPCF7_FormTagsManager') ) {
            return '<!-- Contact Form 7 not available -->';
        }

        $tags = \WPCF7_FormTagsManager::get_instance()->scan($form_content);
        $html = '<div class="wbsycf7-page-and-images wbsycf7-form-builder">';
        $html .= '<div class="wbsycf7-section">';
        $html .= '<div class="wbsycf7-column wbsycf7-hidden"></div>';
        $html .= '<div class="wbsycf7-column wbsycf7-column-active">';

        $disabled = ' disabled';
        foreach ( $tags as $tag ) {
            $type = rtrim($tag['type'], '*');
            $name = $tag['name'];
            $required = substr($tag['type'], -1) === '*' ? 'required' : '';
            $options = $tag['options'];
            $values = $tag['values'];
            // Parse options into HTML attributes
            $attributes = '';
            foreach ( $options as $i => $option ) {
                if ( strpos($option, ':') !== false ) {
                    [$attr, $value] = explode(':', $option, 2);
                    $attributes .= ' ' . esc_attr($attr) . '="' . esc_attr($value) . '"';
                } elseif ( $option === 'placeholder' && isset($values[$i]) ) {
                    $attributes .= ' placeholder="' . esc_attr($values[$i]) . '"';
                }
            }

            $label = ucfirst(str_replace('-', ' ', $name));
            $html .= '<div class="wbsycf7-form-row wbsycf7-label-top wbsycf7-field-' . esc_attr($type) . '">';

            $required_mark = $required ? ' <span class="wbsycf7-required">*</span>' : '';
            if ( $type !== 'submit' ) {
                $html .= '<div class="wbsycf7-label-wrapper"><label class="wbsycf7-field-label">' . esc_html($label) . $required_mark . '</label></div>';
            }
            $html .= '<div class="wbsycf7-field-wrapper">';
            $class_attr = ' class="wbsycf7-field-' . esc_attr($type) . '"';
            // Render the field based on type
            switch ( $type ) {
                case 'text':
                case 'email':
                case 'url':
                case 'tel':
                case 'number':
                case 'range':
                case 'date':
                    $html .= '<input type="' . esc_attr($type) . '" name="' . esc_attr($name) . '" ' . $required . $class_attr . $attributes . $disabled . '>';
                    break;

                case 'textarea':
                    $html .= '<textarea name="' . esc_attr($name) . '" ' . $required . $class_attr . $attributes . $disabled . '></textarea>';
                    break;

                case 'select':
                    $multiple = in_array('multiple', $options) ? ' multiple' : '';
                    $html .= '<select name="' . esc_attr($name) . ($multiple ? '[]' : '') . '" ' . $required . $multiple . $class_attr . $attributes . $disabled . '>';
                    foreach ($values as $val) {
                        $html .= '<option value="' . esc_attr($val) . '">' . esc_html($val) . '</option>';
                    }
                    $html .= '</select>';
                    break;

                case 'checkbox':
                    foreach ($values as $val) {
                        $input = '<input type="checkbox" name="' . esc_attr($name) . '[]" value="' . esc_attr($val) . '" ' . $required . $class_attr . $attributes . $disabled . '>';
                        if (in_array('use_label_element', $options)) {
                            $html .= '<div class="wbsycf7-checkbox-row">' . $input . ' ' . esc_html($val) . '</div>';
                        } else {
                            $html .= $input . ' ' . esc_html($val);
                        }
                    }
                    break;

                case 'radio':
                    foreach ($values as $val) {
                        $input = '<input type="radio" name="' . esc_attr($name) . '" value="' . esc_attr($val) . '" ' . $required . $class_attr . $attributes . $disabled . '>';
                        if (in_array('use_label_element', $options)) {
                            $html .= '<div class="wbsycf7-radio-row">' . $input . ' ' . esc_html($val) . '</div>';
                        } else {
                            $html .= $input . ' ' . esc_html($val);
                        }
                    }
                    break;

                case 'file':
                    $html .= '<input type="file" name="' . esc_attr($name) . '" ' . $required . $class_attr . $attributes . $disabled . '>';
                    break;

                case 'acceptance':
                    $html .= '<label><input type="checkbox" name="' . esc_attr($name) . '" ' . $required . $class_attr . $attributes . $disabled . '> ' . esc_html($values[0] ?? 'I accept') . '</label>';
                    break;

                case 'quiz':
                    $question = $values[0] ?? 'Quiz Question';
                    $html .= '<label>' . esc_html($question) . '<input type="text" name="' . esc_attr($name) . '" ' . $required . $class_attr . $attributes . $disabled . '></label>';
                    break;

                case 'submit':
                    $label = $values[0] ?? 'Send';
                    $html .= '<input type="submit" name="' . esc_attr($name) . '" value="' . esc_attr($label) . '" ' . $class_attr . $attributes . $disabled . '>';
                    break;

                case 'recaptcha':
                    $html .= '<div class="g-recaptcha" data-sitekey="YOUR_SITE_KEY"></div>';
                    break;

                default:
                    $html .= '<!-- Unsupported type: ' . esc_html($tag['type']) . ' -->';
            }
            /* End of field wrapper */
            $html .= '</div>';
            $html .= '<div class="wbsycf7-actions">
                            <span class="dashicons dashicons-edit wbsycf7-field-edit" title="Edit Field"></span>
                            <span class="dashicons dashicons-trash wbsycf7-field-trash" title="Delete Field"></span>
                        </div>';
            $html .= '</div>';
        }
        $html .= '</div></div></div>';
        $html .= '<div class="wbsycf7-add-new-page">Add New Page</div>';
        return $html;
    }

    public function reverseConvert($html) {
        libxml_use_internal_errors(true);
        $dom = new \DOMDocument();
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        $xpath = new \DOMXPath($dom);

        $data = [];

        // Helpers
        $getWrapperClass = static function(\DOMNode $node) {
            $a = $node->parentNode;
            while ($a && $a instanceof \DOMElement && !$a->hasAttribute('class')) {
                $a = $a->parentNode;
            }
            return ($a instanceof \DOMElement) ? $a->getAttribute('class') : '';
        };

        $collectOptions = static function(\DOMElement $el) {
            $options = [];
            $classes = [];
            foreach (['id', 'class', 'autocomplete', 'minlength', 'maxlength'] as $attr) {
                if ($el->hasAttribute($attr)) {
                    $val = $el->getAttribute($attr);
                    if ($attr === 'class') {
                        foreach (preg_split('/\s+/', trim($val)) as $cls) {
                            if ($cls !== '') {
                                $options[] = 'class:' . $cls;
                                $classes[] = $cls;
                            }
                        }
                    } else {
                        $options[] = $attr . ':' . $val;
                    }
                }
            }
            if ($el->hasAttribute('placeholder')) {
                $options[] = 'placeholder';
            }
            return [$options, $classes];
        };

        $valueForChoice = static function(\DOMElement $input) {
            if ($input->parentNode instanceof \DOMElement && $input->parentNode->nodeName === 'label') {
                $text = trim($input->parentNode->textContent ?? '');
                $raw  = $input->getAttribute('value');
                if ($raw !== '') {
                    $text = preg_replace('/^\s*' . preg_quote($raw, '/') . '\s*/', '', $text);
                }
                $text = trim($text);
                return $text !== '' ? $text : $raw;
            }
            return $input->getAttribute('value');
        };

        // ---------- PASS 1: radios/checkboxes (descendants) ----------
        /** @var \DOMNodeList $checkRadioNodes */
        $checkRadioNodes = $xpath->query(
            '//div[contains(@class, "wbsycf7-field-wrapper")]//input[@type="radio" or @type="checkbox"]'
        );

        $groups = []; // name => merged meta
        foreach ($checkRadioNodes as $node) {
            if (!$node instanceof \DOMElement) continue;

            $rawName = $node->getAttribute('name') ?: 'unnamed-' . uniqid();
            // remove [] suffix which comes from checkbox/radio inputs
            $name = preg_replace('/\[\]$/', '', $rawName);
            $typeAttr = strtolower($node->getAttribute('type') ?: 'checkbox');
            $cf7_type = ($typeAttr === 'radio') ? 'radio' : 'checkbox';

            $isRequired   = $node->hasAttribute('required');
            $wrapperClass = $getWrapperClass($node);
            [$opts, $classes] = $collectOptions($node);
            $useLabel = ($node->parentNode instanceof \DOMElement && $node->parentNode->nodeName === 'label');

            $choice = $valueForChoice($node);
            if ($choice === '') {
                $choice = 'option-' . substr(md5(spl_object_hash($node)), 0, 6);
            }

            if (!isset($groups[$name])) {
                $groups[$name] = [
                    'name'        => $name,
                    'type'        => $cf7_type,
                    'required'    => $isRequired,
                    'wrapper'     => $wrapperClass,
                    'id'          => $node->getAttribute('id') ?: null,
                    'autocomplete'=> $node->getAttribute('autocomplete') ?: null,
                    'minlength'   => $node->getAttribute('minlength') ?: null,
                    'maxlength'   => $node->getAttribute('maxlength') ?: null,
                    'placeholder' => $node->getAttribute('placeholder') ?: null,
                    'classes'     => $classes,
                    'options'     => $opts,
                    'values'      => [$choice],
                    'use_label'   => $useLabel,
                    'first_node'  => $node,
                    'rest_nodes'  => [],
                ];
            } else {
                $g =& $groups[$name];
                $g['required'] = $g['required'] || $isRequired;
                $g['classes']  = array_values(array_unique(array_merge($g['classes'], $classes)));
                $g['options']  = array_values(array_unique(array_merge($g['options'], $opts)));
                $g['use_label']= $g['use_label'] || $useLabel;
                if (!in_array($choice, $g['values'], true)) {
                    $g['values'][] = $choice;
                }
                $g['rest_nodes'][] = $node;
                unset($g);
            }
        }

        // Build one CF7 tag per group; replace first, remove others (+wrapper label if present)
        foreach ($groups as $g) {
            $reqStar = $g['required'] ? '*' : '';
            $options = $g['options'];
            if ($g['use_label'] && !in_array('use_label_element', $options, true)) {
                $options[] = 'use_label_element';
            }
            $options = array_values(array_unique($options));
            $quotedValues = array_map(static fn($v) => '"' . trim($v) . '"', $g['values']);

            $shortcode = '[' . $g['type'] . $reqStar . ' ' . $g['name'];
            if (!empty($options))     $shortcode .= ' ' . implode(' ', $options);
            if (!empty($quotedValues)) $shortcode .= ' ' . implode(' ', $quotedValues);
            $shortcode .= ']';

            // Replace the first node or its wrapping label
            $first = $g['first_node'];
            $replaceTarget = ($first->parentNode instanceof \DOMElement && $first->parentNode->nodeName === 'label')
                ? $first->parentNode
                : $first;
            if ($replaceTarget->parentNode) {
                $replaceTarget->parentNode->replaceChild($dom->createTextNode($shortcode), $replaceTarget);
            }

            // Remove the rest (remove their label wrapper if present)
            foreach ($g['rest_nodes'] as $extra) {
                $toDelete = ($extra->parentNode instanceof \DOMElement && $extra->parentNode->nodeName === 'label')
                    ? $extra->parentNode
                    : $extra;
                if ($toDelete->parentNode) {
                    $toDelete->parentNode->removeChild($toDelete);
                }
            }

            // metadata (one record per group)
            $key = $g['name'];
            $data[$key] = [
                'name'        => $g['name'],
                'type'        => $g['type'],
                'required'    => $g['required'],
                'id'          => $g['id'],
                'class'       => $g['classes'],
                'autocomplete'=> $g['autocomplete'],
                'minlength'   => $g['minlength'],
                'maxlength'   => $g['maxlength'],
                'placeholder' => $g['placeholder'],
                'wrapper'     => $g['wrapper'],
                'original'    => 'input:' . $g['type'],
                'values'      => $g['values'],
                'cf7_options' => $options,
            ];
        }

        // ---------- PASS 2: all other fields (descendants) ----------
        /** @var \DOMNodeList $fieldNodes */
        $fieldNodes = $xpath->query(
            '//div[contains(@class, "wbsycf7-field-wrapper")]//textarea | ' .
            '//div[contains(@class, "wbsycf7-field-wrapper")]//select | ' .
            '//div[contains(@class, "wbsycf7-field-wrapper")]//input[not(@type="radio") and not(@type="checkbox")] | ' .
            '//div[contains(@class, "g-recaptcha")]'
        );

        foreach ($fieldNodes as $field) {
            if (!$field instanceof \DOMElement) continue;

            $tagName  = $field->nodeName;
            $typeAttr = $tagName === 'input' ? strtolower($field->getAttribute('type') ?: 'text') : $tagName;
            $wrapperClass = $getWrapperClass($field);

            // Determine CF7 type (your original mapping)
            if (str_contains($wrapperClass, 'wbsycf7-field-quiz')) {
                $cf7_type = 'quiz';
            } elseif ($tagName === 'div' && str_contains($field->getAttribute('class'), 'g-recaptcha')) {
                $cf7_type = 'recaptcha';
            } else {
                $cf7_type = match ($typeAttr) {
                    'text', 'email', 'tel', 'url', 'number', 'range', 'date', 'file' => $typeAttr,
                    'submit' => 'submit',
                    default => $tagName === 'textarea' ? 'textarea'
                        : ($tagName === 'select' ? 'select' : 'text')
                };
            }

            $isRequired = $field->hasAttribute('required');
            $required   = $isRequired ? '*' : '';
            $name       = $field->getAttribute('name') ?: 'unnamed-' . uniqid();

            [$options, $classes] = $collectOptions($field);

            $values = [];
            if ($cf7_type === 'select') {
                foreach ($field->getElementsByTagName('option') as $option) {
                    $values[] = '"' . trim($option->nodeValue) . '"';
                }
            } elseif ($cf7_type === 'submit') {
                if ($field->hasAttribute('value')) {
                    $values[] = '"' . $field->getAttribute('value') . '"';
                }
            } elseif ($cf7_type === 'quiz') {
                // Extract question label from sibling .label-wrapper
                $labelText = '';
                $parentRow = $field->parentNode;
                while ($parentRow && $parentRow instanceof \DOMElement && !$parentRow->hasAttribute('class')) {
                    $parentRow = $parentRow->parentNode;
                }
                if ($parentRow instanceof \DOMElement) {
                    foreach ($parentRow->childNodes as $child) {
                        if ($child instanceof \DOMElement && $child->nodeName === 'div'
                            && str_contains($child->getAttribute('class') ?? '', 'label-wrapper')) {
                            $labelTag = $child->getElementsByTagName('label')->item(0);
                            if ($labelTag) {
                                $labelText = trim($labelTag->textContent);
                                break;
                            }
                        }
                    }
                }
                $answer = $field->getAttribute('value') ?: '???';
                $values[] = '"' . $labelText . ' | ' . $answer . '"';
            }

            // Build shortcode
            $shortcode = '[' . $cf7_type . $required . ' ' . $name;
            if (!empty($options)) $shortcode .= ' ' . implode(' ', array_unique($options));
            if (!empty($values))  $shortcode .= ' ' . implode(' ', array_unique($values));
            $shortcode .= ']';

            // Replace the field node (if textarea/select/input is wrapped by a label, we still replace the field itself)
            $field->parentNode->replaceChild($dom->createTextNode($shortcode), $field);

            // Metadata
            $key = $name;
            $metaValues = array_map(static fn($v) => trim($v, '"'), $values);
            $data[$key] = [
                'name'        => $name,
                'type'        => $cf7_type,
                'required'    => $isRequired,
                'id'          => $field->getAttribute('id') ?: null,
                'class'       => $classes,
                'autocomplete'=> $field->getAttribute('autocomplete') ?: null,
                'minlength'   => $field->getAttribute('minlength') ?: null,
                'maxlength'   => $field->getAttribute('maxlength') ?: null,
                'placeholder' => $field->getAttribute('placeholder') ?: null,
                'wrapper'     => $wrapperClass,
                'original'    => $tagName . ($tagName === 'input' ? ':' . $typeAttr : ''),
                'values'      => $metaValues,
                'cf7_options' => array_values(array_unique($options)),
            ];
        }

        // Final cleanup
        $out = preg_replace('/^<\?xml[^>]+>\s*/', '', $dom->saveHTML());
        $out = preg_replace('/>\s+</', '><', $out);
        $this->reverse_data = $data;

        return [
            'html' => trim($out),
            'data' => $data,
        ];
    }


    private function buildAttributes( $element ) {
        $ignore = ['type', 'name', 'required', 'disabled', 'value'];
        $parts = [];
        foreach ( $element->attributes as $attr ) {
            if ( ! in_array( $attr->name, $ignore ) ) {
                $parts[] = $attr->name . ':' . $attr->value;
            }
        }
        return implode(' ', $parts);
    }
}
