<?php
namespace WBSY\CF7\Submissions;

if (!defined('ABSPATH')) exit;

/**
 * Handles exporting CF7 submissions to CSV or XLSX.
 * - If POST has submission_ids[]: exports only those (for the given form).
 * - Else: exports all submissions for the form.
 */
class Exporter {

    public function register(): void {
        if (is_admin()) {
            add_action('admin_post_wblscf7_export_submissions', [$this, 'handle']);
        }
    }

    public function handle(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You are not allowed to do this.', 'cf7-builder'));
        }
        if ( ! isset($_POST['_wpnonce']) ||
            ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'wblscf7_export_submissions')) {
            wp_die(esc_html__('Invalid request.', 'cf7-builder'));
        }

        $form_id = isset($_POST['form_id']) ? (int) $_POST['form_id'] : 0;
        $format  = isset($_POST['format']) ? sanitize_text_field(wp_unslash($_POST['format'])) : 'csv';

        // Always exclude private meta (_keys)
        $include_private = false;

        $selected_ids = isset($_POST['submission_ids']) && is_array($_POST['submission_ids'])
            ? array_map('intval', $_POST['submission_ids']) : [];

        if ($form_id <= 0) {
            wp_die(esc_html__('Form ID is required for export.', 'cf7-builder'));
        }

        // If user checked rows, only export those. Filter to ensure they belong to this form.
        $ids = [];
        if (!empty($selected_ids)) {
            $ids = $this->filter_valid_ids_for_form($selected_ids, $form_id);
        }

        // Determine columns from the chosen dataset (selected IDs or entire form)
        if (!empty($ids)) {
            $field_keys = $this->discover_field_keys_for_ids($ids, $include_private);
        } else {
            $field_keys = $this->discover_field_keys($form_id, $include_private);
        }

        $headers   = array_merge(['ID', 'Date'], $field_keys);

        if ($format === 'csv') {
            if (!empty($ids)) $this->stream_csv_ids($ids, $headers, $field_keys);
            else $this->stream_csv_form($form_id, $headers, $field_keys);
            return;
        }

        if ($format === 'xlsx') {
            if (!empty($ids)) $this->stream_xlsx_ids($ids, $headers, $field_keys);
            else $this->stream_xlsx_form($form_id, $headers, $field_keys);
            return;
        }

        wp_die(esc_html__('Unknown format.', 'cf7-builder'));
    }

    /* ===== CSV (all vs selected) ===== */

    private function stream_csv_form(int $form_id, array $headers, array $field_keys): void {
        $filename = 'cf7-submissions-form-' . $form_id . '-' . date('Ymd-His') . '.csv';
        ignore_user_abort(true); @set_time_limit(0); nocache_headers();
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        echo "\xEF\xBB\xBF";
        $out = fopen('php://output', 'w');
        fputcsv($out, $headers);
        $this->walk_submissions($form_id, function($post_id) use ($out, $field_keys){
            fputcsv($out, $this->row_for_post($post_id, $field_keys));
        });
        fclose($out); exit;
    }

    private function stream_csv_ids(array $ids, array $headers, array $field_keys): void {
        $filename = 'cf7-submissions-selected-' . date('Ymd-His') . '.csv';
        ignore_user_abort(true); @set_time_limit(0); nocache_headers();
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        echo "\xEF\xBB\xBF";
        $out = fopen('php://output', 'w');
        fputcsv($out, $headers);
        $this->walk_ids($ids, function($post_id) use ($out, $field_keys){
            fputcsv($out, $this->row_for_post($post_id, $field_keys));
        });
        fclose($out); exit;
    }

    /* ===== XLSX (all vs selected) ===== */

    private function stream_xlsx_form(int $form_id, array $headers, array $field_keys): void {
        ignore_user_abort(true); @set_time_limit(0); nocache_headers();
        $rows_xml = [];
        $rows_xml[] = $this->xlsx_row_xml(1, $headers);
        $rowNum = 2;
        $this->walk_submissions($form_id, function($post_id) use (&$rows_xml, &$rowNum, $field_keys){
            $rows_xml[] = $this->xlsx_row_xml($rowNum++, $this->row_for_post($post_id, $field_keys));
        });
        $this->output_xlsx_zip($rows_xml, $headers, 'cf7-submissions-form-' . $form_id);
    }

    private function stream_xlsx_ids(array $ids, array $headers, array $field_keys): void {
        ignore_user_abort(true); @set_time_limit(0); nocache_headers();
        $rows_xml = [];
        $rows_xml[] = $this->xlsx_row_xml(1, $headers);
        $rowNum = 2;
        $this->walk_ids($ids, function($post_id) use (&$rows_xml, &$rowNum, $field_keys){
            $rows_xml[] = $this->xlsx_row_xml($rowNum++, $this->row_for_post($post_id, $field_keys));
        });
        $this->output_xlsx_zip($rows_xml, $headers, 'cf7-submissions-selected');
    }

    private function output_xlsx_zip(array $rows_xml, array $headers, string $basename): void {
        $sheet_xml = $this->xlsx_sheet_xml($rows_xml, count($headers), count($rows_xml)); // rows_xml includes header
        $parts = [
            '[Content_Types].xml'        => $this->xlsx_content_types(),
            '_rels/.rels'                => $this->xlsx_root_rels(),
            'xl/workbook.xml'            => $this->xlsx_workbook(),
            'xl/_rels/workbook.xml.rels' => $this->xlsx_workbook_rels(),
            'xl/styles.xml'              => $this->xlsx_styles(),
            'xl/worksheets/sheet1.xml'   => $sheet_xml,
        ];
        $xlsx_path = $this->zip_parts_to_xlsx($parts);
        $filename  = $basename . '-' . date('Ymd-His') . '.xlsx';
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . filesize($xlsx_path));
        readfile($xlsx_path);
        @unlink($xlsx_path);
        exit;
    }

    /* ===== Column discovery ===== */

    private function discover_field_keys(int $form_id, bool $include_private = false): array {
        $keys = []; $paged = 1;
        do {
            $q = new \WP_Query([
                'post_type'      => 'wblscf7_submission',
                'post_status'    => 'publish',
                'fields'         => 'ids',
                'meta_key'       => '_form_id',
                'meta_value'     => $form_id,
                'posts_per_page' => 500,
                'paged'          => $paged,
                'no_found_rows'  => false,
            ]);
            if (!$q->have_posts()) break;
            foreach ($q->posts as $post_id) {
                foreach (get_post_meta($post_id) as $k => $v) {
                    if (!$include_private && strpos($k, '_') === 0) continue;
                    if (!in_array($k, $keys, true)) $keys[] = $k;
                }
            }
            $paged++;
        } while ($paged <= $q->max_num_pages);

        sort($keys, SORT_NATURAL | SORT_FLAG_CASE);
        return $keys;
    }

    private function discover_field_keys_for_ids(array $ids, bool $include_private = false): array {
        $keys = [];
        foreach ($ids as $post_id) {
            foreach (get_post_meta($post_id) as $k => $v) {
                if (!$include_private && strpos($k, '_') === 0) continue;
                if (!in_array($k, $keys, true)) $keys[] = $k;
            }
        }
        sort($keys, SORT_NATURAL | SORT_FLAG_CASE);
        return $keys;
    }

    /* ===== Iterators ===== */

    private function walk_submissions(int $form_id, callable $cb): void {
        $paged = 1;
        do {
            $q = new \WP_Query([
                'post_type'      => 'wblscf7_submission',
                'post_status'    => 'publish',
                'fields'         => 'ids',
                'meta_key'       => '_form_id',
                'meta_value'     => $form_id,
                'orderby'        => 'date',
                'order'          => 'DESC',
                'posts_per_page' => 500,
                'paged'          => $paged,
                'no_found_rows'  => false,
            ]);
            if (!$q->have_posts()) break;
            foreach ($q->posts as $post_id) $cb($post_id);
            $paged++;
        } while ($paged <= $q->max_num_pages);
    }

    private function walk_ids(array $ids, callable $cb): void {
        $ids = array_values(array_unique(array_map('intval', $ids)));
        if (!$ids) return;
        // Fetch in chunks to avoid memory issues
        $chunks = array_chunk($ids, 500);
        foreach ($chunks as $chunk) {
            $q = new \WP_Query([
                'post_type'      => 'wblscf7_submission',
                'post_status'    => 'publish',
                'fields'         => 'ids',
                'posts_per_page' => -1,
                'post__in'       => $chunk,
                'orderby'        => 'post__in',
                'no_found_rows'  => true,
            ]);
            foreach ($q->posts as $post_id) $cb($post_id);
        }
    }

    private function filter_valid_ids_for_form(array $ids, int $form_id): array {
        $ids = array_values(array_unique(array_map('intval', $ids)));
        if (!$ids) return [];
        $q = new \WP_Query([
            'post_type'      => 'wblscf7_submission',
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => -1,
            'post__in'       => $ids,
            'meta_key'       => '_form_id',
            'meta_value'     => $form_id,
            'no_found_rows'  => true,
        ]);
        return $q->posts;
    }

    /* ===== Rows ===== */

    private function row_for_post(int $post_id, array $field_keys): array {
        $post = get_post($post_id);
        $row  = [];
        $row[] = (string)$post_id;
        $row[] = mysql2date('Y-m-d H:i:s', $post->post_date, false);
        foreach ($field_keys as $key) {
            $val = get_post_meta($post_id, $key, true);
            if (is_array($val)) $val = wp_json_encode($val, JSON_UNESCAPED_UNICODE);
            $row[] = (string)$val;
        }
        return $row;
    }

    /* ===== XLSX pieces + zipping (ZipArchive → PclZip) ===== */

    private function xlsx_content_types(): string {
        return '<?xml version="1.0" encoding="UTF-8"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
  <Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
  <Default Extension="xml" ContentType="application/xml"/>
  <Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>
  <Override PartName="/xl/worksheets/sheet1.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>
  <Override PartName="/xl/styles.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml"/>
</Types>';
    }

    private function xlsx_root_rels(): string {
        return '<?xml version="1.0" encoding="UTF-8"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>
</Relationships>';
    }

    private function xlsx_workbook(): string {
        return '<?xml version="1.0" encoding="UTF-8"?>
<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">
  <sheets>
    <sheet name="Submissions" sheetId="1" r:id="rId1"/>
  </sheets>
</workbook>';
    }

    private function xlsx_workbook_rels(): string {
        return '<?xml version="1.0" encoding="UTF-8"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/sheet1.xml"/>
  <Relationship Id="rId2" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/styles" Target="styles.xml"/>
</Relationships>';
    }

    private function xlsx_styles(): string {
        return '<?xml version="1.0" encoding="UTF-8"?>
<styleSheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">
  <fonts count="1"><font><sz val="11"/><name val="Calibri"/></font></fonts>
  <fills count="1"><fill><patternFill patternType="none"/></fill></fills>
  <borders count="1"><border/></borders>
  <cellStyleXfs count="1"><xf numFmtId="0" fontId="0" fillId="0" borderId="0"/></cellStyleXfs>
  <cellXfs count="1"><xf numFmtId="0" fontId="0" fillId="0" borderId="0" xfId="0"/></cellXfs>
  <cellStyles count="1"><cellStyle name="Normal" xfId="0" builtinId="0"/></cellStyles>
</styleSheet>';
    }

    private function xlsx_sheet_xml(array $rows_xml, int $cols, int $rows): string {
        $lastRef = $this->xlsx_cell_ref($cols, $rows);
        $sheetData = implode('', $rows_xml);
        return '<?xml version="1.0" encoding="UTF-8"?>'
            . '<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">'
            . '<dimension ref="A1:' . $lastRef . '"/>'
            . '<sheetData>' . $sheetData . '</sheetData>'
            . '</worksheet>';
    }

    private function xlsx_row_xml(int $rowIndex, array $cells): string {
        $xml = '<row r="' . $rowIndex . '">';
        $colIndex = 1;
        foreach ($cells as $val) {
            $ref = $this->xlsx_col($colIndex++) . $rowIndex;
            $t = $this->xml_text((string)$val);
            $xml .= '<c r="' . $ref . '" t="inlineStr"><is><t xml:space="preserve">' . $t . '</t></is></c>';
        }
        $xml .= '</row>';
        return $xml;
    }

    private function xlsx_cell_ref(int $col, int $row): string { return $this->xlsx_col($col) . $row; }

    private function xlsx_col(int $index): string {
        $s = '';
        while ($index > 0) { $m = ($index - 1) % 26; $s = chr(65 + $m) . $s; $index = intdiv($index - 1, 26); }
        return $s;
    }

    private function xml_text(string $s): string {
        $s = preg_replace('/[^\x09\x0A\x0D\x20-\x{D7FF}\x{E000}-\x{FFFD}]/u', '', $s);
        return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }

    private function zip_parts_to_xlsx(array $parts): string {
        $xlsx_path = wp_tempnam('wbsy_cf7_export.xlsx');
        if (!$xlsx_path) wp_die(esc_html__('Cannot create temporary file for XLSX.', 'cf7-builder'));

        if (class_exists('\ZipArchive')) {
            $zip = new \ZipArchive();
            if ($zip->open($xlsx_path, \ZipArchive::OVERWRITE) !== true) {
                wp_die(esc_html__('Failed to open ZIP for XLSX.', 'cf7-builder'));
            }
            foreach ($parts as $name => $content) $zip->addFromString($name, $content);
            $zip->close();
            return $xlsx_path;
        }

        if (!class_exists('\PclZip')) {
            if (file_exists(ABSPATH . 'wp-admin/includes/class-pclzip.php')) {
                require_once ABSPATH . 'wp-admin/includes/class-pclzip.php';
            } else {
                wp_die(esc_html__('ZipArchive is disabled and PclZip is not available.', 'cf7-builder'));
            }
        }

        $uuid    = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : uniqid('', true);
        $workdir = trailingslashit(get_temp_dir()) . 'wbsy_cf7_xlsx_' . $uuid;
        if ( ! wp_mkdir_p($workdir) ) wp_die(esc_html__('Failed to create temp directory.', 'cf7-builder'));

        foreach ($parts as $name => $content) {
            $full = $workdir . '/' . $name; $dir = dirname($full);
            if ( ! is_dir($dir) && ! wp_mkdir_p( $dir ) ) {
                $this->rrmdir($workdir);
                wp_die(esc_html__('Failed to create XLSX structure.', 'cf7-builder'));
            }
            file_put_contents($full, $content);
        }

        $zip = new \PclZip($xlsx_path);
        $list = $zip->create($workdir, PCLZIP_OPT_REMOVE_PATH, $workdir);
        if ($list === 0) { $err = method_exists($zip, 'errorInfo') ? $zip->errorInfo(true) : 'PclZip error'; $this->rrmdir($workdir); wp_die(esc_html($err)); }

        $this->rrmdir($workdir);
        return $xlsx_path;
    }

    private function rrmdir(string $dir): void {
        if (!is_dir($dir)) return;
        $it = new \RecursiveDirectoryIterator($dir, \FilesystemIterator::SKIP_DOTS);
        $ri = new \RecursiveIteratorIterator($it, \RecursiveIteratorIterator::CHILD_FIRST);
        foreach ($ri as $file) { $file->isDir() ? @rmdir($file->getPathname()) : @unlink($file->getPathname()); }
        @rmdir($dir);
    }
}
