<?php
namespace WBSY\CF7\Submissions;

class AdminPage {
    public function register_menu() {
        add_action("admin_menu", [$this, 'add_menu_page']);

        add_action('admin_post_wblscf7_delete_submissions', function () {
            if ( ! current_user_can('manage_options' ) ||
                ! isset($_GET['form_id']) ||
                ! wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'wblscf7_delete_submissions')) {
                wp_die(esc_html__('You are not allowed to do this.', 'cf7-builder'));
            }

            $form_id = (int) $_GET['form_id'];

            $submissions = get_posts([
                'post_type'  => 'wblscf7_submission',
                'meta_key'   => '_form_id',
                'meta_value' => $form_id,
                'posts_per_page' => -1,
                'fields'     => 'ids'
            ]);

            foreach ($submissions as $sub_id) {
                wp_trash_post($sub_id);
            }

            wp_redirect(admin_url('admin.php?page=wblscf7-submissions'));
            exit;
        });

        add_action('admin_post_wblscf7_delete_submission', function () {
            if ( ! current_user_can('manage_options' ) ||
                ! isset($_GET['submission_id']) ||
                ! wp_verify_nonce(sanitize_text_field($_GET['_wpnonce']), 'wblscf7_delete_submission')) {
                wp_die(esc_html__('You are not allowed to do this.', 'cf7-builder'));
            }

            wp_trash_post((int) $_GET['submission_id']);
            wp_redirect(wp_get_referer());
            exit;
        });

        add_action('admin_post_wblscf7_bulk_delete_submissions', function () {
            if ( !current_user_can('manage_options') ) {
                wp_die(esc_html__('You are not allowed to do this.', 'cf7-builder'));
            }

            if ( !empty($_POST['submission_ids']) && is_array($_POST['submission_ids']) ) {
                foreach ($_POST['submission_ids'] as $id) {
                    wp_trash_post((int) $id);
                }
            }

            wp_redirect(wp_get_referer());
            exit;
        });
    }

    public function add_menu_page() {
        add_submenu_page(
            'wpcf7',                           // Parent slug (CF7 main menu)
            'Form Submissions',               // Page title
            'Submissions',                    // Menu title
            'manage_options',                 // Capability
            'wblscf7-submissions',              // Menu slug
            [$this,'wblscf7_render_submissions_page']   // Callback function
        );
    }

    public function wblscf7_render_submissions_page() {

        $form_id = isset($_GET['form_id']) ? intval($_GET['form_id']) : 0;

        echo '<div class="wrap"><h1>Form Submissions</h1>';
        echo '<p>Here you can see the submissions for your CF7 forms.</p>';
        if ($form_id) {
            $this->render_submission_table($form_id);
            echo '<p><a href="' . admin_url('admin.php?page=wblscf7-submissions') . '">&larr; ' . __('Back to forms list', 'cf7-builder') . '</a></p>';
        } else {
            $this->render_forms_list();
        }
        echo '</div>';
    }

    private function render_forms_list() {
        $table = new SubmissionsList();
        $table->process_bulk_action();
        $table->prepare_items();

        echo '<div class="wrap">';
        echo '<h1>' . __('Contact Forms', 'cf7-builder') . '</h1>';
        echo '<form method="post">';
        $table->display();
        echo '</form>';
        echo '</div>';
    }

    private function render_submission_table($form_id) {
        wp_enqueue_script(
            'wbsycf7-admin-js',
            WBSY_CF7_PLUGIN_URL . 'assets/admin/js/submissions.js',
            ['jquery'],
            WBSY_CF7_VERSION,
            true
        );

        $orderby = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'date';
        $order   = isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC';

        $per_page = 20; 
        $paged    = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;

        $args = [
            'post_type'      => 'wblscf7_submission',
            'meta_key'       => '_form_id',
            'meta_value'     => $form_id,
            'posts_per_page' => $per_page,
            'paged'          => $paged,
            'orderby'        => $orderby,
            'order'          => $order
        ];

        $query = new \WP_Query($args);
        $submissions = $query->posts;

        if (empty($submissions)) {
            echo '<p>' . __('No submissions found for this form.', 'cf7-builder') . '</p>';
            return;
        }

        // Collect all field keys (columns)
        $field_keys = [];
        foreach ($submissions as $sub) {
            $meta = get_post_meta($sub->ID);
            foreach ($meta as $key => $value) {
                if (!str_starts_with($key, '_') && !in_array($key, $field_keys)) {
                    $field_keys[] = $key;
                }
            }
        }

        echo '<form id="wblscf7-export-form" method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin:12px 0;">';
        wp_nonce_field('wblscf7_export_submissions', '_wpnonce');
        echo '<input type="hidden" name="action" value="wblscf7_export_submissions">';
        echo '<input type="hidden" name="form_id" value="' . esc_attr($form_id) . '">';
        echo '<button type="submit" class="wblscf7-export-button button button-primary" name="format" value="csv" data-export="csv">' . esc_html__('Export CSV', 'cf7-builder') . '</button> ';
        echo '<button type="submit" class="wblscf7-export-button button" name="format" value="xlsx" data-export="xlsx">' . esc_html__('Export XLSX', 'cf7-builder') . '</button>';
        echo '</form>';

        // Bulk delete form
        echo '<form method="post">';
        echo '<input type="hidden" name="action" value="wblscf7_bulk_delete_submissions">';
        echo '<input type="hidden" name="form_id" value="' . esc_attr($form_id) . '">';

        echo '<table class="widefat striped">';
        echo '<thead><tr>';
        echo '<td><input type="checkbox" id="cb-select-all"></td>';

        // Sortable date column
        $date_sort_url = add_query_arg([
            'orderby' => 'date',
            'order'   => $order === 'ASC' ? 'DESC' : 'ASC',
        ]);
        echo '<th><a href="' . esc_url($date_sort_url) . '"><span>' . __('Date', 'cf7-builder') . '</span></a></th>';

        foreach ($field_keys as $key) {
            echo '<th>' . esc_html($key) . '</th>';
        }

        echo '<th>' . __('Actions', 'cf7-builder') . '</th>';
        echo '</tr></thead><tbody>';

        foreach ($submissions as $sub) {
            echo '<tr>';
            echo '<td><input type="checkbox" name="submission_ids[]" value="' . esc_attr($sub->ID) . '"></td>';
            echo '<td>' . esc_html(get_the_date('', $sub)) . '</td>';
            foreach ($field_keys as $key) {
                $val = get_post_meta($sub->ID, $key, true);
                echo '<td>' . esc_html($val) . '</td>';
            }

            $delete_url = wp_nonce_url(admin_url('admin-post.php?action=wblscf7_delete_submission&submission_id=' . $sub->ID), 'wblscf7_delete_submission');
            echo '<td><a href="' . esc_url($delete_url) . '" onclick="return confirm(\'Delete this submission?\')">' . __('Delete', 'cf7-builder') . '</a></td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        echo '<p><input type="submit" class="button action" value="' . __('Delete Selected', 'cf7-builder') . '" onclick="return confirm(\'Are you sure you want to delete selected submissions?\')"></p>';
        echo '</form>';

        // Select all checkbox JS
        echo <<<HTML
        <script>
        document.getElementById('cb-select-all').addEventListener('click', function() {
            document.querySelectorAll('input[name="submission_ids[]"]').forEach(cb => cb.checked = this.checked);
        });
        </script>
        HTML;

        $total_pages = $query->max_num_pages;
        if ($total_pages > 1) {
            $base_url = remove_query_arg('paged');
            echo '<div class="tablenav"><div class="tablenav-pages">';
            echo paginate_links([
                'base'      => add_query_arg('paged', '%#%', $base_url),
                'format'    => '',
                'current'   => $paged,
                'total'     => $total_pages,
                'prev_text' => __('« Prev'),
                'next_text' => __('Next »'),
                'type'      => 'plain',
            ]);
            echo '</div></div>';
        }
    }
}
