<?php

namespace WBSY\CF7\Frontend;

class Override {
    public function register() {
        add_action('init', function () {
            add_rewrite_rule('^wbsycf7-preview/([0-9]+)/?', 'index.php?wbsycf7_preview_form_id=$matches[1]', 'top');
            add_rewrite_tag('%wbsycf7_preview_form_id%', '([0-9]+)');
        });

        add_action('template_redirect', function () {
            $form_id = get_query_var('wbsycf7_preview_form_id');

            if ($form_id) {
                $post = get_post($form_id);
                if ($post && $post->post_type === 'wpcf7_contact_form') {
                    echo '<html><head>';
                    wp_head();
                    echo '</head><body>';
                    echo do_shortcode('[contact-form-7 id="' . esc_attr($form_id) . '"]');
                    wp_footer();
                    echo '</body></html>';
                    exit;
                } else {
                    wp_die('Form not found.');
                }
            }
        });

        // Pro start (unchanged) ...
        add_filter('do_shortcode_tag', function ($output, $tag, $attr) {
            if ($tag !== 'contact-form-7' || empty($attr['id'])) {
                return $output;
            }

            $form_id  = absint($attr['id']);
            $settings = get_post_meta($form_id, '_wblscf7_builder_settings', true);
            if (empty($settings)) return $output;

            $settings_json = esc_attr(json_encode([
                'success_action'        => $settings['success_action'] ?? 'default',
                'redirect_page_id'      => $settings['redirect_page_id'] ?? '',
                'custom_success_message'=> $settings['custom_success_message'] ?? '',
            ]));

            $output = preg_replace(
                '/<form(.*?)>/',
                '<form$1 data-wblscf7-settings="'.$settings_json.'">',
                $output
            );

            if (!empty($settings['redirect_page_id']) && $settings['success_action'] === 'redirect') {
                $redirect_url = get_permalink($settings['redirect_page_id']);
                $hidden_input = '<input type="hidden" name="redirect_url" value="' . esc_url($redirect_url) . '">';
                $output = str_replace('</form>', $hidden_input . '</form>', $output);
            }

            return $output;
        }, 10, 3);
        // Pro end

        add_filter('wpcf7_validate', [$this, 'handle_conditional_validation'], 99, 2);
    }

    public function handle_conditional_validation( $result, $tags ) {
        // Get current form + posted data
        $submission = \WPCF7_Submission::get_instance();
        $cf         = function_exists('\wpcf7_get_current_contact_form') ? \wpcf7_get_current_contact_form() : null;

        if ( ! $submission || ! ( $cf instanceof \WPCF7_ContactForm ) ) {
            return $result;
        }

        $posted = $submission->get_posted_data();
        $form_id = (int) $cf->id();

        // 1) Get hidden target names from frontend hidden input; fallback to server-side compute
        $hidden = $this->get_hidden_names_from_request();
        if (empty($hidden)) {
            $hidden = $this->compute_hidden_targets_server_side( $cf, $posted );
        }
        if (empty($hidden)) {
            return $result; // nothing to do
        }

        // Normalize keys like "field" and "field[]"
        $hidden_lut = [];
        foreach ($hidden as $n) {
            $hidden_lut[$n] = true;
            $hidden_lut[$n.'[]'] = true;
        }

        // 2) Remove validation errors for those hidden fields
        // Newer CF7 has $result->remove_error( $name ) (remove by field name)
        if (method_exists($result, 'remove_error')) {
            foreach ($hidden_lut as $name => $_) {
                $result->remove_error( $name );
            }
            return $result;
        }

        // Older CF7: manipulate invalid_fields directly (fallback)
        if (method_exists($result, 'get_invalid_fields')) {
            $invalid = (array) $result->get_invalid_fields();
            if (!empty($invalid)) {
                foreach (array_keys($invalid) as $key) {
                    if (isset($hidden_lut[$key])) {
                        unset($invalid[$key]);
                    }
                }
                // write back via reflection to private property
                $ref = new \ReflectionClass($result);
                if ($ref->hasProperty('invalid_fields')) {
                    $prop = $ref->getProperty('invalid_fields');
                    $prop->setAccessible(true);
                    $prop->setValue($result, $invalid);
                }
            }
        }

        return $result;
    }

    /**
     * Read hidden target names sent by frontend (JSON array in hidden input).
     */
    private function get_hidden_names_from_request(): array {
        $hidden_json = isset($_POST['wbsycf7_hidden_names']) ? wp_unslash($_POST['wbsycf7_hidden_names']) : '[]';
        $hidden = json_decode($hidden_json, true);
        return is_array($hidden) ? array_values(array_unique(array_filter($hidden, 'strlen'))) : [];
    }

    /**
     * Server-side fallback: recompute which targets are hidden based on saved conditions and posted values.
     * Supports operators: is, is_not, empty, not_empty; group logic all/any; group action show/hide.
     */
    private function compute_hidden_targets_server_side(\WPCF7_ContactForm $cf, array $posted): array {
        $form_id = (int) $cf->id();
        $conds   = get_post_meta($form_id, '_wbsycf7_conditions', true);
        if ( ! is_array($conds) || ! $conds ) return [];

        $get = function(string $name) use ($posted) {
            if (array_key_exists($name, $posted)) return $posted[$name];
            if (array_key_exists($name.'[]', $posted)) return $posted[$name.'[]'];
            return '';
        };
        $is_empty = function($v): bool {
            return is_array($v) ? count($v) === 0 : trim((string)$v) === '';
        };
        $match_item = function(array $item) use ($get, $is_empty): bool {
            $name = (string)($item['fieldName'] ?? '');
            $op   = (string)($item['op'] ?? 'is');
            $v    = $get($name);

            if ($op === 'empty')     return $is_empty($v);
            if ($op === 'not_empty') return ! $is_empty($v);

            $want = $item['value'] ?? '';
            if (is_array($v)) {
                if (is_array($want)) {
                    $set = array_flip(array_map('strval', $v));
                    foreach ($want as $w) {
                        if (isset($set[(string)$w])) return $op === 'is';
                    }
                    return $op !== 'is';
                } else {
                    $has = in_array((string)$want, array_map('strval', $v), true);
                    return $op === 'is' ? $has : ! $has;
                }
            } else {
                if (is_array($want)) {
                    $any = in_array((string)$v, array_map('strval', $want), true);
                    return $op === 'is' ? $any : ! $any;
                } else {
                    $eq = (string)$v === (string)$want;
                    return $op === 'is' ? $eq : ! $eq;
                }
            }
        };

        $decisions = []; // name => ['action'=>'show|hide','res'=>bool]
        foreach ($conds as $g) {
            $items = isset($g['items']) && is_array($g['items']) ? $g['items'] : [];
            $res   = true;
            if ($items) {
                $checks = array_map($match_item, $items);
                $res = (($g['logic'] ?? 'all') === 'any')
                    ? in_array(true, $checks, true)
                    : !in_array(false, $checks, true);
            }
            $action = ($g['action'] ?? 'show') === 'hide' ? 'hide' : 'show';
            $target = (string)($g['fieldName'] ?? '');
            if ($target === '') continue;
            $decisions[$target] = ['action' => $action, 'res' => $res]; // last group wins
        }

        $hidden = [];
        foreach ($decisions as $name => $d) {
            $should_show = ($d['action'] === 'show') ? (bool)$d['res'] : ! (bool)$d['res'];
            if (!$should_show) $hidden[] = $name;
        }
        return $hidden;
    }
}
