<?php

namespace WBSY\CF7\Admin;

use WBSY\CF7\Utils\Utils;

class ConditionalFields {

    public function register($post) {
        $this->display($post);
    }

    public function display($post) {
        if ($post instanceof \WPCF7_ContactForm) {
            $post_id = (int) $post->id();
        } elseif (is_object($post) && isset($post->ID)) {
            $post_id = (int) $post->ID;
        } elseif (isset($_GET['post'])) {
            $post_id = (int) $_GET['post'];
        } else {
            $post_id = 0;
        }

        // Saved conditions (array of groups)
        $conditions = $post_id ? get_post_meta($post_id, '_wbsycf7_conditions', true) : [];
        if (!is_array($conditions)) $conditions = [];

        // Fields meta produced by reverseConvert() and saved earlier
        // (keyed by field name; contains type, values, etc.)
        $fieldsMeta = get_post_meta($post_id, '_wblscf7_form_fields', true);

        if (!is_array($fieldsMeta)) {
            $fieldsMeta = [];
        }

        // List of field names for the dropdowns
        $allFieldNames = array_keys($fieldsMeta);

        // JSON to preload + hidden field value
        $jsonConditions = wp_json_encode($conditions);

        // Render the UI HTML from saved JSON/meta
        $html = $this->render_conditions_html($conditions, $fieldsMeta, $allFieldNames);

        ob_start();
        ?>
        <p>
            Press Add Condition button to configure the first condition of your form.
            Show/Hide select box represents the action which will be completed, if all or any of the condition statements are fulfilled.
            Use the second drop-down menu to select the field which will be shown or hidden.
            Click the little Plus (+) icon to add the statement of your form condition.
        </p>

        <span class="wbsycf7-condition-add-group button button-primary">Add Condition</span>

        <?php
        // Conditions UI (built from JSON)
        echo wp_kses( $html, Utils::allowed_html_tags() );
        ?>

        <!-- Hidden field that will carry your JS object to PHP -->
        <input type="hidden"
               id="wbsycf7_conditions_input"
               name="wbsycf7_conditions_input"
               value="<?php echo esc_attr($jsonConditions ?: '[]'); ?>">

        <?php
        echo wp_kses( ob_get_clean(), Utils::allowed_html_tags() );
    }



    /**
     * Render Conditions UI from saved JSON.
     *
     * @param array $conditions  Array of groups as saved (id, action, fieldName, logic, items[]).
     * @param array $fieldsMeta  Map: fieldName => ['type'=>'radio|checkbox|select|text|email|...', 'values'=>[...], ...]
     * @param array $allFieldNames List of selectable field names for the dropdowns (order as you prefer).
     * @return string HTML
     */
    public function render_conditions_html(array $conditions, array $fieldsMeta, array $allFieldNames): string
    {
        // Helpers
        $e  = 'esc_html';
        $ea = 'esc_attr';

        $renderSelect = function(string $class, string $name, array $options, $selected = null, array $attrs = [] ) use ($ea, $e): string {
            $attrStr = '';
            foreach ($attrs as $k => $v) {
                if (is_bool($v)) {
                    if ($v) $attrStr .= ' ' . $ea($k);
                } else {
                    $attrStr .= ' ' . $ea($k) . '="' . $ea((string)$v) . '"';
                }
            }
            $html = '<select class="'.$ea($class).'" name="'.$ea($name).'"'.$attrStr.'>';
            $selected = is_array($selected) ? array_map('strval', $selected) : (string)$selected;

            foreach ($options as $val => $label) {
                $isSel = is_array($selected) ? in_array((string)$val, $selected, true) : ((string)$val === (string)$selected);
                $html .= '<option value="'.$ea((string)$val).'"'.($isSel?' selected':'').'>'.$e((string)$label).'</option>';
            }
            $html .= '</select>';
            return $html;
        };

        $renderInput = function(string $class, string $name, string $type, $value = '', array $attrs = []) use ($ea): string {
            $attrStr = '';
            foreach ($attrs as $k => $v) {
                if (is_bool($v)) {
                    if ($v) $attrStr .= ' ' . $ea($k);
                } else {
                    $attrStr .= ' ' . $ea($k) . '="' . $ea((string)$v) . '"';
                }
            }
            return '<input class="'.$ea($class).'" name="'.$ea($name).'" type="'.$ea($type).'" value="'.$ea((string)$value).'"'.$attrStr.'>';
        };

        $operatorOptions = [
                'is'         => 'is',
                'is_not'     => 'is not',
                'empty'      => 'empty',
                'not_empty'  => 'not empty',
        ];
        $showHideOptions = [ 'show' => 'Show', 'hide' => 'Hide' ];
        $logicOptions    = [ 'all'  => 'All',  'any'  => 'Any'  ];

        // Build once: options map for field selects
        $fieldOptions = [];
        foreach ($allFieldNames as $fname) {
            $fieldOptions[$fname] = $fname;
        }

        // Start HTML
        $out = '<div class="wbsycf7-conditions-container">';

        foreach ($conditions as $group) {
            $gid       = isset($group['id']) ? (int)$group['id'] : 0;
            $action    = (isset($group['action']) && $group['action'] === 'hide') ? 'hide' : 'show';
            $logic     = (isset($group['logic']) && $group['logic'] === 'any') ? 'any' : 'all';
            $groupField= isset($group['fieldName']) ? (string)$group['fieldName'] : '';
            $items     = isset($group['items']) && is_array($group['items']) ? $group['items'] : [];

            $out .= '<div class="wbsycf7-conditionGroup-row" data-id="'.$ea((string)$gid).'">';
            $out .=   '<div class="wbsycf7-conditionGroup-title-row">';

            // Show/Hide
            $out .= $renderSelect('wbsycf7-condition-show-hide', 'wbsycf7_condition_show_hide"', $showHideOptions, $action);

            // Group field select (disabled if items exist)
            $disabled = !empty($items);
            $out .= $renderSelect(
                    'wbsycf7-form-fields',
                    'wbsycf7_form_fields',
                    $fieldOptions,
                    $groupField,
                    $disabled ? ['disabled' => true] : []
            );

            $out .= '<span>if</span>';

            // All/Any
            $out .= $renderSelect('wbsycf7-condition-and-or', 'wbsycf7_condition_and_or', $logicOptions, $logic);

            $out .= '<span>of the following match:</span>';

            // Group actions
            $out .= '<div class="wbsycf7-condition-actions">';
            $out .=   '<span class="dashicons dashicons-plus wbsycf7-condition-item-add" title="Add Condition"></span>';
            $out .=   '<span class="dashicons dashicons-trash wbsycf7-condition-group-delete" title="Delete"></span>';
            $out .= '</div>'; // actions

            $out .= '</div>'; // title row

            // Items
            $out .= '<div class="wbsycf7-conditionGroup-items">';

            foreach ($items as $item) {
                $iid   = isset($item['id']) ? (int)$item['id'] : 0;
                $iField= isset($item['fieldName']) ? (string)$item['fieldName'] : '';
                $op    = isset($item['op']) ? (string)$item['op'] : 'is';
                $val   = $item['value'] ?? '';

                // Determine field meta
                $key = preg_replace('/\[\]$/', '', $iField);
                $meta = $fieldsMeta[$iField] ?? $fieldsMeta[$key] ?? ['type' => 'text', 'values' => []];
                $type = strtolower((string)($meta['type'] ?? 'text'));
                $values = [];
                // choices list: use 'values' (strings), fallback 'options' if ever present
                if (!empty($meta['values']) && is_array($meta['values'])) {
                    foreach ($meta['values'] as $v) { $values[(string)$v] = (string)$v; }
                } elseif (!empty($meta['options']) && is_array($meta['options'])) {
                    foreach ($meta['options'] as $opt) {
                        if (is_array($opt)) {
                            $vv = (string)($opt['value'] ?? ($opt['label'] ?? ''));
                            $ll = (string)($opt['label'] ?? $vv);
                            $values[$vv] = $ll;
                        } else {
                            $values[(string)$opt] = (string)$opt;
                        }
                    }
                }

                // Build item row
                $out .= '<div class="wbsycf7-conditionGroup-item-row" data-id="'.$ea((string)$iid).'">';

                // Item field select (pruned to remove the group field)
                $itemFieldOptions = $fieldOptions;
                if ($groupField && isset($itemFieldOptions[$groupField])) {
                    unset($itemFieldOptions[$groupField]);
                }
                $out .= $renderSelect('wbsycf7-form-items-fields', 'wbsycf7_form_items_fields', $itemFieldOptions, $iField);

                // Operator
                $out .= $renderSelect('wbsycf7-condition-equal', 'wbsycf7_condition_equal', $operatorOptions, $op);

                // Value control
                if ($op === 'empty' || $op === 'not_empty') {
                    // No editor needed: keep hidden
                    $out .= $renderInput('wbsycf7-condition-value', 'wbsycf7_condition_value', 'hidden', '');
                } else {
                    // Decide control type based on field type
                    if ($type === 'checkbox') {
                        // multiple select
                        $out .= $renderSelect(
                                'wbsycf7-condition-value',
                                'wbsycf7_condition_value',
                                $values,
                                is_array($val) ? $val : (is_string($val) ? preg_split('/\r?\n|,/', $val, -1, PREG_SPLIT_NO_EMPTY) : []),
                                ['multiple' => true]
                        );
                    } elseif ($type === 'radio' || $type === 'select') {
                        $out .= $renderSelect('wbsycf7-condition-value', 'wbsycf7_condition_value', $values, is_array($val)? reset($val) : (string)$val);
                    } else {
                        // map input type
                        $map = ['email'=>'email','tel'=>'tel','url'=>'url','number'=>'number','range'=>'range','date'=>'date'];
                        $htmlType = $map[$type] ?? 'text';
                        $out .= $renderInput('wbsycf7-condition-value', 'wbsycf7_condition_value', $htmlType, is_array($val)? implode(',', $val) : (string)$val);
                    }
                }

                // Item actions
                $out .= '<div class="wbsycf7-condition-actions">';
                $out .=   '<span class="dashicons dashicons-trash wbsycf7-condition-item-delete" title="Delete"></span>';
                $out .= '</div>';

                $out .= '</div>'; // .wbsycf7-conditionGroup-item-row
            }

            $out .= '</div>'; // .wbsycf7-conditionGroup-items
            $out .= '</div>'; // .wbsycf7-conditionGroup-row
        }

        $out .= '</div>'; // .wbsycf7-conditions-container

        return $out;
    }

    /**
     * Write the conditions JS file for a CF7 form and return its URL.
     *
     * @param int $form_id
     * @return string|false URL on success, false on failure
     */
    public function generate_conditions_js( int $form_id ) {
        $conditions = get_post_meta( $form_id, '_wbsycf7_conditions', true );
        if ( ! is_array( $conditions ) ) {
            $conditions = [];
        }

        // Build a list of driver field names we need to listen to
        $driver_names = [];
        foreach ( $conditions as $g ) {
            if ( empty( $g['items'] ) || !is_array($g['items']) ) continue;
            foreach ( $g['items'] as $it ) {
                if ( !empty($it['fieldName']) ) {
                    $driver_names[$it['fieldName']] = true;
                }
            }
        }
        $driver_names = array_keys($driver_names);

        // Data payload the JS will consume
        $payload = [
                'formId'       => $form_id,
                'groups'       => $conditions,
                'drivers'      => $driver_names,
        ];

        // Build the JS string (vanilla JS; no jQuery dependency)
        $js = "(function(){\n".
                "  const CFG = ". wp_json_encode($payload) .";\n".

                "  function findFormsById(id){\n".
                "    const inputs = document.querySelectorAll('form input[name=\"_wpcf7\"][value=\"'+id+'\"]');\n".
                "    const forms = [];\n".
                "    inputs.forEach(inp => { const f = inp.closest('form'); if (f) forms.push(f); });\n".
                "    return forms;\n".
                "  }\n".

                "  function findRowByFieldName(form, name){\n".
                "    const wrap = form.querySelector('.wpcf7-form-control-wrap[data-name=\"'+name+'\"]');\n".
                "    return wrap ? wrap.closest('.wbsycf7-form-row') : null;\n".
                "  }\n".

                "  function getFieldValue(form, name){\n".
                "    const els = form.querySelectorAll('[name=\"'+name+'\"], [name=\"'+name+'[]\"]');\n".
                "    if(!els.length){\n".
                "      const wrap = form.querySelector('.wpcf7-form-control-wrap[data-name=\"'+name+'\"]');\n".
                "      if (wrap){\n".
                "        const sel = wrap.querySelector('select');\n".
                "        if (sel){\n".
                "          if (sel.multiple){ return Array.from(sel.selectedOptions).map(o=>o.value.trim()); }\n".
                "          return sel.value || '';\n".
                "        }\n".
                "      }\n".
                "      return '';\n".
                "    }\n".
                "    const first = els[0];\n".
                "    const type = (first.getAttribute('type')||'').toLowerCase();\n".
                "    const tag  = first.tagName.toLowerCase();\n".
                "    if (tag === 'select'){\n".
                "      if (first.multiple){ return Array.from(first.selectedOptions).map(o=>o.value.trim()); }\n".
                "      return first.value || '';\n".
                "    }\n".
                "    if (type === 'checkbox'){\n".
                "      const checked = Array.from(els).filter(el=>el.checked);\n".
                "      return checked.map(el=> (el.value||'').trim());\n".
                "    }\n".
                "    if (type === 'radio'){\n".
                "      const picked = Array.from(els).find(el=>el.checked);\n".
                "      return picked ? (picked.value||'').trim() : '';\n".
                "    }\n".
                "    return (first.value||'').trim();\n".
                "  }\n".

                "  function isEmpty(v){\n".
                "    if (Array.isArray(v)) return v.length===0;\n".
                "    return String(v).trim()==='';\n".
                "  }\n".

                "  function matchItem(form, item){\n".
                "    const name = item.fieldName || '';\n".
                "    const op   = item.op || 'is';\n".
                "    const v    = getFieldValue(form, name);\n".
                "    if (op === 'empty') return isEmpty(v);\n".
                "    if (op === 'not_empty') return !isEmpty(v);\n".
                "    const want = item.value;\n".
                "    if (Array.isArray(v)){\n".
                "      if (Array.isArray(want)){\n".
                "        const set = new Set(v.map(String));\n".
                "        const any = want.some(x=> set.has(String(x)));\n".
                "        return (op==='is') ? any : !any;\n".
                "      } else {\n".
                "        const has = v.map(String).includes(String(want));\n".
                "        return (op==='is') ? has : !has;\n".
                "      }\n".
                "    } else {\n".
                "      if (Array.isArray(want)){\n".
                "        const any = want.map(String).includes(String(v));\n".
                "        return (op==='is') ? any : !any;\n".
                "      } else {\n".
                "        const eq = String(v)===String(want);\n".
                "        return (op==='is') ? eq : !eq;\n".
                "      }\n".
                "    }\n".
                "  }\n".

                "  function evalGroups(form){\n".
                "    const decisions = {};\n".
                "    (CFG.groups||[]).forEach(g=>{\n".
                "      const items = Array.isArray(g.items)? g.items : [];\n".
                "      let res;\n".
                "      if (!items.length){ res = true; } else {\n".
                "        const checks = items.map(it=>matchItem(form,it));\n".
                "        if ((g.logic||'all')==='any') res = checks.some(Boolean); else res = checks.every(Boolean);\n".
                "      }\n".
                "      const action = (g.action==='hide') ? 'hide' : 'show';\n".
                "      const target = g.fieldName || '';\n".
                "      if (!target) return;\n".
                "      decisions[target] = { action, res };\n".
                "    });\n".
                "    for (const name in decisions){\n".
                "      const d = decisions[name];\n".
                "      const row = findRowByFieldName(form, name);\n".
                "      if (!row) continue;\n".
                "      const shouldShow = (d.action==='show') ? !!d.res : !d.res;\n".
                "      if (shouldShow) {\n".
                "        row.classList.remove('wbsycf7-conditional-hidden');\n".
                "      } else {\n".
                "        row.classList.add('wbsycf7-conditional-hidden');\n".
                "      }\n".
                "    }\n".
                "  }\n".

                "  function bind(form){\n".
                "    const names = (CFG.drivers||[]);\n".
                "    const seen = new Set();\n".
                "    names.forEach(n=>{\n".
                "      if (!n || seen.has(n)) return; seen.add(n);\n".
                "      const sel = '[name=\"'+n+'\"], [name=\"'+n+'[]\"]';\n".
                "      form.querySelectorAll(sel).forEach(el=>{\n".
                "        ['change','input'].forEach(ev=> el.addEventListener(ev, ()=>evalGroups(form)) );\n".
                "      });\n".
                "      const wrap = form.querySelector('.wpcf7-form-control-wrap[data-name=\"'+n+'\"]');\n".
                "      if (wrap){ const selEl = wrap.querySelector('select'); if (selEl){ selEl.addEventListener('change', ()=>evalGroups(form)); } }\n".
                "    });\n".
                "    // re-evaluate on native reset, after the DOM clears values\n".
                "    form.addEventListener('reset', function(){ setTimeout(()=>evalGroups(form), 0); });\n".
                "    // initial evaluation\n".
                "    setTimeout(()=>evalGroups(form), 0);\n".
                "  }\n".

                "  function init(){\n".
                "    const forms = findFormsById(CFG.formId);\n".
                "    if (!forms.length) return;\n".
                "    forms.forEach(form => {\n".
                "      if (form.dataset.wbsycf7Init === '1') return; // guard against double-bind\n".
                "      form.dataset.wbsycf7Init = '1';\n".
                "      bind(form);\n".
                "    });\n".
                "  }\n".

                "  if (document.readyState==='loading'){\n".
                "    document.addEventListener('DOMContentLoaded', init);\n".
                "  } else { init(); }\n".

                "  // Re-evaluate after CF7 AJAX lifecycle events\n".
                "  document.addEventListener('wpcf7mailsent', function(e){\n".
                "    if (!e || !e.detail) return;\n".
                "    if (String(e.detail.contactFormId) !== String(CFG.formId)) return;\n".
                "    findFormsById(CFG.formId).forEach(f=> setTimeout(()=>evalGroups(f), 0));\n".
                "  });\n".
                "  document.addEventListener('wpcf7invalid', function(e){\n".
                "    if (!e || !e.detail) return;\n".
                "    if (String(e.detail.contactFormId) !== String(CFG.formId)) return;\n".
                "    setTimeout(()=>{ findFormsById(CFG.formId).forEach(f=> evalGroups(f)); }, 0);\n".
                "  });\n".
                "  document.addEventListener('wpcf7mailfailed', function(e){\n".
                "    if (!e || !e.detail) return;\n".
                "    if (String(e.detail.contactFormId) !== String(CFG.formId)) return;\n".
                "    setTimeout(()=>{ findFormsById(CFG.formId).forEach(f=> evalGroups(f)); }, 0);\n".
                "  });\n".
                "  document.addEventListener('wpcf7reset', function(e){\n".
                "    if (!e || !e.detail) return;\n".
                "    if (String(e.detail.contactFormId) !== String(CFG.formId)) return;\n".
                "    setTimeout(()=>{ findFormsById(CFG.formId).forEach(f=> evalGroups(f)); }, 0);\n".
                "  });\n".

                "})();\n";


        // Write to uploads/wbsycf7/
        $uploads = wp_upload_dir();
        if ( ! empty( $uploads['error'] ) ) return false;

        $dir = trailingslashit( $uploads['basedir'] ) . 'wbsycf7';
        if ( ! file_exists( $dir ) ) {
            wp_mkdir_p( $dir );
        }

        $file = trailingslashit( $dir ) . 'wbsycf7-conditions-' . $form_id . '.js';
        $ok = file_put_contents( $file, $js );
        if ( ! $ok ) return false;

        return trailingslashit( $uploads['baseurl'] ) . 'wbsycf7/wbsycf7-conditions-' . $form_id . '.js';
    }






}