<?php
namespace WBSY\CF7\Admin;

use WBSY\CF7\Utils\Parser; // Ensure Parser class is namespaced properly
use WBSY\CF7\Utils\Utils; // Ensure Parser class is namespaced properly

class BuilderTab {

    private static $saving_cf7 = false;

    public function register() {
        add_filter('wpcf7_editor_panels', [$this, 'addTab']); 
        add_action('wpcf7_after_save', [$this, 'onFormAfterSave'], 10, 1);

        // Optional safety net (fires after the post exists)
        add_action('save_post_wpcf7_contact_form', [$this, 'onCF7PostSave'], 20, 3);

        add_action('admin_footer', [$this, 'addPreviewButtonTemplate']);
    }

    
    public function addTab($panels) {    
        wp_enqueue_script('select2-js');
        wp_enqueue_style('select2-css');	
        $new_panel = [
            'wbsy_cf7_builder' => [
                'title'    => __( 'Visual Builder', 'websync-builder-cf7' ),
                'callback' => [ $this, 'renderTab' ],
            ],
            'blcf7_builder_settings' => [
                'title' => __('Visual Builder Settings', 'builder-for-contact-form-7'),
                'callback' => [$this, 'renderSettingsTab'],
            ],
            'blcf7_builder_conditional_fields' => [
                    'title' => __('Conditional Fields', 'builder-for-contact-form-7'),
                    'callback' => [$this, 'renderConditionalFields'],
            ]
        ];

        // Prepend new panel
        return array_merge( $new_panel, $panels );
    }

    public function renderSettingsTab($form) {

        $pages = get_pages([
            'post_status' => 'publish',
            'sort_column' => 'post_title',
            'sort_order'  => 'asc',
        ]);

        $settings = get_post_meta($form->id(), '_wblscf7_builder_settings', true);

        $success_action = $settings['success_action'] ?? 'stay';
        $redirect_page_id = $settings['redirect_page_id'] ?? '';
        $subscribers_only = !empty($settings['subscribers_only']) ? $settings['subscribers_only'] : 0;
        $demo_mode = !empty($settings['demo_mode']) ? $settings['demo_mode'] : 0;
        $skip_mail = !empty($settings['skip_mail']) ? $settings['skip_mail'] : 0;
        $acceptance_as_validation = !empty($settings['acceptance_as_validation']) ? $settings['acceptance_as_validation'] : 0;
        ob_start();
        ?>
        <div class="wblscf7-settings-group">
            <p class="wblscf7-group-title"><?php echo esc_html__('Action after submission', 'websync-builder-cf7'); ?></p>
            <select id="wblscf7-success-action" name="wblscf7_success_action">
                <option value="default" <?php selected($success_action, 'default'); ?>><?php echo esc_html__('Stay on form', 'websync-builder-cf7'); ?></option>
                <option value="redirect" <?php selected($success_action, 'redirect'); ?>><?php echo esc_html__('Redirect to page', 'websync-builder-cf7'); ?></option>
            </select>
            <!-- Redirect to page -->
            <div id="wblscf7-redirect-field" style="display:none;">
                <select id="wblscf7-redirect-page-id" name="wblscf7_redirect_page_id">
                    <option value=""><?php echo esc_html__('Select a page', 'websync-builder-cf7'); ?></option>
                    <?php foreach ($pages as $page): ?>
                        <option value="<?php echo esc_attr($page->ID); ?>" <?php selected($redirect_page_id, $page->ID); ?>>
                            <?php echo esc_html($page->post_title); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="wblscf7-field-option-row">
                <p class="wblscf7-group-title"><?php echo esc_html__('Subscribers-only mode', 'websync-builder-cf7'); ?></p>
                <div class="wblscf7-req-switch-button-cover wblscf7-switch-button-cover">
                    <div class="button b2 wblscf7-switch-button">
                        <input type="checkbox" name="subscribers_only" class="checkbox wblscf7-field-option" data-option="subscribers_only: true" value="1" <?php echo $subscribers_only ? "checked" : ""; ?> />
                        <div class="wblscf7-req-knobs wblscf7-knobs">
                            <span></span>
                        </div>
                        <div class="wblscf7-req-layer wblscf7-switch-layer"></div>
                    </div>
                </div>
                <p class="wblscf7-field-option-description">
                    <?php echo esc_html__('You may want to ensure that only logged-in users can submit your contact form. In such cases, use the subscribers-only mode. In this mode, non-logged-in users can\'t submit the contact form and will see a message informing them that login is required, while logged-in users can use it as usual.', 'websync-builder-cf7'); ?>
                    <?php echo esc_html__('No anti-spam verification will be provided for contact forms in the subscribers-only mode since only welcome people are supposed to be able to use them. If this assumption is not applicable to your site, subscribers-only mode probably isn\'t a good option for you.', 'websync-builder-cf7'); ?>
                </p>
            </div>

            <div class="wblscf7-field-option-row">
                <p class="wblscf7-group-title"><?php echo esc_html__('Demo mode', 'websync-builder-cf7'); ?></p>
                <div class="wblscf7-req-switch-button-cover wblscf7-switch-button-cover">
                    <div class="button b2 wblscf7-switch-button">
                        <input type="checkbox" name="demo_mode" class="checkbox wblscf7-field-option" data-option="demo_mode: on" value="1"  <?php echo $demo_mode ? "checked" : ""; ?> />
                        <div class="wblscf7-req-knobs wblscf7-knobs">
                            <span></span>
                        </div>
                        <div class="wblscf7-req-layer wblscf7-switch-layer"></div>
                    </div>
                </div>
                <p class="wblscf7-field-option-description">
                    <?php echo esc_html__('The contact form will be in the demo mode. In this mode, the contact form will skip the process of sending mail and just display "completed successfully" as a response message.', 'websync-builder-cf7'); ?>
                </p>
            </div>

            <div class="wblscf7-field-option-row">
                <p class="wblscf7-group-title"><?php echo esc_html__('Skipping email', 'websync-builder-cf7'); ?></p>
                <div class="wblscf7-req-switch-button-cover wblscf7-switch-button-cover">
                    <div class="button b2 wblscf7-switch-button">
                        <input type="checkbox" name="skip_mail" class="checkbox wblscf7-field-option" data-option="skip_mail: on" value="1" <?php echo $skip_mail ? "checked" : ""; ?> />
                        <div class="wblscf7-req-knobs wblscf7-knobs">
                            <span></span>
                        </div>
                        <div class="wblscf7-req-layer wblscf7-switch-layer"></div>
                    </div>
                </div>
                <p class="wblscf7-field-option-description">
                    <?php echo esc_html__('The skip_mail setting works in the almost same manner as the demo_mode, but the skip_mail skips the mail sending only. Unlike demo_mode, skip_mail doesn\'t affect other activities like storing messages with Flamingo.', 'websync-builder-cf7'); ?>
                </p>
            </div>

            <div class="wblscf7-field-option-row">
                <p class="wblscf7-group-title"><?php echo esc_html__('Acceptance as validation', 'websync-builder-cf7'); ?></p>
                <div class="wblscf7-req-switch-button-cover wblscf7-switch-button-cover">
                    <div class="button b2 wblscf7-switch-button">
                        <input type="checkbox" name="acceptance_as_validation" class="checkbox wblscf7-field-option" data-option="acceptance_as_validation: on" value="1" <?php echo $acceptance_as_validation ? "checked" : ""; ?> />
                        <div class="wblscf7-req-knobs wblscf7-knobs">
                            <span></span>
                        </div>
                        <div class="wblscf7-req-layer wblscf7-switch-layer"></div>
                    </div>
                </div>
                <p class="wblscf7-field-option-description">
                    <?php echo esc_html__('By default, an acceptance checkbox behaves differently from other types of fields; it does not display a validation error message even when the box is not checked. If you set acceptance_as_validation: on in the Additional Settings field, acceptance checkboxes in the contact form behave in the same way as other form fields.', 'websync-builder-cf7'); ?>
                </p>
            </div>
        </div>
        <?php
        echo wp_kses( ob_get_clean(), Utils::allowed_html_tags() );
    }

    public function renderTab($post) {
        $post_id = $post->id();
        $builder_html = get_post_meta($post_id, '_wbsy_cf7_builder', true);

        if ( ! $builder_html ) {
            $form         = $post->prop( 'form' );
            $builder_html = ( new Parser() )->convert( $form );
        }

        $this->field_buttons();
        if (strpos($builder_html, 'wbsycf7-builder-preview') === false) {
            echo '<div class="wbsycf7-builder-preview">';
            echo wp_kses($builder_html, Utils::allowed_html_tags());
            echo '</div>';
        } else {
            echo wp_kses($builder_html, Utils::allowed_html_tags());
        }
        echo '<textarea name="wbsycf7_builder_html" id="wbsycf7_builder_html">' . esc_textarea($builder_html) . '</textarea>';
    }

    public function field_buttons() {
        ob_start(); ?>
        <div class="wbsycf7-field-toolbar">
            <span id="tag-generator-list" class="hide-if-no-js">
                <?php
                $fields = [
                    'text' => 'text',
                    'email' => 'email',
                    'url' => 'url',
                    'tel' => 'tel',
                    'number' => 'number',
                    'date' => 'date',
                    'textarea' => 'textarea',
                    'menu' => 'drop-down menu',
                    'checkbox' => 'checkboxes',
                    'radio' => 'radio buttons',
                    'acceptance' => 'acceptance',
                    'quiz' => 'quiz',
                    'file' => 'file',
                    'submit' => 'submit',
                ];
                foreach ( $fields as $key => $field ) {
                    printf(
                        '<button type="button" data-taggen="open-dialog" data-target="tag-generator-panel-%1$s" title="Form-tag Generator: %1$s">%2$s</button>',
                        esc_attr( $key ),
                        esc_html( $field ),
                    );
                }
                ?>
            </span>
        </div>
        <?php
        $allowed_html = [
            'div'    => [ 'class' => true ],
            'span'   => [ 'id' => true, 'class' => true ],
            'button' => [
                'type'         => true,
                'data-taggen'  => true,
                'data-target'  => true,
                'title'        => true,
            ],
            'hr' => [],
        ];
        echo wp_kses( ob_get_clean(), $allowed_html );
    }

    public function onFormAfterSave( \WPCF7_ContactForm $contact_form ) {
        $post_id = (int) $contact_form->id();
        if ( $post_id > 0 ) {
            $this->persistFormSave( $post_id, $contact_form );
        }
    }

    public function onCF7PostSave( $post_id, $post, $update ) {
        if ( defined('DOING_AUTOSAVE') && DOING_AUTOSAVE ) return;
        if ( (int) $post_id <= 0 || $post->post_type !== 'wpcf7_contact_form' ) return;

        // If you need the CF7 object:
        $cf = \WPCF7_ContactForm::get_instance( $post_id );
        $this->persistFormSave( $post_id, $cf );
    }

    private function persistFormSave( int $post_id, ?\WPCF7_ContactForm $contact_form = null ) : void {

        if ( ! current_user_can( 'wpcf7_edit_contact_form', $post_id ) ) {
            return;
        }

        // If nonce exists (classic CF7 editor), verify it.
        if ( isset($_POST['_wpnonce']) ) {
            $nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );

            if ( isset($_POST['_wpnonce']) ) {

                // Use the CF7 classic action
                if ( ! wp_verify_nonce( $nonce, 'wpcf7-save-contact-form_' . $post_id ) ) {
                    return; // Stop ONLY if classic editor is used and nonce is invalid
                }
            }
        }
        // KSES clean
        $html = ! empty($_POST['wbsycf7_builder_html']) ? wp_kses( wp_unslash($_POST['wbsycf7_builder_html']), Utils::allowed_html_tags() ) : '';

        // Data will be sanitize in next step
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $conds_raw = isset($_POST['wbsycf7_conditions_input']) ? wp_unslash($_POST['wbsycf7_conditions_input']) : null;
        if ($conds_raw !== null && $conds_raw !== '') {
            $conds = json_decode($conds_raw, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($conds)) {
                $sanitized_conds = $this->recursively_sanitize_array($conds);
                update_post_meta($post_id, '_wbsycf7_conditions', $sanitized_conds);
            }
        } else {
            delete_post_meta($post_id, '_wbsycf7_conditions');
        }

        // Save builder HTML
        update_post_meta( $post_id, '_wbsy_cf7_builder', $html );

        /* Pro Start */
        // Settings
        $settings = [];
        if (isset($_POST['wblscf7_success_action'])) {
            $settings['success_action'] = sanitize_text_field(wp_unslash($_POST['wblscf7_success_action']));
        }
        if (!empty($settings['success_action']) && $settings['success_action'] === 'redirect') {
            $settings['redirect_page_id'] = absint($_POST['wblscf7_redirect_page_id'] ?? 0);
        } elseif (!empty($settings['success_action']) && $settings['success_action'] === 'custom_text') {
            $settings['custom_success_message'] = isset($_POST['wblscf7_custom_success_message']) ? wp_kses_post(wp_unslash($_POST['wblscf7_custom_success_message'])) : '';
        }
        $settings['subscribers_only']         = !empty($_POST['subscribers_only']) ? (int) $_POST['subscribers_only'] : 0;
        $settings['demo_mode']                = !empty($_POST['demo_mode']) ? (int) $_POST['demo_mode'] : 0;
        $settings['skip_mail']                = !empty($_POST['skip_mail']) ? (int) $_POST['skip_mail'] : 0;
        $settings['acceptance_as_validation'] = !empty($_POST['acceptance_as_validation']) ? (int) $_POST['acceptance_as_validation'] : 0;

        update_post_meta($post_id, '_wblscf7_builder_settings', $settings);
        /* Pro End */

        // Clean and reverse-convert to CF7 shortcode + field meta
        $clean = preg_replace('/<div class="wbsycf7-actions">.*?<\/div>/s', '', $html);
        $clean = preg_replace('/<div[^>]*\bwbsycf7-add-new-page\b[^>]*>.*?<\/div>/s', '', $clean);
        // Remove ui-sortable-handle and ui-sortable from class attributes
        $clean = preg_replace('/\s*(ui-sortable-handle|ui-sortable)(?=\s|")/', '', $clean);

        $reverseConvert = (new \WBSY\CF7\Utils\Parser())->reverseConvert($clean);

        if (is_array($reverseConvert) && isset($reverseConvert['data'], $reverseConvert['html'])) {
            update_post_meta($post_id, '_wblscf7_form_fields', $reverseConvert['data']);
            $shortcode = $reverseConvert['html'];

            if ( $contact_form instanceof \WPCF7_ContactForm ) {
                $contact_form->set_properties( [ 'form' => $shortcode ] );

                if ( ! self::$saving_cf7 ) {
                    self::$saving_cf7 = true;
                    $contact_form->save(); // persists properties to the post
                    self::$saving_cf7 = false;
                }
            } else {
                // fallback if we don't have the object
                wp_update_post( [
                        'ID'           => $post_id,
                        'post_content' => $shortcode,
                ] );
            }

        }

        if( $post_id ) {
            (new ConditionalFields())->generate_conditions_js($post_id);
        }
    }

    // Helper method to recursively sanitize array
    private function recursively_sanitize_array($data) {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[sanitize_key($key)] = $this->recursively_sanitize_array($value);
            }
            return $data;
        } elseif (is_string($data)) {
            return sanitize_text_field($data);
        } elseif (is_int($data) || is_float($data)) {
            return $data;
        } elseif (is_bool($data)) {
            return (bool) $data;
        }
        return null;
    }

     public function addPreviewButtonTemplate() {
        global $pagenow;
        // Confirm we're on the CF7 form edit screen via admin.php
        if ($pagenow === 'admin.php' && isset($_GET['page'], $_GET['post']) && $_GET['page'] === 'wpcf7') {
            $post_id = absint($_GET['post']);
            $post = get_post($post_id);

            if (!$post || $post->post_type !== 'wpcf7_contact_form') {
                return;
            }

            // Get the preview URL
            $preview_post = get_page_by_path('preview', OBJECT, 'wblscf7_preview');
            $preview_url = $preview_post
                ? add_query_arg('form_id', $post_id, get_permalink($preview_post))
                : home_url('/wblscf7-preview/?form_id=' . $post_id);

            ?>
            <script type="text/template" id="wblscf7-preview-button-template">
                <a href="<?php echo esc_url($preview_url); ?>" 
                   class="button button-secondary"
                   id="wblscf7-preview-button" 
                   target="_blank">
                    Preview Form
                </a>
            </script>
            <?php
        }
        require_once "Templates.php";
    }

    public function renderConditionalFields($post) {
        (new ConditionalFields())->register($post);
    }
}
