
class WBSYCF7FieldEditor {
    constructor() {
        this.bindEditButtons();
        this.bindInsertTagOverride();
        this.bindUpdateField();
        this.bindPopupClose();
        this.bindInlineLabelEditing();
    }

    // --- Validation & sanitization helpers ---
    sanitizeLabelInput(raw) {
        let s = (raw == null) ? '' : String(raw);

        // collapse multiple spaces and trim ends
        s = s.replace(/\s+/g, ' ').trim();

        // remove control chars
        s = s.replace(/[\u0000-\u0008\u000B\u000C\u000E-\u001F\u007F]/g, '');

        // strip any HTML tags
        s = s.replace(/<[^>]*>/g, '');

        // length cap
        const MAX = 120;
        if (s.length > MAX) s = s.slice(0, MAX);

        return s;
    }

    isValidLabelInput(s) {
        if (!s || !s.trim()) return false;

        // basic blacklist
        const blacklist = [
            /<\s*script/i, /javascript:/i, /on\w+\s*=/i, /<\s*img/i, /<\s*iframe/i
        ];
        if (blacklist.some(rx => rx.test(s))) return false;

        return true;
    }

// --- Inline label editing ---
    bindInlineLabelEditing() {
        const self = this;

        jQuery(document).on('click', '.wbsycf7-label-wrapper .wbsycf7-field-label', function () {
            const $label = jQuery(this);
            if ($label.data('editing')) return;

            self.ensureLabelTextSpan($label);

            const $textSpan = $label.find('.wbsycf7-label-text');
            const current = ($textSpan.text() || '');
            const $input = jQuery('<input type="text" class="wbsycf7-label-editor" />').val(current);

            $label.data('editing', true);
            $textSpan.hide().after($input);
            $input.trigger('focus')[0].setSelectionRange(current.length, current.length);
        });

        jQuery(document).on('keydown', '.wbsycf7-label-editor', function (e) {
            if (e.key === 'Enter') { e.preventDefault(); jQuery(this).blur(); }
            else if (e.key === 'Escape') { e.preventDefault(); jQuery(this).data('cancel', true).blur(); }
        });

        // Only sanitize & validate at the end
        jQuery(document).on('blur', '.wbsycf7-label-editor', function () {
            const $input  = jQuery(this);
            const $label  = $input.closest('.wbsycf7-field-label');
            const $text   = $label.find('.wbsycf7-label-text');
            const cancel  = !!$input.data('cancel');

            if (!cancel) {
                const sanitized = self.sanitizeLabelInput($input.val());
                if (!self.isValidLabelInput(sanitized)) {
                    $input.focus();
                    return;
                }
                $text.text(sanitized); // safe: .text() escapes
            }

            $input.remove();
            $text.show();
            $label.removeData('editing');
        });
    }

    ensureLabelTextSpan($label) {
        if ($label.find('.wbsycf7-label-text').length) return;
        const plainText = $label.clone().children().remove().end().text();
        $label.contents().filter(function(){ return this.nodeType === 3; }).remove();
        const safeText = plainText
            .replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
        $label.prepend(`<span class="wbsycf7-label-text">${safeText}</span> `);
    }

    bindUpdateField() {
        let self = this;
        jQuery(document).on('click', '.wbsycf7-update-button', function(e){
            e.preventDefault();

            const shortcode = jQuery(this).closest('.insert-box').find('input.code').val();
            const html = self.wbsycf7ConvertShortcodeToHTML(shortcode);

            jQuery(".wbsycf7-form-row.wbsycf7-edit-active").replaceWith(html);
            const $panel = jQuery(this).closest('.tag-generator-panel');
            $panel.closest(".tag-generator-dialog").find('.close-button').trigger('click');
        });
    }

    bindEditButtons() {
        jQuery(document).on('click', '.wbsycf7-column .wbsycf7-actions .dashicons-edit', (e) => {
            e.preventDefault();

            //this.emptyPopup();

            const $field = jQuery(e.currentTarget).closest('.wbsycf7-form-row');
            const tagType = this.getTagTypeFromField($field);
            const tagPanelId = '#tag-generator-panel-' + tagType;

            $field.addClass("wbsycf7-edit-active");

            jQuery('#tag-generator-list button[data-target="' + tagPanelId.replace('#', '') + '"]').trigger('click');

            setTimeout(() => {
                this.fillPopupFromField($field, tagPanelId);
                this.forceCF7TagUpdate(jQuery(tagPanelId));
                const $insertBtn = jQuery(tagPanelId).find(".wbsycf7-insert-button");
                $insertBtn.text("Update Field").addClass('wbsycf7-update-button').removeClass('wbsycf7-insert-button');
            }, 200);
        });
    }

    bindInsertTagOverride() {
        let self = this;
        jQuery(document).on('click', '.wbsycf7-insert-button', function (e) {
            e.preventDefault();

            const $panel = jQuery(this).closest('.tag-generator-panel');
            const shortcode = $panel.find('input.code').val().trim();

            if (shortcode) {
                const html = self.wbsycf7ConvertShortcodeToHTML(shortcode);
                if (html) {
                    const $container = jQuery('.wbsycf7-column-active');

                    // Try to find the submit row
                    const $submitRow = $container.find('.wbsycf7-form-row input[type="submit"]').closest('.wbsycf7-form-row');

                    if ($submitRow.length) {
                        // Insert before the submit button row
                        $submitRow.before(html);
                    } else {
                        // No submit row found — append at the end
                        $container.append(html);
                    }
                }
            }

            // Close the modal
            $panel.closest(".tag-generator-dialog").find('.close-button').trigger('click');
        });
    }

    bindPopupClose() {
        jQuery(document).on('click', '.tag-generator-dialog .close-button', function () {
            jQuery(".wbsycf7-edit-active").removeClass("wbsycf7-edit-active");
        });
    }

    wbsycf7ConvertShortcodeToHTML(tag) {
        const blockPattern = /^\[(\w+)(\*?)\s+([^\]]+)\]\s*([^\[]+?)\s*\[\/\1\]$/s; // with content
        const inlinePattern = /^\[(\w+)(\*?)\s+([^\]]+)\]$/; // single line

        let match = tag.trim().match(blockPattern);
        let hasInnerContent = true;

        if (!match) {
            match = tag.trim().match(inlinePattern);
            hasInnerContent = false;
        }

        if (!match) return '';

        const type = match[1].toLowerCase();
        const required = match[2] === '*';
        const raw = match[3];
        const inlineLabel = hasInnerContent ? match[4].trim() : null;

        const parts = raw.match(/"[^"]+"|\S+/g) || [];
        const name = parts.shift();
        let attrs = {};
        let values = [];
        let placeholder = false;

        for (const p of parts) {
            if (p.startsWith('"') && p.endsWith('"')) {
                values.push(p.slice(1, -1)); // remove quotes
            } else if (p.includes(':')) {
                const [key, val] = p.split(':');
                if (!attrs[key]) attrs[key] = val;
            } else if (p === 'placeholder') {
                placeholder = true;
            } else if (p === 'use_label_element') {
                attrs['use_label_element'] = true;
            }
        }

        const attrStr = Object.entries(attrs)
            .filter(([k]) => k !== 'use_label_element')
            .map(([k, v]) => `${k}="${v}"`)
            .join(' ');

        const req = required ? 'required' : '';
        let field = '';
        let labelText = '';

        if (type === 'acceptance') {
            const accLabel = inlineLabel || values[0] || 'I accept';
            field = `<label><input type="checkbox" name="${name}" disabled ${attrStr} ${req}> ${accLabel}</label>`;
            labelText = ''; // don't render separate label
        } else if (type === 'quiz') {
            const quizText = values[0] || '';
            const [question, answer] = quizText.split('|').map(s => s.trim());
            field = `<input type="text" name="${name}" value="${answer}" disabled ${attrStr} ${req} class="wpcf7-quiz">`;
            labelText = question;
        } else {
            // default label logic
            if (jQuery(".wbsycf7-edit-active").length) {
                labelText = jQuery('.wbsycf7-edit-active .wbsycf7-field-label').clone().children().remove().end().text().trim();
            } else {
                labelText = name.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
            }

            switch (type) {
                case 'textarea':
                    const defaultText = values[0] || '';
                    field = `<textarea name="${name}" disabled ${attrStr} ${req}>${defaultText}</textarea>`;
                    break;

                case 'select':
                    const optionTags = values.map(opt => `<option>${opt}</option>`).join('');
                    field = `<select name="${name}" disabled ${attrStr} ${req}>${optionTags}</select>`;
                    break;

                case 'radio':
                case 'checkbox':
                    field = values.map(opt => {
                        const input = `<input type="${type}" name="${name}${type === 'checkbox' ? '[]' : ''}" value="${opt}" disabled ${attrStr} ${req}>`;
                        if (attrs.use_label_element) {
                            return `<label class="wbsycf7-${type}-label">${input} ${opt}</label>`;
                        } else {
                            return input + ' ' + opt;
                        }
                    }).join('');
                    break;

                case 'submit':
                    const btnLabel = values[0] || 'Submit';
                    field = `<button type="submit" class="wbsycf7-button" disabled>${btnLabel}</button>`;
                    labelText = '';
                    break;

                default:
                    const defaultVal = values[0] || '';
                    const placeholderAttr = placeholder ? `placeholder="${defaultVal}"` : '';
                    field = `<input type="${type}" name="${name}" value="${defaultVal}" disabled ${placeholderAttr} ${attrStr} ${req}>`;
            }
        }

        const requiredMark = required ? ' <span class="wbsycf7-required">*</span>' : '';
        const labelHTML = labelText ? `<label class="wbsycf7-field-label">${labelText}${requiredMark}</label>` : '';

        return `<div class="wbsycf7-form-row wbsycf7-label-top wbsycf7-field-${type} ui-sortable-handle">
                    <div class="wbsycf7-label-wrapper">${labelHTML}</div>
                    <div class="wbsycf7-field-wrapper">${field}</div>
                    <div class="wbsycf7-actions">
                        <span class="dashicons dashicons-edit wbsycf7-field-edit" title="Edit Field"></span>
                        <span class="dashicons dashicons-trash wbsycf7-field-trash" title="Delete Field"></span>
                    </div>
                </div>`;
    }

    escapeRegExp(s) {
        return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    stripArraySuffix(name) {
        return (name || '').replace(/\[\]$/, '');
    }

    fillPopupFromField($field, panelId) {
        const self = this;
        const $panel = jQuery(panelId);

        // Prefer the *first* control just to detect base props (type/name/etc.)
        const $el = $field.find('input, textarea, select').first();
        if (!$el.length) return;

        const tag = ($el.prop('tagName') || '').toLowerCase(); // input | textarea | select
        const type = ($el.attr('type') || tag).toLowerCase();  // input type or element tag
        const rawName = $el.attr('name') || '';
        const baseName = self.stripArraySuffix(rawName);
        const className = $el.attr('class') || '';
        const required = $el.prop('required');
        const placeholder = $el.attr('placeholder') || '';
        const autocomplete = $el.attr('autocomplete') || '';

        // Common (unchanged)
        $panel.find('[data-tag-part="name"]').val(baseName).trigger('change'); // use stripped name
        $panel.find('[data-tag-part="type-suffix"]').prop('checked', required).trigger('change');
        $panel.find('[data-tag-option^="class:"]').val(className).trigger('change');
        $panel.find('[data-tag-option="placeholder"]').prop('checked', !!placeholder).trigger('change');

        $panel.find('[data-tag-option^="autocomplete:"]').prop('checked', false);
        if (autocomplete) {
            $panel.find(`[data-tag-option="autocomplete:${autocomplete}"]`).prop('checked', true).trigger('change');
        }

        const minlength = $el.attr('minlength') || '';
        const maxlength = $el.attr('maxlength') || '';
        $panel.find('[data-tag-option="minlength:"]').val(minlength).trigger('change');
        $panel.find('[data-tag-option="maxlength:"]').val(maxlength).trigger('change');

        // Helper: set popup "value" part to a list of quoted choices
        const setChoices = (arr) => {
            const lines = (arr || [])
                .filter(v => v != null && String(v).trim() !== '')
                .map(v => String(v).trim());
            $panel.find('[data-tag-part="value"]').val(lines.join("\n")).trigger('change');
        };

        // Helper: get label text if wrapped, otherwise value attr
        const getChoiceLabel = ($input) => {
            const raw = $input.attr('value') || '';
            const $p = $input.parent();
            if ($p.length && $p.is('label')) {
                let t = ($p.text() || '').trim();
                if (raw) {
                    t = t.replace(new RegExp('^\\s*' + self.escapeRegExp(raw) + '\\s*'), '');
                }
                t = t.trim();
                return t || raw;
            }
            return raw;
        };

        // Field-specific handling
        if ($field.hasClass('wbsycf7-field-quiz')) {
            const rawLabel = $field.find('.wbsycf7-field-label').clone().children().remove().end().text().trim();
            const answer = $el.val(); // e.g., "Rio"
            const quizValue = `${rawLabel} | ${answer}`;
            $panel.find('[data-tag-part="value"]').val(quizValue).trigger('change');

        } else if (tag === 'select' || $field.find('select').length) {
            // Collect ALL option labels (not only selected)
            const $select = $field.find('select').first();
            const choices = [];
            $select.find('option').each(function () {
                const txt = (jQuery(this).text() || '').trim();
                if (txt && !choices.includes(txt)) choices.push(txt);
            });
            setChoices(choices);

        } else if (type === 'radio' || type === 'checkbox' || $field.find('input[type=radio], input[type=checkbox]').length) {
            // Collect ALL inputs in the group by base name (strip [] for matching)
            const inputs = [];
            $field.find('input[type=radio], input[type=checkbox]').each(function () {
                const nm = self.stripArraySuffix(this.name || '');
                if (nm === baseName) inputs.push(this);
            });

            const choices = [];
            let anyWrapped = false;

            inputs.forEach(input => {
                const $input = jQuery(input);
                const label = getChoiceLabel($input);
                if (label && !choices.includes(label)) choices.push(label);
                if ($input.parent().is('label')) anyWrapped = true;
            });

            setChoices(choices);

            // Optionally sync use_label_element if you have such checkbox in popup:
            // $panel.find('[data-tag-option="use_label_element"]').prop('checked', anyWrapped).trigger('change');

        } else if ($field.hasClass('wbsycf7-field-textarea') || $field.find('textarea').length) {
            const value = $field.find('textarea').val() || '';
            $panel.find('[data-tag-part="value"]').val(value).trigger('change');

        } else {
            const value = $el.val() || '';
            $panel.find('[data-tag-part="value"]').val(value).trigger('change');
        }

        // Keep your input event dispatch
        setTimeout(() => {
            $panel.find('input, select, textarea').each(function () {
                this.dispatchEvent(new Event('input', { bubbles: true }));
            });
        }, 100);
    }


    forceCF7TagUpdate($panel) {
        let $nameField = $panel.find('[data-tag-part="name"]');
        if ( !$nameField.length ) {
            $nameField = $panel.find('[data-tag-part="value"]');
        }

        if ( $nameField.length ) {
            let val = $nameField.val();
            $nameField.val(val + ' ');
            $nameField.val(val);
            const el = $nameField[0];
            el.dispatchEvent(new Event('input', {bubbles: true}));
            el.dispatchEvent(new Event('change', {bubbles: true}));
        }
    }

    getTagTypeFromField($field) {
        const $el = $field.find('input, textarea, select, button').first();
        const tagName = $el.prop('tagName').toLowerCase();
        const typeAttr = $el.attr('type');
        const classes = $el.attr('class') || '';

        if ($field.find('div.wpcf7-quiz').length || classes.includes('wpcf7-quiz')) return 'quiz';
        if ($field.find('div.wpcf7-recaptcha').length || classes.includes('wpcf7-recaptcha')) return 'recaptcha';
        if (tagName === 'textarea') return 'textarea';
        if (tagName === 'select') return 'menu';
        if (tagName === 'button' || typeAttr === 'submit') return 'submit';

        if (tagName === 'input') {
            switch (typeAttr) {
                case 'text': return 'text';
                case 'email': return 'email';
                case 'url': return 'url';
                case 'tel': return 'tel';
                case 'number': return 'number';
                case 'date': return 'date';
                case 'checkbox': return classes.includes('wpcf7-acceptance') ? 'acceptance' : 'checkbox';
                case 'radio': return 'radio';
                case 'file': return 'file';
                case 'hidden': return 'hidden';
                default: return 'text';
            }
        }

        return null;
    }
}

jQuery(() => new WBSYCF7FieldEditor());
