/**
 * Conditional logic UI manager for CF7 Builder (admin).
 * - Manages condition groups/items in-memory (this.conditions).
 * - Reflects DOM -> model and model -> hidden input for PHP save.
 * - Restores from server-provided JSON (WBSY_CF7_CONDITIONS.form_conditions).
 * - Builds correct value editors based on fieldsMeta (type, values/options).
 */
class WBSYCF7_CONDITIONS {
    /**
     * Bootstraps state, loads server data, binds events, and wires hidden input.
     */
    constructor() {  // Constructor
        this.conditions = {};
        this.groupIdSeq = 0;
        this.itemIdSeq = 0;

        const cfg = window.WBSY_CF7_CONDITIONS || {};
        this.fieldsMeta = cfg.form_fields || {};
        // preload from PHP
        if (Array.isArray(cfg.form_conditions)) this._loadFromServer(cfg.form_conditions);

        this.init();

        this._ensureHiddenInput();
        this._writeToHidden(); // initial write

        // ensure we write before CF7's form submits (Classic style screen)
        jQuery(document).on('submit', '#post', () => this._writeToHidden());
    }

    /**
     * Bootstraps state, loads server data, binds events, and wires hidden input.
     */
    init() {
        this.registerEvents();
    }

    /**
     * Registers all delegated DOM events for add/remove/sync actions.
     * Keeps a single set of handlers (via .off().on()) to avoid duplicates.
     */
    registerEvents() {
        const $doc = jQuery(document);

        // Add group
        $doc.off("click", ".wbsycf7-condition-add-group")
            .on("click", ".wbsycf7-condition-add-group", (e) => this.addConditionGroup(e.currentTarget));

        // Delete group
        $doc.off("click", ".wbsycf7-condition-group-delete")
            .on("click", ".wbsycf7-condition-group-delete", (e) => this.deleteConditionGroup(e.currentTarget));

        // Add item
        $doc.off("click", ".wbsycf7-condition-item-add")
            .on("click", ".wbsycf7-condition-item-add", (e) => this.addConditionItem(e.currentTarget));

        // Delete item
        $doc.off("click", ".wbsycf7-condition-item-delete")
            .on("click", ".wbsycf7-condition-item-delete", (e) => this.deleteConditionItem(e.currentTarget));

        // Sync group fields
        $doc.off("change", ".wbsycf7-conditionGroup-row .wbsycf7-form-fields")
            .on("change", ".wbsycf7-conditionGroup-row .wbsycf7-form-fields", (e) => {
                const $group = jQuery(e.currentTarget).closest(".wbsycf7-conditionGroup-row");
                this.syncGroupFromDOM($group);

                // If items exist, re-prune their selects based on new group field
                if ($group.find(".wbsycf7-conditionGroup-item-row").length) {
                    this.pruneItemOptionsForGroup($group);
                }
            });

        $doc.off("change", ".wbsycf7-conditionGroup-row .wbsycf7-condition-show-hide, .wbsycf7-conditionGroup-row .wbsycf7-condition-and-or")
            .on("change", ".wbsycf7-conditionGroup-row .wbsycf7-condition-show-hide, .wbsycf7-conditionGroup-row .wbsycf7-condition-and-or", (e) => {
                const $group = jQuery(e.currentTarget).closest(".wbsycf7-conditionGroup-row");
                this.syncGroupFromDOM($group);
            });


        $doc.off("change", ".wbsycf7-conditionGroup-item-row .wbsycf7-form-items-fields, .wbsycf7-conditionGroup-item-row .wbsycf7-condition-equal")
            .on("change", ".wbsycf7-conditionGroup-item-row .wbsycf7-form-items-fields, .wbsycf7-conditionGroup-item-row .wbsycf7-condition-equal", (e) => {
                const $item = jQuery(e.currentTarget).closest(".wbsycf7-conditionGroup-item-row");
                this.renderValueControl($item);   // build/select input based on field type
                this.syncItemFromDOM($item);      // keep model in sync
            });
    }

    /**
     * Ensures hidden input exists on the page for posting JSON to PHP.
     * Creates it under #post when available, else under the first <form>.
     * @private
     */
    _ensureHiddenInput() {
        if (!jQuery('#wbsycf7_conditions_input').length) {
            const $form = jQuery('#post').length ? jQuery('#post') : jQuery('form').first();
            jQuery('<input>', {
                type: 'hidden',
                id: 'wbsycf7_conditions_input',
                name: 'wbsycf7_conditions_input'
            }).appendTo($form);
        }
    }

    /**
     * Serializes the current model and writes JSON into the hidden input.
     * @private
     */
    _writeToHidden() {
        jQuery('#wbsycf7_conditions_input').val(JSON.stringify(this.serialize()));
    }

    /**
     * Debounced notifier that model changed → write to hidden input soon.
     * @private
     */
    _signalModelChanged() {
        clearTimeout(this._wmc);
        this._wmc = setTimeout(() => this._writeToHidden(), 100);
    }

    /**
     * Loads model from server-provided array of groups/items.
     * Resets id sequences to match highest seen values.
     * @param {Array<Object>} arr
     * @private
     */
    _loadFromServer(arr) {
        this.conditions = {};
        this.groupIdSeq = 0;
        this.itemIdSeq  = 0;
        for (const g of arr) {
            const gid = Number(g.id ?? ++this.groupIdSeq);
            this.groupIdSeq = Math.max(this.groupIdSeq, gid);
            this.conditions[gid] = {
                id: gid,
                action: g.action || 'show',
                fieldName: g.fieldName || '',
                logic: g.logic || 'all',
                items: Array.isArray(g.items) ? g.items.map(it => {
                    const iid = Number(it.id ?? ++this.itemIdSeq);
                    this.itemIdSeq = Math.max(this.itemIdSeq, iid);
                    return { id: iid, fieldName: it.fieldName || '', op: it.op || 'is', value: it.value ?? '' };
                }) : []
            };
        }
    }

    /**
     * Adds a new condition group to the DOM and model.
     * @param {HTMLElement} triggerEl The clicked “Add Condition” element
     */
    addConditionGroup(triggerEl) {
        const tpl = jQuery("#wbsycf7-conditionDefault-template").html();
        const $container = jQuery(".wbsycf7-conditions-container");
        $container.append(tpl);

        const $group = $container.find(".wbsycf7-conditionGroup-row").last();

        const gid = ++this.groupIdSeq;
        $group.attr("data-id", gid);

        // Initialize model from current DOM values
        const action = ($group.find(".wbsycf7-condition-show-hide").val() || "Show").toLowerCase() === "show" ? "show" : "hide";
        this.conditions[gid] = {
            id: gid,
            action,
            fieldName: $group.find(".wbsycf7-form-fields").val() || "",
            logic: ($group.find(".wbsycf7-condition-and-or").val() || "All").toLowerCase() === "all" ? "all" : "any",
            items: []
        };
        this._signalModelChanged();
    }

    /**
     * Deletes a condition group from DOM and model.
     * Recomputes hidden input afterward.
     * @param {HTMLElement} triggerEl The clicked delete icon within the group
     */
    deleteConditionGroup(triggerEl) {
        const $group = jQuery(triggerEl).closest(".wbsycf7-conditionGroup-row");
        const gid = Number($group.attr("data-id"));
        delete this.conditions[gid];
        $group.remove();
        this._signalModelChanged();
    }

    /**
     * Appends a new item to an existing group.
     * Disables the group's target field selector and prunes item field options.
     * @param {HTMLElement} triggerEl The clicked “+” icon in the group title
     */
    addConditionItem(triggerEl) {
        const tpl = jQuery("#wbsycf7-conditionItem-template").html();
        const $group = jQuery(triggerEl).closest(".wbsycf7-conditionGroup-row");
        const $itemsWrap = $group.find(".wbsycf7-conditionGroup-items");
        const gid = Number($group.attr("data-id"));

        $itemsWrap.append(tpl);

        const $item = $itemsWrap.find(".wbsycf7-conditionGroup-item-row").last();
        const iid = ++this.itemIdSeq;
        $item.attr("data-id", iid);

        // Disable group-level field selector if at least one item exists
        if ($itemsWrap.find(".wbsycf7-conditionGroup-item-row").length > 0) {
            $group.find(".wbsycf7-form-fields").prop("disabled", true);
        }

        // REMOVE the group's chosen field from this item's select
        this.pruneItemOptionsForGroup($group, $item);

        // Initialize item in model
        const fieldName = $item.find(".wbsycf7-form-items-fields").val() || "";
        const op = $item.find(".wbsycf7-condition-equal").val() || "is";
        const value = $item.find(".wbsycf7-condition-value").val() || "";

        if ( !this.conditions[gid] ) {
            this.conditions[gid] = {
                id: gid,
                action: "show",
                fieldName: "",
                logic: "all",
                items: []
            };
        }
        this.conditions[gid].items.push({ id: iid, fieldName, op, value });

        this.renderValueControl($item);
        this._signalModelChanged();
    }

    /**
     * Removes an item from a group and re-enables the group's target select if empty.
     * @param {HTMLElement} triggerEl The delete icon within the item row
     */
    deleteConditionItem(triggerEl) {
        const $item = jQuery(triggerEl).closest(".wbsycf7-conditionGroup-item-row");
        const $group = $item.closest(".wbsycf7-conditionGroup-row");
        const gid = Number($group.attr("data-id"));
        const iid = Number($item.attr("data-id"));

        const group = this.conditions[gid];
        if (group) group.items = group.items.filter(it => it.id !== iid);

        $item.remove();

        // If no more items left, re-enable the group field select
        if ($group.find(".wbsycf7-conditionGroup-item-row").length === 0) {
            $group.find(".wbsycf7-form-fields").prop("disabled", false);
        }
        this._signalModelChanged();
    }

    /**
     * Synchronizes a group's values (action, target field, logic) from DOM into model.
     * @param {jQuery} $group Group row element
     */
    syncGroupFromDOM($group) {
        const gid = Number($group.attr("data-id"));
        if (!this.conditions[gid]) return;

        this.conditions[gid].action    = ($group.find(".wbsycf7-condition-show-hide").val() || "Show").toLowerCase() === "show" ? "show" : "hide";
        this.conditions[gid].fieldName = $group.find(".wbsycf7-form-fields").val() || "";
        this.conditions[gid].logic     = ($group.find(".wbsycf7-condition-and-or").val() || "All").toLowerCase() === "all" ? "all" : "any";
        this._signalModelChanged();
    }

    /**
     * Prunes (removes) the group target field from item-level field selects to prevent self-reference.
     * If a pruned current value was selected, falls back to the first available option.
     * @param {jQuery} $group The group container
     * @param {jQuery|null} $item Optional: only prune this item, otherwise all items in the group
     */
    pruneItemOptionsForGroup($group, $item = null) {
        const selectedGroupField = $group.find(".wbsycf7-form-fields").val();
        const $targets = $item
            ? $item.find(".wbsycf7-form-items-fields")
            : $group.find(".wbsycf7-form-items-fields");

        $targets.each(function () {
            const $sel = jQuery(this);

            // Cache original options on first run so we can restore if group field changes later
            if (!$sel.data("wbsycf7-original-options")) {
                $sel.data("wbsycf7-original-options", $sel.html());
            } else {
                // Restore to full list before pruning again
                $sel.html($sel.data("wbsycf7-original-options"));
            }

            // Remove the option equal to the group's field
            $sel.find("option").filter(function () {
                return jQuery(this).val() === selectedGroupField;
            }).remove();

            // If current value was removed, fallback to first available
            if (!$sel.find(`option[value="${$sel.val()}"]`).length) {
                const firstVal = $sel.find("option:first").val() || "";
                $sel.val(firstVal).trigger("change");
            }
        });
    }

    /**
     * Builds or rebuilds the item value control:
     * - select (single/multiple) for select/radio/checkbox with choices
     * - input[type=…] for text-like types
     * - hidden input when operator is empty/not_empty
     * Restores previous value where possible.
     * @param {jQuery} $item The item row container
     */
    renderValueControl($item) {
        const fieldName = $item.find(".wbsycf7-form-items-fields").val() || "";
        const op        = $item.find(".wbsycf7-condition-equal").val() || "is";

        const $old = $item.find(".wbsycf7-condition-value"); // existing input (from template)
        const oldVal = $old.val();
        const currentVal = Array.isArray(oldVal) ? oldVal : (oldVal ?? "");

        // If operator is empty/not_empty → no value editor needed
        if (op === "empty" || op === "not_empty") {
            if ($old.length) {
                $old.prop("disabled", true).hide();
            } else {
                $item.append('<input type="hidden" class="wbsycf7-condition-value" />');
            }
            return;
        }

        // Normalize name (strip [] suffix so it matches fieldsMeta keys)
        const key = fieldName.replace(/\[\]$/, "");
        const meta = this.fieldsMeta[fieldName] || this.fieldsMeta[key] || {
            type: "text",
            allows_multiple: false,
            values: []
        };

        const type = String(meta.type || "text").toLowerCase();

        // ---- choices normalization ----
        const normalizeChoices = (raw) => {
            // raw can be: array of strings, array of {value,label}, or a newline-separated string
            let arr;
            if (Array.isArray(raw)) {
                arr = raw.slice();
            } else if (typeof raw === "string") {
                arr = raw.split("\n").map(s => s.trim()).filter(Boolean);
            } else {
                arr = [];
            }

            // map to [{value,label}]
            return arr.map(it => {
                if (it && typeof it === "object") {
                    const v = ("value" in it) ? String(it.value) : ("label" in it ? String(it.label) : "");
                    const l = ("label" in it) ? String(it.label) : v;
                    return { value: v, label: l };
                }
                const s = String(it);
                return { value: s, label: s };
            });
        };

        // Prefer meta.values (your parser output), fallback to meta.options if present
        const choices = normalizeChoices(
            Array.isArray(meta.values) || typeof meta.values === "string" ? meta.values :
                (Array.isArray(meta.options) || typeof meta.options === "string" ? meta.options : [])
        );

        // Multiple: checkboxes always allow multiple; select may too (if you later set meta.allows_multiple)
        const allowsMultiple = !!meta.allows_multiple || type === "checkbox";

        // Decide control: select for select/radio/checkbox with choices; input otherwise
        const needsSelect = (type === "select" || type === "radio" || type === "checkbox") && choices.length > 0;

        // Replace control
        let $new;
        if (needsSelect) {
            $new = jQuery('<select class="wbsycf7-condition-value" name="wbsycf7_condition_value"></select>');
            if (allowsMultiple) $new.attr("multiple", "multiple");

            // Add options
            for (const ch of choices) {
                const $opt = jQuery("<option/>").attr("value", ch.value).text(ch.label);
                $new.append($opt);
            }

            // Restore previous value(s)
            if (allowsMultiple && currentVal) {
                // support array, comma-separated, or newline-separated
                const arr = Array.isArray(currentVal)
                    ? currentVal
                    : String(currentVal)
                        .split(/\r?\n|,/)
                        .map(s => s.trim())
                        .filter(Boolean);
                $new.val(arr);
            } else if (currentVal) {
                $new.val(String(currentVal));
            }
        } else {
            const map = { email: "email", tel: "tel", url: "url", number: "number", range: "range", date: "date", text: "text" };
            const htmlType = map[type] || "text";
            $new = jQuery('<input class="wbsycf7-condition-value" name="wbsycf7_condition_value">')
                .attr("type", htmlType)
                .val(currentVal || "");
        }

        if ($old.length) {
            $old.replaceWith($new);
        } else {
            $item.find(".wbsycf7-condition-equal").after($new);
        }

        $new.off("change input").on("change input", () => this.syncItemFromDOM($item));
    }

    /**
     * Reads a single item row from the DOM and writes its values into the model.
     * Handles string vs array values (multi-select).
     * @param {jQuery} $item
     */
    syncItemFromDOM($item) {
        const iid = Number($item.attr("data-id"));
        const gid = Number($item.closest(".wbsycf7-conditionGroup-row").attr("data-id"));
        const group = this.conditions[gid];
        if (!group) return;
        const it = group.items.find(x => x.id === iid);
        if (!it) return;

        // read field/operator/value from controls
        const fname = $item.find(".wbsycf7-form-items-fields").val() || "";
        const op    = $item.find(".wbsycf7-condition-equal").val() || "is";
        const $valEl = $item.find(".wbsycf7-condition-value");

        let val = null;
        if (op === "empty" || op === "not_empty") {
            val = null; // no value needed
        } else if ($valEl.is("select") && $valEl.prop("multiple")) {
            val = $valEl.val() || []; // array
        } else {
            val = $valEl.val() || "";
        }

        it.fieldName = fname;
        it.op        = op;
        it.value     = val; // NOTE: array for multi-select, string otherwise

        this._signalModelChanged();
    }

    /**
     * Serializes the internal dictionary into an array of groups for saving.
     * @returns {Array<Object>} groups[]
     */
    serialize() {
        // Convert dictionary to array if needed
        return Object.values(this.conditions);
    }
}

let ob = new WBSYCF7_CONDITIONS();