<?php
/**
 * Plugin Name: Webhookify - Send Form Submissions to Webhooks
 * Plugin URI:
 * Description: Send form submissions to webhook URLs
 * Version: 1.0.0
 * Author: Nep Tech Pal Pvt Ltd
 * Author URI: https://neptechpal.com.np
 * License: GPL v2 or later
 * Text Domain: webhookify-send-form-submissions-to-webhooks
 */

if (!defined('ABSPATH')) {
    exit;
}

class Webhookify
{

    private $option_name = 'webhookify_settings';

    public function __init()
    {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));

        add_action('wpcf7_before_send_mail', array($this, 'cf7_webhook_handler'));

        add_action('wpforms_process_complete', array($this, 'wpforms_webhook_handler'), 10, 4);

        add_action('gform_after_submission', array($this, 'gf_webhook_handler'), 10, 2);

        add_action('elementor_pro/forms/new_record', array($this, 'elementor_webhook_handler'), 10, 2);

        add_action('frm_after_create_entry', array($this, 'formidable_webhook_handler'), 30, 2);

        add_action('comment_post', array($this, 'comment_webhook_handler'), 10, 3);
    }

    public function enqueue_admin_styles($hook)
    {
        if ($hook !== 'toplevel_page_webhookify') {
            return;
        }

        wp_add_inline_style('wp-admin', '
            #adminmenu #toplevel_page_webhookify .wp-menu-image img {
                width: 20px;
                height: 20px;
                padding: 6px 0;
            }
            #adminmenu #toplevel_page_webhookify:hover .wp-menu-image img,
            #adminmenu #toplevel_page_webhookify.current .wp-menu-image img {
                opacity: 1;
            }
        ');
    }

    public function add_admin_menu()
    {
        $icon_url = $this->get_menu_icon();

        add_menu_page(
            'Webhookify Settings',
            'Webhookify',
            'manage_options',
            'webhookify',
            array($this, 'settings_page'),
            $icon_url,
            100
        );
    }

    private function get_menu_icon()
    {
        $icon_path = plugin_dir_url(__FILE__) . 'icon.png';
        return $icon_path;
    }

    public function register_settings()
    {
        register_setting(
            'webhookify_settings_group',
            $this->option_name,
            array(
                'sanitize_callback' => array($this, 'sanitize_settings')
            )
        );
    }

    public function sanitize_settings($input)
    {
        $sanitized = array();

        if (isset($input['webhook_url'])) {
            $webhook_url = esc_url_raw($input['webhook_url']);

            // Check if URL starts with webhookify.app and add trailing slash if missing
            if (strpos($webhook_url, 'webhookify.app') !== false && substr($webhook_url, -1) !== '/') {
                $webhook_url .= '/';
            }

            $sanitized['webhook_url'] = $webhook_url;

            // Add success message
            add_settings_error(
                $this->option_name,
                'webhookify_updated',
                'Webhook URL saved successfully!',
                'success'
            );
        }

        if (isset($input['enabled'])) {
            $sanitized['enabled'] = '1';
        } else {
            $sanitized['enabled'] = '';
        }

        return $sanitized;
    }

    public function settings_page()
    {
        $settings = get_option($this->option_name, array('webhook_url' => '', 'enabled' => '1'));
        ?>
        <div class="wrap">
            <h1>
                <span style="display: inline-flex; align-items: center; gap: 10px;">
                    <img src="<?php echo esc_url($this->get_menu_icon()); ?>" alt="webhookify icon" height="20">
                    <b> Webhookify Settings</b>
                </span>
            </h1>
            <p>Send form submissions from your site to any webhook URL in real-time.</p>

            <div style="background: #e7f5fe; border: 1px solid #cce8ff; padding: 15px; margin: 20px 0; border-radius: 5px;">
                <p style="margin-top: 0; font-weight: 600;">🔔 Get instant <strong style="color: #11895E;">AI-powered
                        notifications</strong> on your mobile, Telegram, Discord, Slack, and more!</p>

                <a href="https://webhookify.app" target="_blank" class="button button-primary button-large"
                    style="background-color: #11895E; border-color: #0077cc; margin-right: 15px;">
                    Click to Generate Your Webhook! 🚀
                </a>

                <p style="margin-bottom: 0; margin-top: 15px;">
                    Need help with the setup? View the official tutorial:
                    <a href="https://webhookify.app/guides/webhookify-wordpress-setup-guide"
                        target="_blank">webhookify.app/guides/webhookify-wordpress-setup-guide</a>
                </p>
            </div>

            <?php settings_errors($this->option_name); ?>

            <form method="post" action="options.php">
                <?php settings_fields('webhookify_settings_group'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="webhook_url">Webhook URL</label>
                        </th>
                        <td>
                            <input type="url" id="webhook_url" name="<?php echo esc_attr($this->option_name); ?>[webhook_url]"
                                value="<?php echo esc_attr($settings['webhook_url']); ?>" class="regular-text"
                                placeholder="https://your-webhook-url.com/endpoint">
                            <p class="description">Enter the webhook URL where form data will be sent.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="enabled">Enable Webhookify</label>
                        </th>
                        <td>
                            <input type="checkbox" id="enabled" name="<?php echo esc_attr($this->option_name); ?>[enabled]"
                                value="1" <?php checked($settings['enabled'] ?? '', '1'); ?>>
                            <label for="enabled">Send form submissions to webhook</label>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>

            <hr>

            <h2>Form Plugin Status</h2>
            <p style="margin-bottom: 15px;">Webhookify automatically detects and integrates with the following form plugins
                installed on your site:</p>

            <div
                style="display: grid; grid-template-columns: repeat(auto-fill, minmax(220px, 1fr)); gap: 12px; margin: 20px 0;">
                <?php
                $plugins = array(
                    'Contact Form 7' => array(
                        'active' => defined('WPCF7_VERSION'),
                        'slug' => 'contact-form-7'
                    ),
                    'WPForms' => array(
                        'active' => function_exists('wpforms'),
                        'slug' => 'wpforms-lite'
                    ),
                    'Gravity Forms' => array(
                        'active' => class_exists('GFForms'),
                        'slug' => 'gravityforms'
                    ),
                    'Elementor Forms' => array(
                        'active' => defined('ELEMENTOR_PRO_VERSION'),
                        'slug' => 'elementor-pro'
                    ),
                    'Formidable Forms' => array(
                        'active' => class_exists('FrmAppHelper'),
                        'slug' => 'formidable'
                    )
                );

                foreach ($plugins as $name => $info) {
                    $active = $info['active'];

                    if ($active) {
                        $badge = '<span style="display: inline-block; background: #00a32a; color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600;">ACTIVE</span>';
                        $border_color = '#00a32a';
                        $icon = '✓';
                        $icon_color = '#00a32a';
                    } else {
                        $badge = '<span style="display: inline-block; background: #dba617; color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600;">NOT INSTALLED</span>';
                        $border_color = '#ddd';
                        $icon = '○';
                        $icon_color = '#999';
                    }

                    echo '<div style="padding: 12px; background: #fff; border: 2px solid ' . esc_attr($border_color) . '; border-radius: 6px; display: flex; align-items: center; gap: 10px;">';
                    echo '<span style="font-size: 20px; color: ' . esc_attr($icon_color) . '; font-weight: bold;">' . esc_html($icon) . '</span>';
                    echo '<div style="flex: 1;">';
                    echo '<div style="font-weight: 600; margin-bottom: 4px;">' . esc_html($name) . '</div>';
                    echo wp_kses_post($badge);
                    echo '</div>';
                    echo '</div>';
                }
                ?>
            </div>

            <div style="background: #e7f5fe; border-left: 4px solid #11895E; padding: 12px 15px; margin: 20px 0;">
                <strong style="color: #11895E;">💡 How it works:</strong>
                <p style="margin: 8px 0 0 0; color: #333;">When a form plugin is active, Webhookify automatically captures all
                    submissions and sends them to your webhook URL. No additional configuration needed!</p>
            </div>

            <hr>

            <h2>Test Webhook</h2>
            <button type="button" class="button button-primary" id="test-webhook">Send Test Request</button>
            <div id="test-result" style="margin-top: 10px;"></div>

            <script>
                jQuery(document).ready(function ($) {
                    $('#test-webhook').on('click', function () {
                        var btn = $(this);
                        var result = $('#test-result');

                        btn.prop('disabled', true).text('Sending...');
                        result.html('');

                        $.ajax({
                            url: ajaxurl,
                            method: 'POST',
                            data: {
                                action: 'webhookify_test',
                                nonce: <?php echo wp_json_encode(wp_create_nonce('webhookify_test')); ?>
                            },
                            success: function (response) {
                                if (response.success) {
                                    result.html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                                } else {
                                    result.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                                }
                            },
                            error: function () {
                                result.html('<div class="notice notice-error"><p>Failed to send test request.</p></div>');
                            },
                            complete: function () {
                                btn.prop('disabled', false).text('Send Test Request');
                            }
                        });
                    });
                });
            </script>
        </div>
        <?php
    }

    private function send_to_webhook($data)
    {
        $settings = get_option($this->option_name);

        if (empty($settings['webhook_url']) || empty($settings['enabled'])) {
            return false;
        }

        $webhook_url = $settings['webhook_url'];

        $args = array(
            'body' => json_encode($data),
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'method' => 'POST',
            'timeout' => 30,
        );

        $response = wp_remote_post($webhook_url, $args);

        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Webhookify Error: ' . $response->get_error_message());
            }
            return false;
        }

        return true;
    }

    // Contact Form 7 Handler
    public function cf7_webhook_handler($contact_form)
    {
        $submission = WPCF7_Submission::get_instance();

        if ($submission) {
            $posted_data = $submission->get_posted_data();

            $data = array(
                'form_type' => 'contact_form_7',
                'form_id' => $contact_form->id(),
                'form_title' => $contact_form->title(),
                'timestamp' => current_time('mysql'),
                'site_url' => get_site_url(),
                'fields' => $posted_data
            );

            $this->send_to_webhook($data);
        }
    }

    // WPForms Handler
    public function wpforms_webhook_handler($fields, $entry, $form_data, $entry_id)
    {
        $data = array(
            'form_type' => 'wpforms',
            'form_id' => $form_data['id'],
            'form_title' => $form_data['settings']['form_title'],
            'timestamp' => current_time('mysql'),
            'site_url' => get_site_url(),
            'fields' => $fields,
            'entry_id' => $entry_id
        );

        $this->send_to_webhook($data);
    }

    // Gravity Forms Handler
    public function gf_webhook_handler($entry, $form)
    {
        $data = array(
            'form_type' => 'gravity_forms',
            'form_id' => $form['id'],
            'form_title' => $form['title'],
            'timestamp' => current_time('mysql'),
            'site_url' => get_site_url(),
            'fields' => $entry
        );

        $this->send_to_webhook($data);
    }

    // Elementor Forms Handler
    public function elementor_webhook_handler($record, $handler)
    {
        $form_name = $record->get_form_settings('form_name');
        $raw_fields = $record->get('fields');

        // Format fields properly
        $fields = array();
        foreach ($raw_fields as $id => $field) {
            $fields[$field['title']] = $field['value'];
        }

        $data = array(
            'form_type' => 'elementor_forms',
            'form_name' => $form_name,
            'timestamp' => current_time('mysql'),
            'site_url' => get_site_url(),
            'fields' => $fields
        );

        $this->send_to_webhook($data);
    }

    // Formidable Forms Handler
    public function formidable_webhook_handler($entry_id, $form_id)
    {
        $entry = FrmEntry::getOne($entry_id, true);
        $form = FrmForm::getOne($form_id);

        // Get form fields
        $fields = array();
        if ($entry && isset($entry->metas)) {
            foreach ($entry->metas as $field_id => $value) {
                $field = FrmField::getOne($field_id);
                if ($field) {
                    $field_name = $field->name;
                    $fields[$field_name] = is_array($value) ? $value : (string) $value;
                }
            }
        }

        $data = array(
            'form_type' => 'formidable_forms',
            'form_id' => $form_id,
            'form_name' => $form ? $form->name : 'Unknown',
            'entry_id' => $entry_id,
            'timestamp' => current_time('mysql'),
            'site_url' => get_site_url(),
            'fields' => $fields
        );

        $this->send_to_webhook($data);
    }

    // Comment Handler (optional)
    public function comment_webhook_handler($comment_id, $approved, $commentdata)
    {
        if ($approved === 1) {
            $data = array(
                'form_type' => 'wordpress_comment',
                'comment_id' => $comment_id,
                'timestamp' => current_time('mysql'),
                'site_url' => get_site_url(),
                'fields' => $commentdata
            );

            $this->send_to_webhook($data);
        }
    }
}

// Initialize the plugin
$webhookify = new Webhookify();
$webhookify->__init();

// AJAX handler for test webhook
add_action('wp_ajax_webhookify_test', 'webhookify_test_webhook');
function webhookify_test_webhook()
{
    check_ajax_referer('webhookify_test', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'Unauthorized'));
    }

    $settings = get_option('webhookify_settings');

    if (empty($settings['webhook_url'])) {
        wp_send_json_error(array('message' => 'Please set a webhook URL first.'));
    }

    $test_data = array(
        'form_type' => 'test',
        'message' => 'This is a test webhook from Webhookify',
        'timestamp' => current_time('mysql'),
        'site_url' => get_site_url()
    );

    $args = array(
        'body' => json_encode($test_data),
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'method' => 'POST',
        'timeout' => 30,
    );

    $response = wp_remote_post($settings['webhook_url'], $args);

    if (is_wp_error($response)) {
        wp_send_json_error(array('message' => 'Error: ' . $response->get_error_message()));
    }

    $response_code = wp_remote_retrieve_response_code($response);
    wp_send_json_success(array('message' => 'Test webhook sent successfully! Response code: ' . $response_code));
}
?>