<?php
/**
 * Plugin Name: Web-in-Air: Shop Connect for Telegram
 * Plugin URI:  https://woo-telegram-shop.com/
 * Description: Turn your WooCommerce store into an interactive shopping bot on Telegram.
 * Version:     1.4.1
 * Author:      WebinAir
 * Author URI:  https://web-in-air.com/
 * License:     GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Requires Plugins: woocommerce
 * Text Domain: web-in-air-shop-connect-for-telegram
 * Domain Path: /languages
 */

// Security: prevents direct access to the file.
if ( ! defined( 'ABSPATH' ) ) exit;

// =================================================================
//  Constantes et Initialisation
// =================================================================

// Main plugin constants for easy access to paths and settings.
define( 'WASCT_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'WASCT_PLUGIN_FILE', __FILE__ );
define( 'WASCT_PRODUCTS_PER_PAGE', 10 );
define( 'WASCT_CATEGORIES_PER_PAGE', 10 );
define( 'WASCT_TAGS_PER_PAGE', 10 );

/**
 * Loads the classes necessary for the plugin to function.
 * Runs on the “plugins_loaded” action with a priority of 5 to ensure
 * that the classes are available before the rest of the plugin runs.
 */
function wasct_autoload_classes() {
    require_once WASCT_PLUGIN_DIR . 'includes/class-wasct-api-handler.php';
    require_once WASCT_PLUGIN_DIR . 'includes/class-wasct-utils.php';
    require_once WASCT_PLUGIN_DIR . 'includes/class-wasct-webhook-handler.php';
    require_once WASCT_PLUGIN_DIR . 'admin/class-wasct-admin.php';
}
add_action( 'plugins_loaded', 'wasct_autoload_classes', 5 );


/**
 * Main entry point to launch the plugin's features.
 * Initialises the administration section and the webhook manager.
 */
function wasct_run_plugin() {
    new WASCT_Admin();
    
    $webhook_handler = new WASCT_Webhook_Handler();
    add_action( 'init', [ $webhook_handler, 'handle_webhook_request' ] );
}
add_action( 'plugins_loaded', 'wasct_run_plugin', 10 );


// =================================================================
//  Plugin activation/deactivation
// =================================================================

/**
 * Actions to be performed when the plugin is activated.
 * Mainly configures the Telegram webhook.
 */
function wasct_on_plugin_activate() {
    wasct_setup_telegram_webhook();
}
register_activation_hook( __FILE__, 'wasct_on_plugin_activate' );


/**
 * Actions to be performed when the plugin is activated.
 * Mainly configures the Telegram webhook.
 */
function wasct_on_plugin_deactivate() {
    wasct_delete_telegram_webhook();
}
register_deactivation_hook( __FILE__, 'wasct_on_plugin_deactivate' );


// =================================================================
// Telegram Webhook Management
// =================================================================

/**
 * Register the WordPress site URL with Telegram to receive updates.
 */
function wasct_setup_telegram_webhook() {
    $options = wasct_get_settings();
    $token = $options['telegram_token'] ?? '';
    if ( empty($token) ) return;

    $webhook_url = home_url( '/?telegram_webhook=1' );
    $api_url = "https://api.telegram.org/bot{$token}/setWebhook?url={$webhook_url}";
    
    wp_remote_get( $api_url, ['timeout' => 10] );
}

/**
 * Deletes the webhook from Telegram.
 */
function wasct_delete_telegram_webhook() {
    $options = wasct_get_settings();
    $token = $options['telegram_token'] ?? '';
    if ( empty($token) ) return;

    $api_url = "https://api.telegram.org/bot{$token}/deleteWebhook";

    wp_remote_get( $api_url, ['timeout' => 10] );
}


// =================================================================
//  Settings Management
// =================================================================

/**
 * Retrieves the plugin settings stored in the WordPress options.
 * Uses default values to ensure that the keys always exist.
 *
 * @return array The plugin settings.
 */
function wasct_get_settings() {
    $defaults = [
        'enable_bot'            => '0',
        'telegram_token'        => '',
        'wc_consumer_key'       => '',
        'wc_consumer_secret'    => '',
        'support_username'      => '',
        'enabled_languages'     => [substr(get_locale(), 0, 2)],
        'product_display_mode'  => 'text',
        'variation_columns'     => '5',
        'custom_logo_url'       => '',
        'welcome_text'          => '',
        'enable_tags_button'    => '1',
        'enable_brands_button'  => '1',
        'enable_contact_button' => '1',
    ];

    $settings = get_option( 'wasct_settings', $defaults );

    return wp_parse_args( $settings, $defaults );
}


// =================================================================
//  Cache Management
// =================================================================

/**
 * Invalidates ALL plugin data caches by updating a global version.
 * 
 */
function wasct_invalidate_all_data_caches() {
    update_option('wasct_cache_version', time());
    }
    
    add_action('save_post_product', 'wasct_invalidate_all_data_caches');
    add_action('delete_post', function($post_id) {
        if (get_post_type($post_id) === 'product') {
            wasct_invalidate_all_data_caches();
        }
    });
    add_action('set_object_terms', function($object_id) {
        if (get_post_type($object_id) === 'product') {
            wasct_invalidate_all_data_caches();
        }
    });