<?php
/**
 * Handles Telegram webhook requests.
 * This is the main class that orchestrates the bot's responses.
 *
 */
class WASCT_Webhook_Handler
{
    private static $current_lang_for_api = 'en';

    /**
     * Allows other parts of the plugin to know the active language for API requests.
     * @return string The language code (e.g., ‘en’, ‘fr’).
     */
    public static function get_current_lang_for_api() {
        return self::$current_lang_for_api;
    }
    /**
     * Path to the directory where lock files are stored.
     *
     * @var string
     */
    private $lock_dir;
    
    /**
     * Constructor. Initializes the locking system.
     */
    public function __construct()
    {
        $upload_dir     = wp_upload_dir();
        $this->lock_dir = $upload_dir['basedir'] . '/wasct_locks';
        if (! file_exists($this->lock_dir)) {
            wp_mkdir_p($this->lock_dir);
        }
    }
    
    /**
     * Attempts to acquire a lock for a user to prevent concurrent request processing.
     * The lock automatically expires after 15 seconds.
     *
     * @param int $chat_id The chat ID.
     * @return bool True if the lock was acquired, false otherwise.
     *
     *
     * Note to Plugin Reviewers:
     *
     * The following locking mechanism intentionally uses native PHP file functions
     * (fopen, unlink) instead of the WP_Filesystem API for reliability reasons.
     *
     * An initial implementation using WP_Filesystem proved to be fragile across
     * diverse hosting environments (especially in a public webhook context where
     * credentials cannot be requested), causing the bot to become unresponsive.
     *
     * Given that this is a simple, non-critical lock file with a short time-to-live,
     * the atomicity and universal reliability of fopen() with the 'x' flag was chosen
     * as a pragmatic solution to guarantee the plugin's core functionality.
     */
     
    private function acquire_lock($chat_id)
    {
        $lock_file = $this->lock_dir . '/' . intval($chat_id) . '.lock';
        
        if (file_exists($lock_file) && (time() - filemtime($lock_file) > 15)) {
            @unlink($lock_file);
        }
        
        $handle = @fopen($lock_file, 'x'); 
        if ($handle === false) {
            return false; 
        }

        fclose($handle); 
        return true;
    }
    
    /**
     * Releases the lock for a user once processing is complete.
     *
     * @param int $chat_id The conversation ID.
     */
    private function release_lock($chat_id)
    {
        $lock_file = $this->lock_dir . '/' . intval($chat_id) . '.lock';
        if (file_exists($lock_file)) {
            @unlink($lock_file); 
        }
    }
    
    /**
     * Centralised safeguard for message modification failures.
     * If an error is detected, it resets the conversation for the user.
     *
     * @param mixed $result The result of the API call (may be WP_Error).
     * @param int $chat_id The chat ID.
     * @param int $message_id The ID of the message that failed.
     * @return bool True if an error was handled, false otherwise.
     */
    private function _handle_edit_failure($result, $chat_id, $message_id) {
        if (is_wp_error($result)) {
            error_log('WASCT Safeguard: API edit failed for message ' . $message_id . '. Error: ' . $result->get_error_message());
            
            @WASCT_API_Handler::delete_telegram_message($chat_id, $message_id);
            
            $this->send_menu_only($chat_id);
            
            return true; 
        }
        return false;
    }
    
    /**
     * Determines and loads the user's language centrally.
     * This function follows a priority hierarchy to choose the best language.
     *
     * @param array $update_data Telegram update data.
     * @return string The language code to use for API calls (e.g. “fr”, “en”).
     */
    private function _determine_and_load_language($update_data) {
        $options = wasct_get_settings();
        $admin_enabled_languages = $options['enabled_languages'] ?? ['en'];
        $all_site_languages = WASCT_Utils::get_site_languages();
        $user_lang_code = null;
    
        $chat_id = $update_data['callback_query']['message']['chat']['id'] ?? $update_data['message']['chat']['id'] ?? null;
        if ($chat_id) {
            $saved_lang = get_transient('wasct_user_lang_' . $chat_id);
            if ($saved_lang && in_array($saved_lang, $admin_enabled_languages)) {
                $user_lang_code = $saved_lang;
            }
        }
    
        if (!$user_lang_code) {
            $tg_lang_code = substr($update_data['callback_query']['from']['language_code'] ?? $update_data['message']['from']['language_code'] ?? '', 0, 2);
            if ($tg_lang_code && in_array($tg_lang_code, $admin_enabled_languages)) {
                $user_lang_code = $tg_lang_code;
            }
        }
    
        if (!$user_lang_code) {
            $site_lang_code = substr(get_locale(), 0, 2);
            if (in_array($site_lang_code, $admin_enabled_languages)) {
                $user_lang_code = $site_lang_code;
            }
        }
    
        if (!$user_lang_code) {
            $user_lang_code = in_array('en', $admin_enabled_languages) ? 'en' : ($admin_enabled_languages[0] ?? 'en');
        }
    
        $interface_locale = $all_site_languages[$user_lang_code]['locale'] ?? 'en_US';
    
        switch_to_locale($interface_locale);

    
        return $user_lang_code;
    }
    
    // =================================================================
    //  Main Message Management (Panel)
    // =================================================================

    /**
     * Saves the ID of the user's main message (the ‘panel’).
     * The bot modifies this message instead of sending new ones.
     *
     * @param int $chat_id
     * @param int $message_id
     */
    private function set_panel_message_id($chat_id, $message_id)
    {
        set_transient('wasct_panel_msg_' . $chat_id, $message_id, DAY_IN_SECONDS);
    }
    
     /**
     * Retrieves the ID of the user's main message.
     *
     * @param int $chat_id
     * @return int|false The message ID or false if it does not exist.
     */
    private function get_panel_message_id($chat_id)
    {
        return get_transient('wasct_panel_msg_' . $chat_id);
    }
    
    // =================================================================
    //  Main Webhook Entry Point
    // =================================================================

    /**
     * Entry point for all Telegram requests.
     * It checks the request, determines the user's language, and initiates processing.
     */
    public function handle_webhook_request() {
        if (isset($_GET['telegram_webhook']) && $_GET['telegram_webhook'] === '1') {
            $options = wasct_get_settings();
            if (empty($options['enable_bot']) || $options['enable_bot'] !== '1') {
                return;
            }
    
            $update_data = json_decode(file_get_contents('php://input'), true);
            
            if (!empty($update_data)) {
                self::$current_lang_for_api = $this->_determine_and_load_language($update_data);
                
                $this->process_update($update_data);
    
                if (function_exists('restore_current_locale')) {
                    restore_current_locale();
                }
            }
            wp_send_json_success();
            exit;
        }
    }
    
    // =================================================================
    //  Order Routers
    // =================================================================

    /**
     * Processes button clicks (callback queries).
     * This is the main router for all interactive actions.
     *
     * @param array $callback_query The data from the clicked button.
     * @return void
     */
    private function process_update($update_data)
    {
        if (isset($update_data['callback_query'])) {
            $this->process_callback_query($update_data['callback_query']);
        } elseif (isset($update_data['message']['text'])) {
            $this->process_text_message($update_data['message']);
        }
    }
    
    /**
     * Handles button clicks (callback queries).
     * This is the main router for all interactive actions.
     *
     * @param array $callback_query Data for the clicked button.
     */
    private function process_callback_query($callback_query)
    {
        $chat_id           = $callback_query['message']['chat']['id'];
        $message_id        = $callback_query['message']['message_id'];
        $callback_query_id = $callback_query['id'];
    
        $this->acknowledge_callback_query($callback_query_id);
    
        if (! $this->acquire_lock($chat_id)) {
            return;
        }

            try {
            $panel_id = $this->get_panel_message_id($chat_id);
            if ( ! $panel_id || $panel_id != $message_id ) {
                $this->send_menu_only($chat_id, $message_id);
     
                return;
            }
    
            $data = $callback_query['data'];
            if (strpos($data, 'noop') === 0) {
                return;
            }
            
            if ($data === 'back_to_list') {
                $this->handle_back_to_list_from_details_callback($callback_query);
                return; 
            }
    
            switch ($data) {
                case 'show_products':
                    $panel_id = $this->get_panel_message_id($chat_id);
                    $this->handle_show_products_callback($chat_id, $panel_id);
                    break;
                case 'show_brands':
                    $this->handle_taxonomy_command($chat_id, $message_id, 'brands', __('Here are our product brands:', 'web-in-air-shop-connect-for-telegram'), 'brand_');
                    break;
                case 'prompt_search': $this->handle_search_prompt($chat_id);
                    break;
                case 'cancel_search': $this->handle_cancel_search($callback_query);
                    break;
                case 'contact_support': $this->handle_contact_support($chat_id);
                    break;
                case 'select_language': $this->handle_select_language($callback_query);
                    break;
                case 'back_to_main_menu':
                    $this->send_menu_only($chat_id, $message_id);
                    break;
                default:
                    if (strpos($data, 'set_lang_') === 0) {
                        $this->handle_set_language($callback_query);
                    } 
                    elseif (strpos($data, 'details_') === 0) {
                        $this->handle_simple_product_view($chat_id, $message_id, $data);
                    } 
                    elseif (strpos($data, 'nav_') === 0) {
                        $this->handle_navigation_callback($callback_query);
                    } 
                    elseif (strpos($data, 'select_attr_') === 0) {
                        $this->handle_attribute_selection_callback($callback_query);
                    }
                    elseif (strpos($data, 'back_to_product_') === 0) {
                        $this->handle_back_to_product_callback($callback_query);
                    } 
                    elseif (strpos($data, 'back_to_list_from_details') === 0) {
                        $this->handle_back_to_list_from_details_callback($callback_query);
                    } 
                    elseif (strpos($data, 'category_') === 0) {
                        $this->handle_category_callback($callback_query);
                    }
                    elseif (strpos($data, 'view_products_cat_') === 0) {
                        $this->handle_view_products_for_category_callback($callback_query);
                    }
                    elseif (strpos($data, 'tag_') === 0) {
                        $this->handle_taxonomy_product_list($callback_query, [
                            'slug' => 'tag', 'name' => __('Tag', 'web-in-air-shop-connect-for-telegram'), 'emoji' => '🏷️',
                            'api_slug_plural' => 'tags', 'api_param' => 'tag',
                            /* translators: %s: The name of the tag. */
                            'intro_text' => __("Products with the tag *%s*:\n", 'web-in-air-shop-connect-for-telegram')
                        ]);
                    }
                    elseif (strpos($data, 'brand_') === 0) {
                        $this->handle_taxonomy_product_list($callback_query, [
                            'slug' => 'brand', 'name' => __('Brand', 'web-in-air-shop-connect-for-telegram'), 'emoji' => '™️',
                            'api_slug_plural' => 'brands', 'api_param' => 'brand',
                            /* translators: %s: The name of the brand. */
                            'intro_text' => __("Products from the brand *%s*:\n", 'web-in-air-shop-connect-for-telegram')
                        ]);
                    }
                    elseif (strpos($data, 'products_page_') === 0) {
                        $this->handle_products_pagination_callback($callback_query, 'products_page');
                    }
                     elseif (strpos($data, 'search_page_') === 0) {
                        $this->handle_products_pagination_callback($callback_query, 'search_page');
                    }
                    elseif (strpos($data, 'show_categories_page_') === 0) {
                        $this->handle_show_categories_callback($callback_query, false);
                    } 
                    elseif (strpos($data, 'show_tags_page_') === 0) {
                        $this->handle_show_tags_callback($callback_query, true);
                    }
                    break;
            }
    
        } finally {
            $this->release_lock($chat_id);
        }
    }
    
    /**
     * Processes text messages sent by the user.
     * Mainly handles the /start command and responses to questions (search, etc.).
     *
     * @param array $message The message data.
     */
    private function process_text_message($message)
    {
        $chat_id         = $message['chat']['id'];
        $text            = $message['text'];
        $user_message_id = $message['message_id'];

        WASCT_API_Handler::delete_telegram_message($chat_id, $user_message_id);

        $user_state = get_transient('wasct_user_state_' . $chat_id);
        if ($user_state === 'awaiting_search_query') {
            delete_transient('wasct_user_state_' . $chat_id);
            $this->handle_search_command($chat_id, $text);
        } elseif ($user_state === 'awaiting_contact_message') {
            $this->handle_contact_message_forwarding($message);
        } else {
            if (strtolower(trim($text)) === '/start') {
                
                $panel_id = $this->get_panel_message_id($chat_id);
                if ($panel_id) {
                    @WASCT_API_Handler::delete_telegram_message($chat_id, $panel_id);
                }

                $this->send_menu_only($chat_id);
            }
        }
    }
    
    // =================================================================
    //  Construction of Menus and Main Actions
    // =================================================================

    /**
     * Dynamically builds the main menu keyboard.
     * Buttons are displayed based on options and content availability.
     *
     * @return array Keyboard formatted for the Telegram API.
     */
    private function _build_main_menu_keyboard()
    {
        $options = wasct_get_settings();
        $buttons = [];
        
        $buttons[] = [['text' => __('📦 See our products', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'show_products']];
        $buttons[] = [['text' => __('📁 Our categories', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'show_categories_page_1']];
    
        if (!empty($options['enable_tags_button']) && $options['enable_tags_button'] === '1') {
            $should_show_tags = get_transient('wasct_show_tags_button');
            if (false === $should_show_tags) {
                $tags_data = WASCT_API_Handler::get_woocommerce_tags(['per_page' => 1]);
                $should_show_tags = (!is_wp_error($tags_data) && !empty($tags_data['terms'])) ? 'yes' : 'no';
                set_transient('wasct_show_tags_button', $should_show_tags, 10 * MINUTE_IN_SECONDS);
            }
            if ('yes' === $should_show_tags) {
                $buttons[] = [['text' => __('🏷 Our tags', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'show_tags_page_1']];
            }
        }
    
        if (!empty($options['enable_brands_button']) && $options['enable_brands_button'] === '1') {
            $should_show_brands = get_transient('wasct_show_brands_button');
            if (false === $should_show_brands) {
                $brands_data = WASCT_API_Handler::get_woocommerce_brands(['per_page' => 1]);
                $should_show_brands = (!is_wp_error($brands_data) && !empty($brands_data['terms'])) ? 'yes' : 'no';
                set_transient('wasct_show_brands_button', $should_show_brands, 10 * MINUTE_IN_SECONDS);
            }
            if ('yes' === $should_show_brands) {
                $buttons[] = [['text' => __('™️ Our brands', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'show_brands']];
            }
        }
        
        $buttons[] = [['text' => __('🔍 Search for a product', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'prompt_search']];
        if (!empty($options['enable_contact_button']) && $options['enable_contact_button'] === '1') {
            $buttons[] = [['text' => __('✉️ Contact us', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'contact_support']];
        }
        $enabled_languages = $options['enabled_languages'] ?? ['fr'];
        if (count($enabled_languages) > 1) {
            $buttons[] = [['text' => __('🏳 Change language', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'select_language']];
        }
        $buttons[] = [['text' => __('🌐 Visit our website', 'web-in-air-shop-connect-for-telegram'), 'url' => home_url()]];
        
        return $buttons;
    }
    

    /**
     * Sends or edits the message to display only the main menu.
     * This is the central function for displaying the bot's home page.
     *
     * @param int      $chat_id The conversation ID.
     * @param int|null $message_id_to_edit If provided, the message will be edited. Otherwise, a new one will be sent.
     */
    private function send_menu_only($chat_id, $message_id_to_edit = null)
    {
        $options = wasct_get_settings();
        $site_name = get_bloginfo('name');
        $site_slogan = get_bloginfo('description');
        $welcome_text_template = $options['welcome_text'] ?? '';

        if (empty($welcome_text_template)) {
            $welcome_text_template = __("👋 Welcome to the shop of *{site_name}* !\n\n✨ _{site_slogan}_", 'web-in-air-shop-connect-for-telegram');
        }

        $welcome_text = str_replace(
            ['{site_name}', '{site_slogan}'],
            [esc_html($site_name), esc_html($site_slogan)],
            $welcome_text_template
        );
        
        $buttons = $this->_build_main_menu_keyboard();
        $logo_url = WASCT_Utils::get_logo_url();

        if ($message_id_to_edit) {
            $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id_to_edit, $logo_url, $welcome_text, $buttons);
            if ($this->_handle_edit_failure($result, $chat_id, $message_id_to_edit)) {
                return;
            }
            $this->set_panel_message_id($chat_id, $message_id_to_edit);
        } else {
            $sent_message = WASCT_API_Handler::send_telegram_photo($chat_id, $logo_url, $welcome_text, $buttons);
            if (! is_wp_error($sent_message) && isset($sent_message['message_id'])) {
                $this->set_panel_message_id($chat_id, $sent_message['message_id']);
            }
        }
    }
    
    /**
     * Sends a confirmation to the Telegram API for a callback query.
     * Essential for the user's Telegram client to stop the loading animation.
     *
     * @param string      $callback_query_id The ID of the callback.
     * @param string|null $text              An optional message to display in a pop-up to the user.
     */
    private function acknowledge_callback_query($callback_query_id, $text = null)
    {
        $options = wasct_get_settings();
        $token   = $options['telegram_token'] ?? '';
        if (empty($token)) {
            return;
        }
        $api_url = "https://api.telegram.org/bot{$token}/answerCallbackQuery";
        $params  = [ 'callback_query_id' => $callback_query_id ];
        if ($text) {
            $params['text']       = $text;
            $params['show_alert'] = false;
        }
        wp_remote_post(add_query_arg($params, $api_url), ['blocking' => false]);
    }
    
    // =================================================================
    //  Management of Taxonomy Lists (Categories, Brands, etc.)
    // =================================================================

    /**
     * Manages the display of a list of main categories (root).
     *
     * @param array $callback_query Callback data.
     */
    private function handle_show_categories_callback($callback_query, $is_editing)
    {
        $chat_id = $callback_query['message']['chat']['id'];
        $panel_id = $callback_query['message']['message_id']; 

        $logo_url = WASCT_Utils::get_logo_url();
        $result = WASCT_API_Handler::edit_telegram_message_media(
            $chat_id,
            $panel_id,
            $logo_url,
            __('🔄 Loading categories, please wait...', 'web-in-air-shop-connect-for-telegram'),
            [[['text' => '⏳', 'callback_data' => 'noop']]]
        );
        
         if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return; 
        }   
        
        $parts           = explode('_', str_replace('edit_', 'show_', $callback_query['data']));
        $page            = (int) end($parts);
        $categories_data = WASCT_API_Handler::get_woocommerce_categories(['parent' => 0, 'per_page' => WASCT_CATEGORIES_PER_PAGE, 'page' => $page]);
        
        WASCT_Utils::send_category_list(
            $chat_id,
            $categories_data,
            $page,
            __('Here are our main categories:', 'web-in-air-shop-connect-for-telegram'),
            0,
            $panel_id 
        );
    }
    
    /**
     * Handles clicks on a category: displays subcategories or the list of products.
     *
     * @param array $callback_query Complete Telegram callback data.
     * @return void
     */
    private function handle_category_callback($callback_query)
    {
        $chat_id    = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id'];
        $data       = $callback_query['data'];
        $parts       = explode('_', $data);
        $category_id = (int) $parts[1];
        $page        = (int) ($parts[3] ?? 1);
        $sub_categories_data = WASCT_API_Handler::get_woocommerce_categories(['parent' => $category_id, 'per_page' => WASCT_CATEGORIES_PER_PAGE, 'page' => $page]);
        if (! is_wp_error($sub_categories_data) && ! empty($sub_categories_data['terms'])) {
            $category_info_raw = WASCT_API_Handler::get_woocommerce_categories(['include' => [$category_id]]);
            $category_name     = (!is_wp_error($category_info_raw) && !empty($category_info_raw['terms'])) ? esc_html($category_info_raw['terms'][0]['name']) : __('Category', 'web-in-air-shop-connect-for-telegram');
            /* translators: %s: Category name. */
            $title             = sprintf(__("Subcategories of *%s* :", 'web-in-air-shop-connect-for-telegram'), $category_name);
            WASCT_Utils::send_category_list($chat_id, $sub_categories_data, $page, $title, $category_id, $message_id);
        } else {
         
            $this->handle_product_list_for_category($chat_id, $category_id, 1, $message_id);
        }
    }
    
    /**
     * Manages the display of a list of labels.
     *
     * @param array $callback_query Callback data.
     */
    private function handle_show_tags_callback($callback_query, $is_editing)
    {
        $chat_id = $callback_query['message']['chat']['id'];
        $data    = $callback_query['data'];
        $panel_id = $callback_query['message']['message_id']; 

        $logo_url = WASCT_Utils::get_logo_url();
        $result = WASCT_API_Handler::edit_telegram_message_media(
            $chat_id,
            $panel_id,
            $logo_url,
            __('🏷️ Loading tags, please wait....', 'web-in-air-shop-connect-for-telegram'),
            [[['text' => '⏳', 'callback_data' => 'noop']]]
        );
        if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return;
        }
        $page = 1;
        if (strpos($data, '_page_') !== false) {
            $parts = explode('_page_', $data);
            $page  = (int) end($parts);
        }
        $tags_data = WASCT_API_Handler::get_woocommerce_tags(['per_page' => WASCT_TAGS_PER_PAGE, 'page' => $page, 'orderby' => 'count', 'order' => 'desc']);
        
        WASCT_Utils::send_tag_list(
            $chat_id,
            $tags_data,
            $page,
            __('Here are our product tags:', 'web-in-air-shop-connect-for-telegram'),
            $panel_id
        );
    }
    
    /**
     * Manages the display of a list of generic taxonomies (used for brands).
     *
     * @param int    $chat_id         The conversation ID.
     * @param int    $message_id      The ID of the message to be modified.
     * @param string $taxonomy_slug   The taxonomy slug (e.g. “brands”).
     * @param string $title           The title to display above the list.
     * @param string $callback_prefix The prefix for the buttons (e.g. “brand_”).
     */
    private function handle_taxonomy_command($chat_id, $message_id, $taxonomy_slug, $title, $callback_prefix)
    {
        $panel_id = $message_id; 
        $logo_url = WASCT_Utils::get_logo_url();

        $result = WASCT_API_Handler::edit_telegram_message_media(
            $chat_id,
            $panel_id,
            $logo_url,
            __("Loading, please wait...", 'web-in-air-shop-connect-for-telegram'),
            [[['text' => '⏳', 'callback_data' => 'noop']]]
        );
        
        if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return;
        }
        
        $api_function = 'get_woocommerce_' . $taxonomy_slug;
        if (! method_exists('WASCT_API_Handler', $api_function)) {
            /* translators: %s: Taxonomy slug (e.g., "brands"). */
            $error_message = sprintf(__("Error: The “%s” feature does not exist.", 'web-in-air-shop-connect-for-telegram'), $taxonomy_slug);
            $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $error_message);
            $this->_handle_edit_failure($result, $chat_id, $panel_id);
            return;
        }

        $terms_data = WASCT_API_Handler::$api_function();
        if ( is_wp_error( $terms_data ) || empty( $terms_data['terms'] ) ) {
            /* translators: %s: Taxonomy slug (e.g., "brands"). */
            $error_message = sprintf( __( "Sorry, no items were found for “%s”.", 'web-in-air-shop-connect-for-telegram' ), $taxonomy_slug );
            $keyboard = [[['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu']]];
            $result = WASCT_API_Handler::edit_telegram_message_media( $chat_id, $panel_id, $logo_url, $error_message, $keyboard );
            $this->_handle_edit_failure($result, $chat_id, $panel_id);
            return;
        }

        $keyboard = [];
        foreach ($terms_data['terms'] as $term) {
            $keyboard[] = [['text' => esc_html($term['name']), 'callback_data' => $callback_prefix . $term['id'] . '_page_1']];
        }
        $keyboard[] = [['text' => __('↩️ Back', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu']];
        
        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $title, $keyboard);
        
        if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return;
        }
    }
    
    // =================================================================
    //  Detailed Product View Management
    // =================================================================

   /**
     * Displays the detailed view of a product (single or variable).
     * This is the main view when a user selects a product.
     *
     * @param int    $chat_id The conversation ID.
     * @param int    $message_id The ID of the message to be modified.
     * @param string $data The callback data (e.g. “details_123”).
     * @return void
     */
    private function handle_simple_product_view($chat_id, $message_id, $data) {
        delete_transient('wasct_variation_selection_' . $chat_id);
        
        $product_id = (int) str_replace('details_', '', $data);
        $context = get_transient('wasct_user_context_' . $chat_id);
        if (!$context) { $context = '_from_list_page_1'; }
    
        $product = WASCT_API_Handler::get_woocommerce_product_by_id($product_id);
    
        if (is_wp_error($product) || empty($product)) {
            return;
        }
    
        $short_description = WASCT_Utils::clean_and_truncate_description($product['short_description'] ?: $product['description'], 60);
        
        /* translators: 1: Product name, 2: Product short description. */
        $caption_template = __("✨ *Product details* ✨\n\n📦 *Name*: %1\$s\n\n%2\$s", 'web-in-air-shop-connect-for-telegram');
        $response_caption = sprintf(
            $caption_template,
            html_entity_decode($product['name']),
            $short_description
        );
    
        $image_url = '';
        if ( ! empty($product['images'][0]) ) {
            $image_url = WASCT_Utils::get_optimized_image_url($product['images'][0]);
        }
    
        $buttons = $this->build_simple_product_keyboard($product, $chat_id);
    
        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id, $image_url, $response_caption, $buttons);
        if ($this->_handle_edit_failure($result, $chat_id, $message_id)) {
            return;
        }
    }
    
    /**
     * BUILDS THE KEYBOARD FOR THE MAIN PRODUCT VIEW.
     * This keyboard is dynamic and displays variation options, gallery, etc.
     *
     * @param array  $product Product data.
     * @param string $context Navigation context for the ‘Back’ button.
     * @return array Formatted keyboard.
     */
    private function build_simple_product_keyboard($product, $chat_id) {
        $buttons = [];
        $product_id = $product['id'];
    
        if ($product['type'] !== 'variable') {
            $price_display = WASCT_Utils::format_product_price( $product );
            $buttons[] = [['text' => '💰 ' . $price_display, 'callback_data' => 'noop']];
        }
    
        if ($product['type'] === 'variable' && !empty($product['attributes'])) {
            $options = wasct_get_settings();
            $columns = (int) ($options['variation_columns'] ?? 3);
            $user_selection = get_transient('wasct_variation_selection_' . $chat_id) ?? [];
            
            $variation_attributes = array_filter($product['attributes'], function($attr) {
                return $attr['variation'];
            });
    
            $next_attribute_to_ask = null;
            foreach ($variation_attributes as $attribute) {
                if (!isset($user_selection[$attribute['name']])) {
                    $next_attribute_to_ask = $attribute;
                    break;
                }
            }
    
            if ($next_attribute_to_ask) {
                /* translators: %s: Attribute name (e.g., "Color", "Size"). */
                $buttons[] = [['text' => '⤵️ ' . sprintf(__("Select a %s", 'web-in-air-shop-connect-for-telegram'), $next_attribute_to_ask['name']), 'callback_data' => 'noop']];
                
                $option_buttons = [];
                foreach ($next_attribute_to_ask['options'] as $option) {
                    $callback_data = 'select_attr_' . $product_id . '__' . urlencode($next_attribute_to_ask['name']) . '_' . urlencode($option);
                    $option_buttons[] = ['text' => $option, 'callback_data' => $callback_data];
                }
                
                foreach (array_chunk($option_buttons, $columns) as $row) {
                    $buttons[] = $row;
                }
            }
        }
    
        $full_description_text = trim(wp_strip_all_tags($product['description']));
        $short_description_text = trim(wp_strip_all_tags($product['short_description']));
        $is_long_desc = !empty($full_description_text) || !empty($short_description_text);
        $has_gallery = count($product['images']) > 1;
    
        $action_row = [];
        if ($is_long_desc) {
            $action_row[] = ['text' => __('📄 See description', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => "nav_{$product_id}_desc_0"];
        }
        if ($has_gallery) {
            $action_row[] = ['text' => __('🖼️ Photo gallery', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => "nav_{$product_id}_img_1"];
        }
        
        if (!empty($action_row)) {
            $buttons[] = $action_row;
        }
    
        $buttons[] = [
            ['text' => __('⬅️ Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu'],
            ['text' => __('↩️ Back', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_list']
        ];
    
        return $buttons;
    }
    
    /**
     * Manages the selection of an attribute option for a variable product.
     * Updates the status of the user's selection and refreshes the product view.
     *
     * @param array $callback_query The complete data from the Telegram callback.
     * @return void
     */
    private function handle_attribute_selection_callback($callback_query)
    {
        $this->acknowledge_callback_query($callback_query['id']);
        $this->handle_attribute_selection($callback_query['message']['chat']['id'], $callback_query['message']['message_id'], $callback_query['data'], $callback_query);
    }
    
    /**
     * Manages the main logic for selecting a variation attribute.
     * Updates the user's selection transient and determines whether the selection is complete.
     *
     * @param int    $chat_id The conversation ID.
     * @param int    $message_id The ID of the message to be modified.
     * @param string $data The callback data (e.g. “select_attr_123__Colour_Blue”).
     * @param array  $callback_query The complete Telegram callback data.
     * @return void
     */
    private function handle_attribute_selection($chat_id, $message_id, $data, $callback_query)
    {
        $parts = explode('__', $data, 2);
        $product_id_part = str_replace('select_attr_', '', $parts[0]);
        $product_id = (int) $product_id_part;
        if ($product_id === 0) { return; }
    
        $transient_key = 'wasct_variation_selection_' . $chat_id;
        $user_selection = get_transient($transient_key) ?? [];
    
        if (isset($user_selection['product_id']) && $user_selection['product_id'] != $product_id) {
            $user_selection = [];
        }
    
        $new_selection_part = $parts[1] ?? '';
        if (!empty($new_selection_part)) {
            $selection_parts = explode('_', $new_selection_part);
            $attr_name = urldecode($selection_parts[0]);
            $attr_option = urldecode($selection_parts[1]);
            $user_selection[$attr_name] = $attr_option;
        }
        $user_selection['product_id'] = $product_id;
        set_transient($transient_key, $user_selection, 15 * MINUTE_IN_SECONDS);
    
        $product = WASCT_API_Handler::get_woocommerce_product_by_id($product_id);
        if (is_wp_error($product)) { return; }
    
        $variation_attributes = array_filter($product['attributes'], function($attr) { return $attr['variation']; });
        $selection_complete = count($user_selection) - 1 >= count($variation_attributes);
    
        $photo_url = WASCT_Utils::get_optimized_image_url($product['images'][0] ?? []);
        
        /* translators: %s: Product name. */
        $caption_template = __("✨ *Product details* ✨\n\n📦 *Name* : %s\n\n", 'web-in-air-shop-connect-for-telegram');
        $caption = sprintf($caption_template, html_entity_decode($product['name']));
        $caption .= __("Your selections:\n", 'web-in-air-shop-connect-for-telegram');
        foreach($user_selection as $key => $value) {
            if ($key === 'product_id') continue;
            $caption .= "• *{$key}* : {$value}\n";
        }
    
        $keyboard = [];
    
        if ($selection_complete) {
            $final_variation = null;
            $all_variations = WASCT_API_Handler::get_woocommerce_product_variations($product_id);
            foreach ($all_variations as $variation) {
                $temp_attrs = [];
                foreach ($variation['attributes'] as $attr) {
                    if (!empty($attr['option'])) { $temp_attrs[$attr['name']] = $attr['option']; }
                }
                $is_match = true;
                foreach ($user_selection as $name => $value) {
                    if ($name === 'product_id') continue;
                    if (!isset($temp_attrs[$name]) || strtolower($temp_attrs[$name]) !== strtolower($value)) {
                        $is_match = false;
                        break;
                    }
                }
                if ($is_match) {
                    $final_variation = $variation;
                    break;
                }
            }
    
            if ($final_variation) {
                $photo_url = WASCT_Utils::get_optimized_image_url($final_variation['image'] ?? $product['images'][0] ?? []);
                $price_display = WASCT_Utils::format_product_price($final_variation);
                $keyboard[] = [['text' => '💰 ' . $price_display, 'callback_data' => 'noop']];
                
                $full_description_text = trim(wp_strip_all_tags($product['description']));
                $short_description_text = trim(wp_strip_all_tags($product['short_description']));
                $is_long_desc = !empty($full_description_text) || !empty($short_description_text);
                $has_gallery = count($product['images']) > 1;
    
                $action_row = [];
                if ($is_long_desc) {
                    $action_row[] = ['text' => __('📄 See description', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => "nav_{$product_id}_desc_0"];
                }
                if ($has_gallery) {
                    $action_row[] = ['text' => __('🖼️ Photo gallery', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => "nav_{$product_id}_img_1"];
                }
                if (!empty($action_row)) {
                    $keyboard[] = $action_row;
                }
    
                $keyboard[] = [
                    ['text' => __('⬅️ Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu'],
                    ['text' => '↩️ ' . __('Back', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_product_' . $product_id]
                ];
                
            } else {
                $caption .= "\n" . __( "❌ This combination is not available.", 'web-in-air-shop-connect-for-telegram' );
                $keyboard[] = [
                    ['text' => __('✍️ Change selection', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_product_' . $product_id]
                ];
                $keyboard[] = [
                    ['text' => __('⬅️ Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu']
                ];
            }
    
        } else {
            $keyboard = $this->build_simple_product_keyboard($product, $chat_id);
        }
    
        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id, $photo_url, $caption, $keyboard);
        
        if ($this->_handle_edit_failure($result, $chat_id, $message_id)) {
            return;
        }
    }
    
    
    
    
    /**
     * Builds the keyboard for the detailed view of a fully selected product variation.
     *
     * @param array  $product Parent product data.
     * @param array  $variation Specific variation data.
     * @param string $context Navigation context for the ‘Back’ button.
     * @param int    $quantity Currently selected quantity.
     * @param bool   $item_added Indicates whether the item has just been added to the basket.
     * @return array The keyboard formatted for the Telegram API.
     */
        private function build_variation_details_keyboard($product, $variation, $context, $quantity = 1, $item_added = false)
    {
        $product_id   = $product['id'];
        $buttons      = [];
        
        $price_display = WASCT_Utils::format_product_price( $variation );
        $buttons[] = [['text' => '💰 ' . $price_display, 'callback_data' => 'noop']];

        $buttons[] = [
            ['text' => __('⬅️ Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu'],
            ['text' => '↩️ ' . __('Back', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_product_' . $product_id]
        ];
    
        return $buttons;
    }
    
    /**
     * Manages pagination for product lists (all products, search).
     *
     * @param array  $callback_query Complete Telegram callback data.
     * @param string $command_prefix Callback prefix for building pagination links.
     * @return void
     */
    private function handle_products_pagination_callback($callback_query, $command_prefix)
    {
        $chat_id    = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id'];
        $data       = $callback_query['data'];
        
        $parts        = explode('_', $data);
        $current_page = (int) end($parts);

        $search_query = '';
        $search_hash = ''; 

        if ($command_prefix === 'search_page') {
            if (count($parts) >= 4) {
                $search_hash = $parts[count($parts) - 2];
                $search_query = get_transient('wasct_search_' . $search_hash);
            }
        }
        $options  = wasct_get_settings();
        $mode     = $options['product_display_mode'] ?? 'text';
        $per_page = ($mode === 'visual') ? 1 : WASCT_PRODUCTS_PER_PAGE;

        $args          = ['per_page' => $per_page, 'page' => $current_page, 'search' => $search_query];
        $products_data = WASCT_API_Handler::get_woocommerce_products($args);
        
        /* translators: %s: The user's search query. */
        $intro = ($command_prefix === 'search_page' && $search_query) ? sprintf(__("Results for “%s”:\n", 'web-in-air-shop-connect-for-telegram'), esc_html($search_query)) : __("These are our products:\n", 'web-in-air-shop-connect-for-telegram');
        
        if ($mode === 'visual') {
            WASCT_Utils::send_visual_product_item($chat_id, $products_data, $current_page, $command_prefix, $search_hash, '', '', '', $intro, $message_id);
        } else {
            WASCT_Utils::send_paginated_product_list($chat_id, $products_data, $current_page, $command_prefix, $search_hash, '', '', '', $intro, $message_id);
        }
    }
    
    /**
     * Processes a product search initiated by the user.
     *
     * @param int    $chat_id The conversation ID.
     * @param string $search_query The search term sent by the user.
     * @return void
     */
    private function handle_search_command($chat_id, $search_query)
    {
        $panel_id = $this->get_panel_message_id($chat_id);
        if (! $panel_id) { return; }

        $search_hash = md5($search_query);
        set_transient('wasct_search_' . $search_hash, $search_query, HOUR_IN_SECONDS);
        
        /* translators: %s: The user's search query. */
        $loading_message = sprintf(__("🔍 Searching for “%s”, please wait...", 'web-in-air-shop-connect-for-telegram'), esc_html($search_query));
        $logo_url        = WASCT_Utils::get_logo_url();
        
        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $loading_message, [[['text' => '⏳', 'callback_data' => 'noop']]]);
            if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
                return;
            }
        $options  = wasct_get_settings();
        $mode     = $options['product_display_mode'] ?? 'text';
        $per_page = ($mode === 'visual') ? 1 : WASCT_PRODUCTS_PER_PAGE;
        $products_data = WASCT_API_Handler::get_woocommerce_products(['search' => $search_query, 'per_page' => $per_page, 'page' => 1]);
        
        /* translators: %s: The user's search query. */
        $intro = sprintf(__("Results for “%s”:\n", 'web-in-air-shop-connect-for-telegram'), esc_html($search_query));
        if ($mode === 'visual') {
            WASCT_Utils::send_visual_product_item($chat_id, $products_data, 1, 'search_page', $search_hash, '', '', '', $intro, $panel_id);
        } else {
            WASCT_Utils::send_paginated_product_list($chat_id, $products_data, 1, 'search_page', $search_hash, '', '', '', $intro, $panel_id);
        }
    }
    
    /**
     * Manages the display of a list of products for a generic taxonomy (labels, brands).
     *
     * @param array $callback_query The complete data for the Telegram callback.
     * @param array $taxonomy_config A configuration array describing the taxonomy.
     * @return void
     */
    private function handle_taxonomy_product_list($callback_query, $taxonomy_config) {
        $chat_id = $callback_query['message']['chat']['id'];
        $data    = $callback_query['data'];
        $message_id_to_edit = $callback_query['message']['message_id'] ?? null;
    
        if ($message_id_to_edit) {
            /* translators: %s: An emoji. */
            $loading_message = sprintf(__("%s Loading products...", 'web-in-air-shop-connect-for-telegram'), $taxonomy_config['emoji']);
            $logo_url = WASCT_Utils::get_logo_url();
            $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id_to_edit, $logo_url, $loading_message, [[['text' => '⏳', 'callback_data' => 'noop']]]);
            if ($this->_handle_edit_failure($result, $chat_id, $message_id_to_edit)) {
                return;
            }
        }
    
        $parts        = explode('_', $data);
        $term_id      = (int) $parts[1];
        $current_page = (isset($parts[3])) ? (int) $parts[3] : 1;
        
        $options  = wasct_get_settings();
        $mode     = $options['product_display_mode'] ?? 'text';
        $per_page = ($mode === 'visual') ? 1 : WASCT_PRODUCTS_PER_PAGE;
    
        $api_function = 'get_woocommerce_' . $taxonomy_config['api_slug_plural'];
        $terms_data_raw = WASCT_API_Handler::$api_function(['include' => [$term_id]]);
        $term_name = (!is_wp_error($terms_data_raw) && !empty($terms_data_raw['terms'])) ? esc_html($terms_data_raw['terms'][0]['name']) : $taxonomy_config['name'];
        
        $products_data = WASCT_API_Handler::get_woocommerce_products([
            $taxonomy_config['api_param'] => $term_id, 
            'per_page' => $per_page, 
            'page' => $current_page
        ]);
        
        $command_prefix = $taxonomy_config['slug'] . '_' . $term_id . '_page';
        $intro          = sprintf($taxonomy_config['intro_text'], $term_name);
    
        $send_function = ($mode === 'visual') ? 'send_visual_product_item' : 'send_paginated_product_list';
        
        WASCT_Utils::$send_function(
            $chat_id, 
            $products_data, 
            $current_page, 
            $command_prefix, 
            '',
            '',
            ($taxonomy_config['slug'] === 'tag') ? $term_id : '', 
            ($taxonomy_config['slug'] === 'brand') ? $term_id : '',
            $intro, 
            $message_id_to_edit
        );
    }

    /**
     * Builds the complete navigation keyboard for a product sheet.
     * Manages the variation, photo gallery, and description pagination buttons.
     */
    private function build_product_navigation_keyboard($product, $context, $img_idx, $desc_page)
    {
        $product_id = $product['id'];
        $buttons = [];
    
        if ($img_idx !== null) {
            $total_images = count($product['images']);
            if ($total_images > 1) {
                $gallery_row = [];
                if ($img_idx > 0) {
                    $gallery_row[] = ['text' => '◀️ ' . __('Previous', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => "nav_{$product_id}_img_" . ($img_idx - 1)];
                }
                $gallery_row[] = ['text' => '🖼️ ' . ($img_idx + 1) . '/' . $total_images . ' 🖼️', 'callback_data' => 'noop'];
                if ($img_idx < $total_images - 1) {
                    $gallery_row[] = ['text' => __('Next', 'web-in-air-shop-connect-for-telegram') . ' ▶️', 'callback_data' => "nav_{$product_id}_img_" . ($img_idx + 1)];
                }
                $buttons[] = $gallery_row;
            }
        } elseif ($desc_page !== null) {
            $description_pages = WASCT_Utils::paginate_text_by_sentence($product['description'], 900);
            $total_desc_pages = count($description_pages);
    
            if ($total_desc_pages > 1) {
                $desc_row = [];
                $next_desc_page = $desc_page + 1;
    
                if ($desc_page > 0) {
                    $desc_row[] = ['text' => '🔼 ' . __('Start', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => "nav_{$product_id}_desc_0"];
                }
                if ($desc_page < $total_desc_pages - 1) {
                    $desc_row[] = ['text' => __('See more', 'web-in-air-shop-connect-for-telegram') . ' 🔽', 'callback_data' => "nav_{$product_id}_desc_{$next_desc_page}"];
                }
                if (!empty($desc_row)) {
                    $buttons[] = $desc_row;
                }
            }
        }
    
        $buttons[] = [
            ['text' => __('⬅️ Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu'],
            ['text' => __('↩️ Back', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'details_' . $product_id]
        ];
    
        return $buttons;
    }
    
    /**
     * Manages the display of the product list for a specific category.
     *
     * @param int      $chat_id The conversation ID.
     * @param int      $category_id The ID of the product category to display.
     * @param int      $current_page The page number of the list to display.
     * @param int|null $message_id_to_edit The ID of the message to edit, if available.
     * @return void
     */
    private function handle_product_list_for_category($chat_id, $category_id, $current_page = 1, $message_id_to_edit = null)
    {
        $options  = wasct_get_settings();
        $mode     = $options['product_display_mode'] ?? 'text';
        $per_page = ($mode === 'visual') ? 1 : WASCT_PRODUCTS_PER_PAGE;
        $products_data  = WASCT_API_Handler::get_woocommerce_products(['category' => $category_id, 'per_page' => $per_page, 'page' => $current_page]);
        $categories_raw = WASCT_API_Handler::get_woocommerce_categories(['include' => [$category_id]]);
        $category_name  = (!is_wp_error($categories_raw) && !empty($categories_raw['terms'])) ? esc_html($categories_raw['terms'][0]['name']) : __('Category', 'web-in-air-shop-connect-for-telegram');
        $command_prefix = 'view_products_cat_' . $category_id . '_page';
        /* translators: %s: Category name. */
        $intro          = sprintf(__("Products in category *%s*:\n", 'web-in-air-shop-connect-for-telegram'), $category_name);
        if ($mode === 'visual') {
            WASCT_Utils::send_visual_product_item($chat_id, $products_data, $current_page, $command_prefix, '', $category_id, '', '', $intro, $message_id_to_edit);
        } else {
            WASCT_Utils::send_paginated_product_list($chat_id, $products_data, $current_page, $command_prefix, '', $category_id, '', '', $intro, $message_id_to_edit);
        }
    }

    /**
     * Handles the callback to return to the product page from a variation.
     * @param array $callback_query Callback data.
     */
    private function handle_back_to_product_callback($callback_query)
    {
        $chat_id    = $callback_query['message']['chat']['id'];
        $data       = $callback_query['data'];
        $message_id = $callback_query['message']['message_id'];
        
        $product_id = (int) str_replace('back_to_product_', '', $data);
    
        if ($product_id > 0) {
            $this->handle_simple_product_view($chat_id, $message_id, 'details_' . $product_id);
        }
    }

    /**
     * Handles the callback to display products in a category.
     * @param array $callback_query Callback data.
     */
    private function handle_view_products_for_category_callback($callback_query)
    {
        $chat_id    = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id']; 
        $data       = $callback_query['data'];
        $parts       = explode('_', $data);
        $category_id = (int) ($parts[3] ?? 0);
        $page        = (int) ($parts[5] ?? 1);
        $this->handle_product_list_for_category($chat_id, $category_id, $page, $message_id);
    }
    
    /**
     * Handles the callback to display the list of all products.
     * @param int $chat_id Chat ID.
     */
    private function handle_show_products_callback($chat_id, $message_id_to_edit, $page = 1)
    {
        if ($message_id_to_edit) {
            $logo_url = WASCT_Utils::get_logo_url();
            WASCT_API_Handler::edit_telegram_message_media(
                $chat_id,
                $message_id_to_edit,
                $logo_url,
                __('📦 Loading products, please wait...', 'web-in-air-shop-connect-for-telegram'), 
                [[['text' => '⏳', 'callback_data' => 'noop']]]
            );
        }
        
        $options       = wasct_get_settings();
        $mode          = $options['product_display_mode'] ?? 'text';
        $per_page      = ($mode === 'visual') ? 1 : WASCT_PRODUCTS_PER_PAGE;
        
        $products_data = WASCT_API_Handler::get_woocommerce_products([ 'per_page' => $per_page, 'page' => $page ]);

        if ($mode === 'visual') {
            WASCT_Utils::send_visual_product_item($chat_id, $products_data, $page, 'products_page', '', '', '', '', __("Here are our products:\n", 'web-in-air-shop-connect-for-telegram'), $message_id_to_edit);
        } else {
            WASCT_Utils::send_paginated_product_list($chat_id, $products_data, $page, 'products_page', '', '', '', '', __("Here are our products:\n", 'web-in-air-shop-connect-for-telegram'), $message_id_to_edit);
        }
    }
    
    /**
    * Manages the return from a product page to the corresponding product list.
    */
    private function handle_back_to_list_from_details_callback($callback_query)
    {
        $chat_id    = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id']; 
        
        $context = get_transient('wasct_user_context_' . $chat_id);
        if (!$context) {
            WASCT_API_Handler::delete_telegram_message($chat_id, $message_id);
            $this->send_menu_only($chat_id);
            return;
        }

        $logo_url = WASCT_Utils::get_logo_url();
        WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id, $logo_url, __('Back to list...', 'web-in-air-shop-connect-for-telegram'), [[['text' => '⏳', 'callback_data' => 'noop']]]);

        if (strpos($context, '_from_cat_') !== false) {
            preg_match('/_from_cat_(\d+)_page_(\d+)/', $context, $matches);
            $category_id = (int) ($matches[1] ?? 0);
            $page = (int) ($matches[2] ?? 1);
            $this->handle_product_list_for_category($chat_id, $category_id, $page, $message_id);

        } elseif (strpos($context, '_from_tag_') !== false) {
            preg_match('/_from_tag_(\d+)_page_(\d+)/', $context, $matches);
            $tag_id = (int) ($matches[1] ?? 0);
            $page = (int) ($matches[2] ?? 1);
            $fake_callback_query = ['data' => 'tag_' . $tag_id . '_page_' . $page, 'message' => ['chat' => ['id' => $chat_id], 'message_id' => $message_id]];
            $this->handle_taxonomy_product_list($fake_callback_query, [
                'slug' => 'tag', 'name' => __('Tag', 'web-in-air-shop-connect-for-telegram'), 'emoji' => '🏷️',
                'api_slug_plural' => 'tags', 'api_param' => 'tag',
                /* translators: %s: The name of the tag. */
                'intro_text' => __("Products with the tag *%s* :\n", 'web-in-air-shop-connect-for-telegram')
            ]);

        } elseif (strpos($context, '_from_brand_') !== false) {
            preg_match('/_from_brand_(\d+)_page_(\d+)/', $context, $matches);
            $brand_id = (int) ($matches[1] ?? 0);
            $page = (int) ($matches[2] ?? 1);
            $fake_callback_query = ['data' => 'brand_' . $brand_id . '_page_' . $page, 'message' => ['chat' => ['id' => $chat_id], 'message_id' => $message_id]];
            
            $this->handle_taxonomy_product_list($fake_callback_query, [
                'slug' => 'brand', 'name' => __('Marque', 'web-in-air-shop-connect-for-telegram'), 'emoji' => '™️',
                'api_slug_plural' => 'brands', 'api_param' => 'brand',
                /* translators: %s: The name of the brand. */
                'intro_text' => __("Products from the brand *%s*:\n", 'web-in-air-shop-connect-for-telegram')
            ]);

        } elseif (strpos($context, '_from_search_hash_') !== false) {
            preg_match('/_from_search_hash_([a-f0-9]+)/', $context, $matches);
            $search_hash = $matches[1] ?? '';
            $real_search_query = get_transient('wasct_search_' . $search_hash);
            if ($real_search_query) {
                $this->handle_search_command($chat_id, $real_search_query);
            } else {
                $this->handle_show_products_callback($chat_id, $message_id);
            }
        } else {
            preg_match('/_from_list_page_(\d+)/', $context, $matches);
            $page = (int) ($matches[1] ?? 1);
            $this->handle_show_products_callback($chat_id, $message_id, $page);
        }
    }
    
    /**
     * Handles the cancellation of a variation selection and returns to the main product view.
     *
     * @param array $callback_query The complete Telegram callback data.
     * @return void
     */
    private function handle_cancel_selection_callback($callback_query)
    {
        $chat_id    = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id']; 
        $data       = $callback_query['data'];

        $product_id = (int) str_replace('cancel_selection_and_show_product_', '', $data);
        
        if ($product_id > 0) {
            $context = get_transient('wasct_user_context_' . $chat_id);
            if (!$context) { $context = '_from_list_page_1'; }

            $simple_view_data = 'details_' . $product_id . '_from_' . $context;
            $this->handle_simple_product_view($chat_id, $message_id, $simple_view_data);
        }
        $this->acknowledge_callback_query($callback_query['id']);
    }
    
    /**
     * Displays contact options or redirects to support.
     *
     * @param int $chat_id The conversation ID.
     * @return void
     */
    private function handle_contact_support($chat_id)
    {
        $panel_id = $this->get_panel_message_id($chat_id);
        if (! $panel_id) {
            $this->send_menu_only($chat_id);
            return;
        }
        
        $options          = wasct_get_settings();
        $support_username = $options['support_username'] ?? '';
        $logo_url         = WASCT_Utils::get_logo_url();

        if (empty($support_username)) {
            $error_message = __("Sorry, the contact service is not configured at the moment.", 'web-in-air-shop-connect-for-telegram');
            $keyboard      = [[['text' => '↩️ ' . __('Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu']]];
            $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $error_message, $keyboard);
            if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
                return;
            }
        }

        $message_text = __("To contact our team, please click the button below to start a private conversation.", 'web-in-air-shop-connect-for-telegram');
        $support_url  = 'https://t.me/' . ltrim($support_username, '@'); 
        $buttons = [
            [['text' => '💬 ' . __('Start the conversation', 'web-in-air-shop-connect-for-telegram'), 'url' => $support_url]],
            [['text' => '↩️ ' . __('Menu', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu']]
        ];

        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $message_text, $buttons);
        if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return;
        }
        return;
    }
    
    /**
     * Displays a message prompting the user to enter their search term.
     * Sets the user to “awaiting_search_query” status.
     *
     * @param int $chat_id The conversation ID.
     * @return void
     */
    private function handle_search_prompt($chat_id)
    {
        $panel_id = $this->get_panel_message_id($chat_id);
        if (! $panel_id) {
            $this->send_menu_only($chat_id);
            return;
        }

        $message_text = __("🔍 Please enter your search keyword:", 'web-in-air-shop-connect-for-telegram');
        $keyboard     = [[['text' => '↩️ ' . __('Cancel search', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'cancel_search']]];
        $logo_url     = WASCT_Utils::get_logo_url();

        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $message_text, $keyboard);
        if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return;
        }
        set_transient('wasct_user_state_' . $chat_id, 'awaiting_search_query', 5 * MINUTE_IN_SECONDS);
    }
    
    /**
     * Handles search cancellation and returns to the main menu.
     *
     * @param array $callback_query Complete Telegram callback data.
     * @return void
     */
    private function handle_cancel_search($callback_query)
    {
        $chat_id  = $callback_query['message']['chat']['id'];
        $panel_id = $callback_query['message']['message_id'];
        delete_transient('wasct_user_state_' . $chat_id);

        $menu_text = __("What can I do for you?", 'web-in-air-shop-connect-for-telegram');
        $buttons   = $this->_build_main_menu_keyboard();
        $logo_url  = WASCT_Utils::get_logo_url();

        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $panel_id, $logo_url, $menu_text, $buttons);
        if ($this->_handle_edit_failure($result, $chat_id, $panel_id)) {
            return;
        }
        $this->acknowledge_callback_query($callback_query['id']);
    }

    /**
     * Manages navigation within a product sheet (image gallery, description pagination).
     *
     * @param array $callback_query Complete Telegram callback data.
     * @return void
     */
    private function handle_navigation_callback($callback_query, $callback_query_id = null) {
        $chat_id = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id'];
        $data = $callback_query['data'];
    
        $product = null;
        $image_url = '';
        $response_caption = '';
        $buttons = [];
        $context = get_transient('wasct_user_context_' . $chat_id) ?: '_from_list_page_1';
    
        if (strpos($data, '_desc_') !== false) {
            preg_match('/nav_(\d+)_desc_(\d+)/', $data, $matches);
            if (empty($matches)) return;
    
            $product_id = (int) $matches[1];
            $desc_page = (int) $matches[2];
    
            $product = WASCT_API_Handler::get_woocommerce_product_by_id($product_id);
            if (is_wp_error($product) || empty($product)) return;
    
            if (!empty($product['images'][0])) {
                $image_url = WASCT_Utils::get_optimized_image_url($product['images'][0]);
            }
    
            $title = "✨ *Product description* ✨\n\n";         
            $title .= "📦 *Name* : " . html_entity_decode($product['name']) . "\n\n";
            
            $cleaned_short_desc = WASCT_Utils::clean_full_description($product['short_description']);
            $cleaned_long_desc = WASCT_Utils::clean_full_description($product['description']);
            
            $full_description = $cleaned_short_desc;
            if (!empty($cleaned_short_desc) && !empty($cleaned_long_desc)) {
                $full_description .= "\n\n*⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯⎯*\n\n";
            }
            $full_description .= $cleaned_long_desc;
            
            $description_pages = WASCT_Utils::paginate_text_by_sentence($full_description, 900);
            $caption_text = $description_pages[$desc_page] ?? '';
            $response_caption = $title . $caption_text;
    
            $buttons = $this->build_product_navigation_keyboard($product, $context, null, $desc_page);
    
        } elseif (strpos($data, '_img_') !== false) {
            preg_match('/nav_(\d+)_img_(\d+)/', $data, $matches);
            if (empty($matches)) return;
    
            $product_id = (int) $matches[1];
            $img_idx = (int) $matches[2];
    
            $product = WASCT_API_Handler::get_woocommerce_product_by_id($product_id);
            if (is_wp_error($product) || empty($product)) return;
    
            $response_caption = "📦 *Name* : " . html_entity_decode($product['name']);
    
            if (isset($product['images'][$img_idx])) {
                $image_url = WASCT_Utils::get_optimized_image_url($product['images'][$img_idx]);
            }
    
            $buttons = $this->build_product_navigation_keyboard($product, $context, $img_idx, null);
        }
    
        if ($product) {
            $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id, $image_url, $response_caption, $buttons);
            if ($this->_handle_edit_failure($result, $chat_id, $message_id)) {
                return;
            }
        }
    
        if ($callback_query_id) {
            $this->acknowledge_callback_query($callback_query_id);
        }
    }
    
    
    /**
     * Displays the languages available to the user.
     */
    private function handle_select_language($callback_query) {
        $chat_id = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id'];
        $logo_url = WASCT_Utils::get_logo_url();
        $options = wasct_get_settings();
    
        $admin_enabled_languages = $options['enabled_languages'] ?? ['fr'];
        $current_user_lang = get_transient('wasct_user_lang_' . $chat_id);
    
        $flag_emojis = apply_filters('wasct_language_flag_emojis', $default_emojis);
    
        $all_site_languages = WASCT_Utils::get_site_languages();
        $keyboard = [];
    
        foreach ($all_site_languages as $lang_code => $lang_data) {
            if (in_array($lang_code, $admin_enabled_languages) && $lang_code !== $current_user_lang) {
                $lang_to_country_map = [
                    'en' => 'gb', 
                    'el' => 'gr', 
                    'zh' => 'cn', 
                    'ja' => 'jp', 
                    'ko' => 'kr',
                    'da' => 'dk', 
                    'cs' => 'cz', 
                    'et' => 'ee',
                ];
                
                $country_code_for_flag = $lang_to_country_map[$lang_code] ?? $lang_code;
                
                $flag = WASCT_Utils::get_flag_emoji_from_country_code($country_code_for_flag);
                $keyboard[] = [['text' => $flag . ' ' . $lang_data['name'], 'callback_data' => 'set_lang_' . $lang_code]];
            }
        }
    
        $keyboard[] = [['text' => __('↩️ Back', 'web-in-air-shop-connect-for-telegram'), 'callback_data' => 'back_to_main_menu']];
        
        $text = __('Please select your language:', 'web-in-air-shop-connect-for-telegram');
    
        if (count($keyboard) <= 1) { 
            $text = __("No other languages are available at this time.", 'web-in-air-shop-connect-for-telegram');
        }
        
        $result = WASCT_API_Handler::edit_telegram_message_media($chat_id, $message_id, $logo_url, $text, $keyboard);
        if ($this->_handle_edit_failure($result, $chat_id, $message_id)) {
            return;
        }
    }

    /**
     * Saves the language selected by the user and refreshes the menu.
     */
    private function handle_set_language($callback_query) {
        $chat_id = $callback_query['message']['chat']['id'];
        $message_id = $callback_query['message']['message_id'];
        $data = $callback_query['data'];
        $lang_code = str_replace('set_lang_', '', $data);
    
        set_transient('wasct_user_lang_' . $chat_id, $lang_code, YEAR_IN_SECONDS);
    
        $all_languages = WASCT_Utils::get_site_languages();
        $locale_to_use = $all_languages[$lang_code]['locale'] ?? get_locale();
    
        switch_to_locale($locale_to_use);
        
        $this->acknowledge_callback_query($callback_query['id'], __('Language updated.', 'web-in-air-shop-connect-for-telegram'));
        $this->send_menu_only($chat_id, $message_id);
    }
}