<?php
/**
 * Utility functions for the Web-in-Air: Shop Connect for Telegram
 *
 */
class WASCT_Utils {
    /**
     * Formats and sends the paginated product list.
     *
     * @param int    $chat_id              Telegram chat ID.
     * @param array  $products_data        Product and pagination data.
     * @param int    $current_page         Current page.
     * @param string $command_prefix       Prefix for pagination callbacks.
     * @param string $query_param          Search parameter (if applicable).
     * @param int    $category_id          Category ID (if applicable).
     * @param int    $tag_id               Tag ID (if applicable).
     * @param int    $brand_id             Brand ID (if applicable).
     * @param string $intro_message_prefix Introductory message.
     * @param int|null $message_id_to_edit ID of the message to edit (if applicable).
     */
    public static function send_paginated_product_list( $chat_id, $products_data, $current_page, $command_prefix, $query_param = '', $category_id = '', $tag_id = '', $brand_id = '', $intro_message_prefix = "", $message_id_to_edit = null ) {
    
        $product_list_keyboard = [];
    
        if ( is_wp_error( $products_data ) || empty( $products_data['products'] ) ) {
            $message = __( "Sorry, no products match your search.", 'web-in-air-shop-connect-for-telegram' );
            $keyboard = [
                [['text' => '🔍 ' . __( 'New search', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'prompt_search']],
                [['text' => '↩️ ' . __( 'Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu']]
            ];
            $logo_url = self::get_logo_url();
            if ( $message_id_to_edit ) {
                WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $logo_url, $message, $keyboard );
            } else {
                WASCT_API_Handler::send_telegram_photo( $chat_id, $logo_url, $message, $keyboard );
            }
            return;
        }
        
        $products       = $products_data['products'];
        $total_pages    = $products_data['total_pages'];
        $total_products = $products_data['total_products'];
    
        $response_message = $intro_message_prefix;
        /* translators: 1: Current page number, 2: Total number of pages, 3: Total number of products. */
        $response_message .= sprintf( __( 'Page %1$d/%2$d (Total: %3$d products).', 'web-in-air-shop-connect-for-telegram' ), $current_page, $total_pages, $total_products ) . "\n\n";
        $response_message .= __( "Click on a product for details.", 'web-in-air-shop-connect-for-telegram' );
        
        $callback_context = '_from_list_page_' . $current_page;
        if ( ! empty( $category_id ) ) { $callback_context = '_from_cat_' . $category_id . '_page_' . $current_page; }
        elseif ( ! empty( $tag_id ) ) { $callback_context = '_from_tag_' . $tag_id . '_page_' . $current_page; }
        elseif ( ! empty( $brand_id ) ) { $callback_context = '_from_brand_' . $brand_id . '_page_' . $current_page; }
        elseif ( ! empty( $query_param ) ) { $callback_context = '_from_search_hash_' . $query_param; }
        set_transient('wasct_user_context_' . $chat_id, $callback_context, HOUR_IN_SECONDS);
    
        foreach ( $products as $product ) {
            $product_list_keyboard[] = [['text' => html_entity_decode( $product['name'] ), 'callback_data' => 'details_' . $product['id']]];
        }
        
        $pagination_buttons = [];
        $pagination_callback_base = $command_prefix;
        if ($command_prefix === 'search_page' && !empty($query_param)) {
            $pagination_callback_base .= '_' . $query_param;
        }
    
        if ( $current_page > 1 ) {
            $pagination_buttons[] = ['text' => __( '◀️ Previous', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $pagination_callback_base . '_' . ($current_page - 1)];
        }
        if ( $current_page < $total_pages ) {
            $pagination_buttons[] = ['text' => __( 'Next ▶️', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $pagination_callback_base . '_' . ($current_page + 1)];
        }
        if ( ! empty( $pagination_buttons ) ) { 
            $product_list_keyboard[] = $pagination_buttons; 
        }
        
        $back_button_row = [];
        if ( ! empty( $category_id ) ) {
            $current_cat_info_raw = WASCT_API_Handler::get_woocommerce_categories(['include' => [$category_id]]);
            $back_callback_data = 'show_categories_page_1'; 
            if (!is_wp_error($current_cat_info_raw) && !empty($current_cat_info_raw['terms'])) {
                $parent_id = $current_cat_info_raw['terms'][0]['parent'];
                if ($parent_id > 0) {
                    $back_callback_data = 'category_' . $parent_id . '_page_1'; 
                }
            }
            $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $back_callback_data];
        } elseif ( ! empty( $tag_id ) ) {
            $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_tags_page_1'];
        } elseif ( ! empty( $brand_id ) ) {
            $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_brands'];
        } else {
            $back_button_row[] = ['text' => __( '↩️ Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu'];
        }
        $product_list_keyboard[] = $back_button_row;
    
        $logo_url = self::get_logo_url();
        if ( $message_id_to_edit ) {
            WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $logo_url, $response_message, $product_list_keyboard );
        } else {
            $sent_message = WASCT_API_Handler::send_telegram_photo( $chat_id, $logo_url, $response_message, $product_list_keyboard );
            if ( ! is_wp_error( $sent_message ) && isset( $sent_message['message_id'] ) ) {
                set_transient( 'wasct_panel_msg_' . $chat_id, $sent_message['message_id'], DAY_IN_SECONDS );
            }
        } 
    }
    
    
    /**
     * Retrieves a unified list of languages available for the site.
     * Handles English as the native language, detects Polylang and WPML, and the WP language.
     * @return array An array of available languages.
     */
    public static function get_site_languages() {
    static $languages = null;
        if ($languages !== null) {
            return $languages;
        }
    
        $languages = [];
        if (!function_exists('wp_get_available_translations')) {
            require_once(ABSPATH . 'wp-admin/includes/translation-install.php');
        }
        $wp_translations = wp_get_available_translations();
    
        // Polylang 
        if (function_exists('pll_the_languages')) {
            foreach (pll_the_languages(['raw' => 1]) as $lang_code => $lang_data) {
                $locale = str_replace('-', '_', $lang_data['locale']);
                $languages[$lang_code] = ['code' => $lang_code, 'locale' => $locale, 'name' => $lang_data['name']];
            }
        }
    
        // WPML
        if (defined('ICL_SITEPRESS_VERSION')) {
            $wpml_languages = apply_filters('wpml_active_languages', NULL, 'orderby=name&order=asc');
            if (!empty($wpml_languages)) {
                foreach ($wpml_languages as $lang_code => $lang_data) {
                    if (!isset($languages[$lang_code])) {
                        $locale = str_replace('-', '_', $lang_data['default_locale']);
                        $name = $lang_data['native_name'];
                        $languages[$lang_code] = ['code' => $lang_code, 'locale' => $locale, 'name' => $name];
                    }
                }
            }
        }
    
        $wp_locale_raw = get_locale();
        if ($wp_locale_raw) {
            $wp_lang_code = substr($wp_locale_raw, 0, 2);
            if (!isset($languages[$wp_lang_code])) {
                $locale = str_replace('-', '_', $wp_locale_raw);
                $lang_name = $wp_translations[$wp_locale_raw]['native_name'] ?? $locale;
                $languages[$wp_lang_code] = ['code' => $wp_lang_code, 'locale' => $locale, 'name' => $lang_name];
            }
        }
    
        $text_domain = 'web-in-air-shop-connect-for-telegram';
        $translation_files = glob(WASCT_PLUGIN_DIR . 'languages/' . $text_domain . '-*.mo');
        foreach ($translation_files as $file) {
            if (preg_match('/' . $text_domain . '-(.+)\.mo$/', basename($file), $matches)) {
                $locale_raw = $matches[1];
                $lang_code = substr($locale_raw, 0, 2);
                if (!isset($languages[$lang_code])) {
                    $locale = str_replace('-', '_', $locale_raw);
                    $lang_name = $wp_translations[$locale_raw]['native_name'] ?? $locale;
                    $languages[$lang_code] = ['code' => $lang_code, 'locale' => $locale, 'name' => $lang_name];
                }
            }
        }

        if (empty($languages)) {
            $languages['en'] = ['code' => 'en', 'locale' => 'en_US', 'name' => 'English'];
        }
    
        return $languages;
    }

    /**
     * Cleans a price string formatted by WooCommerce for text display.
     *
     * @param string $price_html The price string containing HTML.
     * @return string The cleaned price.
     */
    public static function clean_price_string( $price_html ) {
        if ( empty( $price_html ) ) {
            return '';
        }
        $cleaned_price = wp_strip_all_tags( $price_html );
        $cleaned_price = html_entity_decode( $cleaned_price, ENT_QUOTES, 'UTF-8' );
        return trim( $cleaned_price );
    }

    /**
     * Formats the price of a WooCommerce product for display in Telegram.
     *
     * @param array $product Product data from the API.
     * @return string The formatted and cleaned price.
     */
    public static function format_product_price( $product ) {
        if ( isset( $product['price_html'] ) && ! empty( $product['price_html'] ) ) {
            $price_display = $product['price_html'];
        } elseif ( isset( $product['price'] ) && is_numeric( $product['price'] ) ) {
            $price_display = wc_price( (float) $product['price'] );
        } else {
            $price_display = __( 'Price not specified', 'web-in-air-shop-connect-for-telegram' );
        }
        return self::clean_price_string($price_display);
    }

    /**
     * Cleans and truncates an HTML description for preview purposes.
     *
     * @param string $raw_description Raw description with HTML.
     * @param int    $limit           Maximum number of characters.
     * @return string The cleaned and truncated description.
     */
    public static function clean_and_truncate_description( $raw_description, $limit = 300 ) {
        if ( empty( $raw_description ) ) {
            return __( 'No description available.', 'web-in-air-shop-connect-for-telegram' );
        }
        $description = $raw_description;
        $description = str_ireplace(['</p>', '</div>', '</ul>', '</ol>'], "\n", $description);
        $description = preg_replace('/<li[^>]*>(.*?)<\/li>/is', "• $1\n", $description);
        $description = str_replace(['<br>', '<br/>', '<br />'], "\n", $description);
        $description = wp_strip_all_tags($description);
        $description = html_entity_decode( $description, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
        $description = preg_replace('/[\r\n]+/', "\n", $description);
        $description = trim( $description ); 

        if (mb_strlen($description) > $limit) {
            $description = mb_substr( $description, 0, $limit );
            $last_space = mb_strrpos( $description, ' ' );
            if ($last_space !== false) {
                 $description = mb_substr( $description, 0, $last_space );
            }
            $description .= '...';
        }
        return $description;
    }

    /**
     * Cleans up a complete HTML description without truncating it and formats the titles.
     *
     * @param string $raw_description The raw description.
     * @return string The description cleaned up and formatted for Telegram.
     */
    public static function clean_full_description( $raw_description ) {
        if ( empty( $raw_description ) ) {
            return __( 'No description available.', 'web-in-air-shop-connect-for-telegram' );
        }

        $safe_html = strip_tags( $raw_description, '<h2><h3><h4><p><li><ul><ol><br>' );
        $description = $safe_html;
        $description = preg_replace('/<h[234][^>]*>(.*?)<\/h[234]>/is', "\n\n-*$1*-\n\n", $description);
        $description = str_ireplace( ['<p>'], "\n", $description );
        $description = str_ireplace( ['<br>', '<br/>', '<br />'], "\n", $description );
        $description = str_ireplace( ['<ul>', '<ol>'], "\n", $description );
        $description = str_ireplace( ['</ul>', '</ol>'], "\n", $description );
        $description = preg_replace('/<li[^>]*>(.*?)<\/li>/is', "• $1", $description);
        $description = wp_strip_all_tags($description); 
        $description = html_entity_decode( $description, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
        $description = preg_replace('/[\r\n]{3,}/', "\n\n", $description);

        return trim( $description );
    }

    /**
     * Formats and sends the paginated list of categories.
     */
    public static function send_category_list( $chat_id, $categories_data, $current_page, $title_message, $parent_category_id = 0, $message_id_to_edit = null ) {
        if ( is_wp_error( $categories_data ) ) {
            /* translators: %s: The error message text. */
            $error_message = sprintf( __( "Sorry, unable to retrieve categories. Error: %s", 'web-in-air-shop-connect-for-telegram' ), $categories_data->get_error_message() );
            WASCT_API_Handler::send_telegram_message( $chat_id, $error_message );
            return;
        }

        $categories_data['terms'] = array_filter($categories_data['terms'], function($category) {
            return $category['count'] > 0;
        });

        $categories       = $categories_data['terms'];
        $total_categories = $categories_data['total_terms'];
        $total_pages = ceil( $total_categories / WASCT_CATEGORIES_PER_PAGE );

        if ( empty( $categories ) && $parent_category_id === 0 ) {
            WASCT_API_Handler::send_telegram_message( $chat_id, __( "No product categories were found.", 'web-in-air-shop-connect-for-telegram' ) );
            return;
        }
        
        $response_message = $title_message;
        if ( $total_pages > 1 ) {
            /* translators: 1: Current page number, 2: Total number of pages, 3: Total number of categories. */
            $response_message .= "\n" . sprintf( __( 'Page %1$d/%2$d (Total: %3$d categories)).', 'web-in-air-shop-connect-for-telegram' ), $current_page, $total_pages, $total_categories );
        }
        $keyboard = [];
        foreach ( $categories as $category ) {
            $callback_data = 'category_' . $category['id'] . '_page_1';
            $keyboard[] = [['text' => '📁 ' . esc_html( $category['name'] ), 'callback_data' => $callback_data]];
        }
        if ( $parent_category_id !== 0 ) {
            $keyboard[] = [[
                'text' => __( '️▶ View products in this category ◀  ', 'web-in-air-shop-connect-for-telegram' ) , 
                'callback_data' => 'view_products_cat_' . $parent_category_id . '_page_1'
            ]];
        }
        
        $callback_prefix = ($parent_category_id === 0) ? 'show_categories_page_' : 'category_' . $parent_category_id . '_page_';
        
        $pagination_buttons = [];
        if ( $current_page > 1 ) {
            $pagination_buttons[] = ['text' => __( '◀️ Previous', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $callback_prefix . ($current_page - 1)];
        }
        if ( $current_page < $total_pages ) {
            $pagination_buttons[] = ['text' => __( 'Next ▶️', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $callback_prefix . ($current_page + 1)];
        }
        if ( !empty($pagination_buttons) ) {
            $keyboard[] = $pagination_buttons;
        }
        
        $back_button_row = [];
        if ( $parent_category_id !== 0 ) {
            $current_cat_info_raw = WASCT_API_Handler::get_woocommerce_categories(['include' => [$parent_category_id]]);
            if (!is_wp_error($current_cat_info_raw) && !empty($current_cat_info_raw['terms'])) {
                $grandparent_id = $current_cat_info_raw['terms'][0]['parent']; 
                if ($grandparent_id > 0) {
                     $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'category_' . $grandparent_id . '_page_1'];
                } else {
                     $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_categories_page_1'];
                }
                $back_button_row[] = ['text' => __( '↩️ Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu'];
            }
        } else {
            $back_button_row[] = ['text' => __( '↩️ Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu'];
        }
        if (!empty($back_button_row)) {
            $keyboard[] = $back_button_row;
        }

        $cache_key_context = $callback_prefix . $current_page;
        $cache_key = 'wasct_rendered_cache_' . $chat_id . '_' . md5($cache_key_context);
        $cache_content = [
            'text' => $response_message,
            'keyboard' => $keyboard,
        ];
        set_transient($cache_key, $cache_content, 10 * MINUTE_IN_SECONDS);

        $logo_url = self::get_logo_url();
        if ( $message_id_to_edit ) {
            WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $logo_url, $response_message, $keyboard );
        } else {
            WASCT_API_Handler::send_telegram_photo( $chat_id, $logo_url, $response_message, $keyboard );
        }    
    }

    /**
     * Formats and sends the paginated list of labels.
     */
    public static function send_tag_list( $chat_id, $tags_data, $current_page, $title_message, $message_id_to_edit = null ) {
        if ( is_wp_error( $tags_data ) || empty( $tags_data['terms'] ) ) {
            $message = __( "No product labels were found.", 'web-in-air-shop-connect-for-telegram' );
            $keyboard = [[['text' => '↩️ ' . __( 'Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu']]];
            $logo_url = self::get_logo_url();
            if ( $message_id_to_edit ) {
                WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $logo_url, $message, $keyboard );
            } else {
                WASCT_API_Handler::send_telegram_photo( $chat_id, $logo_url, $message, $keyboard );
            }
            return;
        }
        
        $tags = $tags_data['terms'];
        $total_tags = $tags_data['total_terms'];
        $total_pages = ceil( $total_tags / WASCT_TAGS_PER_PAGE );
        $response_message = $title_message;
        if ( $total_pages > 1 ) {
            /* translators: 1: Current page number, 2: Total number of pages. */
            $response_message .= "\n" . sprintf( __( 'Page %1$d/%2$d', 'web-in-air-shop-connect-for-telegram' ), $current_page, $total_pages );
        }
        
        $keyboard = [];
        foreach ( $tags as $tag ) {
            $keyboard[] = [['text' => '🏷️ ' . esc_html( $tag['name'] ), 'callback_data' => 'tag_' . $tag['id'] . '_page_1']];
        }
        
        $pagination_buttons = [];
        if ( $current_page > 1 ) {
            $pagination_buttons[] = ['text' => __( '◀️ Previous', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_tags_page_' . ($current_page - 1)];
        }
        if ( $current_page < $total_pages ) {
            $pagination_buttons[] = ['text' => __( 'Next ▶️', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_tags_page_' . ($current_page + 1)];
        }
        if ( !empty($pagination_buttons) ) {
            $keyboard[] = $pagination_buttons;
        }
        $keyboard[] = [['text' => __( '↩️ Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu']];
        
        $cache_key = 'wasct_rendered_cache_' . $chat_id . '_' . md5('show_tags_page_' . $current_page);
        $cache_content = [
            'text' => $response_message,
            'keyboard' => $keyboard,
        ];
        set_transient($cache_key, $cache_content, 10 * MINUTE_IN_SECONDS);

        $logo_url = self::get_logo_url();
        if ( $message_id_to_edit ) {
            WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $logo_url, $response_message, $keyboard );
        } else {
            WASCT_API_Handler::send_telegram_photo( $chat_id, $logo_url, $response_message, $keyboard );
        }
    }

    /**
     * Displays a single product with its photo, in a carousel style.
     */
    public static function send_visual_product_item( $chat_id, $products_data, $current_page, $command_prefix, $query_param = '', $category_id = '', $tag_id = '', $brand_id = '', $intro_message_prefix = "", $message_id_to_edit = null ) {
        if ( is_wp_error( $products_data ) || empty($products_data['products']) ) {
            $message = __( "Sorry, no products match your search.", 'web-in-air-shop-connect-for-telegram' );
            $keyboard = [
                [['text' => '🔍 ' . __( 'New search', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'prompt_search']],
                [['text' => '↩️ ' . __( 'Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu']]
            ];
            $logo_url = self::get_logo_url();
            if ( $message_id_to_edit ) {
                WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $logo_url, $message, $keyboard );
            } else {
                WASCT_API_Handler::send_telegram_photo( $chat_id, $logo_url, $message, $keyboard );
            }
            return;
        }
        
        $product = $products_data['products'][0];
        $total_products = $products_data['total_products'];
        $total_pages = $products_data['total_pages']; 
        $caption = $intro_message_prefix . html_entity_decode( $product['name'] );
        $photo_url = self::get_optimized_image_url( $product['images'][0] ?? [] );
        
        $keyboard = [];
        $nav_buttons = [];
        if ( $current_page > 1 ) {
            $nav_buttons[] = ['text' => '◀️ ' . __( 'Previous', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $command_prefix . '_' . ($current_page - 1)];
        }
        if ( $current_page < $total_pages ) {
            $nav_buttons[] = ['text' => __( 'Next', 'web-in-air-shop-connect-for-telegram' ) . ' ▶️', 'callback_data' => $command_prefix . '_' . ($current_page + 1)];
        }
        if ( ! empty( $nav_buttons ) ) { $keyboard[] = $nav_buttons; }

        $callback_context = '_from_list_page_' . $current_page;
        if ( ! empty( $category_id ) ) { $callback_context = '_from_cat_' . $category_id . '_page_' . $current_page; }
        elseif ( ! empty( $tag_id ) ) { $callback_context = '_from_tag_' . $tag_id . '_page_' . $current_page; }
        elseif ( ! empty( $brand_id ) ) { $callback_context = '_from_brand_' . $brand_id . '_page_' . $current_page; }
        elseif ( ! empty( $query_param ) ) { $callback_context = '_from_search_hash_' . $query_param . '_page_' . $current_page; }
        set_transient('wasct_user_context_' . $chat_id, $callback_context, HOUR_IN_SECONDS);

        $details_callback = 'details_' . $product['id'];
        $keyboard[] = [['text' => '🔍 ' . __( 'See details', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $details_callback]];
        
        $back_button_row = [];
        if ( ! empty( $category_id ) ) {
            $current_cat_info_raw = WASCT_API_Handler::get_woocommerce_categories(['include' => [$category_id]]);
            $back_callback_data = 'show_categories_page_1';
            if (!is_wp_error($current_cat_info_raw) && !empty($current_cat_info_raw['terms'])) {
                $parent_id = $current_cat_info_raw['terms'][0]['parent'];
                if ($parent_id > 0) {
                    $back_callback_data = 'category_' . $parent_id . '_page_1';
                }
            }
            $back_button_row[] = ['text' => '↩️ ' . __( 'Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => $back_callback_data];
        } elseif ( ! empty( $tag_id ) ) {
            $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_tags_page_1'];
        } elseif ( ! empty( $brand_id ) ) {
            $back_button_row[] = ['text' => __( '↩️ Back', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'show_brands'];
        } else {
            $back_button_row[] = ['text' => '↩️ ' . __( 'Menu', 'web-in-air-shop-connect-for-telegram' ), 'callback_data' => 'back_to_main_menu'];
        }
        $keyboard[] = $back_button_row;
        
        if ( $message_id_to_edit ) {
            WASCT_API_Handler::edit_telegram_message_media( $chat_id, $message_id_to_edit, $photo_url, $caption, $keyboard );
        } else {
            $sent_message = WASCT_API_Handler::send_telegram_photo( $chat_id, $photo_url, $caption, $keyboard );
            if ( ! is_wp_error( $sent_message ) && isset( $sent_message['message_id'] ) ) {
                set_transient( 'wasct_panel_msg_' . $chat_id, $sent_message['message_id'], DAY_IN_SECONDS );
            }
        }
    }

    /**
     * Divides a long text into several pages while respecting the lines.
     *
     * @param string $text      The complete text to be paginated.
     * @param int    $max_chars The maximum length of each page.
     * @return array An array containing the pages of text.
     */
    public static function paginate_text_by_sentence( $text, $max_chars = 1000 ) {
        if (mb_strlen($text, 'UTF-8') <= $max_chars) {
            return [$text];
        }
        $pages = [];
        $lines = explode("\n", $text);
        $current_page = '';
        foreach ($lines as $line) {
            if (mb_strlen($line, 'UTF-8') > $max_chars) {
                if (!empty(trim($current_page))) {
                    $pages[] = trim($current_page);
                }
                $pages[] = $line; 
                $current_page = '';
                continue;
            }
            if (mb_strlen($current_page . $line . "\n", 'UTF-8') > $max_chars) {
                $pages[] = trim($current_page);
                $current_page = $line . "\n";
            } else {
                $current_page .= $line . "\n";
            }
        }
        if (!empty(trim($current_page))) {
            $pages[] = trim($current_page);
        }
        return $pages;
    }

    /**
     * Retrieves the URL of the site logo with fallback solutions.
     * @return string The URL of the logo.
     */
    public static function get_logo_url() {
        $options = wasct_get_settings();
        $logo_url = '';
    
        if ( !empty($options['custom_logo_url']) ) {
            $logo_url = $options['custom_logo_url'];
        }
    
        if ( empty($logo_url) ) {
            if (function_exists('get_custom_logo') && has_custom_logo()) {
                $logo_id   = get_theme_mod('custom_logo');
                $logo_data = wp_get_attachment_image_src($logo_id, 'full');
                if ($logo_data && !empty($logo_data[0])) {
                    $logo_url = $logo_data[0];
                }
            }
        }
    
        if ( empty($logo_url) ) {
            $logo_url = plugins_url('assets/images/logo.webp', WASCT_PLUGIN_FILE);
        }
    
        $final_url = str_replace('http://', 'https://', $logo_url);
    
        return $final_url;
    }

    /**
     * Retrieves the best available image URL, favouring smaller sizes.
     *
     * @param array $image_data Image data from the WooCommerce API.
     * @return string The optimised image URL or an empty string.
     */
    public static function get_optimized_image_url( $image_data ) {
        $final_url = '';

        if ( ! empty( $image_data['id'] ) ) {
            $image_src = wp_get_attachment_image_src( $image_data['id'], 'medium' );
            if ( $image_src && ! empty( $image_src[0] ) ) {
                $final_url = $image_src[0];
            }
        }
        
        if ( empty( $final_url ) && ! empty( $image_data['src'] ) ) {
            $final_url = $image_data['src'];
        }

        if ( empty( $final_url ) ) {
            return '';
        }
        
        return str_replace('http://', 'https://', $final_url);
    }
    /**
     * Converts a two-letter country code to its corresponding flag emoji.
     *
     * @param string $country_code The two-letter country code (e.g. “FR”, “TH”).
     * @return string The flag emoji.
     */
    public static function get_flag_emoji_from_country_code( $country_code ) {
        if (strlen($country_code) !== 2) {
            return '🏳️';
        }
        
        $country_code = mb_strtoupper($country_code);

        $offset = 127397;
        $regional_indicator_1 = mb_chr(ord($country_code[0]) + $offset);
        $regional_indicator_2 = mb_chr(ord($country_code[1]) + $offset);

        return $regional_indicator_1 . $regional_indicator_2;
    }
}
?>